use anyhow::Result;
use notify::{Op, RawEvent, RecursiveMode, Watcher};
use std::{
    fs::{self, File},
    path::{Path, PathBuf},
    sync::mpsc,
};

const LOCKFILE: &str = "lock";

pub struct TaskLock(PathBuf);

pub fn accquire_task_lock<P>(task_path: P) -> Result<TaskLock>
where
    P: AsRef<Path>,
{
    let mut task_lock_path = PathBuf::from(task_path.as_ref());
    task_lock_path.push(LOCKFILE);

    if !task_lock_path.exists() {
        return create_lock(task_lock_path);
    }

    let (tx, rx) = mpsc::channel();
    let mut watcher = notify::raw_watcher(tx)?;
    watcher.watch(&task_lock_path, RecursiveMode::NonRecursive)?;

    loop {
        if let Ok(RawEvent {
            op: Ok(Op::REMOVE), ..
        }) = rx.recv()
        {
            break;
        }
    }

    create_lock(task_lock_path)
}

fn create_lock(path: PathBuf) -> Result<TaskLock> {
    File::create(&path)?;
    Ok(TaskLock(path))
}

impl Drop for TaskLock {
    fn drop(&mut self) {
        let _ = fs::remove_file(&self.0);
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::util::tests::run_in_temp_dir;

    #[test]
    fn releases_correctly() {
        let task_path = run_in_temp_dir();

        let result = accquire_task_lock(&task_path);
        assert!(
            matches!(result, Ok(TaskLock(..))),
            "expected to accquire lockfile"
        );
        drop(result);

        let result = accquire_task_lock(&task_path);
        assert!(
            matches!(result, Ok(TaskLock(..))),
            "expected to accquire lockfile again"
        );
        drop(result);

        let file_count = std::fs::read_dir(task_path)
            .expect("failed to read current dir")
            .count();
        assert_eq!(file_count, 0, "expected task path to be empty");
    }

    #[test]
    fn parallel_locks_work() {
        let task_path = run_in_temp_dir();

        let first = PathBuf::from(task_path.as_ref());
        let handle = std::thread::spawn(move || {
            let result = accquire_task_lock(first);
            assert!(
                matches!(result, Ok(TaskLock(..))),
                "expected to accquire lockfile again"
            );

            // Wait so the other lock will have to wait
            std::thread::sleep(std::time::Duration::from_millis(50));
        });

        // Wait so the thread has time to accquire the lock
        std::thread::sleep(std::time::Duration::from_millis(10));
        let result = accquire_task_lock(&task_path);
        assert!(
            matches!(result, Ok(TaskLock(..))),
            "expected to accquire lockfile again"
        );
        drop(result);

        handle.join().expect("failed to join handle");

        let file_count = std::fs::read_dir(task_path)
            .expect("failed to read current dir")
            .count();
        assert_eq!(file_count, 0, "expected task path to be empty");
    }
}
