use anyhow::Result;
use std::{
    fs::{self, File},
    hash::Hasher,
    io::Write,
    path::Path,
};
use twox_hash::XxHash64;

use crate::{env::ResolvedEnv, files::FilesReader, tasks::Task};

pub fn find_existing_hash(path: &Path) -> Result<Option<u64>> {
    if !path.is_file() {
        return Ok(None);
    }

    let contents = fs::read_to_string(path)?;
    let parsed = contents.parse::<u64>()?;
    Ok(Some(parsed))
}

pub fn save_hash(path: &Path, hash: u64) -> Result<()> {
    let mut output = File::create(&path)?;
    write!(output, "{}", hash)?;

    Ok(())
}

pub fn calc_hash<R, P>(
    workdir: P,
    task: &Task,
    global_env: &ResolvedEnv,
    task_env: &ResolvedEnv,
) -> Result<u64>
where
    R: FilesReader,
    P: AsRef<Path>,
{
    if task.files.is_empty()
        && task.outputs.is_empty()
        && task.script.is_none()
        && global_env.is_empty()
        && task_env.is_empty()
    {
        return Ok(0);
    }

    let mut hash = XxHash64::with_seed(0);

    hash.write(task.outputs.join(",").as_bytes());

    if let Some(script) = task.script.as_ref() {
        hash.write(script.as_bytes());
    }

    for path in R::get_paths_from_patterns(workdir, &task.files)? {
        let contents = R::read_file(path)?;
        hash.write(&contents);
    }

    for (key, value) in global_env {
        hash.write(key.as_bytes());
        hash.write(value.as_bytes());
    }

    for (key, value) in task_env {
        hash.write(key.as_bytes());
        hash.write(value.as_bytes());
    }

    Ok(hash.finish())
}

#[cfg(test)]
mod tests {
    use lazy_static::lazy_static;
    use std::{path::PathBuf, str::FromStr};

    use super::*;
    use crate::util::tests::random_string;

    struct TestReader;

    impl FilesReader for TestReader {
        fn get_paths_from_patterns<P>(_: P, patterns: &[String]) -> Result<Vec<PathBuf>>
        where
            P: AsRef<Path>,
        {
            Ok(patterns
                .iter()
                .map(|pattern| PathBuf::from_str(pattern).expect("failed to create path"))
                .collect())
        }

        fn read_file<P>(path: P) -> Result<Vec<u8>>
        where
            P: AsRef<Path>,
        {
            Ok(path
                .as_ref()
                .to_str()
                .expect("invalid path")
                .as_bytes()
                .to_vec())
        }
    }

    lazy_static! {
        static ref EMPTY_ENV: ResolvedEnv = ResolvedEnv::new();
        static ref EMPTY_TASK: Task = Task::default();
    }

    #[test]
    fn stable_empty_hash() {
        let empty = calc_hash::<TestReader, _>("", &EMPTY_TASK, &EMPTY_ENV, &EMPTY_ENV)
            .expect("failed to hash");

        assert_eq!(empty, 0, "expected empty hash to be 0");

        let empty = calc_hash::<TestReader, _>("", &EMPTY_TASK, &EMPTY_ENV, &EMPTY_ENV)
            .expect("failed to hash");

        assert_eq!(empty, 0, "expected empty hash to be stable");
    }

    #[test]
    fn script_changes() {
        let task_before = Task {
            script: Some(String::from("before")),
            ..Task::default()
        };

        let task_after = Task {
            script: Some(String::from("after")),
            ..Task::default()
        };

        let before = calc_hash::<TestReader, _>("", &task_before, &EMPTY_ENV, &EMPTY_ENV)
            .expect("failed to hash");
        let again = calc_hash::<TestReader, _>("", &task_before, &EMPTY_ENV, &EMPTY_ENV)
            .expect("failed to hash");
        let after = calc_hash::<TestReader, _>("", &task_after, &EMPTY_ENV, &EMPTY_ENV)
            .expect("failed to hash");

        assert_eq!(before, again, "expected script hash to be stable");
        assert_ne!(before, after, "expected script hash to reflect changes");
    }

    #[test]
    fn file_changes() {
        let task_before = Task {
            files: vec!["file1".to_owned(), "file2".to_owned()],
            ..Task::default()
        };

        let task_after = Task {
            files: vec!["file1".to_owned()],
            ..Task::default()
        };

        let before = calc_hash::<TestReader, _>("", &task_before, &EMPTY_ENV, &EMPTY_ENV)
            .expect("failed to hash");
        let again = calc_hash::<TestReader, _>("", &task_before, &EMPTY_ENV, &EMPTY_ENV)
            .expect("failed to hash");
        let after = calc_hash::<TestReader, _>("", &task_after, &EMPTY_ENV, &EMPTY_ENV)
            .expect("failed to hash");

        assert_eq!(before, again, "expected files hash to be stable");
        assert_ne!(before, after, "expected files hash to reflect changes");
    }

    #[test]
    fn global_env_changes() {
        let mut env = ResolvedEnv::new();
        env.insert(random_string(), random_string());

        let before =
            calc_hash::<TestReader, _>("", &EMPTY_TASK, &env, &EMPTY_ENV).expect("failed to hash");
        let again =
            calc_hash::<TestReader, _>("", &EMPTY_TASK, &env, &EMPTY_ENV).expect("failed to hash");

        env.insert(random_string(), random_string());
        let after =
            calc_hash::<TestReader, _>("", &EMPTY_TASK, &env, &EMPTY_ENV).expect("failed to hash");

        assert_eq!(before, again, "expected global env hash to be stable");
        assert_ne!(before, after, "expected global env hash to reflect changes");
    }

    #[test]
    fn task_env_changes() {
        let mut env = ResolvedEnv::new();
        env.insert(random_string(), random_string());

        let before =
            calc_hash::<TestReader, _>("", &EMPTY_TASK, &EMPTY_ENV, &env).expect("failed to hash");
        let again =
            calc_hash::<TestReader, _>("", &EMPTY_TASK, &EMPTY_ENV, &env).expect("failed to hash");

        env.insert(random_string(), random_string());
        let after =
            calc_hash::<TestReader, _>("", &EMPTY_TASK, &EMPTY_ENV, &env).expect("failed to hash");

        assert_eq!(before, again, "expected task env hash to be stable");
        assert_ne!(before, after, "expected task env hash to reflect changes");
    }
}
