use std::collections::HashMap;

use anyhow::{Context, Result};

use crate::FasterError;

pub type ResolvedEnv = HashMap<String, String>;

pub fn resolve_env(values: &HashMap<String, String>) -> Result<ResolvedEnv> {
    let mut result = ResolvedEnv::new();

    for (key, expr) in values {
        let echo_cmd = format!("echo -n \"{}\"", expr);
        let output = duct::cmd!("sh", "-c", echo_cmd)
            .stderr_capture()
            .stdout_capture()
            .unchecked()
            .run()
            .with_context(|| {
                format!("failed to read output of environment expression `{}`", expr)
            })?;

        match output.status.code() {
            Some(0) => {
                let output = String::from_utf8(output.stdout)
                    .context("failed to read output for env expression")?;

                result.insert(key.to_owned(), output);
            }
            _ => {
                let stderr = String::from_utf8(output.stdout)
                    .context("failed to read stderr after failed env expression")?;

                return Err(FasterError::EnvExpressionError {
                    expr: expr.clone(),
                    stderr,
                }
                .into());
            }
        };
    }

    Ok(result)
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::util::tests::random_string;

    // We set and read random values here to
    // decouple the tests from previously set
    // environment variables.

    #[test]
    fn resolves_literal() {
        let value = random_string();

        let mut config = HashMap::new();
        config.insert("foo".to_owned(), value.clone());

        let resolved = resolve_env(&config).expect("failed to resolve env");
        assert_eq!(resolved["foo"], value);
    }

    #[test]
    fn resolves_var() {
        let key = format!("FASTER_TEST_{}", random_string());
        let value = random_string();
        std::env::set_var(&key, &value);

        let mut config = HashMap::new();
        config.insert("foo".to_owned(), format!("${{{}}}", &key));

        let resolved = resolve_env(&config).expect("failed to resolve env");
        assert_eq!(resolved["foo"], value);

        std::env::remove_var(key);
    }

    #[test]
    fn resolves_command() {
        let value = random_string();

        let mut config = HashMap::new();
        config.insert("foo".to_owned(), format!("$(echo \"{}\")", &value));

        let resolved = resolve_env(&config).expect("failed to resolve env");
        assert_eq!(resolved["foo"], value);
    }

    #[test]
    fn invalid_command_fails() {
        let mut config = HashMap::new();
        config.insert("foo".to_owned(), "$((".to_owned());

        let resolved = resolve_env(&config);
        assert!(matches!(resolved, Err(_)));
    }
}
