use anyhow::Result;
use console::style;
use crossbeam_channel::{select, Receiver, Sender};
use indicatif::{MultiProgress, ProgressBar, ProgressDrawTarget, ProgressStyle};
use lazy_static::lazy_static;
use std::{
    collections::HashSet,
    fs::{self, File},
    io::Write,
    path::{Path, PathBuf},
    thread::{self, JoinHandle},
    time::{Instant, SystemTime},
};

use crate::{
    files::GlobFilesReader,
    hashes, lock, outputs,
    tasks::{Task, TaskGraph},
    util, Config, FasterError,
};

lazy_static! {
    static ref PROGRESS_STYLE: ProgressStyle =
        ProgressStyle::with_template("  {spinner} {wide_msg}")
            .expect("failed to create progress style")
            .tick_chars("⠋⠙⠹⠸⠼⠴⠦⠧⠇⠏");
    static ref STATUS_STYLE: ProgressStyle = ProgressStyle::with_template("  {prefix} {wide_msg}")
        .expect("failed to create status style");
    static ref DONE_STYLE: ProgressStyle =
        ProgressStyle::with_template("  {prefix} {wide_msg}").expect("failed to create done style");
}

struct TaskMessage(String, Task);

#[derive(Clone)]
enum CommandResult {
    Success(String),
    Failed(String),
    Cached,
    Cancelled,
    Skipped,
}

pub fn run<P>(
    workdir: P,
    config: &Config,
    task_graph: &mut TaskGraph,
    draw_target: ProgressDrawTarget,
    job_count: usize,
    ctrlc_rx: Receiver<()>,
) -> Result<()>
where
    P: AsRef<Path>,
{
    let started = Instant::now();

    let progress = MultiProgress::new();
    progress.set_draw_target(draw_target);

    let job_count = std::cmp::min(task_graph.task_count(), job_count);

    let (job_tx, job_rx) = crossbeam_channel::unbounded::<TaskMessage>();
    let (done_tx, done_rx) = crossbeam_channel::unbounded::<(String, Result<CommandResult>)>();
    let (abort_tx, abort_rx) = crossbeam_channel::bounded::<()>(job_count);

    let worker_handles = (0..job_count)
        .map(|_| {
            let job_rx = job_rx.clone();
            let done_tx = done_tx.clone();
            let abort_rx = abort_rx.clone();
            let progress = progress.clone();
            let workdir = PathBuf::from(workdir.as_ref());

            thread::spawn(move || {
                run_worker(
                    workdir,
                    &progress,
                    job_rx.clone(),
                    done_tx.clone(),
                    abort_rx.clone(),
                )
            })
        })
        .collect::<Vec<JoinHandle<_>>>();

    let mut running_tasks = HashSet::<String>::new();
    let mut task_results = Vec::<(String, Result<CommandResult>)>::new();

    while task_graph.has_tasks() {
        let next_targets = task_graph
            .runnable_tasks()
            .filter(|name| !running_tasks.contains(name.as_str()))
            .collect::<Vec<String>>();

        for task_name in next_targets {
            let task = config
                .tasks
                .get(task_name.as_str())
                .ok_or_else(|| FasterError::TargetNotFound(task_name.to_owned()))?;

            running_tasks.insert(task_name.clone());
            job_tx.send(TaskMessage(task_name, task.clone()))?;
        }

        let should_abort;

        // Wait for any thread to finish its task, or an abort signal to arrive
        select! {
            recv(done_rx) -> msg => {
                let (task_name, task_result) = msg.expect("Failed to receive task message");
                should_abort = matches!(task_result, Err(_) | Ok(CommandResult::Failed(_)));

                running_tasks.remove(task_name.as_str());
                task_graph.complete_task(task_name.as_str());
                task_results.push((task_name, task_result));
            }
            recv(ctrlc_rx) -> _ => {
                should_abort = true;
            }
        }

        if should_abort {
            // Print the remaining tasks and mark them as skipped
            let remaining_tasks = task_graph
                .all_tasks()
                .filter(|name| !running_tasks.contains(name.as_str()))
                .collect::<Vec<String>>();

            for task_name in &remaining_tasks {
                let bar = progress.add(ProgressBar::new(1));

                bar.set_style(DONE_STYLE.clone());
                bar.set_prefix(format!("{}", style("✗").black().dim()));
                bar.finish_with_message(format!(
                    "{} {}",
                    style("skipped").black().dim(),
                    task_name
                ));

                task_results.push((task_name.to_string(), Ok(CommandResult::Skipped)));
            }

            (0..job_count).for_each(|_| abort_tx.send(()).expect("failed to send abort message"));

            break;
        }
    }

    // Close the channels so the workers quit after
    // we're done
    drop(job_tx);

    for handle in worker_handles {
        handle.join().expect("failed to join handle");
    }

    // Insert a newline after the progress bars
    print!("\n\n");

    for result in &task_results {
        match result {
            (task_name, Ok(CommandResult::Failed(output))) => {
                println!("Script for task `{}` failed:", style(task_name).bold());
                println!("---\n{}\n---\n", output);
            }
            (task_name, Err(err)) => {
                println!("Failed to run task `{}`", style(task_name).bold());
                println!("---\n{}\n---\n", err);
            }
            _ => {}
        }
    }

    // TODO: Print detailed report
    let elapsed = util::format_millis(started.elapsed().as_millis() as u64);
    println!("Took {}\n", elapsed);

    Ok(())
}

fn run_worker<P>(
    workdir: P,
    progress: &MultiProgress,
    rx: Receiver<TaskMessage>,
    done_tx: Sender<(String, Result<CommandResult>)>,
    abort_rx: Receiver<()>,
) where
    P: AsRef<Path>,
{
    let mut should_abort = false;

    while let Ok(TaskMessage(task_name, task)) = rx.recv() {
        let bar = progress.add(ProgressBar::new(1));

        let started = Instant::now();
        let result = run_task(&workdir, &bar, &task_name, &task, &abort_rx);

        match &result {
            Ok(CommandResult::Cancelled) => {
                bar.set_style(DONE_STYLE.clone());
                bar.set_prefix(format!("{}", style("✗").yellow()));
                bar.finish_with_message(format!(
                    "{} {}",
                    style("cancelled").yellow().dim(),
                    task_name
                ));
            }
            Ok(CommandResult::Success(_output)) => {
                let elapsed = started.elapsed().as_millis() as u64;

                bar.set_style(DONE_STYLE.clone());
                bar.set_prefix(format!("{}", style("✔").green()));
                bar.finish_with_message(format!(
                    "{} {}",
                    style(util::format_millis(elapsed)).green().dim(),
                    task_name
                ));
            }
            Ok(CommandResult::Failed(_output)) => {
                bar.set_style(DONE_STYLE.clone());
                bar.set_prefix(format!("{}", style("✗").red()));
                bar.finish_with_message(format!(
                    "{} {}",
                    style("failed").red().bold().dim(),
                    task_name
                ));

                should_abort = true;
            }
            Ok(CommandResult::Cached) => {
                bar.set_style(DONE_STYLE.clone());
                bar.set_prefix(format!("{}", style("✔").green()));

                bar.finish_with_message(format!("{} {}", style("cached").black().dim(), task_name));
            }
            Err(_) => {
                bar.set_style(DONE_STYLE.clone());
                bar.set_prefix(format!("{}", style("✗").red()));
                bar.finish_with_message(format!(
                    "{} {}",
                    style("failed").red().bold().dim(),
                    task_name
                ));

                should_abort = true;
            }
            // This only ever happens when the task is not even run
            Ok(CommandResult::Skipped) => unreachable!(),
        }

        done_tx
            .send((task_name.to_owned(), result))
            .expect("failed to send done message");

        if should_abort {
            break;
        }
    }
}

fn run_task<P>(
    workdir: P,
    bar: &ProgressBar,
    task_name: &str,
    task: &Task,
    abort_rx: &Receiver<()>,
) -> Result<CommandResult>
where
    P: AsRef<Path>,
{
    bar.set_style(STATUS_STYLE.clone());
    bar.set_prefix("⌛");
    bar.set_message(format!("{} {}", style("waiting").white().dim(), task_name));

    if let Ok(()) = abort_rx.try_recv() {
        return Ok(CommandResult::Cancelled);
    }

    let task_path = util::ensure_task_file_path(&workdir, task_name)?;
    let _lock = lock::accquire_task_lock(&task_path);

    // TODO: Show an ellipsized command here? Split into lines?
    bar.set_style(PROGRESS_STYLE.clone());
    bar.set_message(format!("{} {}", style("running").blue().dim(), task_name));

    let hash_path = task_path.join(util::SUM_FILE);
    let current_hash = hashes::calc_hash::<GlobFilesReader, _>(&workdir, task)?;

    let has_changes = match hashes::find_existing_hash(&hash_path)? {
        Some(previous_hash) => current_hash != previous_hash,
        _ => true,
    };

    if !has_changes {
        outputs::restore_outputs(&workdir, &task_path, current_hash, task)?;
        return Ok(CommandResult::Cached);
    }

    // Run the actual task
    if let Some(script) = &task.script {
        // TODO: Make the run command configurable
        // TODO: Add default windows version for this
        let args = vec!["-c", script];
        let handle = duct::cmd("sh", &args)
            .stderr_to_stdout()
            .stdout_capture()
            .unchecked()
            .start()?;

        let mut last_instant = Instant::now();

        loop {
            // TODO: Can we somehow select over these?
            if let Ok(()) = abort_rx.try_recv() {
                // We don't really care about the child here,
                // even if it doesn't stop we don't want to block
                let _ = handle.kill();
                return Ok(CommandResult::Cancelled);
            }

            match handle.try_wait() {
                Ok(Some(result)) => {
                    // TODO: Can we do this without cloning stdout? Might use up a lot
                    // of memory when there's a lot of output from the command
                    let logs = String::from_utf8(result.stdout.clone())?;
                    save_and_clean_logs(&task_path, current_hash, &logs)?;

                    if !result.status.success() {
                        return Ok(CommandResult::Failed(logs));
                    }

                    // TODO: Run clean command/clean outputs

                    // Only save hash and outputs if the script succeeded
                    hashes::save_hash(&hash_path, current_hash)?;
                    outputs::save_outputs::<GlobFilesReader, _, _>(
                        workdir,
                        task_name,
                        &task_path,
                        current_hash,
                        task,
                    )?;

                    return Ok(CommandResult::Success(logs));
                }
                Err(err) => {
                    return Err(err.into());
                }

                Ok(None) => {
                    // TODO: Add a configurable command timeout?
                    if last_instant.elapsed().as_millis() < 100 {
                        continue;
                    }

                    bar.inc(1);
                    last_instant = Instant::now();
                }
            }
        }
    }

    unreachable!();
}

fn save_and_clean_logs(task_path: &Path, current_hash: u64, output: &str) -> Result<()> {
    let mut log_path = PathBuf::from(task_path);
    log_path.push(format!("{}.{}", current_hash, util::LOG_SUFFIX));

    let mut file = File::create(log_path.clone())?;
    file.write_all(output.as_bytes())?;

    // Now, stat all files in the directory and delete all except the n-1 newest
    // (since we keep the newest one by default)
    let mut log_files = fs::read_dir(task_path)?
        // Just filter out invalid entries here
        .filter_map(|entry| entry.ok())
        .filter_map(|entry| entry.metadata().ok().map(|metadata| (entry, metadata)))
        .filter_map(|(entry, metadata)| {
            entry
                .path()
                .extension()
                .map(|ext| (entry, metadata, ext.to_string_lossy().into_owned()))
        })
        .filter(|(entry, metadata, ext)| {
            // Filter out the newest hash
            entry.path() != log_path &&
            // Filter out directories
            metadata.is_file() &&
            // Filter out non-logfiles
            ext.as_str() == util::LOG_SUFFIX
        })
        .map(|(entry, metadata, _)| (entry, metadata))
        .collect::<Vec<_>>();

    // Sort from oldest to newest
    log_files.sort_by_key(|(_, metadata)| metadata.created().unwrap_or(SystemTime::UNIX_EPOCH));
    log_files.reverse();

    log_files
        .iter()
        .skip(util::KEEP_LAST_RESULTS - 1)
        .for_each(|(entry, _)| {
            // TODO: Log the error in verbose mode here, this is
            // not a blocker though
            let _ = std::fs::remove_file(entry.path());
        });

    Ok(())
}

#[cfg(test)]
mod tests {
    use crate::{run::*, util::tests::run_in_temp_dir};

    #[test]
    fn run_single() {
        let workdir = run_in_temp_dir();

        let config: crate::Config = serde_yaml::from_str(
            r#"
            tasks:
              build:
                script: "echo 'single build'"
            "#,
        )
        .unwrap();

        let mut task_graph =
            TaskGraph::single(&config, "build").expect("failed to build task graph");

        let (_, rx) = crossbeam_channel::unbounded::<()>();
        let result = run(
            &workdir,
            &config,
            &mut task_graph,
            indicatif::ProgressDrawTarget::hidden(),
            1,
            rx,
        );

        assert!(matches!(result, Ok(())));
    }

    #[test]
    fn run_multi() {
        let workdir = run_in_temp_dir();

        let config: crate::Config = serde_yaml::from_str(
            r#"
            tasks:
              generate:
                script: "echo 'multi generate'"

              lint:
                script: "echo 'multi lint'"

              build:
                script: "echo 'multi build'"
                dependencies:
                - generate
                - lint
            "#,
        )
        .unwrap();

        let mut task_graph =
            TaskGraph::build(&config, "build").expect("failed to build task graph");

        let (_, rx) = crossbeam_channel::unbounded::<()>();
        let result = run(
            &workdir,
            &config,
            &mut task_graph,
            indicatif::ProgressDrawTarget::hidden(),
            1,
            rx,
        );

        assert!(matches!(result, Ok(())));
    }
}
