use anyhow::{Context, Result};
use clap::{Parser, Subcommand};
use console::style;
use files::{FilesReader, GlobFilesReader};
use indicatif::ProgressDrawTarget;
use serde::Deserialize;
use std::{collections::HashMap, path::Path};
use thiserror::Error;

pub mod files;
pub mod hashes;
pub mod lock;
pub mod outputs;
pub mod run;
pub mod tasks;
pub mod util;

#[derive(Debug, Clone, Deserialize)]
pub struct Config {
    pub tasks: HashMap<String, tasks::Task>,
}

#[derive(Error, Debug)]
pub enum FasterError {
    #[error("cycle found during dependency resolution")]
    DependencyCycle,

    #[error("target `{0}` not found")]
    TargetNotFound(String),

    #[error("target `{task:?}` not found, referenced by `{parent:?}`")]
    DependencyTargetNotFound { parent: String, task: String },

    #[error("no outputs were matched for task `{task:?}`")]
    NoOutputsFound { task: String },
}

/// Simple and smart language-agnostic task runner
#[derive(Parser, Debug)]
#[clap(author, version, about)]
struct Args {
    #[clap(subcommand)]
    subcommand: Commands,

    /// Provide a custom config file path
    #[clap(short, long, default_value = "faster.yaml")]
    config: String,
}

#[derive(Subcommand, Debug)]
enum Commands {
    /// List all defined tasks
    Tasks,

    /// List all input files for a specific task
    Files {
        /// Task name for which files should be printed
        task_name: String,

        /// Print files for all direct and transitive dependencies of the task
        #[clap(short, long)]
        include_deps: bool,
    },

    /// Run a task and all its dependencies
    Run {
        /// Task name to be run
        task_name: String,

        /// Don't run dependencies of the task
        #[clap(short, long)]
        skip_dependencies: bool,

        /// Number of tasks to run in parallel
        #[clap(short, long, default_value_t = 4)]
        parallel_tasks: usize,
    },

    /// Parse and validate all defined tasks
    Validate,
}

fn main() -> Result<()> {
    let (ctrlc_tx, ctrlc_rx) = crossbeam_channel::unbounded();
    ctrlc::set_handler(move || ctrlc_tx.send(()).expect("Failed to send abort signal"))
        .expect("Failed to set abort signal handler");

    let args = Args::parse();
    let workdir = std::env::current_dir().context("failed to get current dir")?;

    match args.subcommand {
        Commands::Run {
            task_name,
            skip_dependencies,
            parallel_tasks,
        } => {
            let config = read_config(&args.config)?;

            let mut task_graph = if skip_dependencies {
                let task_graph = tasks::TaskGraph::single(&config, task_name.as_str())?;

                println!(
                    "{}",
                    style(format!(
                        "Running task `{}` without dependencies\n",
                        task_name,
                    ))
                    .dim(),
                );

                task_graph
            } else {
                let task_graph = tasks::TaskGraph::build(&config, task_name.as_str())?;
                let resolved_tasks = task_graph.resolved_task_names();

                println!(
                    "{}",
                    style(format!(
                        "Resolved {} dependencies for task `{}`\n",
                        resolved_tasks.len(),
                        task_name,
                    ))
                    .dim(),
                );

                task_graph
            };

            run::run(
                &workdir,
                &config,
                &mut task_graph,
                ProgressDrawTarget::stderr(),
                parallel_tasks,
                ctrlc_rx,
            )?;
        }
        Commands::Tasks => {
            let config = read_config(&args.config)?;
            for task_name in config.tasks.keys() {
                println!("{}", task_name);
            }
        }
        Commands::Files {
            task_name,
            include_deps,
        } => {
            let config = read_config(&args.config)?;

            if !include_deps {
                let task = config
                    .tasks
                    .get(&task_name)
                    .ok_or(FasterError::TargetNotFound(task_name))?;

                if task.files.is_none() {
                    return Ok(());
                }

                let patterns = task.files.as_ref().unwrap();
                let files = GlobFilesReader::get_paths_from_patterns(&workdir, patterns)?;
                for file in files {
                    println!("{}", file.to_string_lossy());
                }

                return Ok(());
            }

            let task_graph = tasks::TaskGraph::build(&config, task_name.as_str())?;

            let mut all_paths = Vec::<String>::new();
            for task_name in task_graph.all_tasks() {
                let task = config
                    .tasks
                    .get(&task_name)
                    .ok_or(FasterError::TargetNotFound(task_name))?;

                let patterns = task.files.as_ref().unwrap();
                let files = GlobFilesReader::get_paths_from_patterns(&workdir, patterns)?;
                for file in files {
                    all_paths.push(file.to_string_lossy().to_string());
                }
            }

            all_paths.dedup();
            for path in &all_paths {
                println!("{}", path);
            }
        }
        Commands::Validate => {
            let config = read_config(&args.config)?;
            let mut errs = HashMap::<&str, anyhow::Error>::new();

            for task_name in config.tasks.keys() {
                let result = tasks::TaskGraph::build(&config, task_name.as_str());
                if let Err(err) = result {
                    errs.insert(task_name, err);
                }
            }

            if errs.is_empty() {
                println!("No errors found!");
                return Ok(());
            }

            for (task_name, err) in &errs {
                println!("Error in task `{}`: {} ", task_name, err)
            }
        }
    }

    Ok(())
}

fn read_config<P>(path: P) -> Result<Config>
where
    P: AsRef<Path>,
{
    let contents = std::fs::read_to_string(path).context("Failed to open config file")?;
    serde_yaml::from_str(&contents).context("Failed to parse config file")
}
