use anyhow::Result;
use std::{
    fs::{self, File},
    hash::Hasher,
    io::Write,
    path::Path,
};
use twox_hash::XxHash64;

use crate::{files::FilesReader, tasks::Task};

pub fn find_existing_hash(path: &Path) -> Result<Option<u64>> {
    if !path.is_file() {
        return Ok(None);
    }

    let contents = fs::read_to_string(path)?;
    let parsed = contents.parse::<u64>()?;
    Ok(Some(parsed))
}

pub fn save_hash(path: &Path, hash: u64) -> Result<()> {
    let mut output = File::create(&path)?;
    write!(output, "{}", hash)?;

    Ok(())
}

pub fn calc_hash<R, P>(workdir: P, task: &Task) -> Result<u64>
where
    R: FilesReader,
    P: AsRef<Path>,
{
    let files_empty = task.files.as_ref().map_or(true, |it| it.is_empty());
    let outputs_empty = task.outputs.as_ref().map_or(true, |it| it.is_empty());

    if files_empty && outputs_empty && task.script.is_none() {
        return Ok(0);
    }

    let mut hash = XxHash64::with_seed(0);

    if let Some(script) = task.script.as_ref() {
        hash.write(script.as_bytes());
    }

    if let Some(outputs) = task.outputs.as_ref() {
        hash.write(outputs.join(",").as_bytes());
    }

    if !files_empty {
        let patterns = task.files.as_ref().unwrap();
        let paths = R::get_paths_from_patterns(workdir, patterns)?;

        for path in &paths {
            let contents = R::read_file(path)?;
            hash.write(&contents);
        }
    }

    Ok(hash.finish())
}

#[cfg(test)]
mod tests {
    use std::{path::PathBuf, str::FromStr};

    use crate::hashes::*;

    struct TestReader;

    impl FilesReader for TestReader {
        fn get_paths_from_patterns<P>(_: P, patterns: &[String]) -> Result<Vec<PathBuf>>
        where
            P: AsRef<Path>,
        {
            Ok(patterns
                .iter()
                .map(|pattern| PathBuf::from_str(pattern).expect("failed to create path"))
                .collect())
        }

        fn read_file<P>(path: P) -> Result<Vec<u8>>
        where
            P: AsRef<Path>,
        {
            Ok(path
                .as_ref()
                .to_str()
                .expect("invalid path")
                .as_bytes()
                .to_vec())
        }
    }

    #[test]
    fn hashing_detects_changes() {
        let empty_task = Task::default();
        let empty = calc_hash::<TestReader, _>("", &empty_task).expect("failed to hash");
        assert_eq!(empty, 0, "expected empty hash to be 0");

        let task_before = Task {
            script: Some(String::from("before")),
            ..Task::default()
        };

        let task_after = Task {
            script: Some(String::from("after")),
            ..Task::default()
        };

        let before = calc_hash::<TestReader, _>("", &task_before).expect("failed to hash");
        let again = calc_hash::<TestReader, _>("", &task_before).expect("failed to hash");
        let after = calc_hash::<TestReader, _>("", &task_after).expect("failed to hash");

        assert_eq!(before, again, "expected script hash to be stable");
        assert_ne!(before, after, "expected script hash to reflect changes");

        let task_before = Task {
            files: Some(vec!["file1".to_owned(), "file2".to_owned()]),
            ..Task::default()
        };

        let task_after = Task {
            files: Some(vec!["file1".to_owned()]),
            ..Task::default()
        };

        let before = calc_hash::<TestReader, _>("", &task_before).expect("failed to hash");
        let again = calc_hash::<TestReader, _>("", &task_before).expect("failed to hash");
        let after = calc_hash::<TestReader, _>("", &task_after).expect("failed to hash");

        assert_eq!(before, again, "expected files hash to be stable");
        assert_ne!(before, after, "expected files hash to reflect changes");
    }
}
