use anyhow::Result;
use std::path::{Path, PathBuf};

pub trait FilesReader {
    fn get_paths_from_patterns<P>(workdir: P, patterns: &[String]) -> Result<Vec<PathBuf>>
    where
        P: AsRef<Path>;
    fn read_file<P>(path: P) -> Result<Vec<u8>>
    where
        P: AsRef<Path>;
}

pub struct GlobFilesReader;

impl FilesReader for GlobFilesReader {
    fn get_paths_from_patterns<P>(workdir: P, patterns: &[String]) -> Result<Vec<PathBuf>>
    where
        P: AsRef<Path>,
    {
        let mut result = vec![];
        let mut directories = vec![];

        for pattern in patterns {
            let absolute_pattern = workdir.as_ref().join(pattern);
            let files = glob::glob(&absolute_pattern.to_string_lossy())?;

            for file in files {
                let file = file?;

                if file.is_file() {
                    result.push(file);
                    continue;
                }

                if file.is_dir() {
                    directories.push(file);
                    continue;
                }

                // TODO: Add an option to enable following symlinks
            }
        }

        for dir in directories.iter_mut() {
            let dir_pattern = dir.join("**/*");
            let files = glob::glob(&dir_pattern.to_string_lossy())?;

            for file in files {
                let file = file?;

                if file.is_file() {
                    result.push(file);
                    continue;
                }
            }
        }

        result.dedup();
        Ok(result)
    }

    fn read_file<P>(path: P) -> Result<Vec<u8>>
    where
        P: AsRef<Path>,
    {
        Ok(std::fs::read(path)?)
    }
}
