use crate::{BruteForceFastMotionPlanner, HeuristicFastMotionPlanner, mp, objects, trajectory, solvers};

use std::collections::HashMap;

///// PYTHON INTERFACE
use pyo3::prelude::*;
use pyo3::exceptions;
use pyo3::PyIterProtocol;


#[pyclass(name="BruteForceFastMotionPlanner")]
pub struct BruteForceFastMotionPlannerPy {
    fmp: BruteForceFastMotionPlanner,
    solution: Option<mp::MPSolution>
}

#[pymethods]
impl BruteForceFastMotionPlannerPy {
    #[new]
    fn new(dt: usize, epsilon: f64) -> Self {
        Self {
            fmp: BruteForceFastMotionPlanner::new(dt, epsilon),
            solution: None
        }
    }

    #[args(
        n = "None",
    )]
    fn solve(
        &mut self,
        time: f64,
        x: mp::MPComponent,
        y: mp::MPComponent,
        z: mp::MPComponent,
        objects: objects::ObjectList,
        n: Option<usize>, // Number of results
    ) -> PyResult<mp::MPSolution>{
        let mp = mp::MPProblem {time, x, y, z, objects};
        let sol = match n {
            Some(n) => mp::MPSolution::from_vec(self.fmp.solver(mp).into_iter().take(n).collect()),
            None => self.fmp.solve(mp)
        };
        self.solution = Some(sol.clone());
        Ok(sol)
    }

    #[args(
        randomise = "true",
    )]
    fn solve_iter(
        &self,
        time: f64,
        x: mp::MPComponent,
        y: mp::MPComponent,
        z: mp::MPComponent,
        objects: objects::ObjectList,
        randomise: bool
    ) -> PyResult<BruteForceSolveIteratorPy>{
        let mp = mp::MPProblem {time, x, y, z, objects};
        let solver = match randomise {
            true => self.fmp.solver(mp),
            false => self.fmp.random_solver(mp)
        };
        Ok(BruteForceSolveIteratorPy{inner:solver.into_iter()})
    }

    #[args(
        randomise = "true",
    )]
    fn solve_trajectories_iter(
        &self,
        time: f64,
        x: mp::MPComponent,
        y: mp::MPComponent,
        z: mp::MPComponent,
        objects: objects::ObjectList,
        num_trajectory_samples: usize,
        randomise: bool
    ) -> PyResult<BruteForceSolveTrajectoryIteratorPy> {
        let mp = mp::MPProblem {time, x, y, z, objects};
        let solver = match randomise {
            true => self.fmp.solver(mp),
            false => self.fmp.random_solver(mp)
        };
        Ok(BruteForceSolveTrajectoryIteratorPy{inner:solver.into_iter(), time_start: time, num_samples: num_trajectory_samples})
    }

    fn generate_trajectories(&self, start_time: f64, num_samples: usize) -> PyResult<trajectory::FourPLTrajectories>{
        match &self.solution {
            None => Err(exceptions::PyTypeError::new_err("Solve function must be run")), // Err(PyErr::new("SHDLKFSJDLF")),
            Some(sol) => {
                let trajs = sol.compute_trajectories(start_time, num_samples);
                Ok(trajs)
            }
        }
    }

    fn generate_trajectories_random_sample(&self, start_time: f64, num_samples: usize, num_traj: usize) -> PyResult<trajectory::FourPLTrajectories>{
        match &self.solution {
            None => Err(exceptions::PyTypeError::new_err("Solve function must be run before")), // Err(PyErr::new("SHDLKFSJDLF")),
            Some(sol) => {
                let trajs = sol.compute_trajectories_random_sample(start_time, num_samples, num_traj);
                Ok(trajs)
            }
        }
    }

    fn generate_trajectories_iter(&self, start_time: f64, num_samples: usize) -> PyResult<FourPLTrajectoryGeneratorIteratorPy>{
        match &self.solution {
            None => Err(exceptions::PyTypeError::new_err("Solve function must be run before")), // Err(PyErr::new("SHDLKFSJDLF")),
            Some(sol) => {
                let iter = sol.clone().into_traj_iter(start_time, num_samples);
                Ok(FourPLTrajectoryGeneratorIteratorPy{inner: iter})
            }
        }
    }
}

#[pyclass(name="HeuristicFastMotionPlanner")]
pub struct HeuristicFastMotionPlannerPy {
    fmp: HeuristicFastMotionPlanner,
    solution: Option<mp::MPSolution>
}

#[pymethods]
impl HeuristicFastMotionPlannerPy {
    #[new]
    fn new(dt: usize, epsilon: f64) -> Self {
        Self {
            fmp: HeuristicFastMotionPlanner::new(dt, epsilon),
            solution: None
        }
    }

    #[args(
        n = "None",
    )]
    fn solve(
        &mut self,
        time: f64,
        x: mp::MPComponent,
        y: mp::MPComponent,
        z: mp::MPComponent,
        objects: objects::ObjectList,
        n: Option<usize>, // Number of results
    ) -> PyResult<mp::MPSolution>{
        let mp = mp::MPProblem {time, x, y, z, objects};
        let sol = match n {
            Some(n) => mp::MPSolution::from_vec(self.fmp.solver(mp).into_iter().take(n).collect()),
            None => self.fmp.solve(mp)
        };
        self.solution = Some(sol.clone());
        Ok(sol)
    }

    #[args(
        randomise = "true",
    )]
    fn solve_iter(
        &self,
        time: f64,
        x: mp::MPComponent,
        y: mp::MPComponent,
        z: mp::MPComponent,
        objects: objects::ObjectList,
        randomise: bool
    ) -> PyResult<HeuristicSolveIteratorPy>{
        let mp = mp::MPProblem {time, x, y, z, objects};
        let solver = match randomise {
            true => self.fmp.solver(mp),
            false => self.fmp.random_solver(mp)
        };
        Ok(HeuristicSolveIteratorPy{inner:solver.into_iter()})
    }

    #[args(
        randomise = "true",
    )]
    fn solve_trajectories_iter(
        &self,
        time: f64,
        x: mp::MPComponent,
        y: mp::MPComponent,
        z: mp::MPComponent,
        objects: objects::ObjectList,
        num_trajectory_samples: usize,
        randomise: bool
    ) -> PyResult<HeuristicSolveTrajectoryIteratorPy> {
        let mp = mp::MPProblem {time, x, y, z, objects};
        let solver = match randomise {
            true => self.fmp.solver(mp),
            false => self.fmp.random_solver(mp)
        };
        Ok(HeuristicSolveTrajectoryIteratorPy{inner:solver.into_iter(), time_start: time, num_samples: num_trajectory_samples})
    }

    fn generate_trajectories(&self, start_time: f64, num_samples: usize) -> PyResult<trajectory::FourPLTrajectories>{
        match &self.solution {
            None => Err(exceptions::PyTypeError::new_err("Solve function must be run")), // Err(PyErr::new("SHDLKFSJDLF")),
            Some(sol) => {
                let trajs = sol.compute_trajectories(start_time, num_samples);
                Ok(trajs)
            }
        }
    }

    fn generate_trajectories_random_sample(&self, start_time: f64, num_samples: usize, num_traj: usize) -> PyResult<trajectory::FourPLTrajectories>{
        match &self.solution {
            None => Err(exceptions::PyTypeError::new_err("Solve function must be run before")), // Err(PyErr::new("SHDLKFSJDLF")),
            Some(sol) => {
                let trajs = sol.compute_trajectories_random_sample(start_time, num_samples, num_traj);
                Ok(trajs)
            }
        }
    }

    fn generate_trajectories_iter(&self, start_time: f64, num_samples: usize) -> PyResult<FourPLTrajectoryGeneratorIteratorPy>{
        match &self.solution {
            None => Err(exceptions::PyTypeError::new_err("Solve function must be run before")), // Err(PyErr::new("SHDLKFSJDLF")),
            Some(sol) => {
                let iter = sol.clone().into_traj_iter(start_time, num_samples);
                Ok(FourPLTrajectoryGeneratorIteratorPy{inner: iter})
            }
        }
    }
}

impl IntoPy<PyObject> for mp::MPSolutionElem {
    fn into_py(self, py: Python) -> PyObject {
        let mut elems = HashMap::new();
        elems.insert("x".to_string(), self.x.to_vec());
        elems.insert("y".to_string(), self.y.to_vec());
        elems.insert("z".to_string(), self.z.to_vec());
        elems.into_py(py)
    }
}

impl ToPyObject for mp::MPSolutionElem {
    fn to_object(&self, py: Python) -> PyObject {
        let mut elems = HashMap::new();
        elems.insert("x".to_string(), self.x.to_vec());
        elems.insert("y".to_string(), self.y.to_vec());
        elems.insert("z".to_string(), self.z.to_vec());
        elems.to_object(py)
    }
}

impl IntoPy<PyObject> for mp::MPSolution {
    fn into_py(self, py: Python) -> PyObject {
        let a: Vec<PyObject> = self.solutions.iter().map(|x| x.into_py(py)).collect();
        a.into_py(py)
    }
}

impl ToPyObject for mp::MPSolution {
    fn to_object(&self, py: Python) -> PyObject {
        let a: Vec<PyObject> = self.solutions.iter().map(|x| x.into_py(py)).collect();
        a.to_object(py)
    }
}

#[pyclass]
struct BruteForceSolveIteratorPy {
    pub inner: solvers::BruteForceSolverIterator2
}
#[pyproto]
impl PyIterProtocol for BruteForceSolveIteratorPy {
    fn __iter__(slf: PyRef<Self>) -> PyRef<Self> {
        slf
    }

    fn __next__(mut slf: PyRefMut<Self>) -> Option<mp::MPSolutionElem> {
        slf.inner.next()
    }
}

#[pyclass]
struct BruteForceSolveTrajectoryIteratorPy {
    inner: solvers::BruteForceSolverIterator2,
    time_start: f64,
    num_samples: usize
}
#[pyproto]
impl PyIterProtocol for BruteForceSolveTrajectoryIteratorPy {
    fn __iter__(slf: PyRef<Self>) -> PyRef<Self> {
        slf
    }

    fn __next__(mut slf: PyRefMut<Self>) -> Option<(mp::MPSolutionElem, trajectory::FourPLTrajectory)> {
        match slf.inner.next() {
            Some(elem) => Some((elem, elem.generate_fourpl_trajectory(slf.time_start, slf.num_samples))),
            None => None
        }
    }
}

#[pyclass]
struct HeuristicSolveIteratorPy {
    pub inner: solvers::HeuristicIterator
}
#[pyproto]
impl PyIterProtocol for HeuristicSolveIteratorPy {
    fn __iter__(slf: PyRef<Self>) -> PyRef<Self> {
        slf
    }

    fn __next__(mut slf: PyRefMut<Self>) -> Option<mp::MPSolutionElem> {
        slf.inner.next()
    }
}

#[pyclass]
struct HeuristicSolveTrajectoryIteratorPy {
    inner: solvers::HeuristicIterator,
    time_start: f64,
    num_samples: usize
}
#[pyproto]
impl PyIterProtocol for HeuristicSolveTrajectoryIteratorPy {
    fn __iter__(slf: PyRef<Self>) -> PyRef<Self> {
        slf
    }

    fn __next__(mut slf: PyRefMut<Self>) -> Option<(mp::MPSolutionElem, trajectory::FourPLTrajectory)> {
        match slf.inner.next() {
            Some(elem) => Some((elem, elem.generate_fourpl_trajectory(slf.time_start, slf.num_samples))),
            None => None
        }
    }
}

// Trajectory Conversion
#[pyclass]
struct FourPLTrajectoryGeneratorIteratorPy {
    inner: trajectory::FourPLTrajectoryGeneratorIterator
}

#[pyproto]
impl PyIterProtocol for FourPLTrajectoryGeneratorIteratorPy {
    fn __iter__(slf: PyRef<Self>) -> PyRef<Self> {
        slf
    }

    fn __next__(mut slf: PyRefMut<Self>) -> Option<trajectory::FourPLTrajectory> {
        slf.inner.next()
    }
}

impl IntoPy<PyObject> for trajectory::FourPLTrajectorySingle {
    fn into_py(self, py: Python) -> PyObject {
        self.into_vector_hashmap().into_py(py)
    }
}

impl IntoPy<PyObject> for trajectory::FourPLTrajectory {
    fn into_py(self, py: Python) -> PyObject {
        let mut elems = HashMap::new();
        elems.insert("x".to_string(), self.x.into_py(py));
        elems.insert("y".to_string(), self.y.into_py(py));
        elems.insert("z".to_string(), self.z.into_py(py));
        elems.into_py(py)
    }
}

impl IntoPy<PyObject> for trajectory::FourPLTrajectories {
    fn into_py(self, py: Python) -> PyObject {
        self.trajs.into_py(py)
    }
}