/*
MIT License

Copyright (c) 2021 University of Bristol Flight Laboratory

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

Author: Mickey Li <mickeyhli@outlook.com>
*/

//! Module containing useful, generic functions for array intersections, reading obstacle files, reading csvs and sorting arrays for testing.
use ndarray::{Array, Array1};
use std::f64::NEG_INFINITY;

/// Purpose: Range/interval intersection
///
/// A and B two ranges of closed intervals written
/// as vectors [lowerbound1 upperbound1 lowerbound2 upperbound2]
/// or as matrix [lowerbound1, lowerbound2, lowerboundn;
///               upperbound1, upperbound2, upperboundn]
/// A and B have to be sorted in ascending order
///
/// out is the mathematical intersection A n B
///
/// > NOTE: This is a reimplemntation of the matlab file using indexes instead of array slices
///
/// EXAMPLE USAGE:
///   >> out=range_intersection([1 3 5 9],[2 9])
///   	out =  [2 3 5 9]
///   >> out=range_intersection([40 44 55 58], [42 49 50 52])
///   	out =  [42 44]
pub fn range_intersection<T>(_first: &Array1<T>, _second: &Array1<T>) -> Array1<T>
    where
        T: Copy + PartialEq + PartialOrd,
{
    let mut out:Vec<T> = Vec::new();

    let mut first = _first;
    let mut second = _second;
    let mut f_idx = 0;
    let mut s_idx = 0;
    while (f_idx+1 < first.len()) && (s_idx+1 < second.len()) {
        // Make sure that first is always larger
        // If not switch both index and array vars
        if first[f_idx] > second[s_idx] {
            let temp = s_idx;
            s_idx = f_idx;
            f_idx = temp;

            let temp_arr = second;
            second = first;
            first = temp_arr;
        }

        if first[f_idx+1] < second[s_idx] {
            f_idx += 2;
        } else if first[f_idx+1] < second[s_idx] {
            out.push(second[s_idx]);
            out.push(second[s_idx]);
            f_idx += 2;
        } else if first[f_idx+1] == second[f_idx+1] {
            out.push(second[s_idx]);
            out.push(second[s_idx+1]);
            f_idx += 2;
            s_idx += 2;
        } else if first[f_idx+1] < second[s_idx+1] {
            out.push(second[s_idx]);
            out.push(first[f_idx+1]);
            f_idx += 2;
        } else {
            out.push(second[s_idx]);
            out.push(second[s_idx+1]);
            s_idx+=2;
        }

    }
    Array::from_vec(out)
}

/// Returns true on intersection
pub fn intersection_test_3(_first: &Array1<f64>, _second: &Array1<f64>, _third: &Array1<f64>) -> bool {
    let intersect_yz = range_intersection(_second, _third);
    let intersect_xyz = range_intersection(_first, &intersect_yz);
    let check_intersect = intersect_xyz.len() == 2 && intersect_xyz.iter().all(|x| *x == NEG_INFINITY);
    !(intersect_xyz.is_empty() || check_intersect)
}

use crate::mp::MPComponentSolutionElem;
/// Returns true on intersection with any object
pub fn intersection_test_3_component_solution_elems(_first: &MPComponentSolutionElem, _second: &MPComponentSolutionElem, _third: &MPComponentSolutionElem, num_objects: usize) -> bool{
    for obs_id in 0..num_objects {
        if intersection_test_3(
            &_first.intersects[obs_id].points.to_array(),
            &_second.intersects[obs_id].points.to_array(),
            &_third.intersects[obs_id].points.to_array()) {
                return true;
        }
    }
    false
}

use ndarray::{Array2};
pub fn compare_row_by_column_lt(arr: &Array2<f64>, i: usize, j:usize) -> bool {
    let row1 = arr.row(i);
    let row2 = arr.row(j);
    for (r1, r2) in row1.iter().zip(row2.iter()) {
        if r1 < r2 {
            return true;
        } else if r1 > r2 {
            return false;
        }
    }
    false
}

use ndarray_csv::{Array2Reader};
use csv::{ReaderBuilder};
use std::error::Error;
use std::fs::File;
pub fn read_ndarray_csv(file_path: String, shape: (usize, usize)) -> Result<Array2<f64>, Box<dyn Error>> {
    // Read an array back from the file
    let file = File::open(file_path)?;
    let mut reader = ReaderBuilder::new().has_headers(false).delimiter(b' ').from_reader(file);
    let array_read: Array2<f64> = reader.deserialize_array2(shape)?;
    Ok(array_read)
}

use crate::extras::sortaxis::*;
use ndarray::{Axis};
pub fn sort_axis_columnwise(arr: Array2<f64>) -> Array2<f64> {
    let perm = arr.sort_axis_by(Axis(0), |i, j| compare_row_by_column_lt(&arr, i, j));
    arr.permute_axis(Axis(0), &perm)
}

use crate::objects::{Object, ObjectList, Bounds};
pub fn read_objects_from_file(file_path: String) -> Result<ObjectList, Box<dyn Error>> {
    let mut objects = ObjectList::new();
    let mut rdr = csv::ReaderBuilder::new()
                    .has_headers(true)
                    .from_path(file_path)?;
    for res in rdr.records() {
        let record = res?;
        let xl: f64 = record[0].parse()?;
        let xu: f64 = record[1].parse()?;
        let yl: f64 = record[2].parse()?;
        let yu: f64 = record[3].parse()?;
        let zl: f64 = record[4].parse()?;
        let zu: f64 = record[5].parse()?;
        let to_init: f64 = record[6].parse()?;
        let to_end: f64 = record[7].parse()?;
        objects.push(Object {
            xbounds: Bounds::new(xl, xu),
            ybounds: Bounds::new(yl, yu),
            zbounds: Bounds::new(zl, zu),
            to_init, to_end
        })
    }
    Ok(objects)
}

#[cfg(test)]
mod tests {

    use ndarray::{array};

    #[test]
    fn test_range_intersection_1() {
        let out = super::range_intersection(&array![1, 3, 5, 9], &array![2, 9]);
        let correct = array![2, 3, 5, 9];
        assert_eq!(out, correct, "Not correct\nexpecting: {}\ngot {}:", correct, out);
    }

    //   >> out=range_intersection([40 44 55 58], [42 49 50 52])
//   	out =  [42 44]
    #[test]
    fn test_range_intersection_2() {
        let out = super::range_intersection(&array![40, 44, 55, 58], &array![42, 49, 50, 52]);
        let correct = array![42, 44];
        assert_eq!(out, correct, "Not correct\nexpecting: {}\ngot {}:", correct, out);
    }

    #[test]
    fn test_range_intersection_float_3() {
        let out = super::range_intersection(&array![1.56, 2.76], &array![1.769, 2.76]);
        let correct = array![1.7690, 2.76];
        assert_eq!(out, correct, "Not correct\nexpecting: {}\ngot {}:", correct, out);
    }

    #[test]
    fn test_range_intersection_float_4() {
        let out = super::range_intersection(&array![1.56, 10./3.0], &array![1.769, 10./3.0]);
        let correct = array![1.7690, 20./6.];
        assert_eq!(out, correct, "Not correct\nexpecting: {}\ngot {}:", correct, out);
    }

    #[test]
    fn test_range_intersection_float_inf_5() {
        let out = super::range_intersection(&array![f64::NEG_INFINITY, f64::NEG_INFINITY, 2.345, f64::INFINITY], &array![f64::NEG_INFINITY, f64::NEG_INFINITY, 2.6, f64::INFINITY]);
        let correct = array![f64::NEG_INFINITY, f64::NEG_INFINITY, 2.6, f64::INFINITY];
        assert_eq!(out, correct, "Not correct\nexpecting: {}\ngot {}:", correct, out);
    }
}