/*
MIT License

Copyright (c) 2021 University of Bristol Flight Laboratory

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

Author: Mickey Li <mickeyhli@outlook.com>
*/

//! Module containing structures and algorithms for generating trajectories
use crate::mp;

use std::collections::HashMap;
use ndarray::{Array, Array1, Array2, Array3, stack, Axis};

/// A Single Trajectory generated from a single Solution Set Element along a single Component Axis
pub struct FourPLTrajectorySingle {
    times: Array1<f64>,
    pos: Array1<f64>,
    vels: Array1<f64>,
    accels: Array1<f64>,
    jerks: Array1<f64>
}

impl FourPLTrajectorySingle {
    /// Generate a trajectory from a given solution set with parameters
    /// - `time_start`: The time the trajectory begins
    /// - `time_goal`: The time the trajectory ends
    /// - `num_samples`: The resolution of the resulting trajectory
    pub fn from_solution(sol: &mp::ss::SolutionSetVectorElem, time_start: f64, time_goal:f64, num_samples: usize) -> Self {
        let t = Array::linspace(time_start, time_goal, num_samples);

        let time_diff = &t - time_start + sol.td;
        let time_over_c = ( &time_diff / sol.cs).mapv(|x| x.powf(2.0).powf(sol.bs));
        let x = sol.xg + ( (sol.xi - sol.xg) / (1. + &time_over_c));

        let v = (2. * sol.bs * (sol.xg - sol.xi) * &time_over_c)
            / ( &time_diff * (1. + &time_over_c).mapv(|x|x.powf(2.)));

        let a = (
            (2. * sol.bs * (sol.xg - sol.xi) * &time_over_c)
            / ( &time_diff.mapv(|x|x.powf(2.)) * (1. + &time_over_c).mapv(|x|x.powf(2.)))
        ) * ( (2.*sol.bs - 1.) - (
            (4. * sol.bs * &time_over_c) / (1. + &time_over_c)
        ));

        let jterm2_temp = ( &time_diff / sol.cs).mapv(|x| {
            let _x1 = x.powf(4.0);
            _x1.powf(sol.bs)
        });
        let j1 = (2. * sol.bs * (sol.xg - sol.xi) * &time_over_c)
            / ( (&time_diff.mapv(|x|x.powf(3.))) *  (1. + &time_over_c).mapv(|x|x.powf(2.)) );
        let j2 = (24. * sol.bs.powf(2.) * &jterm2_temp)
            /  (1. + &time_over_c).mapv(|x|x.powf(2.));
        let j3 = (12. * sol.bs * (2. * sol.bs - 1.) * &time_over_c)
            /  (1. + &time_over_c);
        let j4 = (2.* sol.bs-1.) * (2.*sol.bs - 2.);
        let j = j1 * (j2 - j3 + j4);
        Self {times: t, pos: x, vels: v, accels:a, jerks: j}
    }

    /// Returns trajectory as stacked 2d array
    pub fn as_2d_array(&self) -> Array2<f64> {
        stack![Axis(0),self.times,self.pos, self.vels, self.accels,self.jerks]
    }

    /// Returns trajectory as a dictionary/hashmap
    pub fn into_vector_hashmap(self) -> HashMap<String, Vec<f64>> {
        let mut elems = HashMap::new();
        elems.insert("times".to_string(), self.times.to_vec());
        elems.insert("pos".to_string(), self.pos.to_vec());
        elems.insert("vels".to_string(), self.vels.to_vec());
        elems.insert("accels".to_string(), self.accels.to_vec());
        elems.insert("jerks".to_string(), self.jerks.to_vec());
        elems
    }
}

/// A 3D Trajectory generated from a Solution Set containg [`FourPLTrajectorySingle`](FourPLTrajectorySingle)
pub struct FourPLTrajectory {
    pub x: FourPLTrajectorySingle,
    pub y: FourPLTrajectorySingle,
    pub z: FourPLTrajectorySingle
}
impl FourPLTrajectory {
    pub fn as_3d_array(&self) -> Array3<f64> {
        stack![Axis(0), self.x.as_2d_array(), self.y.as_2d_array(), self.z.as_2d_array()]
    }
}

/// A Container holding trajectories [`FourPLTrajectories`](FourPLTrajectories), can be used as an iterator.
pub struct FourPLTrajectories {
    pub num_samples: usize,
    pub trajs: Vec<FourPLTrajectory>
}
impl FourPLTrajectories {
    pub fn new(num_samples: usize) -> Self {
        Self {num_samples, trajs: Vec::new()}
    }
    pub fn push(&mut self, traj: FourPLTrajectory) {
        self.trajs.push(traj);
    }
}
impl IntoIterator for FourPLTrajectories {
    type Item = FourPLTrajectory;
    type IntoIter = std::vec::IntoIter<Self::Item>;

    fn into_iter(self) -> Self::IntoIter {
        self.trajs.into_iter()
    }
}

/// An iterator which lazily generates [`FourPLTrajectories`](FourPLTrajectories)
pub struct FourPLTrajectoryGeneratorIterator {
    sols: Vec<mp::MPSolutionElem>,
    idx: usize,
    time_start:f64,
    num_samples: usize
}
impl FourPLTrajectoryGeneratorIterator {
    pub fn new(sols: Vec<mp::MPSolutionElem>, time_start:f64, num_samples: usize) -> Self {
        Self{sols, idx:0, time_start, num_samples}
    }
}
impl Iterator for FourPLTrajectoryGeneratorIterator {
    type Item = FourPLTrajectory;
    fn next(&mut self) -> Option<Self::Item> {
        if self.idx >= self.sols.len() {
            return None;
        }
        let t = self.sols[self.idx].generate_fourpl_trajectory(self.time_start, self.num_samples);
        self.idx += 1;
        Some(t)
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use ndarray::{array};
    #[test]
    fn test_trajectory_1() {
        let ssve = mp::ss::SolutionSetVectorElem {
            xi: -2.5120,
            xg: 4.0,
            bs: 8.7648,
            cs: 9.2586,
            td: 9.0161,
            tg: 4.3813,
        };

        let traj = FourPLTrajectorySingle::from_solution(&ssve, 0.0, ssve.tg, 11);
        let as_2d = traj.as_2d_array();

        let correct = array![
            [0.0       ,   0.4381 ,   0.8763 ,   1.3144 ,   1.7525 ,   2.1906  ,  2.6288  ,  3.0669 ,   3.5050 ,   3.9431   , 4.3813],
            [0.0000  ,  1.3342  ,  2.4466  ,  3.1676  ,  3.5697  ,  3.7787  ,  3.8851  ,  3.9394  ,  3.9674 ,   3.9821   , 3.9900],
            [3.0000  ,  2.9194  ,  2.0961  ,  1.2319  ,  0.6542  ,  0.3343  ,  0.1699  ,  0.0871  ,  0.0454 ,   0.0241   , 0.0131],
            [1.0000  , -1.2900  , -2.1541  , -1.6753  , -0.9850  , -0.5173  , -0.2613  , -0.1313  , -0.0666 ,  -0.0343   ,-0.0180],
            [-5.1517 ,  -4.1469 ,   0.0404 ,   1.6494 ,   1.3604 ,   0.7927  ,  0.4110  ,  0.2053 ,   0.1021 ,   0.0512 ,   0.0261]
        ];

        assert!(
            as_2d.abs_diff_eq(&correct, 1e-3),
            "Calculated Trajectory was:\n{}\nvs\n{}", as_2d, correct
        );

    }
    #[test]
    fn test_trajectory_2() {
        let ssve = mp::ss::SolutionSetVectorElem {
            xi:    -2.0177,
            xg:     4.0000,
            bs:     2.7716,
            cs:     2.8038,
            td:     2.4782,
            tg:     6.4142,
        };

        let traj = FourPLTrajectorySingle::from_solution(&ssve, 0.0, ssve.tg, 11);
        let as_2d = traj.as_2d_array();

        let correct = array![
           [ 0.0       ,  0.6414  ,  1.2828 ,   1.9243  ,  2.5657  ,  3.2071  ,  3.8485  ,  4.4900  ,  5.1314  ,  5.7728   , 6.4142],
           [ 0.0000  ,  1.8561  ,  3.0126 ,   3.5440  ,  3.7765  ,  3.8828  ,  3.9346  ,  3.9615  ,  3.9763  ,  3.9849   , 3.9900],
           [ 3.0000  ,  2.4523  ,  1.2165 ,   0.5307  ,  0.2365  ,  0.1121  ,  0.0567  ,  0.0304  ,  0.0172  ,  0.0101   , 0.0062],
           [ 1.0000  , -2.0388  , -1.5281 ,  -0.6875  , -0.2876  , -0.1247  , -0.0575  , -0.0282  , -0.0147  , -0.0080   ,-0.0046],
           [-6.7611  , -1.2031  ,  1.6007 ,   0.9289  ,  0.3861  ,  0.1567  ,  0.0666  ,  0.0300  ,  0.0144  ,  0.0073   , 0.0039],
        ];

        assert!(
            as_2d.abs_diff_eq(&correct, 1e-3),
            "Calculated Trajectory was:\n{}\nvs\n{}", as_2d, correct
        );

    }

    #[test]
    fn test_trajectory_3() {
        let ssve = mp::ss::SolutionSetVectorElem {
            xi:      0.3908,
            xg:      3.0000,
            bs:      4.3722,
            cs:      1.9239,
            td:      1.8586,
            tg:      6.4142,
        };

        let traj = FourPLTrajectorySingle::from_solution(&ssve, 0.0, ssve.tg, 11);
        let as_2d = traj.as_2d_array();

        let correct = array![
        [    0.000000000000000 ,  0.641400000000000 ,  1.282800000000000 ,  1.924200000000000 ,  2.565600000000000 ,  3.207000000000000,    3.848400000000000 ,  4.489800000000000,   5.131200000000000,   5.772600000000000,   6.414000000000000],
        [    1.499919924983955 ,  2.760175931260509 ,  2.964637571073519 ,  2.992957810392111 ,  2.998206188938201 ,  2.999450652760404,    2.999806292011277 ,  2.999923672240367,   2.999967101981124,   2.999984732587434,   2.999992461540741],
        [    3.000058081251758 ,  0.761744586067147 ,  0.097100760205677 ,  0.016234937112071 ,  0.003543017391639 ,  0.000948101037754,    0.000296781914101 ,  0.000105132149280,   0.000041155656691,   0.000017494441406,   0.000007968366981],
        [    0.500794576300787 , -2.479302094727926 , -0.293873212470755 , -0.041618219248403 , -0.007793946782545 , -0.001823117640010,   -0.000506671867799 , -0.000161362851941,  -0.000057373330560,  -0.000022338694382,  -0.000009385991415],
        [  -32.644170162400727 ,  7.505554760604893 ,  0.962831458738734 ,  0.117223133209393 ,  0.018887806217762 ,  0.003864412338289,    0.000953677412277 ,  0.000273075166718,   0.000088188218350,   0.000031451348478,   0.000012190356878],
        ];

        assert!(
            as_2d.abs_diff_eq(&correct, 1e-3),
            "Calculated Trajectory was:\n{}\nvs\n{}", as_2d, correct
        );

    }
}
