/*
MIT License

Copyright (c) 2021 University of Bristol Flight Laboratory

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

Author: Mickey Li <mickeyhli@outlook.com>
*/

//! Module containing structures to describe and work with objects and obstacles
//!
use approx::{RelativeEq, AbsDiffEq};
use ndarray::{Array1, array};
use pyo3::prelude::*;

/// Represent upper and lower bounds for obstacles
#[derive(FromPyObject)]
#[derive(Clone, Copy, Debug)]
pub struct Bounds {
    pub u: f64,
    pub l:f64
}
impl Bounds {
    pub fn new(l: f64, u: f64) -> Self{
        Self {u, l}
    }
}

/// Represents an object or obstacle
#[pyclass]
// #[derive(FromPyObject)]
#[derive(Clone, Copy, Debug)]
pub struct Object {
    /// The time at which the obstacle appears
    pub to_init: f64,
    /// The time at which the obstacle disappears
    pub to_end: f64,
    pub xbounds: Bounds,
    pub ybounds: Bounds,
    pub zbounds: Bounds,
}
impl Object {
    /// Generate a default obstacle from given bounds. Obstacle lasts forever from -inf to inf time
    pub fn new(xbounds: Bounds, ybounds: Bounds, zbounds: Bounds) -> Self {
        Object {xbounds, ybounds, zbounds, to_init: -f64::INFINITY, to_end: f64::INFINITY}
    }
}

/// List of obstacles defining the problem
#[derive(FromPyObject)]
#[derive(Clone, Debug)]
#[pyo3(transparent)]
pub struct ObjectList {
    pub objects: Vec<Object>
}
impl ObjectList {
    pub fn new() -> Self {
        ObjectList {objects: Vec::new()}
    }

    /// Add an obstacle to the list
    pub fn push(&mut self, object: Object) {
        self.objects.push(object);
    }

    /// Converts the list of obstacles into a set of bounds for a given axis
    pub fn to_object_axis_list(&self, axis: String) -> ObjectAxisList{
        let mut oax = ObjectAxisList::new();
        self.objects.iter().for_each(|x| {
            if axis == "x" {
                oax.push(x.to_init, x.to_end, x.xbounds);
            } else if axis == "y" {
                oax.push(x.to_init, x.to_end, x.ybounds);
            } else if axis == "z" {
                oax.push(x.to_init, x.to_end, x.zbounds);
            } else {
                panic!("Axis {} does not exist", axis);
            }
        });
        oax
    }

    /// Number of obstacles in the list
    pub fn number(&self) -> usize {
        self.objects.len()
    }
}

/// Defines one obstacles time and space bounds on one axis.
#[derive(Debug, Clone)]
pub struct ObjectAxis {
    pub to_init: f64,
    pub to_end: f64,
    pub bounds: Bounds,
}

/// List of time and space bounds for an obstacle
pub struct ObjectAxisList {
    pub objects: Vec<ObjectAxis>
}
impl ObjectAxisList {
    pub fn new() -> Self {
        Self { objects: Vec::new() }
    }
    pub fn push(&mut self, to_init: f64, to_end: f64, bounds: Bounds) {
        self.objects.push (
            ObjectAxis {to_init, to_end, bounds}
        )
    }
}

/// Defines the type of intersection given by a solutions trajectory and a particular obstacle
#[derive(Debug, Clone)]
pub struct Intersection {
    /// The set of 4 intersection points of the trajectory
    pub points: IntersectionPoints,
    /// The type of intersection based on the intersection points
    pub flag: IntersectionFlag
}
impl Intersection {
    pub fn new() -> Self {
        Self {
            points: IntersectionPoints::set_all(f64::NEG_INFINITY),
            flag: IntersectionFlag::None
        }
    }
}
impl PartialEq for Intersection {
    fn eq(&self, other: &Self) -> bool {
        self.points == other.points && self.flag == other.flag
    }
}
impl AbsDiffEq for Intersection {
    type Epsilon = f64;
    fn default_epsilon() -> Self::Epsilon {
        f64::default_epsilon()
    }
    fn abs_diff_eq(&self, other: &Self, epsilon: Self::Epsilon) -> bool {
        self.points.abs_diff_eq(&other.points, epsilon)
        && self.flag == other.flag
    }
}
impl RelativeEq for Intersection {
    fn default_max_relative() -> Self::Epsilon {
        Self::Epsilon::default_max_relative()
    }
    fn relative_eq(&self, other: &Self, epsilon: Self::Epsilon, max_relative: Self::Epsilon) -> bool {
        self.points.relative_eq(&other.points, epsilon, max_relative)
        && self.flag == other.flag
    }
}

/// Defines the points of intersection by a solution trajectory and a particular obstacle
/// See paper for more details and diagram.
///
/// In short, an intersection can be characterised by four intersection values in time.
/// There are exactly four types of Intersection:
/// 1. Type 1, in which the trajectory passes through the object, and all four values are non infinity
/// 2. Type 2, in which the trajectory goes in and out within the same plane (two intersection points)
/// 3. Type 3, in which the trajectory begins inside the obstacle and remains inside the obstacle
/// 4. Type 4, in which the trajectory misses the obstacle entirely, and all four values are negative infinity
#[derive(Debug, Clone)]
pub struct IntersectionPoints {
    /// The timeo of first intersection
    pub tkl: f64,
    /// The time of second intersection that should be on the opposite side of obstacle as `tkl`.
    pub tkl_u:f64,
    /// The time of third intersection after the peak and should be on the same side as `tkl_u`
    pub tkr_u: f64,
    /// the time of final intersection that should be on opposite side to tkr
    pub tkr: f64
}
impl IntersectionPoints {
    pub fn set_all(val: f64) -> Self {
        Self{tkl: val, tkl_u: val, tkr_u:val, tkr: val}
    }

    /// Set intersection times using an array in order.
    pub fn set_arr(&mut self, arr: Vec<f64>) {
        self.set_raw(arr[0], arr[1], arr[2], arr[3]);
    }

    pub fn set_raw(&mut self, tkl: f64, tkl_u:f64, tkr_u: f64, tkr: f64) {
            self.tkl = tkl;
            self.tkl_u = tkl_u;
            self.tkr_u = tkr_u;
            self.tkr = tkr;
    }

    pub fn to_array(&self) -> Array1<f64> {
        array![self.tkl, self.tkl_u, self.tkr_u, self.tkr]
    }
}
impl PartialEq for IntersectionPoints {
    fn eq(&self, other: &Self) -> bool {
        self.tkl == other.tkl
        && self.tkl_u == other.tkl_u
        && self.tkr_u == other.tkr_u
        && self.tkr == other.tkr
    }
}
impl AbsDiffEq for IntersectionPoints {
    type Epsilon = f64;
    fn default_epsilon() -> Self::Epsilon {
        f64::default_epsilon()
    }
    fn abs_diff_eq(&self, other: &Self, epsilon: Self::Epsilon) -> bool {
        self.tkl.abs_diff_eq(&other.tkl, epsilon)
        && self.tkl_u.abs_diff_eq(&other.tkl_u, epsilon)
        && self.tkr_u.abs_diff_eq(&other.tkr_u, epsilon)
        && self.tkr.abs_diff_eq(&other.tkr, epsilon)
    }
}
impl RelativeEq for IntersectionPoints {
    fn default_max_relative() -> Self::Epsilon {
        Self::Epsilon::default_max_relative()
    }
    fn relative_eq(&self, other: &Self, epsilon: Self::Epsilon, max_relative: Self::Epsilon) -> bool {
        self.tkl.relative_eq(&other.tkl, epsilon, max_relative)
        && self.tkl_u.relative_eq(&other.tkl_u, epsilon, max_relative)
        && self.tkr_u.relative_eq(&other.tkr_u, epsilon, max_relative)
        && self.tkr.relative_eq(&other.tkr, epsilon, max_relative)
    }
}

/// Defines the intersection flags for type of intersection
#[derive(Debug, Clone, PartialEq, Copy)]
pub enum IntersectionFlag {
    /// No Intersections -> 0
    None,           // 0
    /// Trajectory Definitely intersects object
    Complete,       // 1
    /// Trjaectory May Intersect Object
    Partial,        // 2
    /// Extra for multiplication (not used)
    DoublePartial   // >2
}
impl IntersectionFlag {
    pub fn to_num(&self) -> u8 {
        match self {
            Self::None => 0,
            Self::Complete => 1,
            Self::Partial => 2,
            _ => 3
        }
    }
}
impl std::ops::Mul for IntersectionFlag {
    type Output = Self;

    fn mul(self, rhs: Self) -> Self {
        match self {
            Self::None => Self::None,
            Self::Complete => rhs,
            Self::Partial => match rhs {
                Self::None => Self::None,
                Self::Complete => Self::Partial,
                Self::Partial | Self::DoublePartial => Self::DoublePartial
            }
            Self::DoublePartial => Self::DoublePartial
        }
    }
}