/*
MIT License

Copyright (c) 2021 University of Bristol Flight Laboratory

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

Author: Mickey Li <mickeyhli@outlook.com>
*/

//! The `fast_motion_planning` crate is a Rust implementation by Mickey Li (github: mhl787156) of
//! > S. Upadhyay, T. Richardson, and A. Richards, **“Fast Generation of Obstacle-Avoiding Motion Primitives forQuadrotors”**, Accepted in 2021 IEEE/RSJ International Conference on Intelligent Robots and Systems
//! > (IROS 2021).
//!
//! The paper abstract is the following:
//! > *This  work  considers  the  problem  of  generating computationally efficient quadrotor motion primitives betweena  given  pose  (position,  velocity,  and  acceleration)  and  a  goalplane  in  the  presence  of  obstacles.  A  new  motion  primitivetool  based  on  the  logistic  curve  is  proposed  and  a  closed-formanalytic approach is developed to satisfy constraints on startingpose, goal plane, velocity, acceleration, and jerk. The geometricobstacle  avoidance  problem  is  represented  as  a  combinatorialset problem and a heuristic approach is proposed to acceleratethe   solution   search.   Numerical   examples   are   presented   tohighlight the fast motion primitive generation in multi-obstaclepose-to-plane  scenarios*
//!
//! The crate provides a solver api which samples from motion primitives in order to efficiently generate trajectories which avoid specified obstacles, for UAVs to follow.
//! This crate contains the following Rust implementation, as well as Python api using the `pyo3` crate.
//!
//! - [**`MPProblem`**](mp::MPProblem) Defines the planning problem.
//!     - It request the specification of [**`MPComponent`**](mp::MPComponent) which, for a given axis, defines the start position, velocity and acceleration, minimum and maximum bounds on goal position, as well as upper and lower bounds.
//!     - And also [**`ObjectList`**](objects::ObjectList) which is a list of [**`Object`**](objects::Object) specifying the axis [**`Bounds`**](objects::Bounds) describing the location of cuboidal obstacles in the environment.
//! - [**`BruteForceFastMotionPlanner`**](BruteForceFastMotionPlanner) and [**`HeuristicFastMotionPlanner`**](HeuristicFastMotionPlanner): The structures which represent the motion planner using the BruteForce and Heuristic solver respectively given inputs
//! - Both solvers expose apis for generating trajectories in one go, or as a generator/iterator.
//! - [**`generate_component()`**](generate_component) [**`generate_default_object()`**](generate_default_object) and variants are primarily for use with the Python API, refer to github project and instructions.
//!
//! # Using `fast_motion_planning`
//! To use `fast_motion_planning`, ensure you import the following
//! ```rust
//! use fast_motion_planning::{BruteForceFastMotionPlanner, HeuristicFastMotionPlanner, MPProblem, MPComponent, utils};
//! use fast_motion_planning::{Object, ObjectList, Bounds};
//! ```
//!
//! # Examples
//! The api can be used like the following:
//! ```rust
//!  // Define the axis component ranges
//!  // comp id, start position, goal position min, goal position max, start velocity, start acceleration
//!  let x_param = MPComponent::new("x".to_string(), 0.0, 4.0, 4.0, 3.0, 1.0);
//!  let y_param = MPComponent::new("y".to_string(), 1.5, 2.8, 3.2, 3.0, 0.5);
//!  let z_param = MPComponent::new("z".to_string(), 1.0, 1.7, 2.3, 1.0, 0.1);
//!
//!  // Create Objects
//!  let mut objects = ObjectList::new();
//!  objects.push(Object::new(
//!      Bounds::new(3.75, 4.25),
//!      Bounds::new(2.3, 2.8),
//!      Bounds::new(1.7, 2.3)));
//!  objects.push(Object::new(
//!      Bounds::new(3.75, 4.25),
//!      Bounds::new(3.2, 3.7),
//!      Bounds::new(1.7, 2.3)));
//!
//!  // Create the problem
//!  let problem = MPProblem::new(x_param, y_param, z_param, objects);
//!
//!  // Specify parameters
//!  let time_division = 5;
//!  let epsilon = 0.01;
//!
//!  println!("Testing Heuristic solution");
//!  let planner = HeuristicFastMotionPlanner::new(time_division, epsilon);
//!  let solution = planner.solve(problem.clone());
//!  println!("Found {} solutions", solution.len());
//!
//!  println!("Testing BruteForce solution");
//!  let planner = BruteForceFastMotionPlanner::new(time_division, epsilon);
//!  let solution = planner.solve(problem.clone());
//!  println!("Found {} solutions", solution.len());
//!
//!  println!("Testing Heuristic solution");
//!  let planner = HeuristicFastMotionPlanner::new(time_division, epsilon);
//!  let num_sols = planner.solver(problem.clone()).into_iter().count();
//!  println!("Found {} solutions", num_sols);
//!
//!  println!("Testing BruteForce solution iterator");
//!  let planner = BruteForceFastMotionPlanner::new(time_division, epsilon);
//!  let num_sols = planner.solver(problem.clone()).into_iter().count();
//!  println!("Found {} solutions", num_sols);
//!
//!  println!("Testing Heuristic solution and trajectory iterator");
//!  let time_start = 0.0;
//!  let num_samples = 100;
//!  let planner = HeuristicFastMotionPlanner::new(time_division, epsilon);
//!  let num_sols = planner.solver(problem.clone()).into_iter().map(|x| (x, x.generate_fourpl_trajectory(time_start, num_samples))).count();
//!  println!("Found {} solutions", num_sols);
//!
//!  println!("Testing BruteFroce solution and trajectory iterator");
//!  let time_start = 0.0;
//!  let num_samples = 100;
//!  let planner = BruteForceFastMotionPlanner::new(time_division, epsilon);
//!  let num_sols = planner.solver(problem.clone()).into_iter().map(|x| (x, x.generate_fourpl_trajectory(time_start, num_samples))).count();
//!  println!("Found {} solutions", num_sols);
//! ```


mod mp;
mod extras;
mod pywrappers;

// use pywrappers::generate_component;

pub use mp::*;
pub use mp::objects::*;
pub use mp::solvers::*;
use mp::solvers::{MPSolver};

///// PYTHON INTERFACE
use pyo3::prelude::*;

#[pymodule]
fn fast_motion_planning(_py: Python<'_>, m: &PyModule) -> PyResult<()> {
    m.add_class::<pywrappers::BruteForceFastMotionPlannerPy>()?;
    m.add_class::<pywrappers::HeuristicFastMotionPlannerPy>()?;
    m.add_function(wrap_pyfunction!(generate_component, m)?)?;
    m.add_function(wrap_pyfunction!(generate_component_all, m)?)?;
    m.add_function(wrap_pyfunction!(generate_default_object, m)?)?;
    m.add_function(wrap_pyfunction!(generate_object, m)?)?;
    Ok(())
}


/// Helper pyfunction for generating a rust [`MPComponent`](mp::MPComponent)
#[pyfunction]
#[pyo3(text_signature = "(id, pos, pmin, pmax, vel, accel, /)")]
pub fn generate_component(
    id: String, pos: f64, pmin: f64,
    pmax: f64, vel: f64,
    accel:f64,
) -> PyResult<mp::MPComponent> {
    Ok(mp::MPComponent::new(
        id, pos, pmin, pmax, vel, accel
    ))
}

/// Helper pyfunction for generating a rust [`MPComponent`](mp::MPComponent)
#[pyfunction]
#[pyo3(text_signature = "(id, pos, pmin, pmax, vel, vmin, vmax, accel, amin, amax, jmin, jmax, /)")]
pub fn generate_component_all(
    id:     String,
    pos:    f64,
    pmin:   f64,
    pmax:   f64,
    vel:    f64,
    vmin:   f64,
    vmax:   f64,
    accel:  f64,
    amin:   f64,
    amax:   f64,
    jmin:   f64,
    jmax:   f64
) -> PyResult<mp::MPComponent> {
    Ok(mp::MPComponent{
        id, pos, pmin, pmax, vel, vmin, vmax, accel, amin, amax, jmin, jmax
    })
}

/// Helper pyfunction for generating a rust [`Object`](objects::Object)
#[pyfunction]
#[pyo3(text_signature = "(xl, xu, yl, yu, zl, zu, /)")]
pub fn generate_default_object(
    xl:f64,  xu:f64,
    yl: f64, yu: f64,
    zl: f64, zu: f64
) -> PyResult<objects::Object> {
    Ok(objects::Object::new(
        objects::Bounds::new(xl, xu),
        objects::Bounds::new(yl, yu),
        objects::Bounds::new(zl, zu)
    ))
}

/// Helper pyfunction for generating a rust [`Object`](objects::Object)
#[pyfunction]
#[pyo3(text_signature = "(xl, xu, yl, yu, zl, zu, to_init, to_end, /)")]
pub fn generate_object(
    xl:f64,  xu:f64,
    yl: f64, yu: f64,
    zl: f64, zu: f64,
    to_init: f64, to_end: f64,
) -> PyResult<objects::Object> {
    Ok(objects::Object{
        to_init, to_end,
        xbounds: objects::Bounds::new(xl, xu),
        ybounds: objects::Bounds::new(yl, yu),
        zbounds: objects::Bounds::new(zl, zu)
    })
}

/// Interface for solving the motion planning problem using the **Brute Force** solver approach.
///
/// # Examples
///
/// ```
/// // Define the axis component ranges
/// // comp id, start position, goal position min, goal position max, start velocity, start acceleration
/// let x_param = MPComponent::new("x".to_string(), 0.0, 4.0, 4.0, 3.0, 1.0);
/// let y_param = MPComponent::new("y".to_string(), 1.5, 2.8, 3.2, 3.0, 0.5);
/// let z_param = MPComponent::new("z".to_string(), 1.0, 1.7, 2.3, 1.0, 0.1);
///
/// // Create Objects
/// let mut objects = ObjectList::new();
/// objects.push(Object::new(
///     Bounds::new(3.75, 4.25),
///     Bounds::new(2.3, 2.8),
///     Bounds::new(1.7, 2.3)));
/// objects.push(Object::new(
///     Bounds::new(3.75, 4.25),
///     Bounds::new(3.2, 3.7),
///     Bounds::new(1.7, 2.3)));
///
/// // Create the problem
/// let problem = MPProblem::new(x_param, y_param, z_param, objects);
///
/// // Specify parameters
/// let time_division = 5;
/// let epsilon = 0.01;
/// let planner = BruteForceFastMotionPlanner::new(time_division, epsilon);
/// let solution = planner.solve(problem);
/// ```
pub struct BruteForceFastMotionPlanner {
    fmp: FastMotionPlanner
}

impl BruteForceFastMotionPlanner {
    /// Generate a new instance of `BruteForceFastMotionPlanner`.
    /// - `dt` is the number of samples to divide the 2D goal region into, dt => dt x dt gridding.  e.g. dt = 5, grids goal region into 25.
    /// - `epsilon` is the maximum allowed error in the end goal for a trajectory to be considered.
    pub fn new(dt: usize, epsilon: f64) -> BruteForceFastMotionPlanner {
        BruteForceFastMotionPlanner {
            fmp: FastMotionPlanner::new(dt, epsilon, mp::MPMethod::BRUTEFORCE)
        }
    }

    /// Generate all possible solutions, and returns it in the form of a [**`MPSolution`**](mp::MPSolution)
    pub fn solve(&self, problem: mp::MPProblem) -> mp::MPSolution {
        self.fmp.solve(problem)
    }

    /// Returns the [**`BruteForceSolver`**](mp::solvers::BruteForceSolver) which implements the [**`MPSolver`**](mp::solvers::MPSolver) trait
    /// - The [**`BruteForceSolver`**](mp::solvers::BruteForceSolver) implements `.into_iter()` which will generate solutions lazily within an iterator.
    pub fn solver(&self, problem: mp::MPProblem) -> mp::solvers::BruteForceSolver {
        let mpcs = self.fmp.generate_component_solutions(&problem);
        let solver = mp::solvers::BruteForceSolver::new(mpcs, problem.objects.number());
        solver
    }

    /// Identical to `solver()`, except that it also randomly shuffles the Axis Component solutions.
    /// This is so that the subsequent iterator returns a random sample of solutions.
    pub fn random_solver(&self, problem: mp::MPProblem) -> mp::solvers::BruteForceSolver {
        let mut mpcs = self.fmp.generate_component_solutions(&problem);
        mpcs.random_shuffle();
        let solver = mp::solvers::BruteForceSolver::new(mpcs, problem.objects.number());
        solver
    }
}

/// Interface for solving the motion planning problem using the **Heuristic** solver approach.
///
/// # Examples
///
/// ```
/// // Define the axis component ranges
/// // comp id, start position, goal position min, goal position max, start velocity, start acceleration
/// let x_param = MPComponent::new("x".to_string(), 0.0, 4.0, 4.0, 3.0, 1.0);
/// let y_param = MPComponent::new("y".to_string(), 1.5, 2.8, 3.2, 3.0, 0.5);
/// let z_param = MPComponent::new("z".to_string(), 1.0, 1.7, 2.3, 1.0, 0.1);
///
/// // Create Objects
/// let mut objects = ObjectList::new();
/// objects.push(Object::new(
///     Bounds::new(3.75, 4.25),
///     Bounds::new(2.3, 2.8),
///     Bounds::new(1.7, 2.3)));
/// objects.push(Object::new(
///     Bounds::new(3.75, 4.25),
///     Bounds::new(3.2, 3.7),
///     Bounds::new(1.7, 2.3)));
///
/// // Create the problem
/// let problem = MPProblem::new(x_param, y_param, z_param, objects);
///
/// // Specify parameters
/// let time_division = 5;
/// let epsilon = 0.01;
/// let planner = HeuristicFastMotionPlanner::new(time_division, epsilon);
/// let solution = planner.solve(problem);
/// ```
pub struct HeuristicFastMotionPlanner {
    fmp: FastMotionPlanner
}

impl HeuristicFastMotionPlanner {
    /// Generate a new instance of `HeuristicFastMotionPlanner`.
    /// - `dt` is the number of samples to divide the 2D goal region into, dt => dt x dt gridding.  e.g. dt = 5, grids goal region into 25.
    /// - `epsilon` is the maximum allowed error in the end goal for a trajectory to be considered.
    pub fn new(dt: usize, epsilon: f64) -> HeuristicFastMotionPlanner {
        HeuristicFastMotionPlanner {
            fmp: FastMotionPlanner::new(dt, epsilon, mp::MPMethod::HEURISTIC)
        }
    }

    /// Generate all possible solutions, and returns it in the form of a [**`MPSolution`**](mp::MPSolution)
    pub fn solve(&self, problem: mp::MPProblem) -> mp::MPSolution {
        self.fmp.solve(problem)
    }

    /// Returns the [**`HeuristicSolver`**](mp::solvers::HeuristicSolver) which implements the [**`MPSolver`**](mp::solvers::MPSolver) trait
    /// - The [**`HeuristicSolver`**](mp::solvers::HeuristicSolver) implements `.into_iter()` which will generate solutions lazily within an iterator.
    pub fn solver(&self, problem: mp::MPProblem) -> mp::solvers::HeuristicSolver {
        let mpcs = self.fmp.generate_component_solutions(&problem);
        let solver = mp::solvers::HeuristicSolver::new(mpcs, problem.objects.number());
        solver
    }

    /// Identical to `solver()`, except that it also randomly shuffles the Axis Component solutions.
    /// This is so that the subsequent iterator returns a random sample of solutions.
    pub fn random_solver(&self, problem: mp::MPProblem) -> mp::solvers::HeuristicSolver {
        let mut mpcs = self.fmp.generate_component_solutions(&problem);
        mpcs.random_shuffle();
        let solver = mp::solvers::HeuristicSolver::new(mpcs, problem.objects.number());
        solver
    }
}

/// Original attempt at fast motion planner where the method was a parameter of the system
/// The hope was that one could return a genertic solver regardless of method.
/// Unfortunately this did not pan out as the Python interface appears to struggle with `Box<dyn mp::solvers::MPSolver>` (and also the fact that the Box also needed to specify Item and IntoIter)
/// However the `generate_component_solutions()` function was still useful, and thus both the above specific solvers use this struct still
struct FastMotionPlanner {
    params: mp::MPParams
}

impl FastMotionPlanner {

    /// Generate a new instance of `FastMotionPlanner`.
    /// - `dt` is the number of samples to divide the 2D goal region into, dt => dt x dt gridding.  e.g. dt = 5, grids goal region into 25.
    /// - `epsilon` is the maximum allowed error in the end goal for a trajectory to be considered.
    /// - `method` defines the method used (Heuristic or bruteforce)
    pub fn new(dt: usize, epsilon: f64, method: mp::MPMethod) -> FastMotionPlanner {
        FastMotionPlanner { params: mp::MPParams{dt, epsilon, method} }
    }


    /// Generate all possible solutions, and returns it in the form of a [**`MPSolution`**](mp::MPSolution)
    pub fn solve(&self, problem: mp::MPProblem) -> mp::MPSolution {
        let mpcs = self.generate_component_solutions(&problem);
        log::info!("Number of solutions: x: {}, y: {}, z: {}", &mpcs.xsol.solutions.len(), &mpcs.ysol.solutions.len(), &mpcs.zsol.solutions.len());
        let sol = match self.params.method {
            // mp::MPMethod::HEURISTIC => self.heuristic_solver(mpcs, problem.objects.number()),
            mp::MPMethod::HEURISTIC => mp::solvers::HeuristicSolver::new(mpcs, problem.objects.number()).solve(),
            mp::MPMethod::BRUTEFORCE => mp::solvers::BruteForceSolver::new(mpcs, problem.objects.number()).solve()
        };
        sol
    }

    /// Given a problem, this method finds the solutions for each component axis of the problem.
    pub fn generate_component_solutions(&self, problem: &mp::MPProblem) -> mp::MPComponentSolutions {
        let xsol = self.comp_find_solution_set(problem.time, &problem.x, &problem.objects.to_object_axis_list("x".to_string()));
        log::info!("XSolutions is {:#?}", xsol);
        let ysol = self.comp_find_solution_set(problem.time, &problem.y, &problem.objects.to_object_axis_list("y".to_string()));
        log::info!("YSolutions is {:#?}", ysol);
        let zsol = self.comp_find_solution_set(problem.time, &problem.z, &problem.objects.to_object_axis_list("z".to_string()));
        log::info!("YSolutions is {:#?}", zsol);
        mp::MPComponentSolutions::new(xsol, ysol, zsol)
    }

    /// This method finds the solutions for each component axis and returns a component solution
    fn comp_find_solution_set(
        &self, time: f64, comp: &mp::MPComponent, objectaxis: &mp::objects::ObjectAxisList
    ) -> mp::MPComponentSolution {
        let css = mp::ss::ComponentSolutionSet::new(time, comp, &self.params);
        let _comp_solutions = css.generate_feasible_solutions();
        mp::MPComponentSolution::from_ss_with_object_avoidance(time, _comp_solutions, objectaxis)
    }
}

#[cfg(test)]
mod tests {

    use ndarray::{array};
    use super::*;
    extern crate ndarray_csv;

    fn use_case_1_objects() -> objects::ObjectList {
        println!("Using Case 1 Objects");
        let mut objects = objects::ObjectList::new();
        objects.push(objects::Object::new(
            objects::Bounds::new(3.75, 4.25),
            objects::Bounds::new(2.3, 2.8),
            objects::Bounds::new(1.7, 2.3)));
        objects.push(objects::Object::new(
            objects::Bounds::new(3.75, 4.25),
            objects::Bounds::new(3.2, 3.7),
            objects::Bounds::new(1.7, 2.3)));
        objects
    }

    #[test]
    fn find_solution_set_1() {
        let x_param = MPComponent::new("x".to_string(), 0.0, 4.0, 4.0, 3.0, 1.0);
        // Get Objects
        let objects = use_case_1_objects();

        let time_division = 5;
        let epsilon = 0.01;
        let method = mp::MPMethod::BRUTEFORCE;

        let solver = FastMotionPlanner::new(time_division, epsilon, method);
        let xsol = solver.comp_find_solution_set(0.0, &x_param, &objects.to_object_axis_list("x".to_string()));
        let x_xi_xg_tg = xsol.into_xi_xg_tg_array();
        let x_sorted = utils::sort_axis_columnwise(x_xi_xg_tg);

        let octave_sol = array![
            [-2.5120,   4.0000,   4.3813],
            [-2.2649,   4.0000,   5.2182],
            [-2.0177,   4.0000,   6.4142],
            [-1.7705,   4.0000,   8.1921],
        ];
        let corr_sorted = utils::sort_axis_columnwise(octave_sol);

        assert!(
            x_sorted.abs_diff_eq(&corr_sorted, 1e-4),
            "Calculated outputs was:\n{:?}\nvs\n{:?}", x_sorted, corr_sorted
        );
    }

    #[test]
    fn find_solution_set_2() {
        let x_param = MPComponent::new("y".to_string(), 1.5, 2.8, 3.2, 3.0, 0.5);
        let objects = use_case_1_objects();

        let time_division = 5;
        let epsilon = 0.01;
        let method = mp::MPMethod::BRUTEFORCE;

        let solver = FastMotionPlanner::new(time_division, epsilon, method);
        let xsol = solver.comp_find_solution_set(0.0, &x_param, &objects.to_object_axis_list("y".to_string()));
        let x_xi_xg_tg = xsol.into_xi_xg_tg_array();
        let x_sorted = utils::sort_axis_columnwise(x_xi_xg_tg);

        let octave_sol = array![
           [0.10544,   3.20000,   1.79673],
           [0.18139,   3.10000,   1.67467],
           [0.25418,   3.20000,   2.06220],
           [0.25810,   3.00000,   1.55344],
           [0.32217,   3.10000,   1.91794],
           [0.33557,   2.90000,   1.43316],
           [0.39082,   3.00000,   1.77504],
           [0.40291,   3.20000,   2.43311],
           [0.41382,   2.80000,   1.31391],
           [0.46012,   2.90000,   1.63361],
           [0.46295,   3.10000,   2.25704],
           [0.52353,   3.00000,   2.08313],
           [0.53008,   2.80000,   1.49380],
           [0.55165,   3.20000,   2.97348],
           [0.58466,   2.90000,   1.91155],
           [0.64634,   2.80000,   1.74247],
           [0.65625,   3.00000,   2.52949],
           [0.70921,   2.90000,   2.31295],
        ];
        let corr_sorted = utils::sort_axis_columnwise(octave_sol);

        assert!(
            x_sorted.abs_diff_eq(&corr_sorted, 1e-4),
            "Calculated outputs was:\n{:?}\nvs\n{:?}", x_sorted, corr_sorted
        );
    }

    #[test]
    fn find_solution_set_3() {
        let x_param = MPComponent::new("z".to_string(), 1.0, 1.7, 2.3, 1.0, 0.1);
        let objects = use_case_1_objects();

        let time_division = 5;
        let epsilon = 0.01;
        let method = mp::MPMethod::BRUTEFORCE;

        let solver = FastMotionPlanner::new(time_division, epsilon, method);
        let xsol = solver.comp_find_solution_set(0.0, &x_param, &objects.to_object_axis_list("z".to_string()));
        let x_xi_xg_tg = xsol.into_xi_xg_tg_array();
        let x_sorted = utils::sort_axis_columnwise(x_xi_xg_tg);

        let octave_sol = array![
            [-0.031962,   2.300000,   3.862181],
            [-0.021390,   2.150000,   3.014549],
            [ 0.075990,   2.150000,   3.350875],
            [ 0.076518,   2.300000,   4.396402],
            [ 0.100909,   2.000000,   2.571362],
            [ 0.173371,   2.150000,   3.797501],
            [ 0.184998,   2.300000,   5.132053],
            [ 0.186793,   2.000000,   2.846303],
            [ 0.226590,   1.850000,   2.133509],
            [ 0.270751,   2.150000,   4.409648],
            [ 0.272677,   2.000000,   3.209630],
            [ 0.300560,   1.850000,   2.350358],
            [ 0.355794,   1.700000,   1.703010],
            [ 0.358561,   2.000000,   3.704884],
            [ 0.368132,   2.150000,   5.281283],
            [ 0.374531,   1.850000,   2.635257],
            [ 0.417414,   1.700000,   1.865589],
            [ 0.448501,   1.850000,   3.021072],
            [ 0.479034,   1.700000,   2.077673],
            [ 0.540653,   1.700000,   2.362571],
            [ 0.602273,   1.700000,   2.759118],
        ];
        let corr_sorted = utils::sort_axis_columnwise(octave_sol);

        assert!(
            x_sorted.abs_diff_eq(&corr_sorted, 1e-4),
            "Calculated outputs was:\n{:?}\nvs\n{:?}", x_sorted, corr_sorted
        );
    }

    #[test]
    fn solver_use_case_1_bruteforce() {
        // Define the axis component ranges
        // comp id, start position, pmin, pmax, start velocity, start acceleration
        let x_param = MPComponent::new("x".to_string(), 0.0, 4.0, 4.0, 3.0, 1.0);
        let y_param = MPComponent::new("y".to_string(), 1.5, 2.8, 3.2, 3.0, 0.5);
        let z_param = MPComponent::new("z".to_string(), 1.0, 1.7, 2.3, 1.0, 0.1);

        // Get Objects
        let objects = use_case_1_objects();

        let problem = MPProblem::new(x_param, y_param, z_param, objects);

        let time_division = 5;
        let epsilon = 0.01;
        let method = mp::MPMethod::BRUTEFORCE;

        let solver = FastMotionPlanner::new(time_division, epsilon, method);
        let solution = solver.solve(problem);

        let correct = utils::read_ndarray_csv("tests/examples/solver_use_case_1.csv".to_string(), (924, 6)).unwrap();
        let correct_sorted = utils::sort_axis_columnwise(correct);

        let out = solution.get_positions_as_array();
        let out_sorted = utils::sort_axis_columnwise(out);

        assert!(
            out_sorted.abs_diff_eq(&correct_sorted, 1e-4),
            "Calculated outputs was:\n{:?}\nvs\n{:?}", out_sorted, correct_sorted
        );
    }

    #[test]
    fn solver_use_case_1_heuristic() {
        // Define the axis component ranges
        // comp id, start position, pmin, pmax, start velocity, start acceleration
        let x_param = MPComponent::new("x".to_string(), 0.0, 4.0, 4.0, 3.0, 1.0);
        let y_param = MPComponent::new("y".to_string(), 1.5, 2.8, 3.2, 3.0, 0.5);
        let z_param = MPComponent::new("z".to_string(), 1.0, 1.7, 2.3, 1.0, 0.1);

        // Get Objects
        let objects = use_case_1_objects();

        let problem = MPProblem::new(x_param, y_param, z_param, objects);

        let time_division = 5;
        let epsilon = 0.01;
        let method = mp::MPMethod::HEURISTIC;

        let solver = FastMotionPlanner::new(time_division, epsilon, method);
        let solution = solver.solve(problem);

        let correct = utils::read_ndarray_csv("tests/examples/solver_use_case_1.csv".to_string(), (924, 6)).unwrap();
        let correct_sorted = utils::sort_axis_columnwise(correct);

        let out = solution.get_positions_as_array();
        let out_sorted = utils::sort_axis_columnwise(out);

        assert!(
            out_sorted.abs_diff_eq(&correct_sorted, 1e-4),
            "Calculated outputs was:\n{:?}\nvs\n{:?}", out_sorted, correct_sorted
        );
    }
}