use std::fs::File;
use std::num::NonZeroU32;

use image::codecs::png::PngEncoder;
use image::io::Reader as ImageReader;
use image::{ColorType, GenericImageView};

use fast_image_resize::{
    CpuExtensions, FilterType, ImageData, PixelType, ResizeAlg, Resizer, SrcImageView,
};

fn get_source_image() -> ImageData<'static> {
    let img = ImageReader::open("./data/nasa-4928x3279.png")
        .unwrap()
        .decode()
        .unwrap();
    let width = img.width();
    let height = img.height();
    ImageData::from_vec_u8(
        NonZeroU32::new(width).unwrap(),
        NonZeroU32::new(height).unwrap(),
        img.to_rgba8().into_raw(),
        PixelType::U8x4,
    )
    .unwrap()
}

fn get_small_source_image() -> ImageData<'static> {
    let img = ImageReader::open("./data/nasa-852x567.png")
        .unwrap()
        .decode()
        .unwrap();
    let width = img.width();
    let height = img.height();
    ImageData::from_vec_u8(
        NonZeroU32::new(width).unwrap(),
        NonZeroU32::new(height).unwrap(),
        img.to_rgba8().into_raw(),
        PixelType::U8x4,
    )
    .unwrap()
}

fn get_new_height(src_image: &SrcImageView, new_width: u32) -> u32 {
    let scale = new_width as f32 / src_image.width().get() as f32;
    (src_image.height().get() as f32 * scale).round() as u32
}

const NEW_WIDTH: u32 = 255;
const NEW_BIG_WIDTH: u32 = 5016;

fn save_result(image: &SrcImageView, name: &str) {
    std::fs::create_dir_all("./data/result").unwrap();
    let mut file = File::create(format!("./data/result/{}.png", name)).unwrap();
    let encoder = PngEncoder::new(&mut file);
    encoder
        .encode(
            &image.get_buffer(),
            image.width().get(),
            image.height().get(),
            ColorType::Rgba8,
        )
        .unwrap();
}

#[test]
fn resize_wo_simd_lanczos3_test() {
    let image = get_source_image();
    let mut resizer = Resizer::new(ResizeAlg::Convolution(FilterType::Lanczos3));
    unsafe {
        resizer.set_cpu_extensions(CpuExtensions::None);
    }
    let new_height = get_new_height(&image.src_view(), NEW_WIDTH);
    let mut result = ImageData::new(
        NonZeroU32::new(NEW_WIDTH).unwrap(),
        NonZeroU32::new(new_height).unwrap(),
        image.pixel_type(),
    );
    resizer.resize(&image.src_view(), &mut result.dst_view());
    save_result(&result.src_view(), "lanczos3_wo_simd");
}

#[test]
fn resize_sse4_lanczos3_test() {
    let image = get_source_image();
    let mut resizer = Resizer::new(ResizeAlg::Convolution(FilterType::Lanczos3));
    unsafe {
        resizer.set_cpu_extensions(CpuExtensions::Sse4_1);
    }
    let new_height = get_new_height(&image.src_view(), NEW_WIDTH);
    let mut result = ImageData::new(
        NonZeroU32::new(NEW_WIDTH).unwrap(),
        NonZeroU32::new(new_height).unwrap(),
        image.pixel_type(),
    );
    resizer.resize(&image.src_view(), &mut result.dst_view());
    save_result(&result.src_view(), "lanczos3_sse4");
}

#[test]
fn resize_avx2_lanczos3_test() {
    let image = get_source_image();
    let mut resizer = Resizer::new(ResizeAlg::Convolution(FilterType::Lanczos3));
    unsafe {
        resizer.set_cpu_extensions(CpuExtensions::Avx2);
    }
    let new_height = get_new_height(&image.src_view(), NEW_WIDTH);
    let mut result = ImageData::new(
        NonZeroU32::new(NEW_WIDTH).unwrap(),
        NonZeroU32::new(new_height).unwrap(),
        image.pixel_type(),
    );
    resizer.resize(&image.src_view(), &mut result.dst_view());
    save_result(&result.src_view(), "lanczos3_avx2");
}

#[test]
fn resize_avx2_lanczos3_upscale_test() {
    let image = get_small_source_image();
    let mut resizer = Resizer::new(ResizeAlg::Convolution(FilterType::Lanczos3));
    unsafe {
        resizer.set_cpu_extensions(CpuExtensions::Avx2);
    }
    let new_height = get_new_height(&image.src_view(), NEW_BIG_WIDTH);
    let mut result = ImageData::new(
        NonZeroU32::new(NEW_BIG_WIDTH).unwrap(),
        NonZeroU32::new(new_height).unwrap(),
        image.pixel_type(),
    );
    resizer.resize(&image.src_view(), &mut result.dst_view());
    save_result(&result.src_view(), "lanczos3_avx2_upscale");
}

#[test]
fn resize_nearest_test() {
    let image = get_source_image();
    let mut resizer = Resizer::new(ResizeAlg::Nearest);
    unsafe {
        resizer.set_cpu_extensions(CpuExtensions::None);
    }
    let new_height = get_new_height(&image.src_view(), NEW_WIDTH);
    let mut result = ImageData::new(
        NonZeroU32::new(NEW_WIDTH).unwrap(),
        NonZeroU32::new(new_height).unwrap(),
        image.pixel_type(),
    );
    resizer.resize(&image.src_view(), &mut result.dst_view());
    save_result(&result.src_view(), "nearest_wo_simd");
}

#[test]
fn resize_super_sampling_test() {
    let image = get_source_image();
    let mut resizer = Resizer::new(ResizeAlg::SuperSampling(FilterType::Lanczos3, 2));
    unsafe {
        resizer.set_cpu_extensions(CpuExtensions::Avx2);
    }
    let new_height = get_new_height(&image.src_view(), NEW_WIDTH);
    let mut result = ImageData::new(
        NonZeroU32::new(NEW_WIDTH).unwrap(),
        NonZeroU32::new(new_height).unwrap(),
        image.pixel_type(),
    );
    resizer.resize(&image.src_view(), &mut result.dst_view());
    save_result(&result.src_view(), "super_sampling_avx2");
}

#[test]
fn resize_with_cropping() {
    let src_image = get_source_image();
    let mut resizer = Resizer::new(ResizeAlg::Convolution(FilterType::Lanczos3));
    unsafe {
        resizer.set_cpu_extensions(CpuExtensions::None);
    }
    let mut dst_image = ImageData::new(
        NonZeroU32::new(1024).unwrap(),
        NonZeroU32::new(256).unwrap(),
        src_image.pixel_type(),
    );
    let mut src_view = src_image.src_view();
    src_view.set_crop_box_to_fit_dst_size(dst_image.width(), dst_image.height(), None);
    resizer.resize(&src_view, &mut dst_image.dst_view());
    save_result(&dst_image.src_view(), "cropping_lanczos3");
}
