use super::*;

mod once {
    use super::*;

    static TEST: &str = "Hello, {{name}}, nice to meet you.";

    #[derive(Debug, Render)]
    struct Replacements {
        name: String,
    }

    #[test]
    fn ok() {
        let found = find(TEST).unwrap();
        let replacements = Replacements {
            name: "Charles".to_owned(),
        };

        let s = found.replace(&replacements);

        assert_eq!(s, "Hello, Charles, nice to meet you.");
    }

    #[test]
    fn err_missing() {
        static TEST: &str = "Hello, Charles, nice to meet you.";

        let e = find::<_, Replacements>(TEST).unwrap_err();

        let expected = Errors {
            inner: vec![Error::Missing("name".to_owned())],
        };

        assert_eq!(e, expected);
    }

    #[test]
    fn err_extra() {
        #[derive(Debug, Render)]
        struct Replacements {}

        let e = find::<_, Replacements>(TEST).unwrap_err();

        let expected = Errors {
            inner: vec![Error::Extra("name".to_owned())],
        };

        assert_eq!(e, expected);
    }

    #[test]
    fn err_wrong() {
        static TEST: &str = "Hello, {{eman}}, nice to meet you.";

        let e = find::<_, Replacements>(TEST).unwrap_err();

        let expected = Errors {
            inner: vec![
                Error::Missing("name".to_owned()),
                Error::Extra("eman".to_owned()),
            ],
        };

        assert_eq!(e, expected);
    }

    #[test]
    fn allow_missing() {
        static TEST: &str = "Hello, Charles, nice to meet you.";

        let found = find_with_mode(TEST, Mode::AllowMissing).unwrap();
        let replacements = Replacements {
            name: "Charles".to_owned(),
        };

        let s = found.replace(&replacements);

        assert_eq!(s, "Hello, Charles, nice to meet you.");
    }
}

mod empty_key {
    use super::*;

    #[derive(Debug, Render)]
    struct Replacements {
        adjective: String,
        // Empty key names currently aren't representable with the derive macro
    }

    #[test]
    fn extra_empty_key() {
        static TEST: &str = "Keys with no name like {{}} are {{adjective}}";

        let e = find::<_, Replacements>(TEST).unwrap_err();

        // This is pretty weird behavior, but it's significantly less cursed
        // than it was before. Aside from creating a specific error case to
        // disallow empty keys, this is probably the best option.
        let expected = Errors {
            inner: vec![Error::Extra("".into())],
        };

        assert_eq!(e, expected);
    }
}

mod escape {
    use super::*;

    #[derive(Debug, Render)]
    struct Replacements {
        action: String,
    }

    #[test]
    fn ok_left() {
        static TEST: &str = "You can {{action}} like this: {{{{}}";

        let found = find(TEST).unwrap();
        let replacements = Replacements {
            action: "escape left double brackets".to_owned(),
        };

        let s = found.replace(&replacements);

        assert_eq!(s, "You can escape left double brackets like this: {{");
    }

    #[test]
    fn ok_right() {
        static TEST: &str = "You can {{action}} like this: {{}}}}";

        let found = find(TEST).unwrap();
        let replacements = Replacements {
            action: "escape right double brackets".to_owned(),
        };

        let s = found.replace(&replacements);

        assert_eq!(s, "You can escape right double brackets like this: }}");
    }

    #[test]
    fn ok_both() {
        static TEST: &str =
            "You can {{action}} like this: {{{{}}key_name{{}}}}";

        let found = find(TEST).unwrap();
        let replacements = Replacements {
            action: "escape both double brackets".to_owned(),
        };

        let s = found.replace(&replacements);

        assert_eq!(
            s,
            "You can escape both double brackets like this: {{key_name}}"
        );
    }
}

mod beginning {
    use super::*;

    static TEST: &str = "{{plural_capitalized_food}} taste good.";

    #[derive(Render)]
    struct Replacements {
        plural_capitalized_food: String,
    }

    #[test]
    fn ok() {
        let replacements = Replacements {
            plural_capitalized_food: "Apples".to_owned(),
        };

        let found = find(TEST).unwrap();
        let s = found.replace(&replacements);

        assert_eq!(s, "Apples taste good.");
    }
}

mod only {
    use super::*;

    static TEST: &str = "{{why}}";

    #[derive(Render)]
    struct Replacements {
        why: String,
    }

    #[test]
    fn ok() {
        let replacements = Replacements {
            why: "would you ever do this".to_owned(),
        };

        let found = find(TEST).unwrap();
        let s = found.replace(&replacements);

        assert_eq!(s, "would you ever do this");
    }
}

mod end {
    use super::*;

    static TEST: &str = "I really love {{something}}";

    #[derive(Render)]
    struct Replacements {
        something: String,
    }

    #[test]
    fn ok() {
        let replacements = Replacements {
            something: "Rust".to_owned(),
        };

        let found = find(TEST).unwrap();
        let s = found.replace(&replacements);

        assert_eq!(s, "I really love Rust");
    }
}

// Dunno why you'd do this either
mod empty {
    use super::*;

    static TEST: &str = "";

    #[derive(Render)]
    struct Replacements {}

    #[test]
    fn ok() {
        let replacements = Replacements {};

        let found = find(TEST).unwrap();
        let s = found.replace(&replacements);

        assert_eq!(s, "");
    }
}

mod two {
    use super::*;

    static TEST: &str = "Hello, {{name}}. You remind me of another {{name}}.";

    #[derive(Render)]
    struct Replacements {
        name: String,
    }

    #[test]
    fn ok() {
        let replacements = Replacements {
            name: "Charles".to_owned(),
        };

        let found = find(TEST).unwrap();
        let s = found.replace(&replacements);

        assert_eq!(s, "Hello, Charles. You remind me of another Charles.");
    }
}

mod twice {
    use super::*;

    static TEST: &str = "{{name}}, why are you writing code at {{time}} again?";

    #[derive(Debug, Render)]
    struct Replacements {
        name: String,
        time: String,
    }

    #[test]
    fn ok() {
        let replacements = Replacements {
            name: "Charles".to_owned(),
            time: "2 AM".to_owned(),
        };

        let found = find(TEST).unwrap();
        let s = found.replace(&replacements);

        assert_eq!(s, "Charles, why are you writing code at 2 AM again?");
    }

    #[test]
    fn err_extra_name() {
        #[derive(Debug, Render)]
        struct Replacements {
            time: String,
        }

        let e = find::<_, Replacements>(TEST).unwrap_err();

        let expected = Errors {
            inner: vec![Error::Extra("name".into())],
        };

        assert_eq!(e, expected);
    }

    #[test]
    fn err_extra_time() {
        #[derive(Debug, Render)]
        struct Replacements {
            name: String,
        }

        let e = find::<_, Replacements>(TEST).unwrap_err();

        let expected = Errors {
            inner: vec![Error::Extra("time".into())],
        };

        assert_eq!(e, expected);
    }

    #[test]
    fn err_extra_both() {
        #[derive(Debug, Render)]
        struct Replacements {}

        let e = find::<_, Replacements>(TEST).unwrap_err();

        let mut errors =
            vec![Error::Extra("name".into()), Error::Extra("time".into())];

        let expected_1 = Errors {
            inner: errors.clone(),
        };

        // Same thing but the other order, since hashmap iterators are random
        errors.swap(0, 1);
        let expected_2 = Errors {
            inner: errors,
        };

        assert!(e == expected_1 || e == expected_2);

        // These will be in the order they show up in the template
        let expected_msg = r#"extraneous keys "name" and "time""#;

        assert_eq!(format!("{}", e), expected_msg);
    }

    #[test]
    fn err_missing_name() {
        static TEST: &str =
            "Charles, why are you writing code at {{time}} again?";

        let e = find::<_, Replacements>(TEST).unwrap_err();

        let expected = Errors {
            inner: vec![Error::Missing("name".into())],
        };

        assert_eq!(e, expected);
    }

    #[test]
    fn err_missing_time() {
        static TEST: &str = "{{name}}, why are you writing code at 2 AM again?";

        let e = find::<_, Replacements>(TEST).unwrap_err();

        let expected = Errors {
            inner: vec![Error::Missing("time".into())],
        };

        assert_eq!(e, expected);
    }

    #[test]
    fn err_missing_both() {
        static TEST: &str = "Charles, why are you writing code at 2 AM again?";

        let e = find::<_, Replacements>(TEST).unwrap_err();

        let mut errors =
            vec![Error::Missing("name".into()), Error::Missing("time".into())];

        let expected_1 = Errors {
            inner: errors.clone(),
        };

        // Same thing but the other order, since hashmap iterators are random
        errors.swap(0, 1);
        let expected_2 = Errors {
            inner: errors,
        };

        assert!(e == expected_1 || e == expected_2);

        // These will be in the order they show up in the template
        let expected_msg = r#"missing keys "name" and "time""#;

        assert_eq!(format!("{}", e), expected_msg);
    }
}

mod extraneous_keys {
    use super::*;

    static TEST: &str = "{{wow}}{{lots}}{{of}}{{keys}}";

    #[derive(Debug, Render)]
    struct Replacements {}

    #[test]
    fn ok() {
        let e = find::<_, Replacements>(TEST).unwrap_err();

        assert_eq!(
            format!("{}", e),
            r#"extraneous keys "wow", "lots", "of", and "keys""#,
        );
    }
}

// This could really either be interpreted as nothing or an attempt to create a
// replace point. I think the latter is more likely to be the expected behavior.
mod unclosed {
    use super::*;

    static TEST: &str = "Hello, {{name";

    #[derive(Debug, Render)]
    struct Replacements {
        name: String,
    }

    #[test]
    fn ok() {
        let e = find::<_, Replacements>(TEST).unwrap_err();

        assert_eq!(
            e,
            Errors {
                inner: vec![Error::Unclosed(9)]
            }
        );
    }
}

// This is sort of a weird way to error about mismatched delimiters, but I guess
// it's workable. I'm open to better ideas though, given a working
// implementation to back up those ideas.
mod mismatched {
    use super::*;

    static TEST: &str = "Hello, {{name. You smell like {{smell}}.";

    #[derive(Debug, Render)]
    struct Replacements {
        name: String,
        smell: String,
    }

    #[test]
    fn ok() {
        let e = find::<_, Replacements>(TEST).unwrap_err();

        let mut errors = vec![
            Error::Missing("smell".into()),
            Error::Missing("name".into()),
            Error::Extra("name. You smell like {{smell".into()),
        ];

        let expected_1 = Errors {
            inner: errors.clone(),
        };

        // `Extra`s are swapped because hashmap iterators are random
        errors.swap(0, 1);
        let expected_2 = Errors {
            inner: errors,
        };

        assert!(e == expected_1 || e == expected_2);
    }
}
