extern crate faf_replay_parser;

use std::fs::File;
use std::io::{BufReader, Read, Seek, SeekFrom};

use faf_replay_parser::scfa::parser::parse_body_ticks;
use faf_replay_parser::scfa::*;
use faf_replay_parser::ReplayReadError;

#[test]
fn default_config() {
    let parser = Parser::new();
    let mut file = BufReader::new(File::open("tests/data/6176549.scfareplay").unwrap());

    parser.parse(&mut file).unwrap();
}

#[test]
fn all_commands() {
    let builder = ParserBuilder::new().commands_all();
    let parser = builder.build_clone();
    let mut stream = builder.build_stream();
    let file = File::open("tests/data/6176549.scfareplay").unwrap();
    let mut reader = BufReader::new(file);

    let replay1 = parser.parse(&mut reader).unwrap();

    assert_eq!(
        replay1.body.commands.last().unwrap(),
        &ReplayCommand::EndGame
    );

    reader.seek(SeekFrom::Start(0)).unwrap();
    let replay2 = {
        while stream.feed_reader(&mut reader, 8192).unwrap() != 0 {
            match stream.parse() {
                Err(ReplayReadError::IO(err)) => {
                    assert_eq!(err.kind(), std::io::ErrorKind::UnexpectedEof)
                }
                res => res.unwrap(),
            }
        }
        assert!(stream.can_finalize());
        stream.finalize().unwrap()
    };

    assert_eq!(
        replay2.body.as_ref().unwrap().commands.last().unwrap(),
        &ReplayCommand::EndGame
    );
    assert_eq!(
        replay1.body.commands.len(),
        replay2.body.unwrap().commands.len()
    );
}

#[test]
fn invalid_1() {
    let parser = ParserBuilder::new().commands_all().build();
    let mut f = BufReader::new(File::open("tests/data/1418712.scfareplay").unwrap());

    match parser.parse(&mut f).unwrap_err() {
        ReplayReadError::IO(e) => assert_eq!(e.kind(), std::io::ErrorKind::UnexpectedEof),
        e => panic!("Wrong error returned {:?}", e),
    }
}

#[test]
fn desynced() {
    let parser = Parser::new();
    let mut f = BufReader::new(File::open("tests/data/8748707-desynced.scfareplay").unwrap());

    match parser.parse(&mut f).unwrap_err() {
        ReplayReadError::Desynced(tick) => assert_eq!(tick, 9105),
        e => panic!("Wrong error returned {:?}", e),
    }
}

#[test]
fn desynced_full_parse() {
    let parser = ParserBuilder::new()
        .commands_default()
        .stop_on_desync(false)
        .build();
    let mut f = BufReader::new(File::open("tests/data/8748707-desynced.scfareplay").unwrap());

    let replay = parser.parse(&mut f).unwrap();

    assert_eq!(replay.body.sim.desync_tick, Some(9105));
    assert_eq!(
        replay.body.sim.desync_ticks,
        Some(vec![
            9105, 9154, 9205, 9254, 9305, 9352, 9405, 9454, 9505, 9554, 9604, 9654, 9705, 9755,
            9805, 9860, 9906, 9967, 10004, 10055, 10104, 10158, 10204, 10255, 10305, 10355, 10404,
            10453, 10504, 10554, 10605, 10652, 10703, 10755, 10806, 10855, 10903, 10955, 11004,
            11054, 11105, 11155, 11206, 11258, 11304, 11353, 11404, 11455, 11504, 11554, 11606,
            11655, 11704, 11754, 11804, 11854, 11905, 11955, 12005, 12055, 12104, 12153, 12205,
            12254, 12304, 12354, 12404, 12455, 12504, 12554, 12604, 12653, 12703, 12752, 12802,
            12854, 12904, 12955, 13004, 13058, 13103, 13155, 13203, 13253, 13304, 13354, 13406,
            13455, 13504, 13554, 13605, 13654, 13704, 13756, 13804, 13855, 13903, 13954, 14005,
            14006, 14006, 14006, 14006, 14006, 14009, 14056, 14104, 14155, 14205, 14253, 14305,
            14353, 14355, 14355, 14355, 14355, 14356, 14357, 14405, 14454, 14503, 14554, 14603,
            14657, 14705, 14753, 14802, 14852, 14904, 14951, 14953, 14953, 14953, 14953, 14953,
            14954, 15005, 15059, 15103, 15155, 15202, 15253, 15304, 15353, 15404, 15452, 15452,
            15452, 15453, 15453, 15453, 15453, 15503, 15554, 15604, 15654, 15703, 15753, 15804,
            15853, 15903, 15952, 16005, 16054, 16103, 16154, 16205, 16254, 16304, 16355, 16405,
            16457, 16504, 16555, 16603, 16656, 16705, 16755, 16805, 16853, 16905, 16961, 17004,
            17055, 17104, 17156, 17202, 17254, 17305, 17354, 17404, 17454, 17505, 17554, 17603,
            17652, 17703, 17753, 17805, 17854, 17903, 17954, 18005, 18052, 18104, 18153, 18205,
            18254, 18304, 18354, 18404, 18456, 18504, 18554, 18603, 18656, 18704, 18752, 18805,
            18854, 18906, 18953, 19002, 19053, 19103, 19154, 19204, 19254, 19304, 19354, 19408,
            19453, 19502, 19555, 19605, 19653, 19653, 19653, 19653, 19653, 19653, 19656, 19705,
            19754, 19804, 19853, 19910, 19958, 20004, 20055, 20105, 20154, 20203, 20258, 20317,
            20354, 20411, 20456, 20505, 20554, 20603, 20655, 20656, 20656, 20656, 20657, 20657,
            20658
        ])
    );
}

#[test]
fn parse_ticks_only() {
    let mut f = BufReader::new(File::open("tests/data/6176549.scfareplay").unwrap());
    let mut data = Vec::new();
    f.read_to_end(&mut data).unwrap();

    // Count the ticks with optimized function
    let start = body_offset(&data).unwrap();
    let fast_ticks = body_ticks(&data[start..]).unwrap();
    let reader_ticks = parse_body_ticks(&mut &data[start..]).unwrap();

    // Count the ticks with a normal parser
    let parser = ParserBuilder::new().commands_all().build();
    let replay = parser.parse(&mut data.as_slice()).unwrap();
    let parser_ticks = replay.body.sim.tick;

    assert_eq!(fast_ticks, 28917);
    assert_eq!(fast_ticks, parser_ticks);
    assert_eq!(fast_ticks, reader_ticks);
}
