pub mod server;
pub mod util;

use std::io::Read;
use std::process::exit;
use std::fs::File;
use std::io::ErrorKind;
use std::path::PathBuf;

fn main() {
	let args: CommandlineArgs = argh::from_env();

	//
	// Step 1: Read the file
	//
	let mut config = match File::open(&args.domains) {
		Ok(file) => file,
		Err(e) => {
			match e.kind() {
				ErrorKind::NotFound => {
					eprintln!("Domains file not found: {}", args.domains.display());
					exit(1001);
				},
				ErrorKind::PermissionDenied => {
					eprintln!(
						"Permission denied to read domain file: {}",
						args.domains.display()
					);
					exit(1002);
				},
				_ => {
					eprintln!(
						"Unexepected error opening domain file: {}",
						args.domains.display()
					);
					exit(1000);
				}
			}
		}
	};
	let mut read_into = String::with_capacity(1_000);
	match config.read_to_string(&mut read_into) {
		Ok(_) => {},
		Err(e) => {
			match e.kind() {
				ErrorKind::BrokenPipe => {
					eprintln!("Broken pipe (filesystem error) while reading {}", args.domains.display());
					exit(1000);
				},
				ErrorKind::InvalidData => {
					eprintln!(
						"Domain file {} contains non-UTF-8 data.  Are you sure this is a domains file?",
						args.domains.display()
					);
					exit(1003);
				}
				_ => {
					eprintln!(
						"Unexepected error opening domain file: {}",
						args.domains.display()
					);
					exit(1000);
				}
			}
		}
	};
	let domains: Vec<_> = read_into.split('\n')
		.filter(|s| !s.is_empty())
		.collect();
	let domains = match util::to_ascii_list(&domains) {
		Ok(d) => d,
		Err((line, e)) => {
			let line = domains[line]; // unwrap safe guaranteed by to_ascii_list()
			let character = line.chars().nth(e.valid_up_to()).unwrap(); // unwrap safe
			eprintln!(
				"The domains file can only contain ASCII characters, but it looks like the domain {line} has a non-ASCII character ({character})"
			);
			exit(1005);
		}
	};

	//
	// Step 2: Run the server
	//
	server::go(&domains, args.port);
}

#[derive(argh::FromArgs)]
/// Run a basic webring hub
struct CommandlineArgs {
    /// a text file listing all domains in the ring, one per line
    #[argh(positional)]
    domains: PathBuf,

    /// the port to bind to (default: 3243)
    #[argh(option, short='p', default="3243")]
    port: u16,
}
