use std::error::Error;
use std::io::{self, BufReader, BufWriter, Read, Write};
use std::path::PathBuf;

use factorio_blueprint::{BlueprintCodec, Container};
use structopt::StructOpt;

use factorio_bitpacker::pack_read;

/// Packs any arbitrary binary blob into a blueprint containing factorio
/// combinators. Uses a dictionary of 256 signals to densely many values into a
/// small space. For inputs which exceed 1KB, a separate set of combinators is
/// created and should be wired separately to avoid interference.
#[derive(Debug, StructOpt)]
#[structopt(verbatim_doc_comment)]
#[structopt(after_help(r#"Copyright © 2021 "micycle".
Licensed GPLv3: GPL version 3 <https://gnu.org/licenses/gpl.html>.
This program is free software: you are free to change and redistribute it under
the terms of the License. There is NO WARRANTY, to the extent permitted by law."#))]
struct Opt {
  /// Binary input file, defaults to stdin if not present
  #[structopt(parse(from_os_str))]
  input: Option<PathBuf>,

  /// Blueprint string output file, defaults to stdout if not present
  #[structopt(parse(from_os_str))]
  output: Option<PathBuf>,
}

fn main() -> Result<(), Box<dyn Error>> {
  let opt = Opt::from_args();

  let input: Box<dyn Read> = match opt.input {
    Some(path) => Box::new(BufReader::new(std::fs::File::open(path)?)),
    None => Box::new(BufReader::new(io::stdin())),
  };

  let output: Box<dyn Write> = match opt.output {
    Some(path) => Box::new(BufWriter::new(std::fs::File::open(path)?)),
    None => Box::new(BufWriter::new(io::stdout())),
  };


  let bp = Container::Blueprint(pack_read(input)?);
  BlueprintCodec::encode(output, &bp)?;

  Ok(())
}
