/// key-point-algorithm from plotters
pub fn plotters(range: std::ops::Range<f64>, max_points: usize) -> Vec<f64> {
    if max_points == 0 {
        return vec![];
    }

    assert!(!(range.start.is_nan() || range.end.is_nan()));

    if (range.start - range.end).abs() < std::f64::EPSILON {
        return vec![range.start as f64];
    }

    let mut scale = (10f64).powf((range.end - range.start).log(10.0).floor());
    let mut digits = -(range.end - range.start).log(10.0).floor() as i32 + 1;
    fn rem_euclid(a: f64, b: f64) -> f64 {
        if b > 0.0 {
            a - (a / b).floor() * b
        } else {
            a - (a / b).ceil() * b
        }
    }

    // At this point we need to make sure that the loop invariant:
    // The scale must yield number of points than requested
    if 1 + ((range.end - range.start) / scale).floor() as usize > max_points {
        scale *= 10.0;
    }

    'outer: loop {
        let old_scale = scale;
        for nxt in [2.0, 5.0, 10.0].iter() {
            let new_left = range.start + scale / nxt - rem_euclid(range.start, scale / nxt);
            let new_right = range.end - rem_euclid(range.end, scale / nxt);

            let npoints = 1 + ((new_right - new_left) / old_scale * nxt) as usize;

            if npoints > max_points {
                break 'outer;
            }

            scale = old_scale / nxt;
        }
        scale = old_scale / 10.0;
        if scale < 1.0 {
            digits += 1;
        }
    }

    let mut ret = vec![];
    let mut left = range.start + scale - rem_euclid(range.start, scale);
    let right = range.end - rem_euclid(range.end, scale);
    while left <= right {
        let size = (10f64).powf(digits as f64 + 1.0);
        let new_left = (left * size).abs() + 1e-3;
        if left < 0.0 {
            left = -new_left.round() / size;
        } else {
            left = new_left.round() / size;
        }
        ret.push(left as f64);
        left += scale;
    }
    return ret;
}
