/// reference: https://rdrr.io/cran/labeling/src/R/labeling.R
pub mod heckbert;
// pub mod matplotlib;
pub mod plotters;
pub mod talbot;
pub mod wilkinson;

pub use heckbert::heckbert;
// pub use matplotlib::matplotlib;
pub use plotters::plotters;
pub use talbot::talbot;
// pub use wilkinson::wilkinson;

/// the size of a `range` should be an integer multiple of `step`.
///
/// ```
/// assert_eq!(linspace(0.0..10.0, 2.5), vec![0., 2.5, 5., 7.5, 10.]);
/// ```
pub fn linspace(range: std::ops::Range<f64>, step: f64) -> Vec<f64> {
    let n = ((range.end - range.start) / step).round() as i32;
    (0..=n).map(|i| range.start + step * i as f64).collect()
}

pub enum LabelingAlgorithm {
    Heckbert,
    Plotters,
    Talbot,
}

impl LabelingAlgorithm {
    pub fn locate(&self, range: std::ops::Range<f64>, num_labels: usize) -> Vec<f64> {
        match self {
            LabelingAlgorithm::Heckbert => heckbert(range, num_labels),
            LabelingAlgorithm::Plotters => plotters(range, num_labels),
            LabelingAlgorithm::Talbot => talbot(range, num_labels),
        }
    }
}

impl Default for LabelingAlgorithm {
    fn default() -> Self {
        Self::Talbot
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    #[test]
    fn it_works() {
        assert_eq!(linspace(0.0..10.0, 2.5), vec![0., 2.5, 5., 7.5, 10.]);
    }
}
