use piet::RenderContext;

use super::camera2::Camera2;

pub mod line;
pub mod scatter;

pub enum Geometry {
    Scatter(scatter::Scatter),
    Line(line::Line),
}

impl Geometry {
    pub fn render<C: RenderContext>(&self, ctx: &mut C, camera: &Camera2) {
        match self {
            Geometry::Scatter(x) => x.render(ctx, camera),
            Geometry::Line(x) => x.render(ctx, camera),
        }
    }

    pub fn bounding_box(&self) -> Option<BoundingBox> {
        match self {
            Geometry::Scatter(x) => x.bounding_box(),
            Geometry::Line(x) => x.bounding_box(),
        }
    }
}

#[derive(Default, Debug)]
pub struct BoundingBox {
    pub min_x: f64,
    pub min_y: f64,
    pub max_x: f64,
    pub max_y: f64,
}

impl BoundingBox {
    pub fn merge(&mut self, other: &Self) {
        if other.min_x < self.min_x {
            self.min_x = other.min_x;
        }
        if other.max_x > self.max_x {
            self.max_x = other.max_x;
        }
        if other.min_y < self.min_y {
            self.min_y = other.min_y;
        }
        if other.max_y > self.max_y {
            self.max_y = other.max_y;
        }
    }

    pub fn x_range(&self) -> std::ops::Range<f64> {
        self.min_x..self.max_x
    }
    pub fn y_range(&self) -> std::ops::Range<f64> {
        self.min_y..self.max_y
    }
    pub fn width(&self) -> f64 {
        self.max_x - self.min_x
    }
    pub fn height(&self) -> f64 {
        self.max_y - self.min_y
    }
}
