use piet::{
    kurbo::{self},
    Color, RenderContext,
};
use polars::prelude::DataFrame;

use crate::{
    cartesian::camera2::Camera2,
    data::{ConstOrColumn, ConstOrColumnTrait},
};

use super::BoundingBox;

pub struct Line {
    pub df: DataFrame,
    pub x: ConstOrColumn<f64>,
    pub y: ConstOrColumn<f64>,
}

impl Line {
    pub fn get(&self, idx: usize) -> (f64, f64) {
        (self.x.get(&self.df, idx), self.y.get(&self.df, idx))
    }
    // using a single BezPath is very slow
    pub fn render<C: RenderContext>(&self, ctx: &mut C, camera: &Camera2) {
        let n = self.df.height();
        let mut p0 = camera.map(self.get(0));
        for i in 1..n {
            let p1 = camera.map(self.get(i));
            ctx.stroke(kurbo::Line::new(p0, p1), &Color::BLUE, 1.0);
            p0 = p1;
        }
    }

    pub fn bounding_box(&self) -> Option<BoundingBox> {
        let x_range = self.x.min_max(&self.df)?;
        let y_range = self.y.min_max(&self.df)?;

        Some(BoundingBox {
            min_x: x_range.0,
            max_x: x_range.1,
            min_y: y_range.0,
            max_y: y_range.1,
        })
    }
}
