//! Represents an object in PHP. Allows for overriding the internal object used by classes,
//! allowing users to store Rust data inside a PHP object.

use std::{
    fmt::Debug,
    mem,
    ops::{Deref, DerefMut},
};

use crate::{
    bindings::{
        ext_php_rs_zend_object_alloc, ext_php_rs_zend_object_std_init, std_object_handlers,
        zend_object, zend_object_handlers,
    },
    php::{class::ClassEntry, execution_data::ExecutionData},
};

pub type ZendObject = zend_object;
pub type ZendObjectHandlers = zend_object_handlers;

/// Implemented by the [`ZendObjectHandler`](ext_php_rs_derive::ZendObjectHandler) macro on a type T
/// which is used as the T type for [`ZendClassObject`].
/// Implements a function `create_object` which is passed to a PHP class entry to instantiate the
/// object that will represent an object.
pub trait ZendObjectOverride {
    /// Creates a new Zend object. Also allocates space for type T on which the trait is
    /// implemented on.
    ///
    /// # Parameters
    ///
    /// * `ce` - The class entry that we are creating an object for.
    extern "C" fn create_object(ce: *mut ClassEntry) -> *mut ZendObject;
}

/// A Zend class object which is allocated when a PHP
/// class object is instantiated. Overrides the default
/// handler when the user provides a type T of the struct
/// they want to override with.
#[repr(C)]
pub struct ZendClassObject<T: Default> {
    obj: T,
    std: zend_object,
}

impl<T: Default> ZendClassObject<T> {
    /// Allocates a new object when an instance of the class is created in the PHP world.
    ///
    /// Internal function. The end user functions are generated by the
    /// [`ZendObjectHandler`](ext_php_rs_derive::ZendObjectHandler) derive macro which generates a
    /// function that wraps this function to be exported to C.
    ///
    /// # Parameters
    ///
    /// * `ce` - The class entry that was created.
    /// * `handlers` - A pointer to the object handlers for the class.
    ///
    /// # Safety
    ///
    /// This function is an internal function which is only called from code which is derived using
    /// the [`ZendObjectHandler`](ext_php_rs_derive::ZendObjectHandler) derive macro. PHP will
    /// guarantee that any pointers given to this function will be valid, therefore we can Unwrap
    /// them with safety.
    pub unsafe fn new_ptr(
        ce: *mut ClassEntry,
        handlers: *mut ZendObjectHandlers,
    ) -> *mut zend_object {
        let obj = {
            let obj = (ext_php_rs_zend_object_alloc(std::mem::size_of::<Self>() as _, ce)
                as *mut Self)
                .as_mut()
                .unwrap();

            ext_php_rs_zend_object_std_init(&mut obj.std, ce);
            obj
        };

        obj.obj = T::default();
        obj.std.handlers = handlers;
        &mut obj.std
    }

    /// Attempts to retrieve the zend class object container from the
    /// zend object contained in the execution data of a function.
    ///
    /// # Parameters
    ///
    /// * `ex` - The execution data of the function.
    pub fn get(ex: &ExecutionData) -> Option<&'static mut Self> {
        // cast to u8 to work in terms of bytes
        let ptr = ex.This.object()? as *mut u8;
        let offset = std::mem::size_of::<T>();
        unsafe {
            let ptr = ptr.offset(0 - offset as isize);
            (ptr as *mut Self).as_mut()
        }
    }
}

impl<T: Default + Debug> Debug for ZendClassObject<T> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.obj.fmt(f)
    }
}

impl<T: Default> Deref for ZendClassObject<T> {
    type Target = T;

    fn deref(&self) -> &Self::Target {
        &self.obj
    }
}

impl<T: Default> DerefMut for ZendClassObject<T> {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.obj
    }
}

impl ZendObjectHandlers {
    /// Creates a new set of object handlers from the standard object handlers, returning a pointer
    /// to the handlers.
    pub fn init<T>() -> *mut ZendObjectHandlers {
        // SAFETY: We are allocating memory for the handlers ourselves, which ensures that
        // we can copy to the allocated memory. We can also copy from the standard handlers
        // as the `std_object_handlers` are not modified.
        unsafe {
            let s = mem::size_of::<Self>();
            let ptr = libc::malloc(s) as *mut Self;
            libc::memcpy(
                ptr as *mut _,
                (&std_object_handlers as *const Self) as *mut _,
                s,
            );
            let offset = mem::size_of::<T>();
            (*ptr).offset = offset as i32;
            ptr
        }
    }
}
