#![warn(missing_docs)]
//! Run some tests for the functions implemented by the `expect_exit` crate.
//! A separate program is needed because the tests involve the process
//! exiting, not merely panicking, which cannot be tested by
//! the standard toolset.

/**
 * Copyright (c) 2020, 2021  Peter Pentchev <roam@ringlet.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
use std::env;
use std::error;
use std::io::Read;
use std::process;

mod lib;

use lib::Expected;
use lib::ExpectedResult;
use lib::ExpectedWithError;

struct UnwindTest {
    value: u32,
}

impl Drop for UnwindTest {
    fn drop(&mut self) {
        println!("UnwindTest.drop() invoked for {}", self.value);
    }
}

#[derive(Debug)]
enum OpMode {
    RNUOk,
    RNUFail,
    RUOk,
    RUFail,
    RFNUOk,
    RFNUFail,
    RFUOk,
    RFUFail,
    ONUOk,
    ONUFail,
    OUOk,
    OUFail,
    OFNUOk,
    OFNUFail,
    OFUOk,
    OFUFail,
    RENBOk,
    RENBFail,
    REBOk,
    REBFail,
    REFNBOk,
    REFNBFail,
    REFBOk,
    REFBFail,
    OENBOk,
    OENBFail,
    OEBOk,
    OEBFail,
    OEFNBOk,
    OEFNBFail,
    OEFBOk,
    OEFBFail,
    RunTest,
}

struct TestCase<'a> {
    arg: &'a str,
    stdout: Vec<&'a str>,
    stderr: Vec<&'a str>,
    result: i32,
}

fn return_ok() -> Result<(), String> {
    Ok(())
}

fn return_err() -> Result<(), String> {
    Err(String::from("this error message should be displayed"))
}

fn o_return_some() -> Option<()> {
    Some(())
}

fn o_return_none() -> Option<()> {
    None
}

fn e_res_handle(success: bool) -> Result<(), String> {
    match success {
        false => Err("oof".to_owned()),
        true => Ok(()),
    }
}

fn e_opt_handle(success: bool) -> Option<u32> {
    match success {
        false => None,
        true => Some(0),
    }
}

fn e_res_nb(success: bool) -> Result<(), lib::ExpectationFailed> {
    e_res_handle(success).expect_result_nb("this error message should be displayed")
}

fn e_res(success: bool) -> Result<(), Box<dyn error::Error>> {
    e_res_handle(success).expect_result("this error message should be displayed")
}

fn e_opt_nb(success: bool) -> Result<(), lib::ExpectationFailed> {
    let res = e_opt_handle(success).expect_result_nb("this error message should be displayed")?;
    if res != 0 {
        panic!("e_opt_handle() did not return 0");
    }
    Ok(())
}

fn e_opt(success: bool) -> Result<(), Box<dyn error::Error>> {
    let res = e_opt_handle(success).expect_result("this error message should be displayed")?;
    if res != 0 {
        panic!("e_opt_handle() did not return 0");
    }
    Ok(())
}

fn usage() -> ! {
    lib::die("Usage: expect-exit nu_ok|nu_fail|ok|fail");
}

fn parse_args() -> (String, OpMode) {
    let args: Vec<String> = env::args().collect();

    (
        String::from(&args[0]),
        match args.len() {
            2 => match args[1].as_str() {
                "nu_ok" => OpMode::RNUOk,
                "nu_fail" => OpMode::RNUFail,
                "ok" => OpMode::RUOk,
                "fail" => OpMode::RUFail,
                "f_nu_ok" => OpMode::RFNUOk,
                "f_nu_fail" => OpMode::RFNUFail,
                "f_ok" => OpMode::RFUOk,
                "f_fail" => OpMode::RFUFail,
                "o_nu_ok" => OpMode::ONUOk,
                "o_nu_fail" => OpMode::ONUFail,
                "o_ok" => OpMode::OUOk,
                "o_fail" => OpMode::OUFail,
                "o_f_nu_ok" => OpMode::OFNUOk,
                "o_f_nu_fail" => OpMode::OFNUFail,
                "o_f_ok" => OpMode::OFUOk,
                "o_f_fail" => OpMode::OFUFail,
                "e_nb_ok" => OpMode::RENBOk,
                "e_nb_fail" => OpMode::RENBFail,
                "e_b_ok" => OpMode::REBOk,
                "e_b_fail" => OpMode::REBFail,
                "e_f_nb_ok" => OpMode::REFNBOk,
                "e_f_nb_fail" => OpMode::REFNBFail,
                "e_f_b_ok" => OpMode::REFBOk,
                "e_f_b_fail" => OpMode::REFBFail,
                "o_e_nb_ok" => OpMode::OENBOk,
                "o_e_nb_fail" => OpMode::OENBFail,
                "o_e_b_ok" => OpMode::OEBOk,
                "o_e_b_fail" => OpMode::OEBFail,
                "o_e_f_nb_ok" => OpMode::OEFNBOk,
                "o_e_f_nb_fail" => OpMode::OEFNBFail,
                "o_e_f_b_ok" => OpMode::OEFBOk,
                "o_e_f_b_fail" => OpMode::OEFBFail,
                "test" => OpMode::RunTest,
                _ => usage(),
            },

            _ => usage(),
        },
    )
}

fn check_output_contains(expected: &[&str], actual: &[&str]) -> bool {
    match expected.len() == actual.len() {
        false => false,
        true => expected
            .iter()
            .zip(actual.iter())
            .all(|(&exp, &act)| act.contains(exp)),
    }
}

fn check_test_output(case: &TestCase, stdout_lines: &[&str], stderr_lines: &[&str]) -> bool {
    check_output_contains(&case.stdout, stdout_lines)
        && check_output_contains(&case.stderr, stderr_lines)
}

fn run_test(prog: &str) {
    let exp_first = "with the value 616";
    let exp_last = "the end";
    let exp_drop = "UnwindTest.drop";
    let exp_error = "should be displayed";
    let exp_trig = "should be triggered";
    let exp_fmt_cb = "format callback";
    let exp_fmt_res = "something about 616";

    let test_cases: Vec<TestCase> = vec![
        TestCase {
            arg: "oof",
            stdout: vec![],
            stderr: vec!["Usage: "],
            result: 1,
        },
        TestCase {
            arg: "nu_ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "nu_fail",
            stdout: vec![exp_first],
            stderr: vec![exp_error],
            result: 1,
        },
        TestCase {
            arg: "f_nu_ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "f_nu_fail",
            stdout: vec![exp_first, exp_fmt_cb],
            stderr: vec![exp_error],
            result: 1,
        },
        TestCase {
            arg: "ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "fail",
            stdout: vec![exp_first, exp_drop],
            stderr: vec![exp_error],
            result: 1,
        },
        TestCase {
            arg: "f_ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "f_fail",
            stdout: vec![exp_first, exp_fmt_cb, exp_drop],
            stderr: vec![exp_error],
            result: 1,
        },
        TestCase {
            arg: "o_nu_ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "o_nu_fail",
            stdout: vec![exp_first],
            stderr: vec![exp_trig],
            result: 1,
        },
        TestCase {
            arg: "o_ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "o_fail",
            stdout: vec![exp_first, exp_drop],
            stderr: vec![exp_trig],
            result: 1,
        },
        TestCase {
            arg: "o_f_nu_ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "o_f_nu_fail",
            stdout: vec![exp_first, exp_fmt_cb],
            stderr: vec![exp_fmt_res],
            result: 1,
        },
        TestCase {
            arg: "o_f_ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "o_f_fail",
            stdout: vec![exp_first, exp_fmt_cb, exp_drop],
            stderr: vec![exp_fmt_res],
            result: 1,
        },
        TestCase {
            arg: "e_nb_ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "e_nb_fail",
            stdout: vec![exp_first],
            stderr: vec![exp_trig],
            result: 1,
        },
        TestCase {
            arg: "e_b_ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "e_b_fail",
            stdout: vec![exp_first],
            stderr: vec![exp_trig],
            result: 1,
        },
        TestCase {
            arg: "e_f_nb_ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "e_f_nb_fail",
            stdout: vec![exp_first, exp_fmt_cb],
            stderr: vec![exp_fmt_res],
            result: 1,
        },
        TestCase {
            arg: "e_f_b_ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "e_f_b_fail",
            stdout: vec![exp_first, exp_fmt_cb],
            stderr: vec![exp_fmt_res],
            result: 1,
        },
        TestCase {
            arg: "o_e_nb_ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "o_e_nb_fail",
            stdout: vec![exp_first],
            stderr: vec![exp_trig],
            result: 1,
        },
        TestCase {
            arg: "o_e_b_ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "o_e_b_fail",
            stdout: vec![exp_first],
            stderr: vec![exp_trig],
            result: 1,
        },
        TestCase {
            arg: "o_e_f_nb_ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "o_e_f_nb_fail",
            stdout: vec![exp_first, exp_fmt_cb],
            stderr: vec![exp_fmt_res],
            result: 1,
        },
        TestCase {
            arg: "o_e_f_b_ok",
            stdout: vec![exp_first, exp_last, exp_drop],
            stderr: vec![],
            result: 0,
        },
        TestCase {
            arg: "o_e_f_b_fail",
            stdout: vec![exp_first, exp_fmt_cb],
            stderr: vec![exp_fmt_res],
            result: 1,
        },
    ];

    println!("Running a test on {}", prog);

    for case in test_cases {
        let desc = &format!("'{} {}'", prog, case.arg);
        println!(
            "\n=====\nrun {} stdout {:?} stderr {:?}",
            desc, case.stdout, case.stderr
        );

        let mut proc = process::Command::new(prog)
            .arg(case.arg)
            .stdout(process::Stdio::piped())
            .stderr(process::Stdio::piped())
            .spawn()
            .unwrap_or_else(|err| panic!("Could not run {}: {}", desc, err));

        let mut stdout_text = String::new();
        proc.stdout
            .take()
            .unwrap_or_else(|| panic!("Could not open the standard output stream of {}", desc))
            .read_to_string(&mut stdout_text)
            .unwrap_or_else(|err| panic!("Could not read the output of {}: {}", desc, err));
        let stdout_lines: Vec<&str> = stdout_text.split_terminator('\n').collect();

        let mut stderr_text = String::new();
        proc.stderr
            .take()
            .unwrap_or_else(|| panic!("Could not open the standard error stream of {}", desc))
            .read_to_string(&mut stderr_text)
            .unwrap_or_else(|err| panic!("Could not read the error output of {}: {}", desc, err));
        let stderr_lines: Vec<&str> = stderr_text.split_terminator('\n').collect();

        println!(
            "Got {} line(s) of output, {} line(s) of error output.",
            stdout_lines.len(),
            stderr_lines.len()
        );
        if !check_test_output(&case, &stdout_lines, &stderr_lines) {
            panic!(
                "Output mismatch for {}: ({:?}, {:?}) vs ({:?}, {:?})",
                desc, case.stdout, case.stderr, stdout_lines, stderr_lines
            );
        }

        let res = proc
            .wait()
            .unwrap_or_else(|err| panic!("Could not wait for {}: {}", desc, err))
            .code()
            .unwrap_or_else(|| panic!("Could not fetch the exit code for {}", desc));
        match res == case.result {
            true => println!("\n{} exited with the expected code {}", desc, case.result),
            false => panic!(
                "{} exited with code {}, expected {}",
                desc, res, case.result
            ),
        };
    }

    println!("\n=====\nThe self-test seems to have run successfully.");
}

fn handle(mode: OpMode) {
    let val = UnwindTest { value: 616 };
    println!("Created a test variable with the value {}", val.value);

    let cb = || {
        println!("The format callback was invoked.");
        format!("something about {}", val.value)
    };

    match mode {
        OpMode::RNUOk => return_ok().or_die_e("This should not be triggered"),
        OpMode::RNUFail => return_err().or_die_e("This should be triggered"),
        OpMode::RUOk => return_ok().or_exit_e("This should not be triggered"),
        OpMode::RUFail => return_err().or_exit_e("This should be triggered"),
        OpMode::RFNUOk => return_ok().or_die_e_f(cb),
        OpMode::RFNUFail => return_err().or_die_e_f(cb),
        OpMode::RFUOk => return_ok().or_exit_e_f(cb),
        OpMode::RFUFail => return_err().or_exit_e_f(cb),
        OpMode::ONUOk => o_return_some().or_die("This should not be triggered"),
        OpMode::ONUFail => o_return_none().or_die("This should be triggered"),
        OpMode::OUOk => o_return_some().or_exit("This should not be triggered"),
        OpMode::OUFail => o_return_none().or_exit("This should be triggered"),
        OpMode::OFNUOk => o_return_some().or_die_f(cb),
        OpMode::OFNUFail => o_return_none().or_die_f(cb),
        OpMode::OFUOk => o_return_some().or_exit_f(cb),
        OpMode::OFUFail => o_return_none().or_exit_f(cb),
        OpMode::RENBOk => e_res_nb(true).or_die_e("This should not be triggered"),
        OpMode::RENBFail => e_res_nb(false).or_die_e("This should be triggered"),
        OpMode::REBOk => e_res(true).or_die_e("This should not be triggered"),
        OpMode::REBFail => e_res(false).or_die_e("This should be triggered"),
        OpMode::REFNBOk => e_res_nb(true).or_die_e_f(cb),
        OpMode::REFNBFail => e_res_nb(false).or_die_e_f(cb),
        OpMode::REFBOk => e_res(true).or_die_e_f(cb),
        OpMode::REFBFail => e_res(false).or_die_e_f(cb),
        OpMode::OENBOk => e_opt_nb(true).or_die_e("This should not be triggered"),
        OpMode::OENBFail => e_opt_nb(false).or_die_e("This should be triggered"),
        OpMode::OEBOk => e_opt(true).or_die_e("This should not be triggered"),
        OpMode::OEBFail => e_opt(false).or_die_e("This should be triggered"),
        OpMode::OEFNBOk => e_opt_nb(true).or_die_e_f(cb),
        OpMode::OEFNBFail => e_opt_nb(false).or_die_e_f(cb),
        OpMode::OEFBOk => e_opt(true).or_die_e_f(cb),
        OpMode::OEFBFail => e_opt(false).or_die_e_f(cb),
        _ => panic!("How did we get here with mode {:?}?", mode),
    };

    println!("This is the end.");
}

fn main() {
    let (prog, mode) = parse_args();
    match mode {
        OpMode::RunTest => run_test(&prog),
        _ => handle(mode),
    };
}
