#![warn(missing_docs)]
//! Display an error message and exit without a panic.
//!
//! The `expect-exit` library defines the [`Expected`], [`ExpectedWithError`],
//! and [`ExpectedResult`] traits and implements them for the standard
//! [`Result`][`std::result::Result`] and [`Option`][`std::option::Option`]
//! types as appropriate. This allows a program to display an error message
//! and exit with a non-zero exit code without invoking a Rust panic, yet
//! optionally unwinding the stack so that various objects may perform some
//! clean-up actions.
//!
//! The methods with an `_e` suffix append an appropriate error message to
//! the supplied one. The methods with an `_f` suffix allow the caller to
//! only build the error message if an error is to be reported, similar to
//! the [`Option.or_else`][`std::option::Option::or_else`] method.
//!
//! ```
//! use expect_exit::{Expected, ExpectedResult};
//!
//! {
//!     env::var(name).or_exit_f(|| format!("{} not specified in the environment", name))
//!
//!     fs::read_to_string(path).or_exit_e_f(|| format!("Could not read {:?}", path))
//!
//!     tx.send(result).await.or_exit_e("Could not tell the main thread");
//!
//!     let config = parse().expect_result("Could not parse the config")?;
//!     Ok(config.value + 1)
//! }
//! ```
//!
//! The traits are currently implemented for the standard
//! [`Option`][`std::option::Option`] and
//! [`Result`][`std::result::Result`] types as appropriate.

/**
 * Copyright (c) 2020, 2021  Peter Pentchev <roam@ringlet.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
use std::error;
use std::fmt;
use std::panic;
use std::process;

struct ExitHelper {
    code: i32,
}

impl Drop for ExitHelper {
    fn drop(&mut self) {
        process::exit(self.code);
    }
}

/// Unwind the stack and end the process with the specified exit code.
pub fn exit_unwind(code: i32) -> ! {
    panic::resume_unwind(Box::new(ExitHelper { code }));
}

fn _die(msg: &str, exit: fn(i32) -> !) -> ! {
    eprintln!("{}", msg);
    exit(1);
}

fn _die_perror(msg: &str, err: impl fmt::Display, exit: fn(i32) -> !) -> ! {
    eprintln!("{}: {}", msg, err);
    exit(1);
}

/// Display the specified message, then unwind the stack and exit.
pub fn exit(msg: &str) -> ! {
    _die(msg, exit_unwind);
}

/// Display the specified message and append an appropriate
/// description of the error, then unwind the stack and exit.
pub fn exit_perror(msg: &str, err: impl fmt::Display) -> ! {
    _die_perror(msg, err, exit_unwind);
}

/// Display the specified message, then exit without unwinding
/// the stack.
pub fn die(msg: &str) -> ! {
    _die(msg, process::exit);
}

/// Display the specified message and append an appropriate
/// description of the error, then exit without unwinding
/// the stack.
pub fn die_perror(msg: &str, err: impl fmt::Display) -> ! {
    _die_perror(msg, err, process::exit);
}

/// The error object returned by the [`ExpectedResult`] methods.
#[derive(Debug)]
pub struct ExpectationFailed {
    message: String,
}

impl fmt::Display for ExpectationFailed {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{}", self.message)
    }
}

impl error::Error for ExpectationFailed {}

/// Unwrap or exit with the specified message.
pub trait Expected<T> {
    /// Test the value. On success, return the unwrapped value.
    /// On error, unwind the stack, display the specified message,
    /// and exit.
    fn or_exit(self, msg: &str) -> T;

    /// Test the value. On success, return the unwrapped value.
    /// On error, unwind the stack, call the supplied function to
    /// obtain an error message, display it,
    /// and exit.
    fn or_exit_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String;

    /// Test the value. On success, return the unwrapped value.
    /// On error, do not unwind the stack, display the specified message,
    /// and exit.
    fn or_die(self, msg: &str) -> T;

    /// Test the value. On success, return the unwrapped value.
    /// On error, do not unwind the stack, call the supplied function to
    /// obtain an error message, display it,
    /// and exit.
    fn or_die_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String;

    /// Alias for [`Expected::or_exit`].
    fn expect_or_exit(self, msg: &str) -> T;

    /// Alias for [`Expected::or_exit_f`].
    fn expect_or_exit_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String;

    /// Alias for [`Expected::or_die`].
    fn expect_or_die(self, msg: &str) -> T;

    /// Alias for [`Expected::or_die_f`].
    fn expect_or_die_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String;
}

/// Unwrap or exit with an appropriate error message.
pub trait ExpectedWithError<T>: Expected<T> {
    /// Test the value. On success, return the unwrapped value.
    /// On error, unwind the stack, display the specified message,
    /// append an appropriate description of the error, and exit.
    fn or_exit_e(self, msg: &str) -> T;

    /// Test the value. On success, return the unwrapped value.
    /// On error, unwind the stack, call the supplied function to
    /// obtain an error message, display it,
    /// append an appropriate description of the error, and exit.
    fn or_exit_e_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String;

    /// Test the value. On success, return the unwrapped value.
    /// On error, do not unwind the stack, display the specified message,
    /// append an appropriate description of the error, and exit.
    fn or_die_e(self, msg: &str) -> T;

    /// Test the value. On success, return the unwrapped value.
    /// On error, do not unwind the stack, call the supplied function to
    /// obtain an error message, display it,
    /// append an appropriate description of the error, and exit.
    fn or_die_e_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String;

    /// Alias for [`ExpectedWithError::or_exit_e`].
    fn expect_or_exit_perror(self, msg: &str) -> T;

    /// Alias for [`ExpectedWithError::or_exit_e_f`].
    fn expect_or_exit_perror_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String;

    /// Alias for [`ExpectedWithError::or_die_e`].
    fn expect_or_die_perror(self, msg: &str) -> T;

    /// Alias for [`ExpectedWithError::or_die_e_f`].
    fn expect_or_die_perror_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String;
}

/// Test the value and return a result object containing either
/// the inner value or an error object that, when displayed, will provide
/// the specified error message.
pub trait ExpectedResult<T> {
    /// Return a result object with a non-boxed `Error` object.
    fn expect_result_nb(self, msg: &str) -> Result<T, ExpectationFailed>;

    /// Return a result object that may be tested using the `?` operator.
    fn expect_result(self, msg: &str) -> Result<T, Box<dyn error::Error>>;

    /// Return a result object with a non-boxed `Error` object.
    /// Invoke the specified function to obtain the error message.
    fn expect_result_nb_f<F>(self, f: F) -> Result<T, ExpectationFailed>
    where
        F: FnOnce() -> String;

    /// Return a result object that may be tested using the `?` operator.
    /// Invoke the specified function to obtain the error message.
    fn expect_result_f<F>(self, f: F) -> Result<T, Box<dyn error::Error>>
    where
        F: FnOnce() -> String;
}

impl<T> Expected<T> for Option<T> {
    fn or_exit(self, msg: &str) -> T {
        match self {
            Some(value) => value,
            None => exit(msg),
        }
    }

    fn or_exit_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String,
    {
        match self {
            Some(value) => value,
            None => exit(&f()),
        }
    }

    fn expect_or_exit(self, msg: &str) -> T {
        self.or_exit(msg)
    }

    fn expect_or_exit_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String,
    {
        self.or_exit_f(f)
    }

    fn or_die(self, msg: &str) -> T {
        match self {
            Some(value) => value,
            None => die(msg),
        }
    }

    fn or_die_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String,
    {
        match self {
            Some(value) => value,
            None => die(&f()),
        }
    }

    fn expect_or_die(self, msg: &str) -> T {
        self.or_die(msg)
    }

    fn expect_or_die_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String,
    {
        self.or_die_f(f)
    }
}

impl<T, E> Expected<T> for Result<T, E>
where
    E: fmt::Display,
{
    fn or_exit(self, msg: &str) -> T {
        match self {
            Ok(value) => value,
            Err(_) => exit(msg),
        }
    }

    fn or_exit_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String,
    {
        match self {
            Ok(value) => value,
            Err(_) => exit(&f()),
        }
    }

    fn expect_or_exit(self, msg: &str) -> T {
        self.or_exit(msg)
    }

    fn expect_or_exit_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String,
    {
        self.or_exit_f(f)
    }

    fn or_die(self, msg: &str) -> T {
        match self {
            Ok(value) => value,
            Err(_) => die(msg),
        }
    }

    fn or_die_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String,
    {
        match self {
            Ok(value) => value,
            Err(_) => die(&f()),
        }
    }

    fn expect_or_die(self, msg: &str) -> T {
        self.or_die(msg)
    }

    fn expect_or_die_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String,
    {
        self.or_die_f(f)
    }
}

impl<T, E> ExpectedWithError<T> for Result<T, E>
where
    E: fmt::Display,
{
    fn or_exit_e(self, msg: &str) -> T {
        match self {
            Err(err) => exit_perror(msg, err),
            Ok(value) => value,
        }
    }

    fn expect_or_exit_perror(self, msg: &str) -> T {
        self.or_exit_e(msg)
    }

    fn or_exit_e_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String,
    {
        match self {
            Err(err) => exit_perror(&f(), err),
            Ok(value) => value,
        }
    }

    fn expect_or_exit_perror_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String,
    {
        self.or_exit_e_f(f)
    }

    fn or_die_e(self, msg: &str) -> T {
        match self {
            Err(err) => die_perror(msg, err),
            Ok(value) => value,
        }
    }

    fn or_die_e_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String,
    {
        match self {
            Err(err) => die_perror(&f(), err),
            Ok(value) => value,
        }
    }

    fn expect_or_die_perror(self, msg: &str) -> T {
        self.or_die_e(msg)
    }

    fn expect_or_die_perror_f<F>(self, f: F) -> T
    where
        F: FnOnce() -> String,
    {
        self.or_die_e_f(f)
    }
}

impl<T> ExpectedResult<T> for Option<T> {
    fn expect_result_nb(self, msg: &str) -> Result<T, ExpectationFailed> {
        match self {
            None => Err(ExpectationFailed {
                message: msg.to_owned(),
            }),
            Some(value) => Ok(value),
        }
    }

    fn expect_result(self, msg: &str) -> Result<T, Box<dyn error::Error>> {
        match self {
            None => Err(Box::new(ExpectationFailed {
                message: msg.to_owned(),
            })),
            Some(value) => Ok(value),
        }
    }
    fn expect_result_nb_f<F>(self, f: F) -> Result<T, ExpectationFailed>
    where
        F: FnOnce() -> String,
    {
        match self {
            None => Err(ExpectationFailed { message: f() }),
            Some(value) => Ok(value),
        }
    }

    fn expect_result_f<F>(self, f: F) -> Result<T, Box<dyn error::Error>>
    where
        F: FnOnce() -> String,
    {
        match self {
            None => Err(Box::new(ExpectationFailed { message: f() })),
            Some(value) => Ok(value),
        }
    }
}

impl<T, E> ExpectedResult<T> for Result<T, E>
where
    E: fmt::Display,
{
    fn expect_result_nb(self, msg: &str) -> Result<T, ExpectationFailed> {
        match self {
            Ok(value) => Ok(value),
            Err(err) => Err(ExpectationFailed {
                message: format!("{}: {}", msg, err),
            }),
        }
    }

    fn expect_result(self, msg: &str) -> Result<T, Box<dyn error::Error>> {
        match self {
            Ok(value) => Ok(value),
            Err(err) => Err(Box::new(ExpectationFailed {
                message: format!("{}: {}", msg, err),
            })),
        }
    }

    fn expect_result_nb_f<F>(self, f: F) -> Result<T, ExpectationFailed>
    where
        F: FnOnce() -> String,
    {
        match self {
            Ok(value) => Ok(value),
            Err(err) => Err(ExpectationFailed {
                message: format!("{}: {}", f(), err),
            }),
        }
    }

    fn expect_result_f<F>(self, f: F) -> Result<T, Box<dyn error::Error>>
    where
        F: FnOnce() -> String,
    {
        match self {
            Ok(value) => Ok(value),
            Err(err) => Err(Box::new(ExpectationFailed {
                message: format!("{}: {}", f(), err),
            })),
        }
    }
}
