use exocore_core::cell::NodeId;
use futures::{prelude::*, sink::Sink};

use crate::{Error, InMessage, OutMessage};

pub type TransportHandleOnStart = Box<dyn Future<Output = ()> + Send + Unpin + 'static>;

/// Handle to a transport to be used by a service of a cell.
pub trait TransportServiceHandle: Future<Output = ()> + Send + Unpin + 'static {
    type Sink: Sink<OutEvent, Error = Error> + Send + Unpin + 'static;
    type Stream: Stream<Item = InEvent> + Send + Unpin + 'static;

    fn on_started(&self) -> TransportHandleOnStart;
    fn get_sink(&mut self) -> Self::Sink;
    fn get_stream(&mut self) -> Self::Stream;
}

/// Component / service of the Exocore architecture to which a message is
/// intended / originating. Ex: Chain service
#[derive(Copy, Clone, Debug, PartialEq, Eq, Hash)]
pub enum ServiceType {
    None = 0,
    Meta = 1,
    Common = 2,
    Chain = 3,
    Store = 4,
    Client = 5,
}

impl ServiceType {
    pub fn from_code(code: u8) -> Option<ServiceType> {
        match code {
            0 => Some(ServiceType::None),
            1 => Some(ServiceType::Meta),
            2 => Some(ServiceType::Common),
            3 => Some(ServiceType::Chain),
            4 => Some(ServiceType::Store),
            5 => Some(ServiceType::Client),
            _ => None,
        }
    }

    pub fn to_code(self) -> u8 {
        self as u8
    }
}

/// Connection status of a remote node via the transport.
#[derive(Copy, Clone, Debug, PartialEq, Eq)]
pub enum ConnectionStatus {
    Connecting,
    Connected,
    Disconnected,
}

/// Identifier of a connection to a peer. A peer may have multiple
/// opened connections, and this identifier can be used to make sure
/// a message is replied on the right connection.
#[derive(Clone, Debug)]
pub enum ConnectionId {
    Any,

    #[cfg(feature = "p2p-base")]
    Libp2p(libp2p::core::connection::ConnectionId),

    #[cfg(feature = "http-server")]
    HttpServer(crate::http::RequestId),

    #[cfg(any(feature = "http-server", feature = "p2p-full"))]
    Either(crate::either::Side, Option<Box<ConnectionId>>),
}

/// Event generated by the transport to be handled by client.
pub enum InEvent {
    Message(InMessage),
    NodeStatus(NodeId, ConnectionStatus),
}

/// Event generated by client of transport to be handled by transport.
pub enum OutEvent {
    Message(OutMessage),
    Reset,
}
