use crate::dasm::instruction::Peek;

#[derive(Clone, Copy, Debug, PartialEq, Eq)]
pub enum OPCode {
    STOP,
    ADD,
    MUL,
    SUB,
    DIV,
    SDIV,
    MOD,
    SMOD,
    ADDMOD,
    MULMOD,
    EXP,
    SIGNEXTEND,
    LT,
    GT,
    SLT,
    SGT,
    EQ,
    ISZERO,
    AND,
    OR,
    XOR,
    NOT,
    BYTE,
    SHL,
    SHR,
    SAR,
    SHA3,
    ADDRESS,
    BALANCE,
    ORIGIN,
    CALLER,
    CALLVALUE,
    CALLDATALOAD,
    CALLDATASIZE,
    CALLDATACOPY,
    CODESIZE,
    CODECOPY,
    GASPRICE,
    EXTCODESIZE,
    EXTCODECOPY,
    RETURNDATASIZE,
    RETURNDATACOPY,
    EXTCODEHASH,
    BLOCKHASH,
    COINBASE,
    TIMESTAMP,
    NUMBER,
    DIFFICULTY,
    GASLIMIT,
    CHAINID,
    SELFBALANCE,
    BASEFEE,
    POP,
    MLOAD,
    MSTORE,
    MSTORE8,
    SLOAD,
    SSTORE,
    JUMP,
    JUMPI,
    PC,
    MSIZE,
    GAS,
    JUMPDEST,
    PUSH1,
    PUSH2,
    PUSH3,
    PUSH4,
    PUSH5,
    PUSH6,
    PUSH7,
    PUSH8,
    PUSH9,
    PUSH10,
    PUSH11,
    PUSH12,
    PUSH13,
    PUSH14,
    PUSH15,
    PUSH16,
    PUSH17,
    PUSH18,
    PUSH19,
    PUSH20,
    PUSH21,
    PUSH22,
    PUSH23,
    PUSH24,
    PUSH25,
    PUSH26,
    PUSH27,
    PUSH28,
    PUSH29,
    PUSH30,
    PUSH31,
    PUSH32,
    DUP1,
    DUP2,
    DUP3,
    DUP4,
    DUP5,
    DUP6,
    DUP7,
    DUP8,
    DUP9,
    DUP10,
    DUP11,
    DUP12,
    DUP13,
    DUP14,
    DUP15,
    DUP16,
    SWAP1,
    SWAP2,
    SWAP3,
    SWAP4,
    SWAP5,
    SWAP6,
    SWAP7,
    SWAP8,
    SWAP9,
    SWAP10,
    SWAP11,
    SWAP12,
    SWAP13,
    SWAP14,
    SWAP15,
    SWAP16,
    LOG0,
    LOG1,
    LOG2,
    LOG3,
    LOG4,
    PUSH,
    DUP,
    SWAP,
    CREATE,
    CALL,
    CALLCODE,
    RETURN,
    DELEGATECALL,
    CREATE2,
    STATICCALL,
    REVERT,
    SELFDESTRUCT,
    INVALID(u8),
}


pub(crate) fn peek_opcode<'a>(bytecode: &'a [u8]) -> Option<Peek<OPCode>> {
    let opcode =  match bytecode.first()? {
        0x00 => Peek::new(OPCode::STOP, 1),
        0x01 => Peek::new(OPCode::ADD, 1),
        0x02 => Peek::new(OPCode::MUL, 1),
        0x03 => Peek::new(OPCode::SUB, 1),
        0x04 => Peek::new(OPCode::DIV, 1),
        0x05 => Peek::new(OPCode::SDIV, 1),
        0x06 => Peek::new(OPCode::MOD, 1),
        0x07 => Peek::new(OPCode::SMOD, 1),
        0x08 => Peek::new(OPCode::ADDMOD, 1),
        0x09 => Peek::new(OPCode::MULMOD, 1),
        0x0A => Peek::new(OPCode::EXP, 1),
        0x0B => Peek::new(OPCode::SIGNEXTEND, 1),
        
        0x10 => Peek::new(OPCode::LT, 1),
        0x11 => Peek::new(OPCode::GT, 1),
        0x12 => Peek::new(OPCode::SLT, 1),
        0x13 => Peek::new(OPCode::SGT, 1),
        0x14 => Peek::new(OPCode::EQ, 1),
        0x15 => Peek::new(OPCode::ISZERO, 1),
        0x16 => Peek::new(OPCode::AND, 1),
        0x17 => Peek::new(OPCode::OR, 1),
        0x18 => Peek::new(OPCode::XOR, 1),
        0x19 => Peek::new(OPCode::NOT, 1),
        0x1A => Peek::new(OPCode::BYTE, 1),
        0x1B => Peek::new(OPCode::SHL, 1),
        0x1C => Peek::new(OPCode::SHR, 1),
        0x1D => Peek::new(OPCode::SAR, 1),

        0x20 => Peek::new(OPCode::SHA3, 1),

        0x30 => Peek::new(OPCode::ADDRESS, 1),
        0x31 => Peek::new(OPCode::BALANCE, 1),
        0x32 => Peek::new(OPCode::ORIGIN, 1),
        0x33 => Peek::new(OPCode::CALLDATALOAD, 1),
        0x34 => Peek::new(OPCode::CALLVALUE, 1),
        0x35 => Peek::new(OPCode::CALLDATALOAD, 1),
        0x36 => Peek::new(OPCode::CALLDATASIZE, 1),
        0x37 => Peek::new(OPCode::CALLDATACOPY, 1),
        0x38 => Peek::new(OPCode::CODESIZE, 1),
        0x39 => Peek::new(OPCode::CODECOPY, 1),
        0x3A => Peek::new(OPCode::GASPRICE, 1),
        0x3B => Peek::new(OPCode::EXTCODESIZE, 1),
        0x3C => Peek::new(OPCode::EXTCODECOPY, 1),
        0x3D => Peek::new(OPCode::RETURNDATASIZE, 1),
        0x3E => Peek::new(OPCode::RETURNDATACOPY, 1),
        0x3F => Peek::new(OPCode::EXTCODEHASH, 1),

        0x40 => Peek::new(OPCode::BLOCKHASH, 1),
        0x41 => Peek::new(OPCode::COINBASE, 1),
        0x42 => Peek::new(OPCode::TIMESTAMP, 1),
        0x43 => Peek::new(OPCode::NUMBER, 1),
        0x44 => Peek::new(OPCode::DIFFICULTY, 1),
        0x45 => Peek::new(OPCode::GASLIMIT, 1),
        0x46 => Peek::new(OPCode::CHAINID, 1),
        0x47 => Peek::new(OPCode::SELFBALANCE, 1),
        0x48 => Peek::new(OPCode::BASEFEE, 1),

        0x50 => Peek::new(OPCode::POP, 1),
        0x51 => Peek::new(OPCode::MLOAD, 1),
        0x52 => Peek::new(OPCode::MSTORE, 1),
        0x53 => Peek::new(OPCode::MSTORE8, 1),
        0x54 => Peek::new(OPCode::SLOAD, 1),
        0x55 => Peek::new(OPCode::SSTORE, 1),
        0x56 => Peek::new(OPCode::JUMP, 1),
        0x57 => Peek::new(OPCode::JUMPI, 1),
        0x58 => Peek::new(OPCode::PC, 1),
        0x59 => Peek::new(OPCode::MSIZE, 1),
        0x5A => Peek::new(OPCode::GAS, 1),
        0x5B => Peek::new(OPCode::JUMPDEST, 1),

        0x60 => Peek::new(OPCode::PUSH1, 1),
        0x61 => Peek::new(OPCode::PUSH2, 1),
        0x62 => Peek::new(OPCode::PUSH3, 1),
        0x63 => Peek::new(OPCode::PUSH4, 1),
        0x64 => Peek::new(OPCode::PUSH5, 1),
        0x65 => Peek::new(OPCode::PUSH6, 1),
        0x66 => Peek::new(OPCode::PUSH7, 1),
        0x67 => Peek::new(OPCode::PUSH8, 1),
        0x68 => Peek::new(OPCode::PUSH9, 1),
        0x69 => Peek::new(OPCode::PUSH10, 1),
        0x6A => Peek::new(OPCode::PUSH11, 1),
        0x6B => Peek::new(OPCode::PUSH12, 1),
        0x6C => Peek::new(OPCode::PUSH13, 1),
        0x6D => Peek::new(OPCode::PUSH14, 1),
        0x6E => Peek::new(OPCode::PUSH15, 1),
        0x6F => Peek::new(OPCode::PUSH16, 1),
        0x70 => Peek::new(OPCode::PUSH17, 1),
        0x71 => Peek::new(OPCode::PUSH18, 1),
        0x72 => Peek::new(OPCode::PUSH19, 1),
        0x73 => Peek::new(OPCode::PUSH20, 1),
        0x74 => Peek::new(OPCode::PUSH21, 1),
        0x75 => Peek::new(OPCode::PUSH22, 1),
        0x76 => Peek::new(OPCode::PUSH23, 1),
        0x77 => Peek::new(OPCode::PUSH24, 1),
        0x78 => Peek::new(OPCode::PUSH25, 1),
        0x79 => Peek::new(OPCode::PUSH26, 1),
        0x7A => Peek::new(OPCode::PUSH27, 1),
        0x7B => Peek::new(OPCode::PUSH28, 1),
        0x7C => Peek::new(OPCode::PUSH29, 1),
        0x7D => Peek::new(OPCode::PUSH30, 1),
        0x7E => Peek::new(OPCode::PUSH31, 1),
        0x7F => Peek::new(OPCode::PUSH32, 1),

        0x80 => Peek::new(OPCode::DUP1, 1),
        0x81 => Peek::new(OPCode::DUP2, 1),
        0x82 => Peek::new(OPCode::DUP3, 1),
        0x83 => Peek::new(OPCode::DUP4, 1),
        0x84 => Peek::new(OPCode::DUP5, 1),
        0x85 => Peek::new(OPCode::DUP6, 1),
        0x86 => Peek::new(OPCode::DUP7, 1),
        0x87 => Peek::new(OPCode::DUP8, 1),
        0x88 => Peek::new(OPCode::DUP9, 1),
        0x89 => Peek::new(OPCode::DUP10, 1),
        0x8A => Peek::new(OPCode::DUP11, 1),
        0x8B => Peek::new(OPCode::DUP12, 1),
        0x8C => Peek::new(OPCode::DUP13, 1),
        0x8D => Peek::new(OPCode::DUP14, 1),
        0x8E => Peek::new(OPCode::DUP15, 1),
        0x8F => Peek::new(OPCode::DUP16, 1),

        0x90 => Peek::new(OPCode::SWAP1, 1),
        0x91 => Peek::new(OPCode::SWAP2, 1),
        0x92 => Peek::new(OPCode::SWAP3, 1),
        0x93 => Peek::new(OPCode::SWAP4, 1),
        0x94 => Peek::new(OPCode::SWAP5, 1),
        0x95 => Peek::new(OPCode::SWAP6, 1),
        0x96 => Peek::new(OPCode::SWAP7, 1),
        0x97 => Peek::new(OPCode::SWAP8, 1),
        0x98 => Peek::new(OPCode::SWAP9, 1),
        0x99 => Peek::new(OPCode::SWAP10, 1),
        0x9A => Peek::new(OPCode::SWAP11, 1),
        0x9B => Peek::new(OPCode::SWAP12, 1),
        0x9C => Peek::new(OPCode::SWAP13, 1),
        0x9D => Peek::new(OPCode::SWAP14, 1),
        0x9E => Peek::new(OPCode::SWAP15, 1),
        0x9F => Peek::new(OPCode::SWAP16, 1),

        0xA0 => Peek::new(OPCode::LOG0, 1),
        0xA1 => Peek::new(OPCode::LOG1, 1),
        0xA2 => Peek::new(OPCode::LOG2, 1),
        0xA3 => Peek::new(OPCode::LOG3, 1),
        0xA4 => Peek::new(OPCode::LOG4, 1),

        0xB0 => Peek::new(OPCode::PUSH, 1),
        0xB1 => Peek::new(OPCode::DUP, 1),
        0xB2 => Peek::new(OPCode::SWAP, 1),

        0xF0 => Peek::new(OPCode::CREATE, 1),
        0xF1 => Peek::new(OPCode::CALL, 1),
        0xF2 => Peek::new(OPCode::CALLCODE, 1),
        0xF3 => Peek::new(OPCode::RETURN, 1),
        0xF4 => Peek::new(OPCode::DELEGATECALL, 1),
        0xF5 => Peek::new(OPCode::CREATE2, 1),
        0xFA => Peek::new(OPCode::STATICCALL, 1),
        0xFD => Peek::new(OPCode::REVERT, 1),
        0xFF => Peek::new(OPCode::SELFDESTRUCT, 1),

        invalid    => Peek::new(OPCode::INVALID(*invalid), 1),
    };

    Some(opcode)
}
