use crate::common::*;

pub(crate) struct Parser<'a> {
  tokens: &'a mut Peekable<Iter<'a, Token>>,
}

impl<'a> Parser<'a> {
  fn new(tokens: &'a mut Peekable<Iter<'a, Token>>) -> Self {
    Parser { tokens }
  }

  pub(crate) fn parse(tokens: Iter<Token>) -> Result<Expr> {
    let mut tokens = tokens.peekable();
    Parser::new(&mut tokens).expr()
  }

  fn primary(&mut self) -> Result<Expr> {
    match self.tokens.next().unwrap() {
      Token::Number(n) => Ok(Expr::Number(*n as i64)),
      Token::RightParen => {
        let expr = self.expr()?;
        Ok(expr)
      }
      Token::Dash => {
        let expr = self.factor()?;
        Ok(Expr::Unary(Op::Neg, Box::new(expr)))
      }
      _ => todo!(),
    }
  }

  fn factor(&mut self) -> Result<Expr> {
    let expr = self.primary()?;

    if let Some(next) = self.tokens.peek() {
      if *next == &Token::Caret {
        self.tokens.next();
        let rhs = self.factor()?;
        return Ok(Expr::Binary(Op::Pow, Box::new(expr), Box::new(rhs)));
      }
    }

    Ok(expr)
  }

  fn term(&mut self) -> Result<Expr> {
    let mut expr = self.factor()?;

    while let Some(next) = self.tokens.peek() {
      match next {
        Token::Asterisk => {
          self.tokens.next();
          let rhs = self.factor()?;
          expr = Expr::Binary(Op::Mul, Box::new(expr), Box::new(rhs));
        }
        Token::Slash => {
          self.tokens.next();
          let rhs = self.factor()?;
          expr = Expr::Binary(Op::Div, Box::new(expr), Box::new(rhs))
        }
        _ => break,
      }
    }

    Ok(expr)
  }

  fn expr(&mut self) -> Result<Expr> {
    let mut expr = self.term()?;

    while let Some(next) = self.tokens.peek() {
      match next {
        Token::Plus => {
          self.tokens.next();
          let rhs = self.term()?;
          expr = Expr::Binary(Op::Add, Box::new(expr), Box::new(rhs))
        }
        Token::Dash => {
          self.tokens.next();
          let rhs = self.term()?;
          expr = Expr::Binary(Op::Sub, Box::new(expr), Box::new(rhs))
        }
        _ => break,
      }
    }

    Ok(expr)
  }
}
