use crate::common::*;

#[derive(Debug)]
pub(crate) struct Lexer<'src> {
  chars:  Chars<'src>,
  index:  usize,
  next:   Option<char>,
  src:    &'src str,
  tokens: Vec<Token>,
}

impl<'src> Lexer<'src> {
  fn new(src: &str) -> Lexer {
    let mut chars = src.chars();

    let next = chars.next();

    Lexer {
      chars,
      index: 0,
      next,
      src,
      tokens: Vec::new(),
    }
  }

  pub(crate) fn lex(src: &'src str) -> Result<Vec<Token>> {
    Lexer::new(src).tokenize()
  }

  fn advance(&mut self) -> Result<()> {
    match self.next {
      Some(_) => self.next = self.chars.next(),
      None => todo!(),
    }
    Ok(())
  }

  fn tokenize(mut self) -> Result<Vec<Token>> {
    while let Some(start) = self.next {
      self.lex_normal(start)?;
    }
    Ok(self.tokens)
  }

  fn lex_normal(&mut self, start: char) -> Result<()> {
    match start {
      '*' => self.lex_single(Asterisk),
      '+' => self.lex_single(Plus),
      '-' => self.lex_single(Dash),
      '/' => self.lex_single(Slash),
      '^' => self.lex_single(Caret),
      ')' => self.lex_single(RightParen),
      '(' => self.lex_single(LeftParen),
      c if c.is_digit(10) => self.lex_single(Token::from(c.to_digit(10).unwrap_or(0))),
      _ => self.advance(),
    }
  }

  fn lex_single(&mut self, token: Token) -> Result<()> {
    self.token(token);
    self.advance()?;
    Ok(())
  }

  fn token(&mut self, token: Token) {
    self.tokens.push(token);
  }
}
