use evercrypt::digest::{self, Digest, Mode};

#[test]
fn test_sha2() {
    let data = b"evercrypt-rust bindings";
    let d = digest::hash(Mode::Sha256, data);
    let expected_digest_256 = [
        0xa5, 0x35, 0xf2, 0x6a, 0xff, 0xbc, 0x1f, 0x08, 0x73, 0xdb, 0x15, 0x15, 0x9d, 0xce, 0xbf,
        0x25, 0x99, 0x64, 0xbe, 0x42, 0xde, 0xa8, 0x4d, 0x29, 0x00, 0x38, 0x4b, 0xee, 0x15, 0x09,
        0xe4, 0x00,
    ];
    let expected_digest_512 = [
        0x36, 0x97, 0x36, 0x7c, 0xc9, 0x1e, 0xda, 0xa7, 0x6d, 0xb8, 0x03, 0x39, 0x61, 0x5f, 0xc2,
        0x12, 0xe1, 0x5e, 0x64, 0x3e, 0x31, 0x30, 0xf7, 0x1f, 0x28, 0xd0, 0x3f, 0x34, 0x3d, 0xf4,
        0x88, 0x0a, 0xd3, 0x6c, 0x63, 0xe5, 0x35, 0x1f, 0x56, 0xe0, 0xf7, 0xe0, 0x4c, 0x24, 0x96,
        0xc0, 0xb3, 0x6b, 0xcf, 0x7c, 0x5d, 0xcb, 0xf3, 0x5e, 0x38, 0xe9, 0xbb, 0x44, 0xf8, 0xa0,
        0xc2, 0x83, 0x42, 0x4e,
    ];
    assert_eq!(d, expected_digest_256);
    assert_eq!(
        digest::hash(Mode::Sha512, data)[..],
        expected_digest_512[..]
    );

    let mut digest = Digest::new(Mode::Sha256).unwrap();
    assert!(digest.update(data).is_ok());
    match digest.finish() {
        Ok(d) => assert_eq!(d, expected_digest_256),
        Err(r) => panic!("Got error in finish {:?}", r),
    }
    assert!(digest.finish().is_err());
    assert!(digest.update(&[]).is_err());

    let mut digest = Digest::new(Mode::Sha512).unwrap();
    assert!(digest.update(data).is_ok());
    match digest.finish() {
        Ok(d) => assert_eq!(d[..], expected_digest_512[..]),
        Err(r) => panic!("Got error in finish {:?}", r),
    }
    assert!(digest.finish().is_err());
    assert!(digest.update(&[]).is_err());
}

#[test]
#[should_panic]
fn invalid_sha3() {
    Digest::new(Mode::Sha3_224).unwrap();
    Digest::new(Mode::Sha3_256).unwrap();
    Digest::new(Mode::Sha3_384).unwrap();
    Digest::new(Mode::Sha3_512).unwrap();
}

#[test]
fn test_sha3() {
    let data = b"evercrypt-rust bindings";
    let expected_digest_256 = [
        0x49, 0x4b, 0xc2, 0xea, 0x73, 0x43, 0x4f, 0x88, 0x62, 0x56, 0x13, 0x39, 0xda, 0x1a, 0x6d,
        0x58, 0x05, 0xee, 0x34, 0x4b, 0x67, 0x5d, 0x18, 0xfb, 0x9a, 0x81, 0xca, 0x65, 0xa7, 0x8f,
        0xeb, 0x6e,
    ];
    let expected_digest_512 = [
        0x7a, 0xaa, 0x97, 0x5c, 0x6b, 0x15, 0x5b, 0x55, 0xd3, 0x7b, 0xa6, 0x99, 0x3f, 0x7e, 0x14,
        0xd9, 0x8c, 0x28, 0x0d, 0x2b, 0x2f, 0xc2, 0x4a, 0xa7, 0x84, 0x07, 0xcf, 0x15, 0x2d, 0x0a,
        0xca, 0xbc, 0x32, 0xf2, 0x11, 0xf4, 0x64, 0x30, 0x19, 0x0a, 0x35, 0x26, 0x94, 0x76, 0x84,
        0x2a, 0x1f, 0x17, 0x41, 0xad, 0x46, 0x06, 0xf6, 0xc8, 0xc6, 0xad, 0x8d, 0x02, 0x2e, 0x85,
        0xb4, 0x9d, 0x6b, 0xd7,
    ];

    assert_eq!(digest::hash(Mode::Sha3_256, data), expected_digest_256);
    assert_eq!(
        digest::hash(Mode::Sha3_512, data)[..],
        expected_digest_512[..]
    );
}

#[test]
fn test_shake() {
    let data = b"evercrypt-rust bindings";
    let expected_digest_128 = [
        0xfd, 0x3b, 0x31, 0x35, 0x35, 0x05, 0x87, 0xd5, 0x36, 0x2a, 0xae, 0x4d, 0x1c, 0x8a, 0x25,
        0xba, 0xa4, 0xec, 0x82, 0xef, 0xff, 0xb8, 0x27, 0x1c, 0x91, 0x20, 0xa2, 0xed, 0x53, 0x17,
        0x2a, 0xcc, 0x97, 0x97, 0x34, 0x65, 0x1e, 0x69, 0xb3, 0xb3, 0x27, 0x09, 0x4c, 0xc0, 0x5e,
        0xde, 0x3b, 0x5d, 0xf9, 0x98, 0xe6, 0x37, 0xce, 0x06, 0xb3, 0xa0, 0x53, 0xdf, 0x81, 0x80,
        0x99, 0x8c, 0xfc, 0x95,
    ];
    let expected_digest_256 = [
        0xf0, 0x85, 0x60, 0x6b, 0xed, 0xca, 0x25, 0xe4, 0x3c, 0x97, 0x05, 0x0f, 0xf2, 0x3e, 0xe0,
        0xd9, 0xe5, 0x89, 0x14, 0xff, 0xbb, 0x30, 0x5a, 0x00, 0x26, 0x30, 0x1c, 0x25, 0x7a, 0x5a,
        0xeb, 0x50, 0x7e, 0x4b, 0x21, 0x19, 0x53, 0x3f, 0xf7, 0x23, 0xc7, 0xe1, 0xad, 0xc5, 0xdf,
        0x2a, 0x62, 0x1d, 0xad, 0x18, 0xa4, 0x46, 0xaf, 0xeb, 0x2a, 0x54, 0xb3, 0xad, 0xfe, 0xc7,
        0x8e, 0x08, 0x6a, 0x6f,
    ];

    assert_eq!(digest::shake128(data, 64)[..], expected_digest_128[..]);
    assert_eq!(digest::shake256(data, 64)[..], expected_digest_256[..]);
}
