use std::{
    any::{Any, TypeId},
    collections::HashMap,
    vec,
};

use async_trait::async_trait;
use futures::future::join_all;

/// Event bus
/// # Using example
/// ```rust
/// use eventbus_rs::{UnMutSubscriber, EventBus};
/// use async_trait::async_trait;
/// use std::any::Any;
/// // Create new topic
/// type Topic = ();
///
/// // Impl our subscriber
/// struct SimpleSubscriber;
/// #[async_trait]
/// impl UnMutSubscriber for SimpleSubscriber {
///     async fn handle(&self, message: &(dyn Any + Send + Sync)) {}
/// }
/// async fn subscribe_and_publish() {
///     // Create event bus
///     let mut event_bus = EventBus::new().await;
///     // Subscribe on our topic
///     event_bus.subscribe_unmut::<Topic>(Box::new(SimpleSubscriber {})).await;
///     // Publish event
///     event_bus.publish_unmut::<Topic>(&"Hello".to_string()).await;
/// }
/// ```
pub struct EventBus(HashMap<TypeId, (Option<UnMutTopic>, Option<MutTopic>)>);

impl EventBus {
    /// Creates new event bus.
    pub async fn new() -> EventBus {
        EventBus { 0: HashMap::new() }
    }

    async fn register_unmut_topic<T: 'static>(&mut self, topic: UnMutTopic) {
        if let Some(topics) = self.0.get_mut(&TypeId::of::<T>()) {
            topics.0 = Some(topic);
        } else {
            self.0.insert(TypeId::of::<T>(), (Some(topic), None));
        }
    }

    async fn create_and_register_unmut_topic<T: 'static>(
        &mut self,
        subscriber: Box<dyn UnMutSubscriber>,
    ) {
        self.register_unmut_topic::<T>(UnMutTopic(vec![subscriber]))
            .await;
    }

    async fn register_mut_topic<T: 'static>(&mut self, topic: MutTopic) {
        if let Some(topics) = self.0.get_mut(&TypeId::of::<T>()) {
            topics.1 = Some(topic);
        } else {
            self.0.insert(TypeId::of::<T>(), (None, Some(topic)));
        }
    }

    async fn create_and_register_mut_topic<T: 'static>(
        &mut self,
        subscriber: Box<dyn MutSubscriber>
    ) {
        self.register_mut_topic::<T>(MutTopic(vec![subscriber])).await;
    }

    /// Subscribes on unmutable topic.
    pub async fn subscribe_unmut<T: 'static>(&mut self, subscriber: Box<dyn UnMutSubscriber>) {
        if let Some(topics) = self.0.get_mut(&TypeId::of::<T>()) {
            if let Some(topic) = &mut topics.0 {
                topic.0.push(subscriber);
            } else {
                self.create_and_register_unmut_topic::<T>(subscriber).await;
            }
        } else {
            self.create_and_register_unmut_topic::<T>(subscriber).await;
        }
    }

    /// Publishes event for unmutable subscribers.
    pub async fn publish_unmut<T: 'static>(&self, message: &(dyn Any + Send + Sync)) {
        if let Some(topics) = self.0.get(&TypeId::of::<T>()) {
            if let Some(topic) = &topics.0 {
                topic.publish(message).await;
            }
        }
    }
    /// Subscribes on mutable topic
    pub async fn subscribe_mut<T: 'static>(&mut self, subscriber: Box<dyn MutSubscriber>) {
        if let Some(topics) = self.0.get_mut(&TypeId::of::<T>()) {
            if let Some(topic) = &mut topics.1 {
                topic.0.push(subscriber);
            } else {
                self.create_and_register_mut_topic::<T>(subscriber).await;
            }
        } else {
            self.create_and_register_mut_topic::<T>(subscriber).await;
        }
    }
    /// Publishes event for mutable subscribers.
    pub async fn publish_mut<T: 'static>(&mut self, message: &(dyn Any + Send + Sync)) {
        if let Some(topics) = self.0.get_mut(&TypeId::of::<T>()) {
            if let Some(topic) = &mut topics.1 {
                topic.publish(message).await;
            }
        }
    }
}

/// Topic for unmutable subscribers in event bus.
struct UnMutTopic(Vec<Box<dyn UnMutSubscriber>>);
impl UnMutTopic {
    /// Publishes event.
    pub async fn publish(&self, message: &(dyn Any + Send + Sync)) {
        let mut futures = vec![];
        for subscriber in self.0.iter() {
            futures.push(subscriber.handle(message));
        }
        join_all(futures).await;
    }
}

/// Topic for mutable subscribers in event bus.
struct MutTopic(Vec<Box<dyn MutSubscriber>>);
impl MutTopic {
    pub async fn publish(&mut self, message: &(dyn Any + Send + Sync)) {
        let mut futures = vec![];
        for subscriber in self.0.iter_mut() {
            futures.push(subscriber.handle(message));
        }
        join_all(futures).await;
    }
}

/// Unmutable subscriber trait
/// # Example
///
/// ```rust
/// use eventbus_rs::UnMutSubscriber;
/// use std::any::Any;
/// use async_trait::async_trait;
///
/// struct SimpleSubscriber;
/// #[async_trait]
/// impl UnMutSubscriber for SimpleSubscriber {
///         async fn handle(&self, message: &(dyn Any + Send + Sync)) {
///             match message.downcast_ref::<String>() {
///                 Some(str) => {
///                     println!("{}", str);
///                 }
///                 None => ()
///             }
///         }
///}
/// ```
#[async_trait]
pub trait UnMutSubscriber {
    async fn handle(&self, message: &(dyn Any + Send + Sync));
}

/// Mutable subscriber trait
/// # Example
/// ```rust
/// use eventbus_rs::MutSubscriber;
/// use std::any::Any;
/// use async_trait::async_trait;
/// 
/// struct SimpleSubscriber;
/// #[async_trait]
/// impl MutSubscriber for SimpleSubscriber {
///     async fn handle(&mut self, message: &(dyn Any + Send + Sync)) {
///         match message.downcast_ref::<String>() {
///             Some(str) => {
///                 println!("{}", str);
///             },
///             None => ()
///         }
///     }
/// }
/// ```
#[async_trait]
pub trait MutSubscriber {
    async fn handle(&mut self, message: &(dyn Any + Send + Sync));
}

#[cfg(test)]
mod tests {

    use futures::executor::block_on;

    use crate::EventBus;

    mod subscribers {
        use std::any::Any;

        use async_trait::async_trait;

        use crate::{UnMutSubscriber, MutSubscriber};

        pub type Topic = ();

        pub struct SimpleUnMutSubscriber;

        #[async_trait]
        impl UnMutSubscriber for SimpleUnMutSubscriber {
            async fn handle(&self, message: &(dyn Any + Send + Sync)) {
                check_message(message);
            }
        }
        
        pub struct SimpleMutSubscriber {}

        #[async_trait]
        impl MutSubscriber for SimpleMutSubscriber {
            async fn handle(&mut self, message: &(dyn Any + Send + Sync)) {
                check_message(message);
            }
        }

        pub fn check_message(message: &(dyn Any + Send + Sync)) {
            match message.downcast_ref::<String>() {
                Some(message) => {
                    if message != "Hello" {
                        panic!("Message != Hello")
                    }
                }
                None => panic!("No message"),
            }
        }
    }

    #[test]
    fn test_event_bus_unmut() {
        block_on(async {
            let event_bus = create_eventbus_and_unmut_topic().await;
            event_bus
                .publish_unmut::<subscribers::Topic>(&"Hello".to_string())
                .await;
        });
    }

    async fn create_eventbus_and_unmut_topic() -> EventBus {
        let mut event_bus = EventBus::new().await;
        event_bus
            .subscribe_unmut::<subscribers::Topic>(Box::new(subscribers::SimpleUnMutSubscriber {}))
            .await;
        event_bus
    }
    #[test]
    #[should_panic]
    fn test_event_bus_unmut2() {
        block_on(async {
            let event_bus = create_eventbus_and_unmut_topic().await;
            event_bus
                .publish_unmut::<subscribers::Topic>(&"Not a hello".to_string())
                .await;
        });
    }

    async fn create_event_bus_and_mut_topic() -> EventBus{
        let mut event_bus = EventBus::new().await;
        event_bus.subscribe_mut::<subscribers::Topic>(Box::new(subscribers::SimpleMutSubscriber {})).await;
        event_bus
    }

    #[test]
    fn test_event_bus_mut() {
        block_on(async {
            let mut event_bus = create_event_bus_and_mut_topic().await;
            event_bus.publish_mut::<subscribers::Topic>(&"Hello".to_string()).await;
        });
    }
    #[test]
    #[should_panic]
    fn test_event_bus_mut2() {
        block_on(async {
            let mut event_bus = create_event_bus_and_mut_topic().await;
            event_bus.publish_mut::<subscribers::Topic>(&"Not a hello".to_string()).await;
        });
    }
}
