use std::{
    any::{Any, TypeId},
    collections::HashMap,
    vec,
};

pub struct EventBus(HashMap<TypeId, Topic>);

impl EventBus {
    /// Creates new event bus.
    pub fn new() -> EventBus {
        EventBus { 0: HashMap::new() }
    }

    fn register_topic<T: 'static>(&mut self, topic: Topic) {
        self.0.insert(TypeId::of::<T>(), topic);
    }

    /// Subscribes on topic.
    pub fn subscribe<T: 'static>(&mut self, subscriber: Box<dyn Subscriber>) {
        match self.0.get_mut(&TypeId::of::<T>()) {
            Some(topic) => topic.0.push(subscriber),
            None => {
                let topic = Topic(vec![subscriber]);
                self.register_topic::<T>(topic);
            }
        }
    }

    /// Publishes event.
    pub fn publish<T: 'static>(&mut self, message: &dyn Any) {
        match self.0.get_mut(&TypeId::of::<T>()) {
            Some(topic) => topic.publish(message),
            None => (),
        }
    }

    /// Creates new topic.
    pub fn new_topic<T: 'static>(&mut self) {
        if !self.0.contains_key(&TypeId::of::<T>()) {
            let topic = Topic(Vec::new());
            self.register_topic::<T>(topic);
        }
    }
}

pub struct Topic(Vec<Box<dyn Subscriber>>);
impl Topic {
    /// Publishes event.
    pub fn publish(&mut self, message: &dyn Any) {
        for i in 0..self.0.len() {
            if let Some(subscriber) = self.0.get_mut(i) {
                subscriber.as_mut().handle(message);
            }
        }
    }
}

pub trait Subscriber {
    fn handle(&mut self, message: &dyn Any);
}

pub struct SubscriberFn(fn(&dyn Any) -> ());
impl Subscriber for SubscriberFn {
    fn handle(&mut self, message: &dyn Any) {
        self.0(message);
    }
}
impl SubscriberFn {
    pub fn new(func: fn(&dyn Any) -> ()) -> SubscriberFn {
        SubscriberFn { 0: func }
    }
}

#[cfg(test)]
mod tests {
    use std::{any::Any};

    use crate::{EventBus, SubscriberFn, Subscriber};

    mod topics {
        pub struct Hello;
    }

    #[test]
    fn test_event_bus_with_fn_handler() {
        let mut event_bus = EventBus::new();
        event_bus.new_topic::<topics::Hello>();
        event_bus.subscribe::<topics::Hello>(Box::new(SubscriberFn::new(topic_subscriber)));
        event_bus.publish::<topics::Hello>(&"Hello".to_string())
    }
    fn topic_subscriber(message: &dyn Any) {
        match message.downcast_ref::<String>() {
            Some(str) => {
                if str != "Hello" {
                    panic!("Message != Hello");
                }
            }
            None => panic!("No message"),
        }
    }
    #[test]
    fn test_event_bus_with_trait_handler() {
        let mut event_bus = EventBus::new();
        event_bus.subscribe::<topics::Hello>(Box::new(Handler {}));
        event_bus.publish::<topics::Hello>(&"Hello".to_string());
        
    }
    struct Handler;
    impl Subscriber for Handler {
        fn handle(&mut self, message: &dyn Any) {
            match message.downcast_ref::<String>() {
                Some(str) => {
                    if str != "Hello" {
                        panic!("Message != Hello");
                    }
                }
                None => panic!("No message")
            }
        }
    }
}
