/// Controller methods and structures
use eva_common::actions::{self, ACTION_TOPIC};
use eva_common::events::RAW_STATE_TOPIC;
use eva_common::prelude::*;
use serde::Deserialize;
use std::time::{Duration, Instant};
use uuid::Uuid;

#[path = "actt.rs"]
pub mod actt;
#[path = "transform.rs"]
pub mod transform;

pub const ERR_NO_PARAMS: &str = "action params not specified";

#[inline]
pub fn format_action_topic(oid: &OID) -> String {
    format!("{}{}", ACTION_TOPIC, oid.to_path())
}

#[inline]
pub fn format_raw_state_topic(oid: &OID) -> String {
    format!("{}{}", RAW_STATE_TOPIC, oid.to_path())
}

#[derive(Deserialize)]
#[serde(rename_all = "lowercase")]
pub enum ItemProp {
    Status,
    Value,
}

/// Controller operation
pub struct Op {
    t: Instant,
    timeout: Duration,
}

impl Op {
    #[inline]
    pub fn new(timeout: Duration) -> Self {
        Self {
            t: Instant::now(),
            timeout,
        }
    }
    #[inline]
    pub fn for_action(action: &Action) -> Self {
        Self {
            t: action.received,
            timeout: action.timeout,
        }
    }
    pub fn is_timed_out(&self) -> bool {
        let el = self.t.elapsed();
        el > self.timeout
    }
    pub fn timeout(&self) -> EResult<Duration> {
        let el = self.t.elapsed();
        if el > self.timeout {
            Err(Error::timeout())
        } else {
            Ok(self.timeout - el)
        }
    }
    #[inline]
    pub fn is_enough(&self, expected: Duration) -> bool {
        self.t.elapsed() + expected < self.timeout
    }
    #[inline]
    pub fn enough(&self, expected: Duration) -> EResult<()> {
        if self.is_enough(expected) {
            Ok(())
        } else {
            Err(Error::timeout())
        }
    }
    #[inline]
    pub fn remaining(&self, timeout: Duration) -> EResult<Duration> {
        let el = self.t.elapsed();
        if el > timeout {
            Err(Error::timeout())
        } else {
            Ok(timeout - el)
        }
    }
}

/// Controller action object
#[derive(Deserialize, Debug)]
#[serde(deny_unknown_fields)]
pub struct Action {
    uuid: Uuid,
    i: OID,
    #[serde(deserialize_with = "eva_common::tools::deserialize_duration_from_micros")]
    timeout: Duration,
    priority: u8,
    params: Option<actions::Params>,
    config: Option<Value>,
    #[serde(skip, default = "Instant::now")]
    received: Instant,
}

impl Action {
    pub fn event_pending(&self) -> actions::ActionEvent {
        actions::ActionEvent {
            uuid: self.uuid,
            status: actions::Status::Pending as u8,
            out: None,
            err: None,
            exitcode: None,
        }
    }
    pub fn event_running(&self) -> actions::ActionEvent {
        actions::ActionEvent {
            uuid: self.uuid,
            status: actions::Status::Running as u8,
            out: None,
            err: None,
            exitcode: None,
        }
    }
    pub fn event_completed(&self, out: Option<Value>) -> actions::ActionEvent {
        actions::ActionEvent {
            uuid: self.uuid,
            status: actions::Status::Completed as u8,
            out,
            err: None,
            exitcode: Some(0),
        }
    }
    pub fn event_failed(
        &self,
        exitcode: i16,
        out: Option<Value>,
        err: Option<Value>,
    ) -> actions::ActionEvent {
        actions::ActionEvent {
            uuid: self.uuid,
            status: actions::Status::Failed as u8,
            out,
            err,
            exitcode: Some(exitcode),
        }
    }
    pub fn event_canceled(&self) -> actions::ActionEvent {
        actions::ActionEvent {
            uuid: self.uuid,
            status: actions::Status::Canceled as u8,
            out: None,
            err: None,
            exitcode: None,
        }
    }
    pub fn event_terminated(&self) -> actions::ActionEvent {
        actions::ActionEvent {
            uuid: self.uuid,
            status: actions::Status::Terminated as u8,
            out: None,
            err: None,
            exitcode: Some(-15),
        }
    }
    pub fn uuid(&self) -> &Uuid {
        &self.uuid
    }
    pub fn oid(&self) -> &OID {
        &self.i
    }
    pub fn timeout(&self) -> Duration {
        self.timeout
    }
    pub fn priority(&self) -> u8 {
        self.priority
    }
    pub fn params(&self) -> Option<&actions::Params> {
        self.params.as_ref()
    }
    pub fn take_params(&mut self) -> Option<actions::Params> {
        self.params.take()
    }
    pub fn take_unit_params(&mut self) -> EResult<actions::UnitParams> {
        if let Some(params) = self.params.take() {
            match params {
                eva_common::actions::Params::Unit(p) => Ok(p),
                eva_common::actions::Params::Lmacro(_) => Err(Error::not_implemented(
                    "can not exec lmacro action with unit",
                )),
            }
        } else {
            Err(Error::invalid_data(ERR_NO_PARAMS))
        }
    }
    pub fn take_lmacro_params(&mut self) -> EResult<actions::LmacroParams> {
        if let Some(params) = self.params.take() {
            match params {
                eva_common::actions::Params::Lmacro(p) => Ok(p),
                eva_common::actions::Params::Unit(_) => Err(Error::not_implemented(
                    "can not exec unit action with lmacro",
                )),
            }
        } else {
            Err(Error::invalid_data(ERR_NO_PARAMS))
        }
    }
    pub fn config(&self) -> Option<&Value> {
        self.config.as_ref()
    }
    pub fn take_config(&mut self) -> Option<Value> {
        self.config.take()
    }
}
