use crate::items::ItemState;
use crate::Error;
use eva_common::prelude::*;
use serde::{Deserialize, Deserializer, Serialize, Serializer};
use std::hash::{Hash, Hasher};
use std::str::FromStr;

pub const RAW_STATE_TOPIC: &str = "RAW/";
pub const LOCAL_STATE_TOPIC: &str = "ST/LOC/";
pub const REMOTE_STATE_TOPIC: &str = "ST/REM/";
pub const REMOTE_ARCHIVE_STATE_TOPIC: &str = "ST/RAR/";
pub const REPLICATION_STATE_TOPIC: &str = "RPL/ST/";
pub const REPLICATION_INVENTORY_TOPIC: &str = "RPL/INVENTORY/";
pub const REPLICATION_NODE_STATE_TOPIC: &str = "RPL/NODE/";

#[derive(Debug)]
#[repr(i8)]
pub enum NodeStatus {
    Online = 1,
    Offline = 0,
    Removed = -1,
}

impl NodeStatus {
    fn as_str(&self) -> &str {
        match self {
            NodeStatus::Online => "online",
            NodeStatus::Offline => "offline",
            NodeStatus::Removed => "removed",
        }
    }
}

impl FromStr for NodeStatus {
    type Err = Error;
    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s {
            "online" => Ok(NodeStatus::Online),
            "offline" => Ok(NodeStatus::Offline),
            "removed" => Ok(NodeStatus::Removed),
            _ => Err(Error::invalid_data(format!("Invalid node status: {}", s))),
        }
    }
}

#[derive(Debug, Serialize, Deserialize)]
#[serde(deny_unknown_fields)]
pub struct NodeStateEvent {
    pub status: NodeStatus,
}

impl Serialize for NodeStatus {
    #[inline]
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        serializer.serialize_str(self.as_str())
    }
}

impl<'de> Deserialize<'de> for NodeStatus {
    #[inline]
    fn deserialize<D>(deserializer: D) -> Result<NodeStatus, D::Error>
    where
        D: Deserializer<'de>,
    {
        let s: String = Deserialize::deserialize(deserializer)?;
        s.parse().map_err(serde::de::Error::custom)
    }
}

#[derive(Debug, Serialize, Deserialize)]
#[serde(deny_unknown_fields)]
pub struct RawStateEvent {
    status: ItemStatus,
    value: Option<Value>,
}

impl RawStateEvent {
    #[inline]
    pub fn new(status: ItemStatus, value: Value) -> Self {
        Self {
            status,
            value: Some(value),
        }
    }
    #[inline]
    pub fn new0(status: ItemStatus) -> Self {
        Self {
            status,
            value: None,
        }
    }
    #[inline]
    pub fn status(&self) -> ItemStatus {
        self.status
    }
    #[inline]
    pub fn value(&self) -> &Option<Value> {
        &self.value
    }
    #[inline]
    pub fn take_value(&mut self) -> Option<Value> {
        self.value.take()
    }
}

#[derive(Debug, Serialize, Deserialize)]
#[serde(deny_unknown_fields)]
pub struct ReplicationStateEvent {
    status: ItemStatus,
    value: Option<Value>,
    act: Option<usize>,
    ieid: IEID,
    t: f64,
    node: Option<String>,
}

impl From<ReplicationInventoryItem> for ReplicationStateEvent {
    fn from(item: ReplicationInventoryItem) -> Self {
        Self {
            status: item.status.unwrap_or_default(),
            value: Some(item.value.unwrap_or_default()),
            act: item.act,
            ieid: item.ieid,
            t: item.t,
            node: None,
        }
    }
}

#[allow(clippy::similar_names)]
impl ReplicationStateEvent {
    #[inline]
    pub fn new(
        status: ItemStatus,
        value: Value,
        act: Option<usize>,
        ieid: IEID,
        t: f64,
        node: &str,
    ) -> Self {
        Self {
            status,
            value: Some(value),
            act,
            ieid,
            t,
            node: Some(node.to_owned()),
        }
    }
    #[inline]
    pub fn status(&self) -> ItemStatus {
        self.status
    }
    #[inline]
    pub fn act(&self) -> Option<usize> {
        self.act
    }
    #[inline]
    pub fn value(&self) -> &Option<Value> {
        &self.value
    }
    #[inline]
    pub fn nvalue(&self) -> &Option<Value> {
        &self.value
    }
    #[inline]
    pub fn take_value(&mut self) -> Option<Value> {
        self.value.take()
    }
    #[inline]
    pub fn time(&self) -> f64 {
        self.t
    }
    #[inline]
    pub fn ieid(&self) -> &IEID {
        &self.ieid
    }
    #[inline]
    pub fn node(&self) -> Option<&str> {
        self.node.as_deref()
    }
}

impl From<ReplicationStateEvent> for ItemState {
    fn from(rse: ReplicationStateEvent) -> Self {
        Self::new(
            rse.status,
            rse.value.unwrap_or_default(),
            rse.act,
            rse.ieid,
            rse.t,
        )
    }
}

// helper object, all fields should be public
#[derive(Debug, Serialize, Deserialize, Clone)]
#[serde(deny_unknown_fields)]
pub struct ReplicationInventoryItem {
    pub oid: OID,
    pub status: Option<ItemStatus>,
    pub value: Option<Value>,
    pub act: Option<usize>,
    pub ieid: IEID,
    pub t: f64,
    pub meta: Option<Value>,
    pub enabled: bool,
}

impl Hash for ReplicationInventoryItem {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.oid.hash(state);
    }
}

impl Eq for ReplicationInventoryItem {}

impl PartialEq for ReplicationInventoryItem {
    fn eq(&self, other: &Self) -> bool {
        self.oid == other.oid
    }
}
