use crate::{EResult, Error};
use elbus::rpc::{Rpc, RpcClient};
use elbus::QoS;
use eva_common::acl::OIDMask;
use eva_common::prelude::*;
use log::error;
use log::trace;
use serde::{Deserialize, Serialize};
use std::collections::HashMap;
use std::sync::Arc;
use std::sync::Mutex;
use std::time::{Duration, Instant};
use tokio::sync::RwLock;
use uuid::Uuid;

pub const ACTION_CREATED: u8 = 0b0000_0000; // created by the core
pub const ACTION_ACCEPTED: u8 = 0b0000_0001; // accepted
pub const ACTION_PENDING: u8 = 0b0000_0010; // queued by the controller
pub const ACTION_RUNNING: u8 = 0b0000_1000; // running by the controller
pub const ACTION_COMPLETED: u8 = 0b0000_1111; // completed successfully
pub const ACTION_FAILED: u8 = 0b1000_0000; // failed to be completed
pub const ACTION_CANCELED: u8 = 0b1000_0001; // canceled in queue
pub const ACTION_TERMINATED: u8 = 0b1000_0010; // terminated while running

pub const ACTION_TOPIC: &str = "ACT/";

const INTERVAL_CLEAN_ACTIONS: Duration = std::time::Duration::from_secs(60);

pub const DEFAULT_KEEP: Duration = Duration::from_secs(600);
pub const DEFAULT_ACTION_PRIORITY: u8 = 100;

pub fn default_action_priority() -> u8 {
    DEFAULT_ACTION_PRIORITY
}

//#[derive(Serialize_repr, Deserialize_repr, Eq, PartialEq, Debug, Copy, Clone)]
#[derive(Serialize, Deserialize, Eq, PartialEq, Debug, Copy, Clone, Hash, PartialOrd)]
#[repr(u8)]
#[serde(rename_all = "lowercase")]
pub enum Status {
    Created = ACTION_CREATED,
    Accepted = ACTION_ACCEPTED,
    Pending = ACTION_PENDING,
    Running = ACTION_RUNNING,
    Completed = ACTION_COMPLETED,
    Failed = ACTION_FAILED,
    Canceled = ACTION_CANCELED,
    Terminated = ACTION_TERMINATED,
}

impl TryFrom<u8> for Status {
    type Error = Error;
    fn try_from(code: u8) -> EResult<Self> {
        match code {
            ACTION_CREATED => Ok(Status::Created),
            ACTION_ACCEPTED => Ok(Status::Accepted),
            ACTION_PENDING => Ok(Status::Pending),
            ACTION_RUNNING => Ok(Status::Running),
            ACTION_COMPLETED => Ok(Status::Completed),
            ACTION_FAILED => Ok(Status::Failed),
            ACTION_CANCELED => Ok(Status::Canceled),
            ACTION_TERMINATED => Ok(Status::Terminated),
            _ => Err(Error::invalid_data(format!(
                "invalid action code: {}",
                code
            ))),
        }
    }
}

#[derive(Serialize, Deserialize)]
pub struct UnitParams {
    status: ItemStatus,
    #[serde(skip_serializing_if = "Option::is_none")]
    value: Option<Value>,
}

#[derive(Serialize, Deserialize)]
pub struct LmacroParams {
    #[serde(skip_serializing_if = "Option::is_none")]
    args: Option<Vec<Value>>,
    #[serde(skip_serializing_if = "Option::is_none")]
    kwargs: Option<HashMap<String, Value>>,
}

#[derive(Serialize, Deserialize)]
#[serde(untagged)]
pub enum Params {
    Unit(UnitParams),
    Lmacro(LmacroParams),
}

#[derive(Deserialize, Debug, Copy, Clone)]
#[serde(rename_all = "lowercase")]
pub enum StatusQuery {
    Waiting,   // not running yet
    Running,   // currently running
    Completed, // completed successfully
    Failed,    // failed, canceled, terminated
    Finished,  // all finished
}

impl StatusQuery {
    fn matches(self, s: Status) -> bool {
        match self {
            StatusQuery::Waiting => s < Status::Running,
            StatusQuery::Running => s == Status::Running,
            StatusQuery::Completed => s == Status::Completed,
            StatusQuery::Failed => s >= Status::Failed,
            StatusQuery::Finished => s >= Status::Completed,
        }
    }
}

#[inline]
fn default_limit() -> u32 {
    1000
}

#[derive(Debug, Deserialize)]
pub struct Filter {
    i: Option<OIDMask>,
    sq: Option<StatusQuery>,
    svc: Option<String>,
    time: Option<f64>,
    #[serde(default = "default_limit")]
    limit: u32,
}

impl Default for Filter {
    fn default() -> Self {
        Self {
            i: None,
            sq: None,
            svc: None,
            time: None,
            limit: default_limit(),
        }
    }
}

impl Filter {
    #[inline]
    fn matches(&self, action: &Action) -> bool {
        if let Some(ref mask) = self.i {
            if !mask.matches(&action.i) {
                return false;
            }
        }
        if let Some(sq) = self.sq {
            if !sq.matches(action.status) {
                return false;
            }
        }
        if let Some(ref svc) = self.svc {
            if svc != &action.target {
                return false;
            }
        }
        if let Some(time) = self.time {
            if let Some(created) = action.time.get(&Status::Created) {
                if created < &time {
                    return false;
                }
            }
        }
        true
    }
}

pub struct UnitAction<'a> {
    pub oid: &'a OID,
    pub status: ItemStatus,
    pub value: Option<Value>,
    pub timeout: Option<Duration>,
    pub priority: u8,
    pub config: Option<Value>,
    pub node: Option<String>,
    pub target: String,
}

pub struct LmacroAction<'a> {
    pub oid: &'a OID,
    pub args: Option<Vec<Value>>,
    pub kwargs: Option<HashMap<String, Value>>,
    pub timeout: Option<Duration>,
    pub priority: u8,
    pub config: Option<Value>,
    pub node: Option<String>,
    pub target: String,
}

#[derive(Serialize)]
pub struct Action {
    uuid: Uuid,
    i: OID,
    #[serde(skip)]
    status: Status,
    #[serde(skip)]
    started: Instant,
    #[serde(skip)]
    time: HashMap<Status, f64>,
    #[serde(
        serialize_with = "crate::time::serialize_opt_duration_as_micros",
        skip_serializing_if = "Option::is_none"
    )]
    timeout: Option<Duration>,
    priority: u8,
    params: Params,
    #[serde(skip_serializing_if = "Option::is_none")]
    config: Option<Value>,
    #[serde(skip_serializing_if = "Option::is_none")]
    node: Option<String>,
    #[serde(skip)]
    out: Option<Value>,
    #[serde(skip)]
    err: Option<Value>,
    #[serde(skip)]
    completed: triggered::Trigger,
    #[serde(skip)]
    core_completed: triggered::Trigger,
    #[serde(skip)]
    target: String,
}

// returned as serialized only
#[derive(Serialize)]
pub struct ActionInfo<'a> {
    uuid: &'a Uuid,
    i: &'a OID,
    status: Status,
    time: &'a HashMap<Status, f64>,
    priority: u8,
    params: &'a Params,
    out: Option<&'a Value>,
    err: Option<&'a Value>,
    node: &'a str,
    svc: &'a str,
}

impl<'a> ActionInfo<'a> {
    fn new(action: &'a Action) -> Self {
        Self {
            uuid: &action.uuid,
            i: &action.i,
            status: action.status,
            time: &action.time,
            priority: action.priority,
            params: &action.params,
            out: action.out.as_ref(),
            err: action.err.as_ref(),
            node: action.node.as_ref().map_or(crate::LOCAL_NODE_ALIAS, |v| v),
            svc: &action.target,
        }
    }
}

impl Action {
    fn create(
        oid: &OID,
        params: Params,
        timeout: Option<Duration>,
        priority: u8,
        config: Option<Value>,
        node: Option<String>,
        target: String,
    ) -> (Self, triggered::Listener, triggered::Listener) {
        let (trigger, listener) = triggered::trigger();
        let (core_trigger, core_listener) = triggered::trigger();
        let mut times = HashMap::new();
        times.insert(Status::Created, crate::time::Time::now().timestamp());
        (
            Self {
                uuid: Uuid::new_v4(),
                i: oid.clone(),
                status: Status::Created,
                started: Instant::now(),
                time: times,
                timeout,
                priority,
                params,
                config,
                out: None,
                err: None,
                completed: trigger,
                core_completed: core_trigger,
                node,
                target,
            },
            listener,
            core_listener,
        )
    }
    #[inline]
    pub fn uuid(&self) -> &Uuid {
        &self.uuid
    }
    #[inline]
    pub fn timeout(&self) -> Option<Duration> {
        self.timeout
    }
    pub fn new_unit(params: UnitAction) -> (Self, triggered::Listener, triggered::Listener) {
        Self::create(
            params.oid,
            Params::Unit(UnitParams {
                status: params.status,
                value: params.value,
            }),
            params.timeout,
            params.priority,
            params.config,
            params.node,
            params.target,
        )
    }
    pub fn new_lmacro(params: LmacroAction) -> (Self, triggered::Listener, triggered::Listener) {
        Self::create(
            params.oid,
            Params::Lmacro(LmacroParams {
                args: params.args,
                kwargs: params.kwargs,
            }),
            params.timeout,
            params.priority,
            params.config,
            params.node,
            params.target,
        )
    }
    fn mark_failed(&mut self, e: Error) {
        if self.status < Status::Completed {
            error!(
                "action {} for {} failed at launch: {}",
                self.uuid, self.i, e
            );
            self.status = Status::Failed;
            self.err = Some(e.to_string().into());
            self.time
                .insert(Status::Failed, crate::time::Time::now().timestamp());
            self.completed.trigger();
            self.core_completed.trigger();
        }
    }
    fn mark_accepted(&mut self) {
        if self.status < Status::Accepted {
            self.status = Status::Accepted;
        }
        self.time
            .insert(Status::Accepted, crate::time::Time::now().timestamp());
    }
}

#[derive(Serialize, Deserialize)]
#[serde(deny_unknown_fields)]
pub struct ActionEvent {
    pub uuid: Uuid,
    pub status: u8,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub out: Option<Value>,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub err: Option<Value>,
}

pub struct Manager {
    actions: Arc<Mutex<HashMap<Uuid, Action>>>,
    keep_for: Mutex<Duration>,
    rpc: RwLock<Option<Arc<RpcClient>>>,
}

impl Default for Manager {
    fn default() -> Self {
        Self {
            actions: <_>::default(),
            keep_for: Mutex::new(DEFAULT_KEEP),
            rpc: <_>::default(),
        }
    }
}

impl Manager {
    #[inline]
    pub async fn set_rpc(&self, rpc: Arc<RpcClient>) {
        self.rpc.write().await.replace(rpc);
    }
    /// # Panics
    ///
    /// Will panic if the mutex is poisoned
    #[inline]
    pub fn rpc(&self) -> &RwLock<Option<Arc<RpcClient>>> {
        &self.rpc
    }
    /// # Panics
    ///
    /// Will panic if the mutex is poisoned
    #[inline]
    pub fn set_keep_for(&self, keep_for: Duration) {
        *self.keep_for.lock().unwrap() = keep_for;
    }
    /// # Panics
    ///
    /// Will panic if the actions mutex is poisoned
    pub fn process_event(&self, event: ActionEvent) -> EResult<()> {
        let mut actions = self.actions.lock().unwrap();
        if let Some(action) = actions.get_mut(&event.uuid) {
            let status: Status = event.status.try_into()?;
            if (action.status as u8) < event.status {
                let need_trigger =
                    event.status >= ACTION_COMPLETED && (action.status as u8) < ACTION_COMPLETED;
                action.status = status;
                action.out = event.out;
                action.err = event.err;
                if need_trigger {
                    action.completed.trigger();
                    action.core_completed.trigger();
                }
            }
            action
                .time
                .insert(status, crate::time::Time::now().timestamp());
        }
        Ok(())
    }
    /// # Panics
    ///
    /// Will panic if the actions mutex is poisoned or rpc is not set
    pub async fn terminate_action(&self, uuid: &Uuid, timeout: Duration) -> EResult<()> {
        #[derive(Serialize)]
        struct ParamsUuid<'a> {
            u: &'a Uuid,
        }
        let (target, payload) = {
            let actions = self.actions.lock().unwrap();
            let action = actions
                .get(uuid)
                .ok_or_else(|| Error::not_found("action not found"))?;
            if action.status >= Status::Completed {
                return Err(Error::failed("action is already finished"));
            }
            (
                action.target.clone(),
                rmp_serde::to_vec_named(&ParamsUuid { u: uuid })?,
            )
        };
        let rpc_c = self.rpc.read().await;
        let rpc = rpc_c.as_ref().unwrap();
        perform(&target, "terminate", &payload, timeout, rpc).await?;
        Ok(())
    }
    /// # Panics
    ///
    /// Will panic if rpc is not set
    pub async fn kill_actions(&self, oid: &OID, target: &str, timeout: Duration) -> EResult<()> {
        #[derive(Serialize)]
        struct ParamsId<'a> {
            i: &'a OID,
        }
        let rpc_c = self.rpc.read().await;
        let rpc = rpc_c.as_ref().unwrap();
        perform(
            target,
            "kill",
            &rmp_serde::to_vec_named(&ParamsId { i: oid })?,
            timeout,
            rpc,
        )
        .await?;
        Ok(())
    }
    /// # Panics
    ///
    /// Will panic if the actions mutex is poisoned
    pub async fn launch_action(
        &self,
        action: Action,
        accepted: triggered::Trigger,
        timeout: Duration,
    ) -> EResult<()> {
        let uuid = action.uuid;
        let payload = rmp_serde::to_vec_named(&action)?;
        let target = action.target.clone();
        let method = match action.params {
            Params::Unit(_) => "action",
            Params::Lmacro(_) => "run",
        };
        self.actions.lock().unwrap().insert(uuid, action);
        accepted.trigger();
        // drop the lock until the launcher is done
        let rpc_c = self.rpc.read().await;
        let rpc = rpc_c.as_ref().unwrap();
        let result = perform(&target, method, &payload, timeout, rpc).await;
        let mut actions = self.actions.lock().unwrap();
        let action = actions
            .get_mut(&uuid)
            .ok_or_else(|| Error::core("action not stored"))?;
        if let Err(e) = result {
            action.mark_failed(e);
        } else {
            action.mark_accepted();
        }
        Ok(())
    }
    /// # Panics
    ///
    /// Will panic if the mutex is poisoned
    pub async fn start(&self) -> EResult<()> {
        let keep_for = *self.keep_for.lock().unwrap();
        let actions = self.actions.clone();
        crate::cleaner!(
            "action_history",
            cleanup_action_history,
            INTERVAL_CLEAN_ACTIONS,
            keep_for,
            &actions
        );
        Ok(())
    }
    /// # Panics
    ///
    /// Will panic if the actions mutex is poisoned
    #[inline]
    pub fn get_action_serialized(&self, uuid: &Uuid) -> EResult<Option<Vec<u8>>> {
        if let Some(action) = self.actions.lock().unwrap().get(uuid) {
            Ok(Some(rmp_serde::to_vec_named(&ActionInfo::new(action))?))
        } else {
            Ok(None)
        }
    }
    /// # Panics
    ///
    /// Will panic if the actions mutex is poisoned
    ///
    #[inline]
    pub fn get_actions_filtered_serialized(&self, filter: &Filter) -> EResult<Vec<u8>> {
        // Actions aren't wrapped to Arc, so they're locked until serialized, no actions can be
        // executed during this period. Consider making filters as light as possible
        let actions = self.actions.lock().unwrap();
        let act = actions
            .values()
            .filter(|a| filter.matches(a))
            .collect::<Vec<&Action>>();
        let len = act.len();
        let result: Vec<ActionInfo> = act
            .iter()
            .skip(if len > filter.limit as usize {
                len - filter.limit as usize
            } else {
                0
            })
            .map(|a| ActionInfo::new(a))
            .collect();
        rmp_serde::to_vec_named(&result).map_err(Into::into)
    }
    /// # Panics
    ///
    /// Will panic if the actions mutex is poisoned
    #[inline]
    pub fn mark_action_timed_out(&self, uuid: &Uuid) {
        if let Some(action) = self.actions.lock().unwrap().get_mut(uuid) {
            action.mark_failed(Error::timeout());
        }
    }
}

#[inline]
async fn perform(
    target: &str,
    method: &str,
    payload: &[u8],
    timeout: Duration,
    rpc: &RpcClient,
) -> EResult<()> {
    tokio::time::timeout(
        timeout,
        rpc.call(target, method, payload.into(), QoS::RealtimeProcessed),
    )
    .await??;
    Ok(())
}

#[allow(clippy::unused_async)]
async fn cleanup_action_history(keep_for: Duration, actions: &Mutex<HashMap<Uuid, Action>>) {
    trace!("cleaning actions");
    let not_before = Instant::now() - keep_for;
    actions
        .lock()
        .unwrap()
        .retain(|_, v| v.started < not_before);
}
