#![cfg_attr(feature = "nostd", no_std)]

#[cfg(feature = "ext")]
#[macro_use]
extern crate lazy_static;

pub use crate::value::to_value;
use serde::{Deserialize, Deserializer, Serialize, Serializer};
use serde_repr::{Deserialize_repr, Serialize_repr};
use std::cmp::Ordering;
use std::collections::{BTreeMap, HashSet};
use std::convert::{TryFrom, TryInto};
use std::fmt;
use std::hash::{Hash, Hasher};
use std::str::FromStr;
pub use value::Value;

pub const LOG_LEVEL_TRACE: u8 = 0;
pub const LOG_LEVEL_DEBUG: u8 = 10;
pub const LOG_LEVEL_INFO: u8 = 20;
pub const LOG_LEVEL_WARN: u8 = 30;
pub const LOG_LEVEL_ERROR: u8 = 40;

#[path = "tools.rs"]
pub mod tools;

#[cfg(feature = "acl")]
#[path = "acl.rs"]
pub mod acl;
#[cfg(feature = "ext")]
#[path = "ext.rs"]
pub mod ext;

#[path = "value/value.rs"]
pub mod value;

pub mod prelude {
    //pub use crate::tools::VecX;
    pub use crate::value::to_value;
    pub use crate::value::Value;
    pub use crate::value::ValueGet;
    pub use crate::EvaError;
    pub use crate::EvaErrorKind;
    pub use crate::EvaItemKind;
    pub use crate::ItemStatus;
    pub use crate::IEID;
    pub use crate::OID;
}

static ERR_INVALID_OID: &str = "Invalid OID format";
static ERR_OID_TOO_LONG: &str = "OID too long";

pub type EvaResult<T> = std::result::Result<T, EvaError>;

pub type ItemStatus = i16;

pub const ITEM_STATUS_ERROR: i16 = -1;

pub const ERR_CODE_NOT_FOUND: i16 = -32001;
pub const ERR_CODE_ACCESS_DENIED: i16 = -32002;
pub const ERR_CODE_SYSTEM_ERROR: i16 = -32003;
pub const ERR_CODE_OTHER: i16 = -32004;
pub const ERR_CODE_NOT_READY: i16 = -32005;
pub const ERR_CODE_UNSUPPORTED: i16 = -32006;
pub const ERR_CODE_CORE_ERROR: i16 = -32007;
pub const ERR_CODE_TIMEOUT: i16 = -32008;
pub const ERR_CODE_INVALID_DATA: i16 = -32009;
pub const ERR_CODE_FUNC_FAILED: i16 = -32010;
pub const ERR_CODE_ABORTED: i16 = -32011;
pub const ERR_CODE_ALREADY_EXISTS: i16 = -32012;
pub const ERR_CODE_BUSY: i16 = -32013;
pub const ERR_CODE_METHOD_NOT_IMPLEMENTED: i16 = -32014;
pub const ERR_CODE_TOKEN_RESTRICTED: i16 = -32015;
pub const ERR_CODE_IO: i16 = -32016;
pub const ERR_CODE_REGISTRY: i16 = -32017;
pub const ERR_CODE_EVAHI_AUTH_REQUIRED: i16 = -32018;

pub const ERR_CODE_PARSE: i16 = -32700;
pub const ERR_CODE_INVALID_REQUEST: i16 = -32600;
pub const ERR_CODE_METHOD_NOT_FOUND: i16 = -32601;
pub const ERR_CODE_INVALID_PARAMS: i16 = -32602;
pub const ERR_CODE_INTERNAL_RPC: i16 = -32603;

pub const ERR_CODE_ELBUS_CLIENT_NOT_REGISTERED: i16 = -32113;
pub const ERR_CODE_ELBUS_DATA: i16 = -32114;
pub const ERR_CODE_ELBUS_IO: i16 = -32115;
pub const ERR_CODE_ELBUS_OTHER: i16 = -32116;
pub const ERR_CODE_ELBUS_NOT_SUPPORTED: i16 = -32117;
pub const ERR_CODE_ELBUS_BUSY: i16 = -32118;
pub const ERR_CODE_ELBUS_NOT_DELIVERED: i16 = -32119;
pub const ERR_CODE_ELBUS_TIMEOUT: i16 = -32120;

#[derive(Serialize_repr, Deserialize_repr, Eq, PartialEq, Debug, Copy, Clone)]
#[repr(i16)]
pub enum EvaErrorKind {
    CoreError = ERR_CODE_CORE_ERROR,
    Unsupported = ERR_CODE_UNSUPPORTED,
    NotReady = ERR_CODE_NOT_READY,
    IOError = ERR_CODE_IO,
    RegistryError = ERR_CODE_REGISTRY,
    InvalidData = ERR_CODE_INVALID_DATA,
    FunctionFailed = ERR_CODE_FUNC_FAILED,
    ResourceNotFound = ERR_CODE_NOT_FOUND,
    ResourceBusy = ERR_CODE_BUSY,
    ResourceAlreadyExists = ERR_CODE_ALREADY_EXISTS,
    AccessDenied = ERR_CODE_ACCESS_DENIED,
    MethodNotImplemented = ERR_CODE_METHOD_NOT_IMPLEMENTED,
    MethodNotFound = ERR_CODE_METHOD_NOT_FOUND,
    InvalidParameter = ERR_CODE_INVALID_PARAMS,
    Timeout = ERR_CODE_TIMEOUT,
    Aborted = ERR_CODE_ABORTED,
    EvaHIAuthenticationRequired = ERR_CODE_EVAHI_AUTH_REQUIRED,
    TokenRestricted = ERR_CODE_TOKEN_RESTRICTED,
    ElbusClientNotRegistered = ERR_CODE_ELBUS_CLIENT_NOT_REGISTERED,
    ElbusData = ERR_CODE_ELBUS_DATA,
    ElbusIo = ERR_CODE_ELBUS_IO,
    ElbusOther = ERR_CODE_ELBUS_OTHER,
    ElbusNotSupported = ERR_CODE_ELBUS_NOT_SUPPORTED,
    ElbusBusy = ERR_CODE_ELBUS_BUSY,
    ElbusNotDelivered = ERR_CODE_ELBUS_NOT_DELIVERED,
    ElbusTimeout = ERR_CODE_ELBUS_TIMEOUT,
    Other = ERR_CODE_OTHER,
}

impl From<i16> for EvaErrorKind {
    fn from(code: i16) -> EvaErrorKind {
        match code {
            x if x == EvaErrorKind::CoreError as i16 => EvaErrorKind::CoreError,
            x if x == EvaErrorKind::Unsupported as i16 => EvaErrorKind::Unsupported,
            x if x == EvaErrorKind::IOError as i16 => EvaErrorKind::IOError,
            x if x == EvaErrorKind::RegistryError as i16 => EvaErrorKind::RegistryError,
            x if x == EvaErrorKind::InvalidData as i16 => EvaErrorKind::InvalidData,
            x if x == EvaErrorKind::FunctionFailed as i16 => EvaErrorKind::FunctionFailed,
            x if x == EvaErrorKind::ResourceNotFound as i16 => EvaErrorKind::ResourceNotFound,
            x if x == EvaErrorKind::ResourceBusy as i16 => EvaErrorKind::ResourceBusy,
            x if x == EvaErrorKind::ResourceAlreadyExists as i16 => {
                EvaErrorKind::ResourceAlreadyExists
            }
            x if x == EvaErrorKind::AccessDenied as i16 => EvaErrorKind::AccessDenied,
            x if x == EvaErrorKind::MethodNotImplemented as i16 => {
                EvaErrorKind::MethodNotImplemented
            }
            x if x == EvaErrorKind::MethodNotFound as i16 => EvaErrorKind::MethodNotFound,
            x if x == EvaErrorKind::InvalidParameter as i16 => EvaErrorKind::InvalidParameter,
            x if x == EvaErrorKind::Timeout as i16 => EvaErrorKind::Timeout,
            x if x == EvaErrorKind::Aborted as i16 => EvaErrorKind::Aborted,
            x if x == EvaErrorKind::EvaHIAuthenticationRequired as i16 => {
                EvaErrorKind::EvaHIAuthenticationRequired
            }
            x if x == EvaErrorKind::TokenRestricted as i16 => EvaErrorKind::TokenRestricted,
            x if x == EvaErrorKind::Other as i16 => EvaErrorKind::Other,
            x if x == EvaErrorKind::NotReady as i16 => EvaErrorKind::NotReady,
            x if x == EvaErrorKind::ElbusClientNotRegistered as i16 => {
                EvaErrorKind::ElbusClientNotRegistered
            }
            x if x == EvaErrorKind::ElbusData as i16 => EvaErrorKind::ElbusData,
            x if x == EvaErrorKind::ElbusIo as i16 => EvaErrorKind::ElbusIo,
            x if x == EvaErrorKind::ElbusOther as i16 => EvaErrorKind::ElbusOther,
            x if x == EvaErrorKind::ElbusNotSupported as i16 => EvaErrorKind::ElbusNotSupported,
            x if x == EvaErrorKind::ElbusBusy as i16 => EvaErrorKind::ElbusBusy,
            x if x == EvaErrorKind::ElbusNotDelivered as i16 => EvaErrorKind::ElbusNotDelivered,
            x if x == EvaErrorKind::ElbusTimeout as i16 => EvaErrorKind::ElbusTimeout,
            _ => EvaErrorKind::Other,
        }
    }
}

impl std::fmt::Display for EvaErrorKind {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(
            f,
            "{}",
            match self {
                EvaErrorKind::CoreError => "Core error",
                EvaErrorKind::Unsupported => "Unsupported",
                EvaErrorKind::IOError => "IO error",
                EvaErrorKind::RegistryError => "Registry error",
                EvaErrorKind::InvalidData => "Invalid data",
                EvaErrorKind::FunctionFailed => "Function failed",
                EvaErrorKind::ResourceNotFound => "Resource not found",
                EvaErrorKind::ResourceBusy => "Resource busy",
                EvaErrorKind::ResourceAlreadyExists => "Resource already exists",
                EvaErrorKind::AccessDenied => "Access denied",
                EvaErrorKind::MethodNotImplemented => "Method not implemented",
                EvaErrorKind::MethodNotFound => "Method not found",
                EvaErrorKind::InvalidParameter => "Invalid parameter",
                EvaErrorKind::Timeout => "Timed out",
                EvaErrorKind::Aborted => "Aborted",
                EvaErrorKind::EvaHIAuthenticationRequired => "EvaHI authentication required",
                EvaErrorKind::TokenRestricted => "Token restricted",
                EvaErrorKind::Other => "Other",
                EvaErrorKind::NotReady => "Not ready",
                EvaErrorKind::ElbusClientNotRegistered => "ELBUS client not registered",
                EvaErrorKind::ElbusData => "ELBUS data error",
                EvaErrorKind::ElbusIo => "ELBUS IO error",
                EvaErrorKind::ElbusOther => "ELBUS error",
                EvaErrorKind::ElbusNotSupported => "ELBUS feature not supported",
                EvaErrorKind::ElbusBusy => "ELBUS busy",
                EvaErrorKind::ElbusNotDelivered => "ELBUS not delivered",
                EvaErrorKind::ElbusTimeout => "ELBUS timed out",
            }
        )
    }
}

#[derive(Debug, Eq, PartialEq)]
pub struct EvaError {
    kind: EvaErrorKind,
    message: Option<String>,
}

impl std::error::Error for EvaError {}

#[cfg(feature = "ext")]
impl From<rmp_serde::encode::Error> for EvaError {
    fn from(error: rmp_serde::encode::Error) -> EvaError {
        EvaError::invalid_data(error)
    }
}

#[cfg(feature = "ext")]
impl From<rmp_serde::decode::Error> for EvaError {
    fn from(error: rmp_serde::decode::Error) -> EvaError {
        EvaError::invalid_data(error)
    }
}

impl From<std::num::ParseIntError> for EvaError {
    fn from(err: std::num::ParseIntError) -> EvaError {
        EvaError::invalid_data(err)
    }
}

impl From<std::num::ParseFloatError> for EvaError {
    fn from(err: std::num::ParseFloatError) -> EvaError {
        EvaError::invalid_data(err)
    }
}

impl From<std::convert::Infallible> for EvaError {
    fn from(_err: std::convert::Infallible) -> EvaError {
        panic!();
    }
}

impl From<ipnetwork::IpNetworkError> for EvaError {
    fn from(err: ipnetwork::IpNetworkError) -> EvaError {
        EvaError::invalid_data(err)
    }
}

impl From<serde_json::Error> for EvaError {
    fn from(error: serde_json::Error) -> Self {
        EvaError::invalid_data(error).into()
    }
}

impl EvaError {
    #[allow(clippy::must_use_candidate)]
    pub fn new<T: fmt::Display>(kind: EvaErrorKind, message: T) -> Self {
        Self {
            kind,
            message: Some(message.to_string()),
        }
    }

    #[allow(clippy::must_use_candidate)]
    pub fn new0(kind: EvaErrorKind) -> Self {
        Self {
            kind,
            message: None,
        }
    }

    #[allow(clippy::must_use_candidate)]
    pub fn newc(kind: EvaErrorKind, message: Option<String>) -> Self {
        Self { kind, message }
    }

    pub fn code(&self) -> i16 {
        self.kind as i16
    }

    #[allow(clippy::must_use_candidate)]
    pub fn e<T: fmt::Display>(kind: EvaErrorKind, message: T) -> Self {
        Self {
            kind,
            message: Some(message.to_string()),
        }
    }

    #[allow(clippy::must_use_candidate)]
    pub fn not_found<T: fmt::Display>(message: T) -> Self {
        Self {
            kind: EvaErrorKind::ResourceNotFound,
            message: Some(message.to_string()),
        }
    }

    #[allow(clippy::must_use_candidate)]
    pub fn not_ready<T: fmt::Display>(message: T) -> Self {
        Self {
            kind: EvaErrorKind::NotReady,
            message: Some(message.to_string()),
        }
    }

    #[allow(clippy::must_use_candidate)]
    pub fn unsupported<T: fmt::Display>(message: T) -> Self {
        Self {
            kind: EvaErrorKind::Unsupported,
            message: Some(message.to_string()),
        }
    }

    #[allow(clippy::must_use_candidate)]
    pub fn registry<T: fmt::Display>(message: T) -> Self {
        Self {
            kind: EvaErrorKind::RegistryError,
            message: Some(message.to_string()),
        }
    }

    #[allow(clippy::must_use_candidate)]
    pub fn busy<T: fmt::Display>(message: T) -> Self {
        Self {
            kind: EvaErrorKind::ResourceBusy,
            message: Some(message.to_string()),
        }
    }

    #[allow(clippy::must_use_candidate)]
    pub fn core<T: fmt::Display>(message: T) -> Self {
        Self {
            kind: EvaErrorKind::CoreError,
            message: Some(message.to_string()),
        }
    }

    #[allow(clippy::must_use_candidate)]
    pub fn io<T: fmt::Display>(message: T) -> Self {
        Self {
            kind: EvaErrorKind::IOError,
            message: Some(message.to_string()),
        }
    }

    #[allow(clippy::must_use_candidate)]
    pub fn duplicate<T: fmt::Display>(message: T) -> Self {
        Self {
            kind: EvaErrorKind::ResourceAlreadyExists,
            message: Some(message.to_string()),
        }
    }

    #[allow(clippy::must_use_candidate)]
    pub fn failed<T: fmt::Display>(message: T) -> Self {
        Self {
            kind: EvaErrorKind::FunctionFailed,
            message: Some(message.to_string()),
        }
    }

    #[allow(clippy::must_use_candidate)]
    pub fn access<T: fmt::Display>(message: T) -> Self {
        Self {
            kind: EvaErrorKind::AccessDenied,
            message: Some(message.to_string()),
        }
    }

    #[allow(clippy::must_use_candidate)]
    pub fn timeout() -> Self {
        Self {
            kind: EvaErrorKind::Timeout,
            message: None,
        }
    }

    #[allow(clippy::must_use_candidate)]
    pub fn aborted() -> Self {
        Self {
            kind: EvaErrorKind::Timeout,
            message: None,
        }
    }

    #[allow(clippy::must_use_candidate)]
    pub fn invalid_data<T: fmt::Display>(message: T) -> Self {
        Self {
            kind: EvaErrorKind::InvalidData,
            message: Some(message.to_string()),
        }
    }

    pub fn kind(&self) -> EvaErrorKind {
        self.kind
    }
    pub fn message(&self) -> Option<&str> {
        self.message.as_deref()
    }
}

impl std::fmt::Display for EvaError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let mut err = self.kind.to_string();
        if let Some(msg) = self.message.as_ref() {
            err += &format!(": {}", msg);
        }
        write!(f, "{}", err)
    }
}

#[derive(Debug, Eq, PartialEq, Clone, Serialize, Deserialize)]
pub struct IEID(u64, u64);

impl IEID {
    #[allow(clippy::must_use_candidate)]
    #[inline]
    pub fn new(b: u64, i: u64) -> Self {
        Self(b, i)
    }
    #[inline]
    pub fn is_phantom(&self) -> bool {
        self.0 == 0
    }
    #[inline]
    pub fn mark_phantom(&mut self) {
        self.0 = 0;
        self.1 = 0;
    }
    #[allow(clippy::must_use_candidate)]
    #[inline]
    pub fn to_value(&self) -> Value {
        let value_b: Value = self.0.into();
        let value_i: Value = self.1.into();
        to_value(vec![value_b, value_i]).unwrap()
    }

    /// Other IEID is newer than current
    #[inline]
    pub fn other_is_newer(&self, other: &IEID) -> bool {
        other.0 > self.0 || (other.0 == self.0 && other.1 > self.1)
    }

    /// Other IEID is less or equal to the current
    #[inline]
    pub fn other_is_less_or_equal(&self, other: &IEID) -> bool {
        other.0 < self.0 || (other.0 == self.0 && other.1 <= self.1)
    }
}

impl TryFrom<&Value> for IEID {
    type Error = EvaError;
    fn try_from(v: &Value) -> EvaResult<Self> {
        if let Value::Seq(s) = v {
            let mut ix = s.iter();
            let ieid_b = if let Some(b) = ix.next() {
                b.try_into()?
            } else {
                return Err(EvaError::invalid_data("First IEID element mismatch"));
            };
            let ieid_i = if let Some(i) = ix.next() {
                i.try_into()?
            } else {
                return Err(EvaError::invalid_data("Second IEID element mismatch"));
            };
            if ix.next().is_some() {
                return Err(EvaError::invalid_data(
                    "Incompatible IEID (more than 2 elements)",
                ));
            }
            Ok(Self(ieid_b, ieid_i))
        } else {
            Err(EvaError::invalid_data("invalid value for IEID"))
        }
    }
}

impl PartialOrd for IEID {
    fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
        match self.0.cmp(&other.0) {
            Ordering::Less => Some(Ordering::Less),
            Ordering::Greater => Some(Ordering::Greater),
            Ordering::Equal => self.1.partial_cmp(&other.1),
        }
    }
}

#[derive(Clone, Eq)]
pub struct OID {
    kind: EvaItemKind,
    oid_str: String,
    tpos: u16,
    grp_pos: Option<u16>,
}

impl PartialEq for OID {
    fn eq(&self, other: &Self) -> bool {
        self.oid_str == other.oid_str
    }
}

impl Ord for OID {
    fn cmp(&self, other: &Self) -> Ordering {
        if self.kind == other.kind {
            self.oid_str.cmp(&other.oid_str)
        } else {
            self.kind.cmp(&other.kind)
        }
    }
}

impl PartialOrd for OID {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        Some(self.cmp(other))
    }
}

const OID_ALLOWED_SYMBOLS: &str = "_.()[]-+\\";

impl OID {
    #[inline]
    fn check_allowed_symbols(s: &str, is_path: bool) -> EvaResult<()> {
        for c in s.chars() {
            if !c.is_alphanumeric() && !(is_path && c == '/') && !OID_ALLOWED_SYMBOLS.contains(c) {
                return Err(EvaError::invalid_data(format!(
                    "Invalid symbol in OID: {}",
                    c
                )));
            }
        }
        Ok(())
    }
    pub fn new(kind: EvaItemKind, group: &str, id: &str) -> EvaResult<Self> {
        OID::check_allowed_symbols(group, true)?;
        OID::check_allowed_symbols(id, false)?;
        if group == "+" || id == "+" {
            return Err(EvaError::invalid_data(
                "OID group or id can not be equal to +",
            ));
        }
        let tp_str = kind.to_string();
        if id.is_empty() || group.is_empty() {
            Err(EvaError::invalid_data(ERR_INVALID_OID))
        } else if tp_str.len() + id.len() + group.len() + 2 > std::u16::MAX as usize {
            Err(EvaError::invalid_data(ERR_OID_TOO_LONG))
        } else {
            let oid_str = format!("{}:{}/{}", kind, group, id);
            let grp_pos = Some((group.len() as u16) + (tp_str.len() as u16) + 1);
            Ok(Self {
                kind,
                oid_str,
                grp_pos,
                tpos: tp_str.len() as u16 + 1,
            })
        }
    }
    pub fn new0(kind: EvaItemKind, id: &str) -> EvaResult<Self> {
        OID::check_allowed_symbols(id, true)?;
        let tp_str = kind.to_string();
        if id.is_empty() {
            Err(EvaError::invalid_data(ERR_INVALID_OID))
        } else if id.len() + tp_str.len() >= std::u16::MAX as usize {
            Err(EvaError::invalid_data(ERR_OID_TOO_LONG))
        } else {
            let grp_pos = id.rfind('/').map(|p| p as u16 + tp_str.len() as u16 + 1);
            let oid_str = format!("{}:{}", kind, id);
            Ok(Self {
                kind,
                oid_str,
                grp_pos,
                tpos: tp_str.len() as u16 + 1,
            })
        }
    }
    #[inline]
    pub fn id(&self) -> &str {
        self.grp_pos.map_or_else(
            || &self.oid_str[self.tpos as usize..],
            |g| &self.oid_str[(g + 1) as usize..],
        )
    }
    #[inline]
    pub fn full_id(&self) -> &str {
        &self.oid_str[self.tpos as usize..]
    }
    #[inline]
    pub fn group(&self) -> Option<&str> {
        self.grp_pos
            .map(|g| &self.oid_str[self.tpos as usize..g as usize])
    }
    #[inline]
    pub fn kind(&self) -> EvaItemKind {
        self.kind
    }
    #[inline]
    pub fn to_path(&self) -> String {
        format!("{}/{}", self.kind, self.full_id())
    }
    #[inline]
    pub fn as_str(&self) -> &str {
        self.oid_str.as_str()
    }
    pub fn serialize_into(&self, target: &mut BTreeMap<Value, Value>) {
        target.insert("oid".into(), self.as_str().into());
        //COMPAT, deprecated, remove in 4.2
        target.insert("full_id".into(), self.full_id().into());
        target.insert("id".into(), self.id().into());
        target.insert(
            "group".into(),
            self.group()
                .map_or_else(|| Value::Option(None), |g| g.into()),
        );
        target.insert("type".into(), self.kind.into());
    }
    pub fn from_str_type(tp: EvaItemKind, s: &str) -> EvaResult<Self> {
        if let Some(tpos) = s.find(':') {
            let otp: EvaItemKind = s[..tpos].parse()?;
            if otp == tp {
                Self::new0(tp, &s[tpos + 1..])
            } else {
                Err(EvaError::invalid_data(format!(
                    "OID type mismatch, expected: {}, found: {}",
                    tp, otp
                )))
            }
        } else {
            OID::new0(tp, s)
        }
    }
}

impl FromStr for OID {
    type Err = EvaError;
    fn from_str(s: &str) -> Result<Self, Self::Err> {
        s.find([':', '/']).map_or(
            Err(EvaError::invalid_data(format!(
                "{}: {}",
                ERR_INVALID_OID, s
            ))),
            |tpos| {
                let tp: EvaItemKind = s[..tpos].parse()?;
                Self::new0(tp, &s[tpos + 1..])
            },
        )
    }
}

impl TryFrom<&Value> for OID {
    type Error = EvaError;
    fn try_from(value: &Value) -> Result<OID, Self::Error> {
        let s: &str = value.try_into()?;
        s.parse()
    }
}

impl Serialize for OID {
    #[inline]
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        serializer.serialize_str(self.as_str())
    }
}

// in case of problems with visitor
#[inline]
pub fn deserialize_oid<'de, D>(deserializer: D) -> Result<OID, D::Error>
where
    D: Deserializer<'de>,
{
    let buf = String::deserialize(deserializer)?;
    buf.parse().map_err(serde::de::Error::custom)
}

impl<'de> Deserialize<'de> for OID {
    #[inline]
    fn deserialize<D>(deserializer: D) -> Result<OID, D::Error>
    where
        D: Deserializer<'de>,
    {
        let s: String = Deserialize::deserialize(deserializer)?;
        s.parse().map_err(serde::de::Error::custom)
    }
}

impl fmt::Display for OID {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{}", self.oid_str)
    }
}

impl fmt::Debug for OID {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{}", self.oid_str)
    }
}

impl Hash for OID {
    fn hash<H: Hasher>(&self, hasher: &mut H) {
        (self.kind as u16).hash(hasher);
        self.full_id().hash(hasher);
    }
}

impl From<OID> for Value {
    fn from(oid: OID) -> Value {
        oid.as_str().into()
    }
}

impl From<&OID> for Value {
    fn from(oid: &OID) -> Value {
        oid.as_str().into()
    }
}

impl TryFrom<Value> for OID {
    type Error = EvaError;
    fn try_from(value: Value) -> EvaResult<OID> {
        match value {
            Value::String(s) => Ok(s.parse()?),
            _ => Err(EvaError::invalid_data("Expected string")),
        }
    }
}

impl TryFrom<Value> for HashSet<OID> {
    type Error = EvaError;
    fn try_from(value: Value) -> EvaResult<HashSet<OID>> {
        match value {
            Value::Seq(vec) => {
                let mut result = HashSet::new();
                for v in vec {
                    result.insert(v.try_into()?);
                }
                Ok(result)
            }
            Value::String(s) => {
                let mut result = HashSet::new();
                for v in s.split(',') {
                    result.insert(v.parse()?);
                }
                Ok(result)
            }
            _ => Err(EvaError::invalid_data("Expected vec or string")),
        }
    }
}

impl From<HashSet<OID>> for Value {
    fn from(v: HashSet<OID>) -> Value {
        Value::Seq(v.iter().map(|oid| to_value(oid).unwrap()).collect())
    }
}

#[derive(Debug, Eq, PartialEq, Copy, Clone, Ord, PartialOrd, Hash)]
#[repr(u16)]
pub enum EvaItemKind {
    Unit = 100,
    Sensor = 101,
    Lvar = 200,
    Lmacro = 300,
}

impl EvaItemKind {
    pub fn as_str(&self) -> &str {
        match self {
            EvaItemKind::Unit => "unit",
            EvaItemKind::Sensor => "sensor",
            EvaItemKind::Lvar => "lvar",
            EvaItemKind::Lmacro => "lmacro",
        }
    }
}

impl fmt::Display for EvaItemKind {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{}", self.as_str())
    }
}

impl From<EvaItemKind> for Value {
    fn from(src: EvaItemKind) -> Value {
        src.to_string().into()
    }
}

impl FromStr for EvaItemKind {
    type Err = EvaError;
    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s {
            "unit" | "U" => Ok(EvaItemKind::Unit),
            "sensor" | "S" => Ok(EvaItemKind::Sensor),
            "lvar" => Ok(EvaItemKind::Lvar),
            "lmacro" | "K" => Ok(EvaItemKind::Lmacro),
            _ => Err(EvaError::new(
                EvaErrorKind::InvalidData,
                format!("Invalid item type: {}", s),
            )),
        }
    }
}

impl TryFrom<&Value> for EvaItemKind {
    type Error = EvaError;
    fn try_from(value: &Value) -> Result<EvaItemKind, Self::Error> {
        TryInto::<&str>::try_into(value)?.parse()
    }
}

impl TryFrom<&Value> for Vec<EvaItemKind> {
    type Error = EvaError;
    fn try_from(value: &Value) -> Result<Vec<EvaItemKind>, Self::Error> {
        let data: Vec<&str> = value.try_into()?;
        let mut result = Vec::new();
        for d in data {
            result.push(TryInto::<&str>::try_into(d)?.parse()?);
        }
        Ok(result)
    }
}

impl TryFrom<Value> for EvaItemKind {
    type Error = EvaError;
    fn try_from(value: Value) -> Result<EvaItemKind, Self::Error> {
        TryInto::<String>::try_into(value)?.parse()
    }
}

impl TryFrom<Value> for Vec<EvaItemKind> {
    type Error = EvaError;
    fn try_from(value: Value) -> Result<Vec<EvaItemKind>, Self::Error> {
        let data: Vec<String> = value.try_into()?;
        let mut result = Vec::new();
        for d in data {
            result.push(TryInto::<String>::try_into(d)?.parse()?);
        }
        Ok(result)
    }
}

#[cfg(feature = "elbus-rpc")]
impl From<elbus::rpc::RpcError> for EvaError {
    fn from(err: elbus::rpc::RpcError) -> Self {
        EvaError {
            kind: err.code().into(),
            message: err
                .data()
                .map(|v| std::str::from_utf8(&v).unwrap_or_default().to_owned()),
        }
    }
}

#[cfg(test)]
mod tests {
    use super::to_value;
    use super::{EvaError, EvaItemKind, Value, IEID, OID};
    use std::convert::TryInto;

    #[test]
    fn test_oid() {
        let oid: OID = "sensor:env/room1/temp1".parse().unwrap();
        assert_eq!(oid.id(), "temp1");
        assert_eq!(oid.full_id(), "env/room1/temp1");
        assert_eq!(oid.group().unwrap(), "env/room1");
        assert_eq!(oid.kind, EvaItemKind::Sensor);
        let oid: OID = "sensor/env/room1/temp1".parse().unwrap();
        assert_eq!(oid.id(), "temp1");
        assert_eq!(oid.full_id(), "env/room1/temp1");
        assert_eq!(oid.group().unwrap(), "env/room1");
        assert_eq!(oid.kind, EvaItemKind::Sensor);
        assert_eq!("sensorx:env/temp1".parse::<OID>().is_err(), true);
        assert_eq!("sensorxenv/temp1".parse::<OID>().is_err(), true);
        assert_eq!("sensorxenv/:temp1".parse::<OID>().is_err(), true);
        assert_eq!("sensor|temp1".parse::<OID>().is_err(), true);
        assert_eq!("sensor:".parse::<OID>().is_err(), true);
        let oid = OID::new0(EvaItemKind::Sensor, "tests/test1").unwrap();
        assert_eq!(oid.id(), "test1");
        assert_eq!(oid.group().unwrap(), "tests");
        assert_eq!(oid.kind(), EvaItemKind::Sensor);
        let oid = OID::new0(EvaItemKind::Sensor, "tests/room1/test1").unwrap();
        assert_eq!(oid.id(), "test1");
        assert_eq!(oid.group().unwrap(), "tests/room1");
        assert_eq!(oid.kind(), EvaItemKind::Sensor);
    }

    #[test]
    fn test_ieid() {
        assert_eq!(IEID::new(1, 1) == IEID::new(1, 1), true);
        assert_eq!(IEID::new(2, 1) > IEID::new(1, 9), true);
        assert_eq!(IEID::new(2, 2) < IEID::new(3, 1), true);
        assert_eq!(IEID::new(2, 4) > IEID::new(2, 2), true);
        assert_eq!(IEID::new(2, 4) < IEID::new(2, 5), true);
    }

    #[test]
    fn test_try_into_vec() {
        let v = vec!["1", "2", "3"];
        let value = to_value(v.clone()).unwrap();
        let result: Vec<&str> = value.as_ref().try_into().unwrap();
        let value2: Value = "1,2,3".into();
        assert_eq!(result, v);
        let result: Vec<&str> = value2.as_ref().try_into().unwrap();
        assert_eq!(result, v);
    }

    #[test]
    fn test_try_into_bool() {
        assert_eq!(
            TryInto::<bool>::try_into(Value::String("True".to_owned())).unwrap(),
            true
        );
        assert_eq!(
            TryInto::<bool>::try_into(Value::String("Trux".to_owned())).is_err(),
            true
        );
        assert_eq!(TryInto::<bool>::try_into(Value::U64(0)).unwrap(), false);
        assert_eq!(TryInto::<bool>::try_into(Value::F64(1.0)).unwrap(), true);
        assert_eq!(TryInto::<bool>::try_into(Value::F64(2.0)).is_err(), true);
    }

    #[test]
    fn test_err() {
        assert_eq!(format!("{}", EvaError::timeout()), "Timed out");
        assert_eq!(
            format!("{}", EvaError::not_found("test")),
            "Resource not found: test"
        );
    }
}
