//! EVM Toolkit Function Selector Database.
//!
//! To roughly quote [4byte.directory](https://www.4byte.directory):
//!
//! > Function calls in the Ethereum Virtual Machine are specified by the first
//! > four bytes of data sent with a transaction. These function selectors are
//! > defined as the first four bytes of the Keccak-256 hash of the canonical
//! > representation of the function signature. Since this is a one-way
//! > operation, it is not possible to derive the human-readable representation
//! > of the function (signature) from the four byte selector. This database is
//! > meant to allow mapping those bytes signatures back to their
//! > human-readable versions.
#![deny(unsafe_code)]
#![deny(missing_docs)]
#![deny(unreachable_pub)]
#![deny(missing_debug_implementations)]

use lazy_static::lazy_static;

use std::collections::BTreeMap;
use std::convert::TryFrom;

const PACKED: &[u8] = include_bytes!("database.br");

lazy_static! {
    static ref SIGNATURES: (BTreeMap<u32, u32>, Vec<Vec<String>>) = {
        let mut input = brotli::Decompressor::new(PACKED, 4096);
        bincode::deserialize_from(&mut input).unwrap()
    };
}

/// Attempt to retrieve the human-readable signature given a selector.
///
/// ## Example
///
/// ```
/// use etk_4byte::reverse_selector;
///
/// let signatures: Vec<_> = reverse_selector(0x3bb2dead).collect();
///
/// assert_eq!(signatures[0], "initialFundsReleaseNumerator()");
/// assert_eq!(signatures[1], "resolveAddressLight(address)");
/// ```
pub fn reverse_selector(selector: u32) -> impl Iterator<Item = &'static str> {
    SIGNATURES
        .0
        .get(&selector)
        .map(|v| usize::try_from(*v).unwrap())
        .map(|v| SIGNATURES.1[v].iter())
        .unwrap_or_else(|| [].iter())
        .map(String::as_str)
}
