use std::{collections::HashMap, fmt, io::BufRead, path::PathBuf, process::Command};

use glob::glob;
use serde::{Deserialize, Serialize};
use thiserror::Error;

use crate::{abi::Abi, types::Bytes};

/// The name of the `solc` binary on the system
const SOLC: &str = "solc";

type Result<T> = std::result::Result<T, SolcError>;

#[derive(Debug, Error)]
pub enum SolcError {
    /// Internal solc error
    #[error("Solc Error: {0}")]
    SolcError(String),
    /// Deserialization error
    #[error(transparent)]
    SerdeJson(#[from] serde_json::Error),
}

#[derive(Clone, Debug, Serialize, Deserialize)]
/// The result of a solc compilation
pub struct CompiledContract {
    /// The contract's ABI
    pub abi: Abi,
    /// The contract's bytecode
    pub bytecode: Bytes,
    /// The contract's runtime bytecode
    pub runtime_bytecode: Bytes,
}

/// Solidity Compiler Bindings
///
/// Assumes that `solc` is installed and available in the caller's $PATH. Any calls
/// will **panic** otherwise.
///
/// By default, it uses 200 optimizer runs and Istanbul as the EVM version
///
/// # Examples
///
/// ```no_run
/// use ethers_core::utils::Solc;
///
/// # fn main() -> Result<(), Box<dyn std::error::Error>> {
/// // Give it a glob
/// let contracts = Solc::new("./contracts/*")
///     .optimizer(Some(200))
///     .build()?;
///
/// // this will return None if the specified contract did not exist in the compiled
/// // files
/// let contract = contracts.get("SimpleStorage").expect("contract not found");
/// # Ok(())
/// # }
/// ```
pub struct Solc {
    /// The path to the Solc binary
    pub solc_path: Option<PathBuf>,

    /// The path where contracts will be read from
    pub paths: Vec<String>,

    /// Number of optimizer runs. None for no optimization
    pub optimizer: Option<usize>,

    /// Evm Version
    pub evm_version: EvmVersion,

    /// Paths for importing other libraries
    pub allowed_paths: Vec<PathBuf>,

    /// Additional arguments to pass to solc
    pub args: Vec<String>,
}

impl Solc {
    /// Instantiates The Solc builder with the provided glob of Solidity files
    pub fn new(path: &str) -> Self {
        // Convert the glob to a vector of string paths
        // TODO: This might not be the most robust way to do this
        let paths = glob(path)
            .expect("could not get glob")
            .map(|path| path.expect("path not found").to_string_lossy().to_string())
            .collect::<Vec<String>>();
        Self::new_with_paths(paths)
    }

    /// Instantiates the Solc builder for the provided paths
    pub fn new_with_paths(paths: Vec<String>) -> Self {
        Self {
            paths,
            solc_path: None,
            optimizer: Some(200), // default optimizer runs = 200
            evm_version: EvmVersion::Istanbul,
            allowed_paths: Vec::new(),
            args: Vec::new(),
        }
    }

    /// Gets the ABI for the contracts
    pub fn build_raw(self) -> Result<HashMap<String, CompiledContractStr>> {
        let path = self.solc_path.unwrap_or_else(|| PathBuf::from(SOLC));

        let mut command = Command::new(&path);
        let version = Solc::version(Some(path));

        command.arg("--combined-json").arg("abi,bin,bin-runtime");

        if (version.starts_with("0.5") && self.evm_version < EvmVersion::Istanbul)
            || !version.starts_with("0.4")
        {
            command
                .arg("--evm-version")
                .arg(self.evm_version.to_string());
        }

        if let Some(runs) = self.optimizer {
            command
                .arg("--optimize")
                .arg("--optimize-runs")
                .arg(runs.to_string());
        }

        command.args(self.args);

        for path in self.paths {
            command.arg(path);
        }

        let command = command.output().expect("could not run `solc`");

        if !command.status.success() {
            return Err(SolcError::SolcError(
                String::from_utf8_lossy(&command.stderr).to_string(),
            ));
        }

        // Deserialize the output
        let mut output: serde_json::Value = serde_json::from_slice(&command.stdout)?;
        let contract_values = output["contracts"].as_object_mut().ok_or_else(|| {
            SolcError::SolcError("no contracts found in `solc` output".to_string())
        })?;

        let mut contracts = HashMap::with_capacity(contract_values.len());

        for (name, contract) in contract_values {
            if let serde_json::Value::String(bin) = contract["bin"].take() {
                let name = name
                    .rsplit(':')
                    .next()
                    .expect("could not strip fname")
                    .to_owned();

                // abi could be an escaped string (solc<=0.7) or an array (solc>=0.8)
                let abi = match contract["abi"].take() {
                    serde_json::Value::String(abi) => abi,
                    val @ serde_json::Value::Array(_) => val.to_string(),
                    val => {
                        return Err(SolcError::SolcError(format!(
                            "Expected abi in solc output, found {:?}",
                            val
                        )))
                    }
                };

                let runtime_bin =
                    if let serde_json::Value::String(bin) = contract["bin-runtime"].take() {
                        bin
                    } else {
                        panic!("no runtime bytecode found")
                    };
                contracts.insert(
                    name,
                    CompiledContractStr {
                        abi,
                        bin,
                        runtime_bin,
                    },
                );
            } else {
                return Err(SolcError::SolcError(
                    "could not find `bin` in solc output".to_string(),
                ));
            }
        }

        Ok(contracts)
    }

    /// Builds the contracts and returns a hashmap for each named contract
    pub fn build(self) -> Result<HashMap<String, CompiledContract>> {
        // Build, and then get the data in the correct format
        let contracts = self
            .build_raw()?
            .into_iter()
            .map(|(name, contract)| {
                // parse the ABI
                let abi = serde_json::from_str(&contract.abi)
                    .expect("could not parse `solc` abi, this should never happen");

                // parse the bytecode
                let bytecode = hex::decode(contract.bin)
                    .expect("solc did not produce valid bytecode")
                    .into();

                // parse the runtime bytecode
                let runtime_bytecode = hex::decode(contract.runtime_bin)
                    .expect("solc did not produce valid runtime-bytecode")
                    .into();
                (
                    name,
                    CompiledContract {
                        abi,
                        bytecode,
                        runtime_bytecode,
                    },
                )
            })
            .collect::<HashMap<String, CompiledContract>>();

        Ok(contracts)
    }

    /// Returns the output of `solc --version`
    ///
    /// # Panics
    ///
    /// If `solc` is not found
    pub fn version(solc_path: Option<PathBuf>) -> String {
        let solc_path = solc_path.unwrap_or_else(|| PathBuf::from(SOLC));
        let command_output = Command::new(&solc_path)
            .arg("--version")
            .output()
            .unwrap_or_else(|_| panic!("`{:?}` not found", solc_path));

        let version = command_output
            .stdout
            .lines()
            .last()
            .expect("expected version in solc output")
            .expect("could not get solc version");

        // Return the version trimmed
        version.replace("Version: ", "")
    }

    /// Sets the EVM version for compilation
    pub fn evm_version(mut self, version: EvmVersion) -> Self {
        self.evm_version = version;
        self
    }

    /// Sets the path to the solc binary
    pub fn solc_path(mut self, path: PathBuf) -> Self {
        self.solc_path = Some(std::fs::canonicalize(path).unwrap());
        self
    }

    /// Sets the optimizer runs (default = 200). None indicates no optimization
    ///
    /// ```rust,no_run
    /// use ethers_core::utils::Solc;
    ///
    /// // No optimization
    /// let contracts = Solc::new("./contracts/*")
    ///     .optimizer(None)
    ///     .build().unwrap();
    ///
    /// // Some(200) is default, optimizer on with 200 runs
    /// // .arg() allows passing arbitrary args to solc command
    /// let optimized_contracts = Solc::new("./contracts/*")
    ///     .optimizer(Some(200))
    ///     .arg("--metadata-hash=none")
    ///     .build().unwrap();
    /// ```
    pub fn optimizer(mut self, runs: Option<usize>) -> Self {
        self.optimizer = runs;
        self
    }

    /// Sets the allowed paths for using files from outside the same directory
    // TODO: Test this
    pub fn allowed_paths(mut self, paths: Vec<PathBuf>) -> Self {
        self.allowed_paths = paths;
        self
    }

    /// Adds an argument to pass to solc
    pub fn arg<T: Into<String>>(mut self, arg: T) -> Self {
        self.args.push(arg.into());
        self
    }

    /// Adds multiple arguments to pass to solc
    pub fn args<I, S>(mut self, args: I) -> Self
    where
        I: IntoIterator<Item = S>,
        S: Into<String>,
    {
        for arg in args {
            self = self.arg(arg);
        }
        self
    }
}

#[derive(Clone, Debug, PartialEq, Eq, PartialOrd, Ord)]
pub enum EvmVersion {
    Homestead,
    TangerineWhistle,
    SpuriusDragon,
    Constantinople,
    Petersburg,
    Istanbul,
    Berlin,
}

impl fmt::Display for EvmVersion {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let string = match self {
            EvmVersion::Homestead => "homestead",
            EvmVersion::TangerineWhistle => "tangerineWhistle",
            EvmVersion::SpuriusDragon => "spuriusDragon",
            EvmVersion::Constantinople => "constantinople",
            EvmVersion::Petersburg => "petersburg",
            EvmVersion::Istanbul => "istanbul",
            EvmVersion::Berlin => "berlin",
        };
        write!(f, "{}", string)
    }
}

#[derive(Clone, Debug, Serialize, Deserialize)]
// Helper struct for deserializing the solc string outputs
struct SolcOutput {
    contracts: HashMap<String, CompiledContractStr>,
    version: String,
}

#[derive(Clone, Debug, Serialize, Deserialize)]
/// Helper struct for deserializing the solc string outputs
pub struct CompiledContractStr {
    /// The contract's raw ABI
    pub abi: String,
    /// The contract's bytecode in hex
    pub bin: String,
    /// The contract's runtime bytecode in hex
    pub runtime_bin: String,
}
