//! Implements artifact format generated by [hardhat-deploy] plugin.
//!
//! There are three distinct artifact formats.
//!
//! First is called "hardhat export", it is a JSON file that contains
//! information about a single network and all contracts deployed on it.
//! It can be generated with `hardhat export` command.
//!
//! Second is called "hardhat multi-export", it contains information about
//! multiple networks, for each network it contains information about
//! all contracts deployed on it. It can be generated with
//! `hardhat export --export-all` command.
//!
//! Third is hardhat's `deployments` directory. It contains more details about
//! contracts than the previous two formats. Specifically, it has info about
//! deployed bytecode, deployment transaction receipt, documentation for
//! contract methods, and some other things. Given that it is a directory,
//! there are obvious issues with loading it over network. For this reason,
//! we don't recommend this export format for public libraries that export
//! contracts.
//!
//! All three formats are supported by [`HardHatLoader`], see its documentation
//! for info and limitations.
//!
//! [hardhat-deploy]: https://github.com/wighawag/hardhat-deploy

use crate::artifact::Artifact;
use crate::contract::Network;
use crate::errors::ArtifactError;
use crate::{Address, Contract, DeploymentInformation, TransactionHash};
use serde::Deserialize;
use serde_json::{from_reader, from_slice, from_str, from_value, Value};
use std::collections::HashMap;
use std::fs::File;
use std::io::{BufReader, Read};
use std::path::Path;

/// Loads hardhat artifacts generated via `--export` and `--export-all`.
///
/// # Limitations
///
/// In hardhat, a contract could have different ABIs on different networks.
/// This could happen when deploying test versions of contracts.
/// Ethcontract does not support this. Parsing such artifact will result
/// in an error. You'll have to rename contracts, or filter out networks
/// with [`networks_allow_list`].
///
/// Another limitation is that hardhat allows having multiple networks
/// with the same chain ID. For example, you can have `rinkeby`
/// and `rinkeby-testing`. Both have chain ID of `4`, but contract addresses
/// and ABIs can be different. Ethcontract does not support this, so you'll
/// have to filter such networks. See [#545] for more info.
///
/// [#545]: https://github.com/gnosis/ethcontract-rs/issues/545.
#[must_use = "hardhat loaders do nothing unless you load them"]
pub struct HardHatLoader {
    /// Override for artifact's origin. If `None`, origin
    /// will be derived automatically.
    pub origin: Option<String>,

    /// List of allowed network names and chain IDs.
    ///
    /// When loading a contract, networks with names that aren't found
    /// in this list will be completely ignored. Contracts from these networks
    /// will not be loaded. You can use this mechanism to bypass
    /// the requirement that a contract must have the same ABI on all networks.
    ///
    /// Empty list means that all networks are allowed.
    pub networks_allow_list: Vec<NetworkEntry>,

    /// List of denied network names and chain IDs.
    ///
    /// When loading a contract, networks with names that are found
    /// in this list will be completely ignored.
    ///
    /// Empty list means that no networks are denied.
    ///
    /// Deny list takes precedence over allow list. That is, if network
    /// appears in both, it will be denied.
    pub networks_deny_list: Vec<NetworkEntry>,

    /// List of allowed contract names.
    ///
    /// When loading artifact, loader will only load contracts if their names
    /// are present in this list.
    ///
    /// Empty list means that all contracts are allowed.
    pub contracts_allow_list: Vec<String>,

    /// List of denied contract names.
    ///
    /// When loading artifact, loader will not load contracts if their names
    /// are present in this list.
    ///
    /// Empty list means that no contracts are denied.
    ///
    /// Deny list takes precedence over allow list. That is, if contract
    /// appears in both, it will be denied.
    pub contracts_deny_list: Vec<String>,
}

impl HardHatLoader {
    /// Creates a new hardhat loader.
    pub fn new() -> Self {
        HardHatLoader {
            origin: None,
            networks_deny_list: Vec::new(),
            networks_allow_list: Vec::new(),
            contracts_allow_list: Vec::new(),
            contracts_deny_list: Vec::new(),
        }
    }

    /// Creates a new hardhat loader and sets an override for artifact's origins.z
    pub fn with_origin(origin: impl Into<String>) -> Self {
        HardHatLoader {
            origin: Some(origin.into()),
            networks_deny_list: Vec::new(),
            networks_allow_list: Vec::new(),
            contracts_allow_list: Vec::new(),
            contracts_deny_list: Vec::new(),
        }
    }

    /// Sets new override for artifact's origin. See [`origin`] for more info.
    ///
    /// [`origin`]: #structfield.origin
    pub fn origin(mut self, origin: impl Into<String>) -> Self {
        self.origin = Some(origin.into());
        self
    }

    /// Adds chain id to the list of [`allowed networks`].
    ///
    /// [`allowed networks`]: #structfield.networks_allow_list
    pub fn allow_network_by_chain_id(mut self, network: impl Into<String>) -> Self {
        self.networks_allow_list
            .push(NetworkEntry::ByChainId(network.into()));
        self
    }

    /// Adds network name to the list of [`allowed networks`].
    ///
    /// [`allowed networks`]: #structfield.networks_allow_list
    pub fn allow_network_by_name(mut self, network: impl Into<String>) -> Self {
        self.networks_allow_list
            .push(NetworkEntry::ByName(network.into()));
        self
    }

    /// Adds chain id to the list of [`denied networks`].
    ///
    /// [`denied networks`]: #structfield.networks_deny_list
    pub fn deny_network_by_chain_id(mut self, network: impl Into<String>) -> Self {
        self.networks_deny_list
            .push(NetworkEntry::ByChainId(network.into()));
        self
    }

    /// Adds network name to the list of [`denied networks`].
    ///
    /// [`denied networks`]: #structfield.networks_deny_list
    pub fn deny_network_by_name(mut self, network: impl Into<String>) -> Self {
        self.networks_deny_list
            .push(NetworkEntry::ByName(network.into()));
        self
    }

    /// Adds contract name to the list of [`allowed contracts`].
    ///
    /// [`allowed contracts`]: #structfield.contracts_allow_list
    pub fn allow_contract(mut self, contract: impl Into<String>) -> Self {
        self.contracts_allow_list.push(contract.into());
        self
    }

    /// Adds contract name to the list of [`denied contracts`].
    ///
    /// [`denied contracts`]: #structfield.contracts_deny_list
    pub fn deny_contract(mut self, contract: impl Into<String>) -> Self {
        self.contracts_deny_list.push(contract.into());
        self
    }

    /// Loads an artifact from a JSON value.
    pub fn load_from_reader(&self, f: Format, v: impl Read) -> Result<Artifact, ArtifactError> {
        self.load_artifact(f, "<unknown>", v, from_reader, from_reader)
    }

    /// Loads an artifact from bytes of JSON text.
    pub fn load_from_slice(&self, f: Format, v: &[u8]) -> Result<Artifact, ArtifactError> {
        self.load_artifact(f, "<unknown>", v, from_slice, from_slice)
    }

    /// Loads an artifact from string of JSON text.
    pub fn load_from_str(&self, f: Format, v: &str) -> Result<Artifact, ArtifactError> {
        self.load_artifact(f, "<unknown>", v, from_str, from_str)
    }

    /// Loads an artifact from a loaded JSON value.
    pub fn load_from_value(&self, f: Format, v: Value) -> Result<Artifact, ArtifactError> {
        self.load_artifact(f, "<unknown>", v, from_value, from_value)
    }

    /// Loads an artifact from disk.
    pub fn load_from_file(
        &self,
        f: Format,
        p: impl AsRef<Path>,
    ) -> Result<Artifact, ArtifactError> {
        let path = p.as_ref();
        let file = File::open(path)?;
        let reader = BufReader::new(file);
        self.load_artifact(f, path.display(), reader, from_reader, from_reader)
    }

    /// Loads an artifact from `deployments` directory.
    pub fn load_from_directory(&self, p: impl AsRef<Path>) -> Result<Artifact, ArtifactError> {
        self._load_from_directory(p.as_ref())
    }

    /// Helper for `load_from_directory`. We use this helper function to avoid
    /// making a big chunk of code generic over `AsRef<Path>`.
    ///
    /// # Implementation note
    ///
    /// Layout of the `deployments` directory looks like this:
    ///
    /// ```text
    /// deployments
    ///  |
    ///  +-- main
    ///  |    |
    ///  |    +-- .chainId
    ///  |    |
    ///  |    +-- Contract1.json
    ///  |    |
    ///  |    +-- Contract2.json
    ///  |    |
    ///  |    ...
    ///  |
    ///  +-- rinkeby
    ///  |    |
    ///  |    +-- .chainId
    ///  |    |
    ///  |    +-- Contract1.json
    ///  |    |
    ///  |    +-- Contract2.json
    ///  |    |
    ///  |    ...
    ///  |
    ///  ...
    ///  ```
    ///
    /// There's a directory for each network. Within network's directory,
    /// there's a `.chainId` file containing chain identifier encoded
    /// as plain text. Next to `.chainId` file, there are JSON files for each
    /// contract deployed to this network.
    fn _load_from_directory(&self, p: &Path) -> Result<Artifact, ArtifactError> {
        let mut artifact = Artifact::with_origin(p.display().to_string());

        let mut chain_id_buf = String::new();

        for chain_entry in p.read_dir()? {
            let chain_entry = chain_entry?;

            let chain_path = chain_entry.path();
            if !chain_path.is_dir() {
                continue;
            }

            let chain_id_file = chain_path.join(".chainId");
            if !chain_id_file.exists() {
                continue;
            }

            chain_id_buf.clear();
            File::open(chain_id_file)?.read_to_string(&mut chain_id_buf)?;
            let chain_id = chain_id_buf.trim().to_string();

            let chain_name = chain_path
                .file_name()
                .ok_or_else(|| {
                    std::io::Error::new(
                        std::io::ErrorKind::Other,
                        format!("unable to get directory name for path {:?}", chain_path),
                    )
                })?
                .to_string_lossy();

            if !self.network_allowed(&chain_id, &chain_name) {
                continue;
            }

            for contract_entry in chain_path.read_dir()? {
                let contract_entry = contract_entry?;

                let contract_path = contract_entry.path();
                if !contract_path.is_file() {
                    continue;
                }

                let mut contract_name = contract_path
                    .file_name()
                    .ok_or_else(|| {
                        std::io::Error::new(
                            std::io::ErrorKind::Other,
                            format!("unable to get file name for path {:?}", contract_path),
                        )
                    })?
                    .to_string_lossy()
                    .into_owned();

                if !contract_name.ends_with(".json") {
                    continue;
                }

                contract_name.truncate(contract_name.len() - ".json".len());

                if !self.contract_allowed(&contract_name) {
                    continue;
                }

                let HardHatContract {
                    address,
                    transaction_hash,
                    mut contract,
                } = {
                    let file = File::open(contract_path)?;
                    let reader = BufReader::new(file);
                    from_reader(reader)?
                };

                contract.name = contract_name;

                self.add_contract_to_artifact(
                    &mut artifact,
                    contract,
                    chain_id.clone(),
                    address,
                    transaction_hash,
                )?;
            }
        }

        Ok(artifact)
    }

    fn load_artifact<T>(
        &self,
        format: Format,
        origin: impl ToString,
        source: T,
        single_loader: impl FnOnce(T) -> serde_json::Result<HardHatExport>,
        multi_loader: impl FnOnce(T) -> serde_json::Result<HardHatMultiExport>,
    ) -> Result<Artifact, ArtifactError> {
        let origin = self.origin.clone().unwrap_or_else(|| origin.to_string());

        let mut artifact = Artifact::with_origin(origin);

        match format {
            Format::SingleExport => {
                let loaded = single_loader(source)?;
                self.fill_artifact(&mut artifact, loaded)?
            }
            Format::MultiExport => {
                let loaded = multi_loader(source)?;
                self.fill_artifact_multi(&mut artifact, loaded)?
            }
        }

        Ok(artifact)
    }

    fn fill_artifact(
        &self,
        artifact: &mut Artifact,
        export: HardHatExport,
    ) -> Result<(), ArtifactError> {
        if self.network_allowed(&export.chain_id, &export.chain_name) {
            for (name, contract) in export.contracts {
                let HardHatContract {
                    address,
                    transaction_hash,
                    mut contract,
                } = contract;

                if !self.contract_allowed(&name) {
                    continue;
                }

                contract.name = name;

                self.add_contract_to_artifact(
                    artifact,
                    contract,
                    export.chain_id.clone(),
                    address,
                    transaction_hash,
                )?;
            }
        }

        Ok(())
    }

    fn fill_artifact_multi(
        &self,
        artifact: &mut Artifact,
        export: HardHatMultiExport,
    ) -> Result<(), ArtifactError> {
        for (_, export) in export.networks {
            for (_, export) in export {
                self.fill_artifact(artifact, export)?;
            }
        }

        Ok(())
    }

    fn add_contract_to_artifact(
        &self,
        artifact: &mut Artifact,
        contract: Contract,
        chain_id: String,
        address: Address,
        transaction_hash: Option<TransactionHash>,
    ) -> Result<(), ArtifactError> {
        let mut contract = match artifact.get_mut(&contract.name) {
            Some(existing_contract) => {
                if existing_contract.abi != contract.abi {
                    return Err(ArtifactError::AbiMismatch(contract.name));
                }

                existing_contract
            }
            None => artifact.insert(contract).inserted_contract,
        };

        let deployment_information = transaction_hash.map(DeploymentInformation::TransactionHash);

        if contract.networks.contains_key(&chain_id) {
            Err(ArtifactError::DuplicateChain(chain_id))
        } else {
            contract.networks_mut().insert(
                chain_id,
                Network {
                    address,
                    deployment_information,
                },
            );

            Ok(())
        }
    }

    fn contract_allowed(&self, name: &str) -> bool {
        !self.contract_explicitly_denied(name)
            && (self.contracts_allow_list.is_empty() || self.contract_explicitly_allowed(name))
    }

    fn contract_explicitly_allowed(&self, name: &str) -> bool {
        self.contracts_allow_list.iter().any(|x| x == name)
    }

    fn contract_explicitly_denied(&self, name: &str) -> bool {
        self.contracts_deny_list.iter().any(|x| x == name)
    }

    fn network_allowed(&self, chain_id: &str, chain_name: &str) -> bool {
        !self.network_explicitly_denied(chain_id, chain_name)
            && (self.networks_allow_list.is_empty()
                || self.network_explicitly_allowed(chain_id, chain_name))
    }

    fn network_explicitly_allowed(&self, chain_id: &str, chain_name: &str) -> bool {
        self.networks_allow_list
            .iter()
            .any(|x| x.matches(chain_id, chain_name))
    }

    fn network_explicitly_denied(&self, chain_id: &str, chain_name: &str) -> bool {
        self.networks_deny_list
            .iter()
            .any(|x| x.matches(chain_id, chain_name))
    }
}

impl Default for HardHatLoader {
    fn default() -> Self {
        HardHatLoader::new()
    }
}

/// Artifact format.
#[derive(Copy, Clone, Debug, Eq, PartialEq)]
pub enum Format {
    /// Contracts for a single network. Generated with `hardhat export`.
    SingleExport,

    /// Contracts for all networks. Generated with `hardhat export --export-all`.
    MultiExport,
}

/// Network allow-deny entry.
#[derive(Clone, Debug)]
pub enum NetworkEntry {
    /// Network identified by chain ID.
    ByChainId(String),

    /// Network identified by its name specified in `hardhat.config.js`.
    ByName(String),
}

impl NetworkEntry {
    fn matches(&self, chain_id: &str, chain_name: &str) -> bool {
        match self {
            NetworkEntry::ByChainId(id) => chain_id == id,
            NetworkEntry::ByName(name) => chain_name == name,
        }
    }
}

#[derive(Deserialize)]
struct HardHatMultiExport {
    #[serde(flatten)]
    networks: HashMap<String, HashMap<String, HardHatExport>>,
}

#[derive(Deserialize)]
struct HardHatExport {
    #[serde(rename = "name")]
    chain_name: String,
    #[serde(rename = "chainId")]
    chain_id: String,
    contracts: HashMap<String, HardHatContract>,
}

#[derive(Deserialize)]
struct HardHatContract {
    address: Address,
    #[serde(rename = "transactionHash")]
    transaction_hash: Option<TransactionHash>,
    #[serde(flatten)]
    contract: Contract,
}

#[cfg(test)]
mod test {
    use super::*;
    use std::path::PathBuf;
    use web3::ethabi::ethereum_types::BigEndianHash;
    use web3::types::{H256, U256};

    fn address(address: u8) -> Address {
        Address::from(H256::from_uint(&U256::from(address)))
    }

    #[test]
    fn load_single() {
        let json = r#"
          {
            "name": "mainnet",
            "chainId": "1",
            "contracts": {
              "A": {
                "address": "0x000000000000000000000000000000000000000A"
              },
              "B": {
                "address": "0x000000000000000000000000000000000000000B"
              }
            }
          }
        "#;

        let artifact = HardHatLoader::new()
            .load_from_str(Format::SingleExport, json)
            .unwrap();

        assert_eq!(artifact.len(), 2);

        let a = artifact.get("A").unwrap();
        assert_eq!(a.name, "A");
        assert_eq!(a.networks.len(), 1);
        assert_eq!(a.networks["1"].address, address(0xA));

        let b = artifact.get("B").unwrap();
        assert_eq!(b.name, "B");
        assert_eq!(b.networks.len(), 1);
        assert_eq!(b.networks["1"].address, address(0xB));
    }

    static MULTI_EXPORT: &str = r#"
      {
        "1": {
          "mainnet": {
            "name": "mainnet",
            "chainId": "1",
            "contracts": {
              "A": {
                "address": "0x000000000000000000000000000000000000000A"
              },
              "B": {
                "address": "0x000000000000000000000000000000000000000B"
              }
            }
          }
        },
        "4": {
          "rinkeby": {
            "name": "rinkeby",
            "chainId": "4",
            "contracts": {
              "A": {
                "address": "0x00000000000000000000000000000000000000AA"
              }
            }
          }
        }
      }
    "#;

    #[test]
    fn load_multi() {
        let artifact = HardHatLoader::new()
            .load_from_str(Format::MultiExport, MULTI_EXPORT)
            .unwrap();

        assert_eq!(artifact.len(), 2);

        let a = artifact.get("A").unwrap();
        assert_eq!(a.name, "A");
        assert_eq!(a.networks.len(), 2);
        assert_eq!(a.networks["1"].address, address(0xA));
        assert_eq!(a.networks["4"].address, address(0xAA));

        let b = artifact.get("B").unwrap();
        assert_eq!(b.name, "B");
        assert_eq!(b.networks.len(), 1);
        assert_eq!(b.networks["1"].address, address(0xB));
    }

    #[test]
    fn load_multi_duplicate_networks_ok() {
        let json = r#"
          {
            "1": {
              "mainnet": {
                "name": "mainnet",
                "chainId": "1",
                "contracts": {
                  "A": {
                    "address": "0x000000000000000000000000000000000000000A"
                  }
                }
              },
              "mainnet_beta": {
                "name": "mainnet_beta",
                "chainId": "1",
                "contracts": {
                  "B": {
                    "address": "0x000000000000000000000000000000000000000B"
                  }
                }
              }
            }
          }
        "#;

        let artifact = HardHatLoader::new()
            .load_from_str(Format::MultiExport, json)
            .unwrap();

        assert_eq!(artifact.len(), 2);

        let a = artifact.get("A").unwrap();
        assert_eq!(a.name, "A");
        assert_eq!(a.networks.len(), 1);
        assert_eq!(a.networks["1"].address, address(0xA));

        let b = artifact.get("B").unwrap();
        assert_eq!(b.name, "B");
        assert_eq!(b.networks.len(), 1);
        assert_eq!(b.networks["1"].address, address(0xB));
    }

    #[test]
    fn load_multi_duplicate_networks_err() {
        let json = r#"
          {
            "1": {
              "mainnet": {
                "name": "mainnet",
                "chainId": "1",
                "contracts": {
                  "A": {
                    "address": "0x000000000000000000000000000000000000000A"
                  }
                }
              },
              "mainnet_beta": {
                "name": "mainnet_beta",
                "chainId": "1",
                "contracts": {
                  "A": {
                    "address": "0x00000000000000000000000000000000000000AA"
                  }
                }
              }
            }
          }
        "#;

        let err = HardHatLoader::new().load_from_str(Format::MultiExport, json);

        match err {
            Err(ArtifactError::DuplicateChain(chain_id)) => assert_eq!(chain_id, "1"),
            Err(unexpected_err) => panic!("unexpected error {:?}", unexpected_err),
            _ => panic!("didn't throw an error"),
        }
    }

    #[test]
    fn load_multi_mismatching_abi() {
        let json = r#"
          {
            "1": {
              "mainnet": {
                "name": "mainnet",
                "chainId": "1",
                "contracts": {
                  "A": {
                    "address": "0x000000000000000000000000000000000000000A",
                    "abi": [
                      {
                        "constant": false,
                        "inputs": [],
                        "name": "foo",
                        "outputs": [],
                        "payable": false,
                        "stateMutability": "nonpayable",
                        "type": "function"
                      }
                    ]
                  }
                }
              }
            },
            "4": {
              "rinkeby": {
                "name": "rinkeby",
                "chainId": "4",
                "contracts": {
                  "A": {
                    "address": "0x00000000000000000000000000000000000000AA",
                    "abi": [
                      {
                        "constant": false,
                        "inputs": [],
                        "name": "bar",
                        "outputs": [],
                        "payable": false,
                        "stateMutability": "nonpayable",
                        "type": "function"
                      }
                    ]
                  }
                }
              }
            }
          }
        "#;

        let err = HardHatLoader::new().load_from_str(Format::MultiExport, json);

        match err {
            Err(ArtifactError::AbiMismatch(name)) => assert_eq!(name, "A"),
            Err(unexpected_err) => panic!("unexpected error {:?}", unexpected_err),
            _ => panic!("didn't throw an error"),
        }
    }

    static NETWORK_CONFLICTS: &str = r#"
      {
        "1": {
          "mainnet": {
            "name": "mainnet",
            "chainId": "1",
            "contracts": {
              "A": {
                "address": "0x000000000000000000000000000000000000000A"
              }
            }
          },
          "mainnet_beta": {
            "name": "mainnet_beta",
            "chainId": "1",
            "contracts": {
              "A": {
                "address": "0x00000000000000000000000000000000000000AA",
                "abi": [
                  {
                    "constant": false,
                    "inputs": [],
                    "name": "test_method",
                    "outputs": [],
                    "payable": false,
                    "stateMutability": "nonpayable",
                    "type": "function"
                  }
                ]
              }
            }
          }
        },
        "4": {
          "rinkeby": {
            "name": "rinkeby",
            "chainId": "4",
            "contracts": {
              "A": {
                "address": "0x00000000000000000000000000000000000000BA"
              }
            }
          }
        }
      }
    "#;

    #[test]
    fn load_multi_allow_by_name() {
        let artifact = HardHatLoader::new()
            .allow_network_by_name("mainnet")
            .allow_network_by_name("rinkeby")
            .load_from_str(Format::MultiExport, NETWORK_CONFLICTS)
            .unwrap();

        assert_eq!(artifact.len(), 1);

        let a = artifact.get("A").unwrap();
        assert_eq!(a.name, "A");
        assert_eq!(a.networks.len(), 2);
        assert_eq!(a.networks["1"].address, address(0xA));
        assert_eq!(a.networks["4"].address, address(0xBA));
    }

    #[test]
    fn load_multi_allow_by_chain_id() {
        let artifact = HardHatLoader::new()
            .allow_network_by_chain_id("4")
            .load_from_str(Format::MultiExport, NETWORK_CONFLICTS)
            .unwrap();

        assert_eq!(artifact.len(), 1);

        let a = artifact.get("A").unwrap();
        assert_eq!(a.name, "A");
        assert_eq!(a.networks.len(), 1);
        assert_eq!(a.networks["4"].address, address(0xBA));
    }

    #[test]
    fn load_multi_deny_by_name() {
        let artifact = HardHatLoader::new()
            .deny_network_by_name("mainnet_beta")
            .load_from_str(Format::MultiExport, NETWORK_CONFLICTS)
            .unwrap();

        assert_eq!(artifact.len(), 1);

        let a = artifact.get("A").unwrap();
        assert_eq!(a.name, "A");
        assert_eq!(a.networks.len(), 2);
        assert_eq!(a.networks["1"].address, address(0xA));
        assert_eq!(a.networks["4"].address, address(0xBA));
    }

    #[test]
    fn load_multi_deny_by_chain_id() {
        let artifact = HardHatLoader::new()
            .deny_network_by_chain_id("1")
            .load_from_str(Format::MultiExport, NETWORK_CONFLICTS)
            .unwrap();

        assert_eq!(artifact.len(), 1);

        let a = artifact.get("A").unwrap();
        assert_eq!(a.name, "A");
        assert_eq!(a.networks.len(), 1);
        assert_eq!(a.networks["4"].address, address(0xBA));
    }

    #[test]
    fn load_multi_allow_contract_name() {
        let artifact = HardHatLoader::new()
            .allow_contract("A")
            .load_from_str(Format::MultiExport, MULTI_EXPORT)
            .unwrap();

        assert_eq!(artifact.len(), 1);

        let a = artifact.get("A").unwrap();
        assert_eq!(a.name, "A");
        assert_eq!(a.networks.len(), 2);
        assert_eq!(a.networks["1"].address, address(0xA));
        assert_eq!(a.networks["4"].address, address(0xAA));

        let artifact = HardHatLoader::new()
            .allow_contract("X")
            .load_from_str(Format::MultiExport, MULTI_EXPORT)
            .unwrap();

        assert_eq!(artifact.len(), 0);
    }

    #[test]
    fn load_multi_deny_contract_name() {
        let artifact = HardHatLoader::new()
            .deny_contract("A")
            .load_from_str(Format::MultiExport, MULTI_EXPORT)
            .unwrap();

        assert_eq!(artifact.len(), 1);

        let a = artifact.get("B").unwrap();
        assert_eq!(a.name, "B");
        assert_eq!(a.networks.len(), 1);
        assert_eq!(a.networks["1"].address, address(0xB));

        let artifact = HardHatLoader::new()
            .deny_contract("X")
            .load_from_str(Format::MultiExport, MULTI_EXPORT)
            .unwrap();

        assert_eq!(artifact.len(), 2);

        let a = artifact.get("A").unwrap();
        assert_eq!(a.name, "A");
        assert_eq!(a.networks.len(), 2);
        assert_eq!(a.networks["1"].address, address(0xA));
        assert_eq!(a.networks["4"].address, address(0xAA));

        let b = artifact.get("B").unwrap();
        assert_eq!(b.name, "B");
        assert_eq!(b.networks.len(), 1);
        assert_eq!(b.networks["1"].address, address(0xB));
    }

    fn hardhat_dir() -> PathBuf {
        let mut path = PathBuf::from(env!("CARGO_MANIFEST_DIR"));
        path.push("../examples/hardhat/deployments");
        path
    }

    #[test]
    fn load_from_directory() {
        let artifact = HardHatLoader::new()
            .load_from_directory(hardhat_dir())
            .unwrap();

        assert_eq!(artifact.len(), 1);

        let a = artifact.get("DeployedContract").unwrap();
        assert_eq!(a.name, "DeployedContract");
        assert_eq!(a.networks.len(), 2);
        assert_eq!(
            a.networks["4"].address,
            "0x4E29B76eC7d20c58A6B156CB464594a4ae39FdEd"
                .parse()
                .unwrap()
        );
        assert_eq!(
            a.networks["4"].deployment_information,
            Some(DeploymentInformation::TransactionHash(
                "0x0122d15a8d394b8f9e45c15b7d3e5365bbf7122a15952246676e2fe7eb858f35"
                    .parse()
                    .unwrap()
            ))
        );
        assert_eq!(
            a.networks["1337"].address,
            "0x29BE0588389993e7064C21f00761303eb51373F5"
                .parse()
                .unwrap()
        );
        assert_eq!(
            a.networks["1337"].deployment_information,
            Some(DeploymentInformation::TransactionHash(
                "0xe0631d7f749fe73f94e59f6e25ff9b925980e8e29ed67b8f862ec76a783ea06e"
                    .parse()
                    .unwrap()
            ))
        );
    }
}
