use anyhow::Result;
use clap::Parser;
use esquel::{
    mermaid::{Flow, Mermaid},
    SqlDialect,
};
use std::fs;
use std::io::Write;

#[derive(Parser, Debug)]
#[clap(version, about, long_about = None)]
struct Args {
    #[clap(short, long)]
    /// Path to sql script or raw sql
    input: String,

    #[clap(arg_enum, short, long, default_value_t = SqlDialect::Generic)]
    dialect: SqlDialect,

    #[clap(arg_enum, short, long, default_value_t = Flow::TB)]
    /// Direction of the flowchart
    flow: Flow,

    #[clap(short, long)]
    /// Path to output file
    out: Option<String>,

    #[clap(long)]
    /// Do not show icons
    no_icons: bool,
}

fn main() -> Result<()> {
    let Args {
        input,
        dialect,
        flow,
        out,
        no_icons,
    } = Args::parse();
    let sql = match fs::read_to_string(&input) {
        Ok(sql) => sql,
        _ => input,
    };
    let mut mermaid = Mermaid::try_new(&sql, dialect)?;
    mermaid.flow(flow).no_icons(no_icons);
    let flowchart = mermaid.to_flowchart();
    if let Some(out) = out {
        let mut out = fs::File::create(out)?;
        write!(out, "{flowchart}")?;
    } else {
        write!(std::io::stdout(), "{flowchart}")?;
    }
    Ok(())
}
