//! Wallet-like features
use derive_more::{From, Into};
use ergo_lib::chain::transaction::TxIoVec;
use wasm_bindgen::prelude::*;

pub mod derivation_path;
pub mod ext_pub_key;
pub mod ext_secret_key;
pub mod mnemonic;

use crate::transaction::TransactionHintsBag;
use crate::{
    box_coll::ErgoBoxes,
    ergo_state_ctx::ErgoStateContext,
    error_conversion::to_js,
    secret_key::{SecretKey, SecretKeys},
    transaction::reduced::ReducedTransaction,
    transaction::Transaction,
    transaction::UnsignedTransaction,
};

/// A collection of secret keys. This simplified signing by matching the secret keys to the correct inputs automatically.
#[wasm_bindgen]
#[derive(From, Into)]
pub struct Wallet(ergo_lib::wallet::Wallet);

#[wasm_bindgen]
impl Wallet {
    /// Create wallet instance loading secret key from mnemonic
    /// Returns None if a DlogSecretKey cannot be parsed from the provided phrase
    #[wasm_bindgen]
    pub fn from_mnemonic(mnemonic_phrase: &str, mnemonic_pass: &str) -> Option<Wallet> {
        Some(ergo_lib::wallet::Wallet::from_mnemonic(mnemonic_phrase, mnemonic_pass)?.into())
    }

    /// Create wallet using provided secret key
    #[wasm_bindgen]
    pub fn from_secrets(secret: &SecretKeys) -> Wallet {
        Wallet(ergo_lib::wallet::Wallet::from_secrets(secret.into()))
    }

    /// Add a secret to the wallets prover
    #[wasm_bindgen]
    pub fn add_secret(&mut self, secret: &SecretKey) {
        self.0.add_secret(secret.clone().into());
    }

    /// Sign a transaction:
    /// `tx` - transaction to sign
    /// `boxes_to_spend` - boxes corresponding to [`UnsignedTransaction::inputs`]
    /// `data_boxes` - boxes corresponding to [`UnsignedTransaction::data_inputs`]
    #[wasm_bindgen]
    pub fn sign_transaction(
        &self,
        _state_context: &ErgoStateContext,
        tx: &UnsignedTransaction,
        boxes_to_spend: &ErgoBoxes,
        data_boxes: &ErgoBoxes,
    ) -> Result<Transaction, JsValue> {
        let boxes_to_spend = TxIoVec::from_vec(boxes_to_spend.clone().into()).map_err(to_js)?;
        let data_boxes = {
            let d: Vec<_> = data_boxes.clone().into();
            if d.is_empty() {
                None
            } else {
                Some(TxIoVec::from_vec(d).map_err(to_js)?)
            }
        };
        let tx_context = ergo_lib::wallet::signing::TransactionContext::new(
            tx.0.clone(),
            boxes_to_spend,
            data_boxes,
        )
        .map_err(to_js)?;
        self.0
            .sign_transaction(tx_context, &_state_context.clone().into(), None)
            .map_err(to_js)
            .map(Transaction::from)
    }

    /// Sign a multi signature transaction:
    /// `tx` - transaction to sign
    /// `boxes_to_spend` - boxes corresponding to [`UnsignedTransaction::inputs`]
    /// `data_boxes` - boxes corresponding to [`UnsignedTransaction::data_inputs`]
    /// `tx_hints` - transaction hints bag corresponding to [`TransactionHintsBag`]
    #[wasm_bindgen]
    pub fn sign_transaction_multi(
        &self,
        _state_context: &ErgoStateContext,
        tx: &UnsignedTransaction,
        boxes_to_spend: &ErgoBoxes,
        data_boxes: &ErgoBoxes,
        tx_hints: &TransactionHintsBag,
    ) -> Result<Transaction, JsValue> {
        let boxes_to_spend = TxIoVec::from_vec(boxes_to_spend.clone().into()).map_err(to_js)?;
        let data_boxes = {
            let d: Vec<_> = data_boxes.clone().into();
            if d.is_empty() {
                None
            } else {
                Some(TxIoVec::from_vec(d).map_err(to_js)?)
            }
        };
        let tx_context = ergo_lib::wallet::signing::TransactionContext::new(
            tx.0.clone(),
            boxes_to_spend,
            data_boxes,
        )
        .map_err(to_js)?;
        self.0
            .sign_transaction(
                tx_context,
                &_state_context.clone().into(),
                Some(&tx_hints.0),
            )
            .map_err(to_js)
            .map(Transaction::from)
    }

    /// Sign a transaction:
    /// `reduced_tx` - reduced transaction, i.e. unsigned transaction where for each unsigned input
    /// added a script reduction result.
    #[wasm_bindgen]
    pub fn sign_reduced_transaction(
        &self,
        reduced_tx: &ReducedTransaction,
    ) -> Result<Transaction, JsValue> {
        self.0
            .sign_reduced_transaction(reduced_tx.clone().into(), None)
            .map_err(to_js)
            .map(Transaction::from)
    }

    /// Sign a multi signature reduced transaction:
    /// `reduced_tx` - reduced transaction, i.e. unsigned transaction where for each unsigned input
    /// added a script reduction result.
    /// `tx_hints` - transaction hints bag corresponding to [`TransactionHintsBag`]
    #[wasm_bindgen]
    pub fn sign_reduced_transaction_multi(
        &self,
        reduced_tx: &ReducedTransaction,
        tx_hints: &TransactionHintsBag,
    ) -> Result<Transaction, JsValue> {
        self.0
            .sign_reduced_transaction(reduced_tx.clone().into(), Some(&tx_hints.0))
            .map_err(to_js)
            .map(Transaction::from)
    }

    /// Generate Commitments for unsigned tx
    #[wasm_bindgen]
    pub fn generate_commitments(
        &self,
        _state_context: &ErgoStateContext,
        tx: &UnsignedTransaction,
        boxes_to_spend: &ErgoBoxes,
        data_boxes: &ErgoBoxes,
    ) -> Result<TransactionHintsBag, JsValue> {
        let boxes_to_spend = TxIoVec::from_vec(boxes_to_spend.clone().into()).map_err(to_js)?;
        let data_boxes = {
            let d: Vec<_> = data_boxes.clone().into();
            if d.is_empty() {
                None
            } else {
                Some(TxIoVec::from_vec(d).map_err(to_js)?)
            }
        };
        let tx_context = ergo_lib::wallet::signing::TransactionContext::new(
            tx.0.clone(),
            boxes_to_spend,
            data_boxes,
        )
        .map_err(to_js)?;
        self.0
            .generate_commitments(tx_context, &_state_context.clone().into())
            .map_err(to_js)
            .map(TransactionHintsBag::from)
    }

    /// Generate Commitments for reduced Transaction
    #[wasm_bindgen]
    pub fn generate_commitments_for_reduced_transaction(
        &self,
        reduced_tx: &ReducedTransaction,
    ) -> Result<TransactionHintsBag, JsValue> {
        self.0
            .generate_commitments_for_reduced_transaction(reduced_tx.clone().into())
            .map_err(to_js)
            .map(TransactionHintsBag::from)
    }
}
