
//!
//! Lookup tools for EPSG Coordinate Reference System data.
//!
//! See https://epsg.org/terms-of-use.html.
//! # Example 
//! ```
//! use epsg::references::get_crs;
//! let wgs84 = get_crs("EPSG:4326").unwrap();
//! assert_eq!(wgs84.coord_ref_sys_code, 4326);
//! assert_eq!(wgs84.coord_ref_sys_name, "WGS 84");
//! assert_eq!(wgs84.coord_ref_sys_kind, "geographic 2D");
//! assert_eq!(wgs84.coord_sys_code, 6422);
//! assert_eq!(wgs84.datum_code, 6326);
//! assert_eq!(wgs84.base_crs_code, 4979);
//! assert_eq!(wgs84.remarks, "");
//! assert_eq!(wgs84.data_source, "EPSG");
//! ```
use phf::{phf_map};
use crate::CRS;

static COORDINATE_REFS: phf::Map<&'static str, CRS> = phf_map! {
    
    "EPSG:2000" => CRS {
        coord_ref_sys_code: 2000,
        coord_ref_sys_name: "Anguilla 1957 / British West Indies Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4600,
        remarks: "",
        information_source: "Ordnance Survey of Great Britain",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2001" => CRS {
        coord_ref_sys_code: 2001,
        coord_ref_sys_name: "Antigua 1943 / British West Indies Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4601,
        remarks: "",
        information_source: "Ordnance Survey of Great Britain",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2041" => CRS {
        coord_ref_sys_code: 2041,
        coord_ref_sys_name: "Abidjan 1987 / UTM zone 30N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4143,
        remarks: "Replaces Locodjo 65 / UTM 30N (EPSG code 2040).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2043" => CRS {
        coord_ref_sys_code: 2043,
        coord_ref_sys_name: "Abidjan 1987 / UTM zone 29N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4143,
        remarks: "Replaces Locodjo 65 / UTM 29N (EPSG code 2042).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2136" => CRS {
        coord_ref_sys_code: 2136,
        coord_ref_sys_name: "Accra / Ghana National Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4404,
        datum_code: -1, 
        base_crs_code: 4168,
        remarks: "Ellipsoid semi-major axis (a)&#x3D;20926201 Gold Coast feet. ProjCRS sometimes found in metric form: 1 Gold Coast foot &#x3D; 0.3047997101815 m.
Replaced by Leigon / Ghana Metric Grid from 1978.",
        information_source: "Ordnance Survey International",
        data_source: "EPSG",
        revision_date: "2000-10-19",
        deprecated: 0, 
    },

    "EPSG:2137" => CRS {
        coord_ref_sys_code: 2137,
        coord_ref_sys_name: "Accra / TM 1 NW",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4168,
        remarks: "",
        information_source: "Various oil industry sources",
        data_source: "EPSG",
        revision_date: "2000-10-19",
        deprecated: 0, 
    },

    "EPSG:2155" => CRS {
        coord_ref_sys_code: 2155,
        coord_ref_sys_name: "American Samoa 1962 / American Samoa Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4497,
        datum_code: -1, 
        base_crs_code: 4169,
        remarks: "Superseded by projCRS 2156 as of mid-2000.",
        information_source: "US National Geodetic Survey (NGS) http://www.ngs.noaa.gov/",
        data_source: "EPSG",
        revision_date: "2000-10-19",
        deprecated: 1, 
    },

    "EPSG:2165" => CRS {
        coord_ref_sys_code: 2165,
        coord_ref_sys_name: "Abidjan 1987 / TM 5 NW",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4143,
        remarks: "",
        information_source: "Various oil industry sources",
        data_source: "EPSG",
        revision_date: "2001-06-05",
        deprecated: 0, 
    },

    "EPSG:2194" => CRS {
        coord_ref_sys_code: 2194,
        coord_ref_sys_name: "American Samoa 1962 / American Samoa Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4497,
        datum_code: -1, 
        base_crs_code: 4169,
        remarks: "Superseded by projCRS 2156 as of mid-2000.",
        information_source: "US National Geodetic Survey (NGS) http://www.ngs.noaa.gov/",
        data_source: "EPSG",
        revision_date: "2001-08-28",
        deprecated: 1, 
    },

    "EPSG:2199" => CRS {
        coord_ref_sys_code: 2199,
        coord_ref_sys_name: "Albanian 1987 / Gauss Kruger zone 4",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4191,
        remarks: "",
        information_source: "EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2001-11-06",
        deprecated: 1, 
    },

    "EPSG:2296" => CRS {
        coord_ref_sys_code: 2296,
        coord_ref_sys_name: "Ammassalik 1958 / Greenland zone 7 east",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 1031,
        datum_code: -1, 
        base_crs_code: 4196,
        remarks: "Historically also found with coordinate system axis abbreviations N/E (CS code 4501); second axis has abbreviation E but is positive to the west.",
        information_source: "Kort &amp; Matrikelstyrelsen, Copenhagen.",
        data_source: "EPSG",
        revision_date: "2012-02-13",
        deprecated: 0, 
    },

    "EPSG:2318" => CRS {
        coord_ref_sys_code: 2318,
        coord_ref_sys_name: "Ain el Abd / Aramco Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4204,
        remarks: "Used by Saudi Aramco when area of interest crosses UTM zone boundary. Adopted by partners for Core Venture 1 (South Ghawar) area.",
        information_source: "Saudi Aramco",
        data_source: "EPSG",
        revision_date: "2002-02-12",
        deprecated: 0, 
    },

    "EPSG:2462" => CRS {
        coord_ref_sys_code: 2462,
        coord_ref_sys_name: "Albanian 1987 / Gauss-Kruger zone 4",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4191,
        remarks: "",
        information_source: "EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2014-11-21",
        deprecated: 0, 
    },

    "EPSG:3102" => CRS {
        coord_ref_sys_code: 3102,
        coord_ref_sys_name: "American Samoa 1962 / American Samoa Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4497,
        datum_code: -1, 
        base_crs_code: 4169,
        remarks: "Replaced by NAD83(HARN) / UTM zone 2S (projCRS 2195) as of mid-2000.",
        information_source: "US National Geodetic Survey (NGS) http://www.ngs.noaa.gov/",
        data_source: "EPSG",
        revision_date: "2005-05-21",
        deprecated: 0, 
    },

    "EPSG:3110" => CRS {
        coord_ref_sys_code: 3110,
        coord_ref_sys_name: "AGD66 / Vicgrid66",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4202,
        remarks: "May also be used to display Victoria, New South Wales, Tasmania and South Australia as a single entity. Replaced by Vicgrid94 (CRS code 3111) with effect from 2nd February 2000. Prior to 2000 this CRS was known as VICGRID.",
        information_source: "Victoria Land Registry; http://www.land.vic.gov.au",
        data_source: "EPSG",
        revision_date: "2016-12-15",
        deprecated: 0, 
    },

    "EPSG:4143" => CRS {
        coord_ref_sys_code: 4143,
        coord_ref_sys_name: "Abidjan 1987",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6143, 
        base_crs_code: -1,
        remarks: "Replaces Locodjo 1965 (EPSG code 4142).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2016-12-15",
        deprecated: 0, 
    },

    "EPSG:4168" => CRS {
        coord_ref_sys_code: 4168,
        coord_ref_sys_name: "Accra",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6168, 
        base_crs_code: -1,
        remarks: "Ellipsoid semi-major axis (a)&#x3D;20926201 exactly Gold Coast feet. 
Replaced by Leigon (code 4250) in 1978.",
        information_source: "Ordnance Survey International",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4169" => CRS {
        coord_ref_sys_code: 4169,
        coord_ref_sys_name: "American Samoa 1962",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6169, 
        base_crs_code: -1,
        remarks: "",
        information_source: "NIMA TR8350.2 revision of January 2000 and ftp://ftp.ngs.noaa.gov/pub/pcsoft/nadcon/samoa_readme.txt",
        data_source: "EPSG",
        revision_date: "2005-05-21",
        deprecated: 0, 
    },

    "EPSG:4191" => CRS {
        coord_ref_sys_code: 4191,
        coord_ref_sys_name: "Albanian 1987",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6191, 
        base_crs_code: -1,
        remarks: "Replaced by KRGJSH-2010.",
        information_source: "EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2014-11-21",
        deprecated: 0, 
    },

    "EPSG:4196" => CRS {
        coord_ref_sys_code: 4196,
        coord_ref_sys_name: "Ammassalik 1958",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6196, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen, Copenhagen.",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4201" => CRS {
        coord_ref_sys_code: 4201,
        coord_ref_sys_name: "Adindan",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6201, 
        base_crs_code: -1,
        remarks: "The 12th parallel traverse of 1966-70 (geogCRS Point 58, code 4620) is connected to the Blue Nile 1958 network in western Sudan. This has given rise to misconceptions that the Blue Nile 1958 network is used in west Africa.",
        information_source: "US Coast and Geodetic Survey via Geophysical Reasearch vol 67 #11, October 1962.",
        data_source: "EPSG",
        revision_date: "2015-04-22",
        deprecated: 0, 
    },

    "EPSG:4202" => CRS {
        coord_ref_sys_code: 4202,
        coord_ref_sys_name: "AGD66",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6202, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Australian Commonwealth Gazette No. 84, 6th October 1966.",
        data_source: "EPSG",
        revision_date: "2019-09-17",
        deprecated: 0, 
    },

    "EPSG:4203" => CRS {
        coord_ref_sys_code: 4203,
        coord_ref_sys_name: "AGD84",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6203, 
        base_crs_code: -1,
        remarks: "National system replacing AGD66 but officially adopted only in Queensland, South Australia and Western Australia. Replaced by GDA94.",
        information_source: "&quot;GDA technical manual v2_2&quot;, Intergovernmental Committee on Surveying and Mapping. www.anzlic.org.au/icsm/gdtm/",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:4204" => CRS {
        coord_ref_sys_code: 4204,
        coord_ref_sys_name: "Ain el Abd",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6204, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4205" => CRS {
        coord_ref_sys_code: 4205,
        coord_ref_sys_name: "Afgooye",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6205, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-05",
        deprecated: 0, 
    },

    "EPSG:4206" => CRS {
        coord_ref_sys_code: 4206,
        coord_ref_sys_name: "Agadez",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6206, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4208" => CRS {
        coord_ref_sys_code: 4208,
        coord_ref_sys_name: "Aratu",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6208, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4209" => CRS {
        coord_ref_sys_code: 4209,
        coord_ref_sys_name: "Arc 1950",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6209, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4289" => CRS {
        coord_ref_sys_code: 4289,
        coord_ref_sys_name: "Amersfoort",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6289, 
        base_crs_code: -1,
        remarks: "Use of geographic2D CRS Amersfoort (code 4289) for spatial referencing is discouraged. Use projected CRS Amersfoort / RD New (code 28992).",
        information_source: "NSGI: Netherlands Partnership of Kadaster, Rijkswaterstaat and Hydrographic Service, http://www.nsgi.nl/.",
        data_source: "EPSG",
        revision_date: "2020-08-31",
        deprecated: 0, 
    },

    "EPSG:4600" => CRS {
        coord_ref_sys_code: 4600,
        coord_ref_sys_name: "Anguilla 1957",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6600, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4601" => CRS {
        coord_ref_sys_code: 4601,
        coord_ref_sys_name: "Antigua 1943",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6601, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:5337" => CRS {
        coord_ref_sys_code: 5337,
        coord_ref_sys_name: "Aratu / UTM zone 25S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4208,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2011-01-25",
        deprecated: 0, 
    },

    "EPSG:5711" => CRS {
        coord_ref_sys_code: 5711,
        coord_ref_sys_name: "AHD height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5111, 
        base_crs_code: -1,
        remarks: "Appropriate for cadastral and local engineering survey including construction or mining. Only suitable onshore. AVWS height (CRS code 9458) is more accurate than AHD height for applications over distances greater than 10 km and also extends offshore.",
        information_source: "GDA2020 Technical Manual v1.4 and Australian Vertical Working Surface (AVWS) Technical Implementation Plan v1.2 (http://www.icsm.gov.au).",
        data_source: "EPSG",
        revision_date: "2020-10-21",
        deprecated: 0, 
    },

    "EPSG:5712" => CRS {
        coord_ref_sys_code: 5712,
        coord_ref_sys_name: "AHD (Tasmania) height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5112, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:5734" => CRS {
        coord_ref_sys_code: 5734,
        coord_ref_sys_name: "AIOC95 depth",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6498,
        datum_code: -1, 
        base_crs_code: 5797,
        remarks: "Also used by AIOC and BP as a height system for engineering survey at Sangachal terminal (see CRS code 5797). AIOC95 datum is 1.7m above Caspian datum and 26.3m below Baltic datum.",
        information_source: "BP",
        data_source: "EPSG",
        revision_date: "2021-04-20",
        deprecated: 0, 
    },

    "EPSG:5775" => CRS {
        coord_ref_sys_code: 5775,
        coord_ref_sys_name: "Antalya height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5173, 
        base_crs_code: -1,
        remarks: "",
        information_source: "EuroGeographics, http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5782" => CRS {
        coord_ref_sys_code: 5782,
        coord_ref_sys_name: "Alicante height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5180, 
        base_crs_code: -1,
        remarks: "",
        information_source: "National Geographic Institute of Spain (IGN), http://www.ign.es",
        data_source: "EPSG",
        revision_date: "2020-07-01",
        deprecated: 0, 
    },

    "EPSG:5797" => CRS {
        coord_ref_sys_code: 5797,
        coord_ref_sys_name: "AIOC95 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5133, 
        base_crs_code: -1,
        remarks: "AIOC95 datum is 1.7m above Caspian datum and 26.3m below Baltic datum. Also used by AIOC and BP as the depth system for all offshore Azerbaijan activities (see CRS code 5734).",
        information_source: "BP",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5825" => CRS {
        coord_ref_sys_code: 5825,
        coord_ref_sys_name: "AGD66 / ACT Standard Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4202,
        remarks: "",
        information_source: "ACT Planning and Land Authority, http://www.actpla.act.gov.au",
        data_source: "EPSG",
        revision_date: "2012-01-28",
        deprecated: 0, 
    },

    "EPSG:6881" => CRS {
        coord_ref_sys_code: 6881,
        coord_ref_sys_name: "Aden 1925",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1135, 
        base_crs_code: -1,
        remarks: "",
        information_source: "US NGA GeoTrans software v3.4 and other industry sources.",
        data_source: "EPSG",
        revision_date: "2014-07-30",
        deprecated: 0, 
    },

    "EPSG:7415" => CRS {
        coord_ref_sys_code: 7415,
        coord_ref_sys_name: "Amersfoort / RD New + NAP height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "http://www.rdnap.nl/",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:9384" => CRS {
        coord_ref_sys_code: 9384,
        coord_ref_sys_name: "AbInvA96_2020-IRF",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1273, 
        base_crs_code: -1,
        remarks: "Intermediate CRS created in 2020 to assist the emulation of the ETRS89 / AbInvA96_2020 SnakeGrid projected CRS through transformation ETRS89 to AbInvA96_2020-IRF (1) (code 9386) used in conjunction with the AbInvA96_2020-TM map projection (code 9385).",
        information_source: "Transport Scotland.",
        data_source: "EPSG",
        revision_date: "2020-06-16",
        deprecated: 0, 
    },

    "EPSG:9387" => CRS {
        coord_ref_sys_code: 9387,
        coord_ref_sys_name: "AbInvA96_2020 Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 9384,
        remarks: "The CRS&#x27;s definition through transformation ETRS89 to AbInvA96_2020-IRF (1) (code 9386) and map projection AbInvA96_2020-TM (code 9385) emulates the AbInvA96_2020 Snake projection applied to ETRS89 as realised through OSNet 2009 CORS.",
        information_source: "Transport Scotland.",
        data_source: "EPSG",
        revision_date: "2020-06-16",
        deprecated: 0, 
    },

    "EPSG:9388" => CRS {
        coord_ref_sys_code: 9388,
        coord_ref_sys_name: "AbInvA96_2020 Grid + ODN height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Transport Scotland.",
        data_source: "EPSG",
        revision_date: "2020-06-16",
        deprecated: 0, 
    },

    "EPSG:20135" => CRS {
        coord_ref_sys_code: 20135,
        coord_ref_sys_name: "Adindan / UTM zone 35N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4201,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2015-04-22",
        deprecated: 0, 
    },

    "EPSG:20136" => CRS {
        coord_ref_sys_code: 20136,
        coord_ref_sys_name: "Adindan / UTM zone 36N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4201,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2015-04-22",
        deprecated: 0, 
    },

    "EPSG:20137" => CRS {
        coord_ref_sys_code: 20137,
        coord_ref_sys_name: "Adindan / UTM zone 37N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4201,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2015-04-22",
        deprecated: 0, 
    },

    "EPSG:20138" => CRS {
        coord_ref_sys_code: 20138,
        coord_ref_sys_name: "Adindan / UTM zone 38N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4201,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2015-04-22",
        deprecated: 0, 
    },

    "EPSG:20248" => CRS {
        coord_ref_sys_code: 20248,
        coord_ref_sys_name: "AGD66 / AMG zone 48",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4202,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:20249" => CRS {
        coord_ref_sys_code: 20249,
        coord_ref_sys_name: "AGD66 / AMG zone 49",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4202,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:20250" => CRS {
        coord_ref_sys_code: 20250,
        coord_ref_sys_name: "AGD66 / AMG zone 50",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4202,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:20251" => CRS {
        coord_ref_sys_code: 20251,
        coord_ref_sys_name: "AGD66 / AMG zone 51",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4202,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:20252" => CRS {
        coord_ref_sys_code: 20252,
        coord_ref_sys_name: "AGD66 / AMG zone 52",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4202,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:20253" => CRS {
        coord_ref_sys_code: 20253,
        coord_ref_sys_name: "AGD66 / AMG zone 53",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4202,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:20254" => CRS {
        coord_ref_sys_code: 20254,
        coord_ref_sys_name: "AGD66 / AMG zone 54",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4202,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:20255" => CRS {
        coord_ref_sys_code: 20255,
        coord_ref_sys_name: "AGD66 / AMG zone 55",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4202,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:20256" => CRS {
        coord_ref_sys_code: 20256,
        coord_ref_sys_name: "AGD66 / AMG zone 56",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4202,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2011-07-20",
        deprecated: 0, 
    },

    "EPSG:20257" => CRS {
        coord_ref_sys_code: 20257,
        coord_ref_sys_name: "AGD66 / AMG zone 57",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4202,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:20258" => CRS {
        coord_ref_sys_code: 20258,
        coord_ref_sys_name: "AGD66 / AMG zone 58",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4202,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:20348" => CRS {
        coord_ref_sys_code: 20348,
        coord_ref_sys_name: "AGD84 / AMG zone 48",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4203,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:20349" => CRS {
        coord_ref_sys_code: 20349,
        coord_ref_sys_name: "AGD84 / AMG zone 49",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4203,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:20350" => CRS {
        coord_ref_sys_code: 20350,
        coord_ref_sys_name: "AGD84 / AMG zone 50",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4203,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:20351" => CRS {
        coord_ref_sys_code: 20351,
        coord_ref_sys_name: "AGD84 / AMG zone 51",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4203,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:20352" => CRS {
        coord_ref_sys_code: 20352,
        coord_ref_sys_name: "AGD84 / AMG zone 52",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4203,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2010-03-30",
        deprecated: 0, 
    },

    "EPSG:20353" => CRS {
        coord_ref_sys_code: 20353,
        coord_ref_sys_name: "AGD84 / AMG zone 53",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4203,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2010-03-30",
        deprecated: 0, 
    },

    "EPSG:20354" => CRS {
        coord_ref_sys_code: 20354,
        coord_ref_sys_name: "AGD84 / AMG zone 54",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4203,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2010-03-30",
        deprecated: 0, 
    },

    "EPSG:20355" => CRS {
        coord_ref_sys_code: 20355,
        coord_ref_sys_name: "AGD84 / AMG zone 55",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4203,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2010-03-30",
        deprecated: 0, 
    },

    "EPSG:20356" => CRS {
        coord_ref_sys_code: 20356,
        coord_ref_sys_name: "AGD84 / AMG zone 56",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4203,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2010-03-30",
        deprecated: 0, 
    },

    "EPSG:20357" => CRS {
        coord_ref_sys_code: 20357,
        coord_ref_sys_name: "AGD84 / AMG zone 57",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4203,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:20358" => CRS {
        coord_ref_sys_code: 20358,
        coord_ref_sys_name: "AGD84 / AMG zone 58",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4203,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:20436" => CRS {
        coord_ref_sys_code: 20436,
        coord_ref_sys_name: "Ain el Abd / UTM zone 36N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4204,
        remarks: "Replaced by MTRF-2000 / UTM zone 36N (CRS code 8836).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2018-08-29",
        deprecated: 0, 
    },

    "EPSG:20437" => CRS {
        coord_ref_sys_code: 20437,
        coord_ref_sys_name: "Ain el Abd / UTM zone 37N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4204,
        remarks: "Replaced by MTRF-2000 / UTM zone 37N (CRS code 8837).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2018-08-29",
        deprecated: 0, 
    },

    "EPSG:20438" => CRS {
        coord_ref_sys_code: 20438,
        coord_ref_sys_name: "Ain el Abd / UTM zone 38N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4204,
        remarks: "Known in Kuwait as &quot;KOC UTM&quot;. Used by KOC for engineering but not explorartion (see KOC Lambert, CRS code 24600). In Saudi Arabia, replaced by MTRF-2000 / UTM zone 38N (CRS code 8838).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2018-08-29",
        deprecated: 0, 
    },

    "EPSG:20439" => CRS {
        coord_ref_sys_code: 20439,
        coord_ref_sys_name: "Ain el Abd / UTM zone 39N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4204,
        remarks: "Known in Kuwait as &quot;KOC UTM&quot;. Used by KOC for engineering but not explorartion (see KOC Lambert, CRS code 24600). In Saudi Arabia, replaced by MTRF-2000 / UTM zone 39N (CRS code 8839).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2018-08-29",
        deprecated: 0, 
    },

    "EPSG:20440" => CRS {
        coord_ref_sys_code: 20440,
        coord_ref_sys_name: "Ain el Abd / UTM zone 40N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4204,
        remarks: "Replaced by MTRF-2000 / UTM zone 40N (CRS code 8840).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2018-08-29",
        deprecated: 0, 
    },

    "EPSG:20499" => CRS {
        coord_ref_sys_code: 20499,
        coord_ref_sys_name: "Ain el Abd / Bahrain Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4204,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:20538" => CRS {
        coord_ref_sys_code: 20538,
        coord_ref_sys_name: "Afgooye / UTM zone 38N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4205,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:20539" => CRS {
        coord_ref_sys_code: 20539,
        coord_ref_sys_name: "Afgooye / UTM zone 39N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4205,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:20822" => CRS {
        coord_ref_sys_code: 20822,
        coord_ref_sys_name: "Aratu / UTM zone 22S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4208,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-07-21",
        deprecated: 0, 
    },

    "EPSG:20823" => CRS {
        coord_ref_sys_code: 20823,
        coord_ref_sys_name: "Aratu / UTM zone 23S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4208,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-07-21",
        deprecated: 0, 
    },

    "EPSG:20824" => CRS {
        coord_ref_sys_code: 20824,
        coord_ref_sys_name: "Aratu / UTM zone 24S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4208,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-07-21",
        deprecated: 0, 
    },

    "EPSG:20934" => CRS {
        coord_ref_sys_code: 20934,
        coord_ref_sys_name: "Arc 1950 / UTM zone 34S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4209,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1997-11-13",
        deprecated: 0, 
    },

    "EPSG:28991" => CRS {
        coord_ref_sys_code: 28991,
        coord_ref_sys_name: "Amersfoort / RD Old",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4289,
        remarks: "Replaced by 28992 (Amersfoort / RD New).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:28992" => CRS {
        coord_ref_sys_code: 28992,
        coord_ref_sys_name: "Amersfoort / RD New",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4289,
        remarks: "Replaces 28991 (Amersfoort / RD Old).",
        information_source: "NSGI: Netherlands Partnership of Kadaster, Rijkswaterstaat and Hydrographic Service, http://www.nsgi.nl/.",
        data_source: "EPSG",
        revision_date: "2020-08-31",
        deprecated: 0, 
    },

    "EPSG:61436405" => CRS {
        coord_ref_sys_code: 61436405,
        coord_ref_sys_name: "Abidjan 1987 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4143,
        remarks: "See CRS code 4143 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61686405" => CRS {
        coord_ref_sys_code: 61686405,
        coord_ref_sys_name: "Accra (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4168,
        remarks: "See CRS code 4168 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61696405" => CRS {
        coord_ref_sys_code: 61696405,
        coord_ref_sys_name: "American Samoa 1962 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4169,
        remarks: "See CRS code 4169 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61916405" => CRS {
        coord_ref_sys_code: 61916405,
        coord_ref_sys_name: "Albanian 1987 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4191,
        remarks: "See CRS code 4191 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61966405" => CRS {
        coord_ref_sys_code: 61966405,
        coord_ref_sys_name: "Ammassalik 1958 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4196,
        remarks: "See CRS code 4196 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62016405" => CRS {
        coord_ref_sys_code: 62016405,
        coord_ref_sys_name: "Adindan (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4201,
        remarks: "See CRS code 4201 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62026405" => CRS {
        coord_ref_sys_code: 62026405,
        coord_ref_sys_name: "AGD66 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4202,
        remarks: "See CRS code 4202 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62036405" => CRS {
        coord_ref_sys_code: 62036405,
        coord_ref_sys_name: "AGD84 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4203,
        remarks: "See CRS code 4203 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2003-09-01",
        deprecated: 1, 
    },

    "EPSG:62046405" => CRS {
        coord_ref_sys_code: 62046405,
        coord_ref_sys_name: "Ain el Abd (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4204,
        remarks: "See CRS code 4204 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62056405" => CRS {
        coord_ref_sys_code: 62056405,
        coord_ref_sys_name: "Afgooye (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4205,
        remarks: "See CRS code 4205 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62066405" => CRS {
        coord_ref_sys_code: 62066405,
        coord_ref_sys_name: "Agadez (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4206,
        remarks: "See CRS code 4206 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62086405" => CRS {
        coord_ref_sys_code: 62086405,
        coord_ref_sys_name: "Aratu (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4208,
        remarks: "See CRS code 4208 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62096405" => CRS {
        coord_ref_sys_code: 62096405,
        coord_ref_sys_name: "Arc 1950 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4209,
        remarks: "See CRS code 4209 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62896405" => CRS {
        coord_ref_sys_code: 62896405,
        coord_ref_sys_name: "Amersfoort (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4289,
        remarks: "See CRS code 4289 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:66006405" => CRS {
        coord_ref_sys_code: 66006405,
        coord_ref_sys_name: "Anguilla 1957 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4600,
        remarks: "See CRS code 4600 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:66016405" => CRS {
        coord_ref_sys_code: 66016405,
        coord_ref_sys_name: "Antigua 1943 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4601,
        remarks: "See CRS code 4601 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:2188" => CRS {
        coord_ref_sys_code: 2188,
        coord_ref_sys_name: "Azores Occidental 1939 / UTM zone 25N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4182,
        remarks: "Replaced by PTRA08 / UTM zone 25N (CRS code 5014).",
        information_source: "Instituto Geografico e Cadastral Lisbon via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2010-03-31",
        deprecated: 0, 
    },

    "EPSG:2189" => CRS {
        coord_ref_sys_code: 2189,
        coord_ref_sys_name: "Azores Central 1948 / UTM zone 26N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4183,
        remarks: "",
        information_source: "Instituto Geografico e Cadastral Lisbon via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 0, 
    },

    "EPSG:2190" => CRS {
        coord_ref_sys_code: 2190,
        coord_ref_sys_name: "Azores Oriental 1940 / UTM zone 26N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4184,
        remarks: "",
        information_source: "Instituto Geografico e Cadastral Lisbon via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 0, 
    },

    "EPSG:2200" => CRS {
        coord_ref_sys_code: 2200,
        coord_ref_sys_name: "ATS77 / New Brunswick Stereographic (ATS77)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4122,
        remarks: "In use from 1979. Replaces NAD27 / NB Stereographic (CRS code 5588). Replaced by NAD83(CSRS) / NB Stereographic (CRS code 2953).",
        information_source: "New Brunswick Geographic Information Corporation land and water information standards manual.",
        data_source: "EPSG",
        revision_date: "2011-10-08",
        deprecated: 0, 
    },

    "EPSG:2219" => CRS {
        coord_ref_sys_code: 2219,
        coord_ref_sys_name: "ATS77 / UTM zone 19N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4122,
        remarks: "In use from 1979. To be phased out in late 1990&#x27;s.",
        information_source: "New Brunswick Geographic Information Corporation land and water information standards manual.",
        data_source: "EPSG",
        revision_date: "1997-07-22",
        deprecated: 0, 
    },

    "EPSG:2220" => CRS {
        coord_ref_sys_code: 2220,
        coord_ref_sys_name: "ATS77 / UTM zone 20N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4122,
        remarks: "In use from 1979. To be phased out in late 1990&#x27;s.",
        information_source: "New Brunswick Geographic Information Corporation land and water information standards manual.",
        data_source: "EPSG",
        revision_date: "1997-07-22",
        deprecated: 0, 
    },

    "EPSG:2290" => CRS {
        coord_ref_sys_code: 2290,
        coord_ref_sys_name: "ATS77 / Prince Edward Isl. Stereographic (ATS77)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4496,
        datum_code: -1, 
        base_crs_code: 4122,
        remarks: "In use from 1979. To be phased out in late 1990&#x27;s.",
        information_source: "Geomatics Centre; Nova Scotia Ministry of Housing and Municipal Affairs.",
        data_source: "EPSG",
        revision_date: "1997-11-13",
        deprecated: 0, 
    },

    "EPSG:2294" => CRS {
        coord_ref_sys_code: 2294,
        coord_ref_sys_name: "ATS77 / MTM Nova Scotia zone 4",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4122,
        remarks: "In use from 1979. To be phased out in late 1990&#x27;s.",
        information_source: "Geomatics Centre; Nova Scotia Ministry of Housing and Municipal Affairs.",
        data_source: "EPSG",
        revision_date: "1997-11-13",
        deprecated: 0, 
    },

    "EPSG:2295" => CRS {
        coord_ref_sys_code: 2295,
        coord_ref_sys_name: "ATS77 / MTM Nova Scotia zone 5",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4122,
        remarks: "In use from 1979. To be phased out in late 1990&#x27;s.",
        information_source: "Geomatics Centre; Nova Scotia Ministry of Housing and Municipal Affairs.",
        data_source: "EPSG",
        revision_date: "1997-11-13",
        deprecated: 0, 
    },

    "EPSG:2308" => CRS {
        coord_ref_sys_code: 2308,
        coord_ref_sys_name: "Batavia / TM 109 SE",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4211,
        remarks: "Used by Arco and BP for ONWJ.",
        information_source: "BP Indonesia.",
        data_source: "EPSG",
        revision_date: "2002-02-12",
        deprecated: 0, 
    },

    "EPSG:2431" => CRS {
        coord_ref_sys_code: 2431,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 102E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 45 (code 2421). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 102E (code 2379).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2432" => CRS {
        coord_ref_sys_code: 2432,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 105E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 35 (code 2411). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 105E (code 2380).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2006-07-22",
        deprecated: 0, 
    },

    "EPSG:2433" => CRS {
        coord_ref_sys_code: 2433,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 108E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 36 (code 2412). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 108E (code 2381).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2434" => CRS {
        coord_ref_sys_code: 2434,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 111E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 37 (code 2413). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 111E (code 2382).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2006-07-22",
        deprecated: 0, 
    },

    "EPSG:2931" => CRS {
        coord_ref_sys_code: 2931,
        coord_ref_sys_name: "Beduaram / TM 13 NE",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4213,
        remarks: "Used by Elf in 1986.",
        information_source: "TotalFinaElf",
        data_source: "EPSG",
        revision_date: "2003-08-30",
        deprecated: 0, 
    },

    "EPSG:3001" => CRS {
        coord_ref_sys_code: 3001,
        coord_ref_sys_name: "Batavia / NEIEZ",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4211,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2003-01-16",
        deprecated: 0, 
    },

    "EPSG:3062" => CRS {
        coord_ref_sys_code: 3062,
        coord_ref_sys_name: "Azores Oriental 1995 / UTM zone 26N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4664,
        remarks: "Replaced by PTRA08 / UTM zone 26N (CRS code 5015).",
        information_source: "Instituto Geografico e Cadastral Lisbon; http://www.igeo.pt",
        data_source: "EPSG",
        revision_date: "2010-03-31",
        deprecated: 0, 
    },

    "EPSG:3063" => CRS {
        coord_ref_sys_code: 3063,
        coord_ref_sys_name: "Azores Central 1995 / UTM zone 26N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4665,
        remarks: "Replaced by PTRA08 / UTM zone 26N (CRS code 5015).",
        information_source: "Instituto Geografico e Cadastral Lisbon; http://www.igeo.pt",
        data_source: "EPSG",
        revision_date: "2010-03-31",
        deprecated: 0, 
    },

    "EPSG:3770" => CRS {
        coord_ref_sys_code: 3770,
        coord_ref_sys_name: "BDA2000 / Bermuda 2000 National Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4762,
        remarks: "Replaces Bermuda 1957 / UTM zone 20N (alias Bermuda National Grid) CRS code 3769).",
        information_source: "Department of Lands Buildings and Surveys.",
        data_source: "EPSG",
        revision_date: "2007-12-12",
        deprecated: 0, 
    },

    "EPSG:4122" => CRS {
        coord_ref_sys_code: 4122,
        coord_ref_sys_name: "ATS77",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6122, 
        base_crs_code: -1,
        remarks: "In use from 1979. To be phased out in late 1990&#x27;s.",
        information_source: "New Brunswick Geographic Information Corporation land and water information standards manual.",
        data_source: "EPSG",
        revision_date: "2019-09-17",
        deprecated: 0, 
    },

    "EPSG:4176" => CRS {
        coord_ref_sys_code: 4176,
        coord_ref_sys_name: "Australian Antarctic",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6176, 
        base_crs_code: 4931,
        remarks: "",
        information_source: "EPSG. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2007-08-27",
        deprecated: 0, 
    },

    "EPSG:4182" => CRS {
        coord_ref_sys_code: 4182,
        coord_ref_sys_name: "Azores Occidental 1939",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6182, 
        base_crs_code: -1,
        remarks: "Replaced by PTRA08 (CRS code 5013).",
        information_source: "Instituto Geografico e Cadastral Lisbon via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2010-03-31",
        deprecated: 0, 
    },

    "EPSG:4183" => CRS {
        coord_ref_sys_code: 4183,
        coord_ref_sys_name: "Azores Central 1948",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6183, 
        base_crs_code: -1,
        remarks: "Replaced by 1995 system (CRS code 4665).",
        information_source: "Instituto Geografico e Cadastral Lisbon via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4184" => CRS {
        coord_ref_sys_code: 4184,
        coord_ref_sys_name: "Azores Oriental 1940",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6184, 
        base_crs_code: -1,
        remarks: "Replaced by 1995 system (CRS code 4664).",
        information_source: "Instituto Geografico e Cadastral Lisbon via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4210" => CRS {
        coord_ref_sys_code: 4210,
        coord_ref_sys_name: "Arc 1960",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6210, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4211" => CRS {
        coord_ref_sys_code: 4211,
        coord_ref_sys_name: "Batavia",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6211, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2010-03-30",
        deprecated: 0, 
    },

    "EPSG:4212" => CRS {
        coord_ref_sys_code: 4212,
        coord_ref_sys_name: "Barbados 1938",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6212, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2011-06-30",
        deprecated: 0, 
    },

    "EPSG:4213" => CRS {
        coord_ref_sys_code: 4213,
        coord_ref_sys_name: "Beduaram",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6213, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4214" => CRS {
        coord_ref_sys_code: 4214,
        coord_ref_sys_name: "Beijing 1954",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6214, 
        base_crs_code: -1,
        remarks: "In 1982 replaced by Xian 1980 (CRS code 4610) and New Beijing (CRS code 4555).",
        information_source: "Chinese Science Bulletin, 2009, 54:2714-2721.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:4215" => CRS {
        coord_ref_sys_code: 4215,
        coord_ref_sys_name: "BD50",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6215, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IGN Brussels.",
        data_source: "EPSG",
        revision_date: "2021-06-17",
        deprecated: 0, 
    },

    "EPSG:4313" => CRS {
        coord_ref_sys_code: 4313,
        coord_ref_sys_name: "BD72",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6313, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IGN Brussels.",
        data_source: "EPSG",
        revision_date: "2021-06-17",
        deprecated: 0, 
    },

    "EPSG:4339" => CRS {
        coord_ref_sys_code: 4339,
        coord_ref_sys_name: "Australian Antarctic (3D)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6401,
        datum_code: 6176, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Standards Australia",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:4340" => CRS {
        coord_ref_sys_code: 4340,
        coord_ref_sys_name: "Australian Antarctic (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6176, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Standards Australia",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:4664" => CRS {
        coord_ref_sys_code: 4664,
        coord_ref_sys_name: "Azores Oriental 1995",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6664, 
        base_crs_code: -1,
        remarks: "Replaces 1948 system (CRS code 4184). Replaced by PTRA08 (CRS code 5013).",
        information_source: "Instituto Geografico e Cadastral Lisbon; http://www.igeo.pt",
        data_source: "EPSG",
        revision_date: "2010-03-31",
        deprecated: 0, 
    },

    "EPSG:4665" => CRS {
        coord_ref_sys_code: 4665,
        coord_ref_sys_name: "Azores Central 1995",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6665, 
        base_crs_code: -1,
        remarks: "Replaces 1948 system (CRS code 4183). Replaced by PTRA08 (CRS code 5013).",
        information_source: "Instituto Geografico e Cadastral Lisbon; http://www.igeo.pt",
        data_source: "EPSG",
        revision_date: "2010-03-31",
        deprecated: 0, 
    },

    "EPSG:4710" => CRS {
        coord_ref_sys_code: 4710,
        coord_ref_sys_name: "Astro DOS 71",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6710, 
        base_crs_code: -1,
        remarks: "Used between 1972 and 2015. Replaced by SHGD2015 (CRS code 7886) from 2015.",
        information_source: "DMA / NIMA / NGA TR8350.2 (3rd edition, Amendment 1, 3 January 2000) and St. Helena Government, Environment and Natural Resources Directorate (ENRD).",
        data_source: "EPSG",
        revision_date: "2016-12-15",
        deprecated: 0, 
    },

    "EPSG:4712" => CRS {
        coord_ref_sys_code: 4712,
        coord_ref_sys_name: "Ascension Island 1958",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6712, 
        base_crs_code: -1,
        remarks: "",
        information_source: "DMA / NIMA / NGA TR8350.2 (3rd edition, January 2000).",
        data_source: "EPSG",
        revision_date: "2006-01-26",
        deprecated: 0, 
    },

    "EPSG:4713" => CRS {
        coord_ref_sys_code: 4713,
        coord_ref_sys_name: "Ayabelle Lighthouse",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6713, 
        base_crs_code: -1,
        remarks: "",
        information_source: "DMA / NIMA / NGA TR8350.2 (3rd edition, January 2000).",
        data_source: "EPSG",
        revision_date: "2006-01-26",
        deprecated: 0, 
    },

    "EPSG:4762" => CRS {
        coord_ref_sys_code: 4762,
        coord_ref_sys_name: "BDA2000",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6762, 
        base_crs_code: 4887,
        remarks: "Replaces Bermuda 1957 (CRS code 4216).",
        information_source: "OGP. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2007-12-12",
        deprecated: 0, 
    },

    "EPSG:4809" => CRS {
        coord_ref_sys_code: 4809,
        coord_ref_sys_name: "BD50 (Brussels)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6809, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IGN Brussels.",
        data_source: "EPSG",
        revision_date: "2021-06-17",
        deprecated: 0, 
    },

    "EPSG:4813" => CRS {
        coord_ref_sys_code: 4813,
        coord_ref_sys_name: "Batavia (Jakarta)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6813, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4886" => CRS {
        coord_ref_sys_code: 4886,
        coord_ref_sys_name: "BDA2000",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6762, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Department of Lands Buildings and Surveys.",
        data_source: "EPSG",
        revision_date: "2007-12-12",
        deprecated: 0, 
    },

    "EPSG:4887" => CRS {
        coord_ref_sys_code: 4887,
        coord_ref_sys_name: "BDA2000",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6762, 
        base_crs_code: 4886,
        remarks: "",
        information_source: "Department of Lands Buildings and Surveys.",
        data_source: "EPSG",
        revision_date: "2007-12-12",
        deprecated: 0, 
    },

    "EPSG:4901" => CRS {
        coord_ref_sys_code: 4901,
        coord_ref_sys_name: "ATF (Paris)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6403,
        datum_code: 6901, 
        base_crs_code: -1,
        remarks: "ProjCRS covering all mainland France based on this datum used Bonne projection. In Alsace, suspected to be an extension of core network based on transformation of old German system.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2007-11-01",
        deprecated: 0, 
    },

    "EPSG:4930" => CRS {
        coord_ref_sys_code: 4930,
        coord_ref_sys_name: "Australian Antarctic",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6176, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Standards Australia",
        data_source: "EPSG",
        revision_date: "2004-01-05",
        deprecated: 0, 
    },

    "EPSG:4931" => CRS {
        coord_ref_sys_code: 4931,
        coord_ref_sys_name: "Australian Antarctic",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6176, 
        base_crs_code: 4930,
        remarks: "",
        information_source: "Standards Australia",
        data_source: "EPSG",
        revision_date: "2007-08-27",
        deprecated: 0, 
    },

    "EPSG:5330" => CRS {
        coord_ref_sys_code: 5330,
        coord_ref_sys_name: "Batavia (Jakarta) / NEIEZ",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4813,
        remarks: "Replaced by Greenwich-based Batavia / NEIEZ (CRS code 3001).",
        information_source: "Indonesian Army Map Service.",
        data_source: "EPSG",
        revision_date: "2010-11-17",
        deprecated: 0, 
    },

    "EPSG:5612" => CRS {
        coord_ref_sys_code: 5612,
        coord_ref_sys_name: "Baltic 1977 depth",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6498,
        datum_code: -1, 
        base_crs_code: 5705,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-04-20",
        deprecated: 0, 
    },

    "EPSG:5705" => CRS {
        coord_ref_sys_code: 5705,
        coord_ref_sys_name: "Baltic 1977 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5105, 
        base_crs_code: -1,
        remarks: "The adjustment also included the Czech and Slovak Republics but not adopted there, with earlier 1957 adjustment remaining in use: see CRS code 8357.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2018-02-16",
        deprecated: 0, 
    },

    "EPSG:5752" => CRS {
        coord_ref_sys_code: 5752,
        coord_ref_sys_name: "Bandar Abbas height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5150, 
        base_crs_code: -1,
        remarks: "Replaces Fao height (CRS code 5751) for national map agency work in Iran. At time of record creation NIOC data still generally referenced to Fao.",
        information_source: "National Cartographic Cenre of Iran",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 0, 
    },

    "EPSG:5759" => CRS {
        coord_ref_sys_code: 5759,
        coord_ref_sys_name: "Auckland 1946 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5157, 
        base_crs_code: -1,
        remarks: "",
        information_source: "OSG Technical Report #10, February 2001.",
        data_source: "EPSG",
        revision_date: "2011-01-25",
        deprecated: 0, 
    },

    "EPSG:5786" => CRS {
        coord_ref_sys_code: 5786,
        coord_ref_sys_name: "Baltic 1982 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5184, 
        base_crs_code: -1,
        remarks: "",
        information_source: "EuroGeographics, http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5800" => CRS {
        coord_ref_sys_code: 5800,
        coord_ref_sys_name: "Astra Minas Grid",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 6507,
        datum_code: 9300, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:5801" => CRS {
        coord_ref_sys_code: 5801,
        coord_ref_sys_name: "Barcelona Grid B1",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 4500,
        datum_code: 9301, 
        base_crs_code: -1,
        remarks: "Centre of the gateway of San Cristobal chuch; Plaza Boyaca; Barcelona. 500000E; 300000N at 10°08&#x27;06&quot;N, 64°41&#x27;17&quot;W.",
        information_source: "Ministry of Mines standards manual of 1974.",
        data_source: "EPSG",
        revision_date: "2008-06-23",
        deprecated: 0, 
    },

    "EPSG:5802" => CRS {
        coord_ref_sys_code: 5802,
        coord_ref_sys_name: "Barcelona Grid B2",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 4500,
        datum_code: 9301, 
        base_crs_code: -1,
        remarks: "Centre of the gateway of San Cristobal chuch; Plaza Boyaca; Barcelona. 500000E; 300000N at 10°08&#x27;06&quot;N, 64°41&#x27;07.5&quot;W.",
        information_source: "Ministry of Mines standards manual of 1974.",
        data_source: "EPSG",
        revision_date: "2008-06-23",
        deprecated: 0, 
    },

    "EPSG:5816" => CRS {
        coord_ref_sys_code: 5816,
        coord_ref_sys_name: "Barinas west base",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 4500,
        datum_code: 9311, 
        base_crs_code: -1,
        remarks: "Grid coordinates 200000N; 200000E at the west base monument in Barinas.",
        information_source: "Ministry of Mines standards manual of 1974.",
        data_source: "EPSG",
        revision_date: "2000-06-23",
        deprecated: 0, 
    },

    "EPSG:6190" => CRS {
        coord_ref_sys_code: 6190,
        coord_ref_sys_name: "BD72 / Belgian Lambert 72 + Ostend height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IGN Brussels; www.ngi.be/",
        data_source: "EPSG",
        revision_date: "2021-06-17",
        deprecated: 0, 
    },

    "EPSG:6643" => CRS {
        coord_ref_sys_code: 6643,
        coord_ref_sys_name: "ASVD02 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1125, 
        base_crs_code: -1,
        remarks: "Replaces Tutuila 1962 height (CRS code 6638). Replaced by Pago Pago 2020 height after ASVD02 benchmarks destroyed by earthquake activity.",
        information_source: "National Geodetic Survey, http://www.ngs.noaa.gov",
        data_source: "EPSG",
        revision_date: "2020-12-14",
        deprecated: 0, 
    },

    "EPSG:7877" => CRS {
        coord_ref_sys_code: 7877,
        coord_ref_sys_name: "Astro DOS 71 / SHLG71",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4710,
        remarks: "Replaced by SHMG2015 (CRS code 7887) from 2015.",
        information_source: "St. Helena Government, Environment and Natural Resources Directorate (ENRD).",
        data_source: "EPSG",
        revision_date: "2016-11-25",
        deprecated: 0, 
    },

    "EPSG:7878" => CRS {
        coord_ref_sys_code: 7878,
        coord_ref_sys_name: "Astro DOS 71 / UTM zone 30S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4710,
        remarks: "Replaced by SHMG2015 (CRS code 7887) from 2015.",
        information_source: "St. Helena Government, Environment and Natural Resources Directorate (ENRD).",
        data_source: "EPSG",
        revision_date: "2016-11-20",
        deprecated: 0, 
    },

    "EPSG:7954" => CRS {
        coord_ref_sys_code: 7954,
        coord_ref_sys_name: "Astro DOS 71 / UTM zone 30S + Jamestown 1971 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "St. Helena Government, Environment and Natural Resources Directorate (ENRD).",
        data_source: "EPSG",
        revision_date: "2016-11-25",
        deprecated: 0, 
    },

    "EPSG:8357" => CRS {
        coord_ref_sys_code: 8357,
        coord_ref_sys_name: "Baltic 1957 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1202, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Land Survey Office (ZU), Prague; and Geodesy, Cartography and Cadastre Authority of Slovak Republic (UGKK SR).",
        data_source: "EPSG",
        revision_date: "2018-02-13",
        deprecated: 0, 
    },

    "EPSG:8358" => CRS {
        coord_ref_sys_code: 8358,
        coord_ref_sys_name: "Baltic 1957 depth",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6498,
        datum_code: -1, 
        base_crs_code: 8357,
        remarks: "",
        information_source: "Land Survey Office (ZU), Prague; and Geodesy, Cartography and Cadastre Authority of Slovak Republic (UGKK SR).",
        data_source: "EPSG",
        revision_date: "2021-04-20",
        deprecated: 0, 
    },

    "EPSG:9307" => CRS {
        coord_ref_sys_code: 9307,
        coord_ref_sys_name: "ATRF2014",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1291, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Australian Terrestrial Reference Frame (ATRF) Technical Implementation Plan v2.3, 12th February 2020, ANZLIC Intergovernmental Committee on Surveying and Mapping (ICSM) (http://www.icsm.gov.au).",
        data_source: "EPSG",
        revision_date: "2020-08-17",
        deprecated: 0, 
    },

    "EPSG:9308" => CRS {
        coord_ref_sys_code: 9308,
        coord_ref_sys_name: "ATRF2014",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1291, 
        base_crs_code: 9307,
        remarks: "",
        information_source: "Australian Terrestrial Reference Frame (ATRF) Technical Implementation Plan v2.3, 12th February 2020, ANZLIC Intergovernmental Committee on Surveying and Mapping (ICSM) (http://www.icsm.gov.au).",
        data_source: "EPSG",
        revision_date: "2020-08-17",
        deprecated: 0, 
    },

    "EPSG:9309" => CRS {
        coord_ref_sys_code: 9309,
        coord_ref_sys_name: "ATRF2014",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1291, 
        base_crs_code: 9308,
        remarks: "",
        information_source: "Australian Terrestrial Reference Frame (ATRF) Technical Implementation Plan v2.3, 12th February 2020, ANZLIC Intergovernmental Committee on Surveying and Mapping (ICSM) (http://www.icsm.gov.au).",
        data_source: "EPSG",
        revision_date: "2020-08-17",
        deprecated: 0, 
    },

    "EPSG:6178" => CRS {
        coord_ref_sys_code: 6178,
        coord_ref_sys_name: "Cais da Pontinha - Funchal height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1101, 
        base_crs_code: -1,
        remarks: "",
        information_source: "General Directorate of Land, http://www.dgterritorio.pt/",
        data_source: "EPSG",
        revision_date: "2013-03-15",
        deprecated: 0, 
    },

    "EPSG:9458" => CRS {
        coord_ref_sys_code: 9458,
        coord_ref_sys_name: "AVWS height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1292, 
        base_crs_code: -1,
        remarks: "For cadastral and local engineering applications see AHD height (CRS code 5711). AVWS is more accurate than AHD for applications over distances greater than 10 km.",
        information_source: "Australian Vertical Working Surface (AVWS) Technical Implementation Plan v1.2, 26th August 2020, ANZLIC Intergovernmental Committee on Surveying and Mapping (ICSM) (http://www.icsm.gov.au).",
        data_source: "EPSG",
        revision_date: "2021-01-18",
        deprecated: 0, 
    },

    "EPSG:9650" => CRS {
        coord_ref_sys_code: 9650,
        coord_ref_sys_name: "Baltic 1986 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1296, 
        base_crs_code: -1,
        remarks: "Initially valid until 2019-12-31, but that extended to 2023-12-31. Will be replaced by EVRF2007-PL height (CRS 9651) after 2023-12-31.",
        information_source: "Glowny Urzad Geodezji i Kartografii (GUGiK), http://www.gugik.gov.pl",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:20935" => CRS {
        coord_ref_sys_code: 20935,
        coord_ref_sys_name: "Arc 1950 / UTM zone 35S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4209,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1997-11-13",
        deprecated: 0, 
    },

    "EPSG:20936" => CRS {
        coord_ref_sys_code: 20936,
        coord_ref_sys_name: "Arc 1950 / UTM zone 36S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4209,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1997-11-13",
        deprecated: 0, 
    },

    "EPSG:21035" => CRS {
        coord_ref_sys_code: 21035,
        coord_ref_sys_name: "Arc 1960 / UTM zone 35S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4210,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1997-11-13",
        deprecated: 0, 
    },

    "EPSG:21036" => CRS {
        coord_ref_sys_code: 21036,
        coord_ref_sys_name: "Arc 1960 / UTM zone 36S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4210,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1997-11-13",
        deprecated: 0, 
    },

    "EPSG:21037" => CRS {
        coord_ref_sys_code: 21037,
        coord_ref_sys_name: "Arc 1960 / UTM zone 37S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4210,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1997-11-13",
        deprecated: 0, 
    },

    "EPSG:21095" => CRS {
        coord_ref_sys_code: 21095,
        coord_ref_sys_name: "Arc 1960 / UTM zone 35N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4210,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1997-11-13",
        deprecated: 0, 
    },

    "EPSG:21096" => CRS {
        coord_ref_sys_code: 21096,
        coord_ref_sys_name: "Arc 1960 / UTM zone 36N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4210,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1997-11-13",
        deprecated: 0, 
    },

    "EPSG:21097" => CRS {
        coord_ref_sys_code: 21097,
        coord_ref_sys_name: "Arc 1960 / UTM zone 37N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4210,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1997-11-13",
        deprecated: 0, 
    },

    "EPSG:21100" => CRS {
        coord_ref_sys_code: 21100,
        coord_ref_sys_name: "Batavia (Jakarta) / NEIEZ",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4813,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:21148" => CRS {
        coord_ref_sys_code: 21148,
        coord_ref_sys_name: "Batavia / UTM zone 48S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4211,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:21149" => CRS {
        coord_ref_sys_code: 21149,
        coord_ref_sys_name: "Batavia / UTM zone 49S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4211,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:21150" => CRS {
        coord_ref_sys_code: 21150,
        coord_ref_sys_name: "Batavia / UTM zone 50S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4211,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:21291" => CRS {
        coord_ref_sys_code: 21291,
        coord_ref_sys_name: "Barbados 1938 / British West Indies Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4212,
        remarks: "Replaced by Barbados National Grid (code 21292) from 1983.",
        information_source: "Ordnance Survey of Great Britain",
        data_source: "EPSG",
        revision_date: "2011-06-30",
        deprecated: 0, 
    },

    "EPSG:21292" => CRS {
        coord_ref_sys_code: 21292,
        coord_ref_sys_name: "Barbados 1938 / Barbados National Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4212,
        remarks: "Replaces Barbados 1938 / BWI Grid (code 21291) from 1983.",
        information_source: "Ordnance Survey of Great Britain",
        data_source: "EPSG",
        revision_date: "2011-06-30",
        deprecated: 0, 
    },

    "EPSG:21500" => CRS {
        coord_ref_sys_code: 21500,
        coord_ref_sys_name: "BD50 (Brussels) / Belge Lambert 50",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4809,
        remarks: "Replaced by BD72 / Belge Lambert 72 (code 31300) and BD72 / Belgian Lambert 72 (code 31370).",
        information_source: "IGN Brussels.",
        data_source: "EPSG",
        revision_date: "2021-06-17",
        deprecated: 0, 
    },

    "EPSG:27500" => CRS {
        coord_ref_sys_code: 27500,
        coord_ref_sys_name: "ATF (Paris) / Nord de Guerre",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4901,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:31300" => CRS {
        coord_ref_sys_code: 31300,
        coord_ref_sys_name: "BD72 / Belge Lambert 72",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4313,
        remarks: "Replaces 21500 (BD50 / Belge Lambert 50). An alternative - BD72 / Belgian Lambert 72 (code 31370) - was introduced in 2000 to use the standard Lambert Conic Conformal (2SP) projection method (code 9802). EPSG recommends this alternative.",
        information_source: "IGN Brussels; www.ngi.be/",
        data_source: "EPSG",
        revision_date: "2021-06-17",
        deprecated: 0, 
    },

    "EPSG:31370" => CRS {
        coord_ref_sys_code: 31370,
        coord_ref_sys_name: "BD72 / Belgian Lambert 72",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4313,
        remarks: "Introduced in 2000 as an alternative to CRS code 31300 to avoid the special projection method LCC (2SP Belgium) (code 9803). If software cannot handle latitude of origin 90°N, use latitude of origin &#x3D; 50°47&#x27;57.704&quot;N with Nf &#x3D; 165372.956 m.",
        information_source: "IGN Brussels; www.ngi.be/",
        data_source: "EPSG",
        revision_date: "2021-06-17",
        deprecated: 0, 
    },

    "EPSG:61226405" => CRS {
        coord_ref_sys_code: 61226405,
        coord_ref_sys_name: "ATS77 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4122,
        remarks: "See CRS code 4122 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61766405" => CRS {
        coord_ref_sys_code: 61766405,
        coord_ref_sys_name: "Australian Antarctic (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4176,
        remarks: "See CRS code 4176 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61766413" => CRS {
        coord_ref_sys_code: 61766413,
        coord_ref_sys_name: "Australian Antarctic (3D deg)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6413,
        datum_code: -1, 
        base_crs_code: 4339,
        remarks: "See CRS code 4339 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:61826405" => CRS {
        coord_ref_sys_code: 61826405,
        coord_ref_sys_name: "Azores Occidental 1939 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4182,
        remarks: "See CRS code 4182 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61836405" => CRS {
        coord_ref_sys_code: 61836405,
        coord_ref_sys_name: "Azores Central 1948 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4183,
        remarks: "See CRS code 4183 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61846405" => CRS {
        coord_ref_sys_code: 61846405,
        coord_ref_sys_name: "Azores Oriental 1940 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4184,
        remarks: "See CRS code 4184 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62106405" => CRS {
        coord_ref_sys_code: 62106405,
        coord_ref_sys_name: "Arc 1960 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4210,
        remarks: "See CRS code 4210 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62116405" => CRS {
        coord_ref_sys_code: 62116405,
        coord_ref_sys_name: "Batavia (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4211,
        remarks: "See CRS code 4211 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62126405" => CRS {
        coord_ref_sys_code: 62126405,
        coord_ref_sys_name: "Barbados 1938 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4212,
        remarks: "See CRS code 4212 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62136405" => CRS {
        coord_ref_sys_code: 62136405,
        coord_ref_sys_name: "Beduaram (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4213,
        remarks: "See CRS code 4213 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62146405" => CRS {
        coord_ref_sys_code: 62146405,
        coord_ref_sys_name: "Beijing 1954 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "See CRS code 4214 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:66646405" => CRS {
        coord_ref_sys_code: 66646405,
        coord_ref_sys_name: "Azores Oriental 1995 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4664,
        remarks: "See CRS code 4664 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2003-08-14",
        deprecated: 1, 
    },

    "EPSG:66656405" => CRS {
        coord_ref_sys_code: 66656405,
        coord_ref_sys_name: "Azores Central 1995 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4665,
        remarks: "See CRS code 4665 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2003-08-14",
        deprecated: 1, 
    },

    "EPSG:68136405" => CRS {
        coord_ref_sys_code: 68136405,
        coord_ref_sys_name: "Batavia (Jakarta) (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4813,
        remarks: "See CRS code 4813 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:2401" => CRS {
        coord_ref_sys_code: 2401,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 25",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 75E (code 2422). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 25 (code 2349).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2402" => CRS {
        coord_ref_sys_code: 2402,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 26",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 78E (code 2423). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 26 (code 2350).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2403" => CRS {
        coord_ref_sys_code: 2403,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 27",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 81E (code 2424). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 27 (code 2351).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2404" => CRS {
        coord_ref_sys_code: 2404,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 28",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 84E (code 2425). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 28 (code 2352).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2405" => CRS {
        coord_ref_sys_code: 2405,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 29",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 87E (code 2426). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 29 (code 2353).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2406" => CRS {
        coord_ref_sys_code: 2406,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 30",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 90E (code 2427). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 30 (code 2354).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2407" => CRS {
        coord_ref_sys_code: 2407,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 31",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 93E (code 2428). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 31 (code 2355).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2408" => CRS {
        coord_ref_sys_code: 2408,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 32",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 96E (code 2429). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 32 (code 2356).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2409" => CRS {
        coord_ref_sys_code: 2409,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 33",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 99E (code 2430). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 33 (code 2357).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2410" => CRS {
        coord_ref_sys_code: 2410,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 34",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 102E (code 2431). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 34 (code 2358).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2411" => CRS {
        coord_ref_sys_code: 2411,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 35",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 105E (code 2432). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 35 (code 2359).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2412" => CRS {
        coord_ref_sys_code: 2412,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 36",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 108E (code 2433). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 36 (code 2360).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2413" => CRS {
        coord_ref_sys_code: 2413,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 37",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 111E (code 2434). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 37 (code 2361).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2414" => CRS {
        coord_ref_sys_code: 2414,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 38",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 114E (code 2435). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 38 (code 2362).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2415" => CRS {
        coord_ref_sys_code: 2415,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 39",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 117E (code 2436). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 39 (code 2363).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2416" => CRS {
        coord_ref_sys_code: 2416,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 40",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 120E (code 2437). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 40 (code 2364).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2417" => CRS {
        coord_ref_sys_code: 2417,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 41",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 123E (code 2438). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 41 (code 2365).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2418" => CRS {
        coord_ref_sys_code: 2418,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 42",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 126E (code 2439). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 42 (code 2366).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2419" => CRS {
        coord_ref_sys_code: 2419,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 43",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 129E (code 2440). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 43 (code 2367).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2420" => CRS {
        coord_ref_sys_code: 2420,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 44",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 132E (code 2441). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 44 (code 2368).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2421" => CRS {
        coord_ref_sys_code: 2421,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger zone 45",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / 3-degree Gauss-Kruger CM 135E (code 2442). Replaced by Xian 1980 / 3-degree Gauss-Kruger zone 45 (code 2369).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2422" => CRS {
        coord_ref_sys_code: 2422,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 75E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 25 (code 2401). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 75E (code 2370).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2006-07-22",
        deprecated: 0, 
    },

    "EPSG:2423" => CRS {
        coord_ref_sys_code: 2423,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 78E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 26 (code 2402). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 78E (code 2371).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2424" => CRS {
        coord_ref_sys_code: 2424,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 81E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 27 (code 2403). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 81E (code 2372).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2006-07-22",
        deprecated: 0, 
    },

    "EPSG:2425" => CRS {
        coord_ref_sys_code: 2425,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 84E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 28 (code 2404). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 84E (code 2373).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2426" => CRS {
        coord_ref_sys_code: 2426,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 87E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 29 (code 2405). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 87E (code 2374).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2006-07-22",
        deprecated: 0, 
    },

    "EPSG:2427" => CRS {
        coord_ref_sys_code: 2427,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 90E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 30 (code 2406). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 90E (code 2375).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2428" => CRS {
        coord_ref_sys_code: 2428,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 93E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 31 (code 2407). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 93E (code 2376).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2006-07-22",
        deprecated: 0, 
    },

    "EPSG:2429" => CRS {
        coord_ref_sys_code: 2429,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 96E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 32 (code 2408). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 96E (code 2377).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2430" => CRS {
        coord_ref_sys_code: 2430,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 99E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 33 (code 2409). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 99E (code 2378).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2006-07-22",
        deprecated: 0, 
    },

    "EPSG:2435" => CRS {
        coord_ref_sys_code: 2435,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 114E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 38 (code 2414). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 114E (code 2383).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2436" => CRS {
        coord_ref_sys_code: 2436,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 117E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 39 (code 2415). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 117E (code 2384).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2006-07-22",
        deprecated: 0, 
    },

    "EPSG:2437" => CRS {
        coord_ref_sys_code: 2437,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 120E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 40 (code 2416). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 120E (code 2385).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2438" => CRS {
        coord_ref_sys_code: 2438,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 123E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 41 (code 2417). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 123E (code 2386).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2006-07-22",
        deprecated: 0, 
    },

    "EPSG:2439" => CRS {
        coord_ref_sys_code: 2439,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 126E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 42 (code 2418). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 126E (code 2387).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2440" => CRS {
        coord_ref_sys_code: 2440,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 129E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 43 (code 2419). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 129E (code 2388).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2006-07-22",
        deprecated: 0, 
    },

    "EPSG:2441" => CRS {
        coord_ref_sys_code: 2441,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 132E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 44 (code 2420). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 132E (code 2389).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2442" => CRS {
        coord_ref_sys_code: 2442,
        coord_ref_sys_name: "Beijing 1954 / 3-degree Gauss-Kruger CM 135E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / 3-degree Gauss-Kruger zone 45 (code 2421). Replaced by Xian 1980 / 3-degree Gauss-Kruger CM 135E (code 2390).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2006-07-22",
        deprecated: 0, 
    },

    "EPSG:3769" => CRS {
        coord_ref_sys_code: 3769,
        coord_ref_sys_name: "Bermuda 1957 / UTM zone 20N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4216,
        remarks: "Replaced by BDA2000 / Bermuda 2000 National Grid (alias BNG2000) (CRS code 3770).",
        information_source: "Department of Lands Buildings and Surveys.",
        data_source: "EPSG",
        revision_date: "2007-12-12",
        deprecated: 0, 
    },

    "EPSG:4165" => CRS {
        coord_ref_sys_code: 4165,
        coord_ref_sys_name: "Bissau",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6165, 
        base_crs_code: -1,
        remarks: "",
        information_source: "NIMA",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4216" => CRS {
        coord_ref_sys_code: 4216,
        coord_ref_sys_name: "Bermuda 1957",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6216, 
        base_crs_code: -1,
        remarks: "Replaced by BDA2000 (CRS code 4762).",
        information_source: "Department of Lands Buildings and Surveys.",
        data_source: "EPSG",
        revision_date: "2011-01-25",
        deprecated: 0, 
    },

    "EPSG:4306" => CRS {
        coord_ref_sys_code: 4306,
        coord_ref_sys_name: "Bern 1938",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6306, 
        base_crs_code: -1,
        remarks: "Used for the geographic coordinates overprinted on topographic maps constructed on the CH1903 / LV03 projected CS (code 21781).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4714" => CRS {
        coord_ref_sys_code: 4714,
        coord_ref_sys_name: "Bellevue",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6714, 
        base_crs_code: -1,
        remarks: "",
        information_source: "DMA / NIMA / NGA TR8350.2 (3rd edition, January 2000).",
        data_source: "EPSG",
        revision_date: "2006-07-29",
        deprecated: 0, 
    },

    "EPSG:4801" => CRS {
        coord_ref_sys_code: 4801,
        coord_ref_sys_name: "Bern 1898 (Bern)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6801, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:5732" => CRS {
        coord_ref_sys_code: 5732,
        coord_ref_sys_name: "Belfast height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5131, 
        base_crs_code: -1,
        remarks: "Malin Head height (CRS code 5731) used for medium and small scale topographic mapping.",
        information_source: "Ordnance Survey of Northern Ireland.",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:6882" => CRS {
        coord_ref_sys_code: 6882,
        coord_ref_sys_name: "Bekaa Valley 1920",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1137, 
        base_crs_code: -1,
        remarks: "",
        information_source: "US NGA GeoTrans software v3.4.",
        data_source: "EPSG",
        revision_date: "2014-07-30",
        deprecated: 0, 
    },

    "EPSG:6883" => CRS {
        coord_ref_sys_code: 6883,
        coord_ref_sys_name: "Bioko",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1136, 
        base_crs_code: -1,
        remarks: "",
        information_source: "US NGA GeoTrans software v3.4.",
        data_source: "EPSG",
        revision_date: "2014-07-30",
        deprecated: 0, 
    },

    "EPSG:7796" => CRS {
        coord_ref_sys_code: 7796,
        coord_ref_sys_name: "BGS2005",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1167, 
        base_crs_code: -1,
        remarks: "Adopted 2010-07-29.",
        information_source: "Geodesy, Cartography and Geoinformation Systems Directorate (GCGIS).",
        data_source: "EPSG",
        revision_date: "2016-04-27",
        deprecated: 0, 
    },

    "EPSG:7797" => CRS {
        coord_ref_sys_code: 7797,
        coord_ref_sys_name: "BGS2005",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1167, 
        base_crs_code: 7796,
        remarks: "Adopted 2010-07-29.",
        information_source: "Geodesy, Cartography and Geoinformation Systems Directorate (GCGIS).",
        data_source: "EPSG",
        revision_date: "2016-04-27",
        deprecated: 0, 
    },

    "EPSG:7798" => CRS {
        coord_ref_sys_code: 7798,
        coord_ref_sys_name: "BGS2005",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1167, 
        base_crs_code: 7797,
        remarks: "Adopted 2010-07-29. Replaces earlier systems.",
        information_source: "IOGP. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2016-04-27",
        deprecated: 0, 
    },

    "EPSG:7799" => CRS {
        coord_ref_sys_code: 7799,
        coord_ref_sys_name: "BGS2005 / UTM zone 34N (N-E)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4531,
        datum_code: -1, 
        base_crs_code: 7798,
        remarks: "Adopted 2010-07-29, replacing earlier systems. See BGS2005 / CCS2005 (CRS code 7801) for cadastral and other large scale mapping purposes. See BGS2005 / UTM zone 34N (CRS code 7803) for similar CRS with east-north axis order used for GIS purposes.",
        information_source: "Geodesy, Cartography and Geoinformation Systems Directorate (GCGIS).",
        data_source: "EPSG",
        revision_date: "2020-07-01",
        deprecated: 0, 
    },

    "EPSG:7800" => CRS {
        coord_ref_sys_code: 7800,
        coord_ref_sys_name: "BGS2005 / UTM zone 35N (N-E)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4531,
        datum_code: -1, 
        base_crs_code: 7798,
        remarks: "Adopted 2010-07-29, replacing earlier systems. See BGS2005 / CCS2005 (CRS code 7801) for cadastral and other large scale mapping purposes. See BGS2005 / UTM zone 35N (CRS code 9391) for similar CRS with east-north axis order used for GIS purposes.",
        information_source: "Geodesy, Cartography and Geoinformation Systems Directorate (GCGIS).",
        data_source: "EPSG",
        revision_date: "2020-07-01",
        deprecated: 0, 
    },

    "EPSG:7801" => CRS {
        coord_ref_sys_code: 7801,
        coord_ref_sys_name: "BGS2005 / CCS2005",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4531,
        datum_code: -1, 
        base_crs_code: 7798,
        remarks: "Adopted 2010-07-29, See BGS2005 / CS2005 zones 34 and 35 (CRS codes 7799 and 7800) for smaller scale mapping purposes.",
        information_source: "Geodesy, Cartography and Geoinformation Systems Directorate (GCGIS).",
        data_source: "EPSG",
        revision_date: "2016-04-27",
        deprecated: 0, 
    },

    "EPSG:7803" => CRS {
        coord_ref_sys_code: 7803,
        coord_ref_sys_name: "BGS2005 / UTM zone 34N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7798,
        remarks: "See BGS2005 / UTM zone 34N (N-E) (CRS code 7799) for similar CRS with north-east axis order defined as the official system in Bulgaria.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2016-04-27",
        deprecated: 0, 
    },

    "EPSG:7804" => CRS {
        coord_ref_sys_code: 7804,
        coord_ref_sys_name: "BGS2005 / UTM zone 35N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7798,
        remarks: "See BGS2005 / UTM zone 35N (N-E) (CRS code 7800) for similar CRS with north-east axis order defined as the official system in Bulgaria.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2016-04-27",
        deprecated: 1, 
    },

    "EPSG:7805" => CRS {
        coord_ref_sys_code: 7805,
        coord_ref_sys_name: "BGS2005 / UTM zone 36N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7798,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2016-04-27",
        deprecated: 0, 
    },

    "EPSG:9391" => CRS {
        coord_ref_sys_code: 9391,
        coord_ref_sys_name: "BGS2005 / UTM zone 35N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7798,
        remarks: "See BGS2005 / UTM zone 35N (N-E) (CRS code 7800) for similar CRS with north-east axis order defined as the official system in Bulgaria.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2020-07-01",
        deprecated: 0, 
    },

    "EPSG:9451" => CRS {
        coord_ref_sys_code: 9451,
        coord_ref_sys_name: "BI height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1288, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Ordnance Survey Limited",
        data_source: "EPSG",
        revision_date: "2020-07-24",
        deprecated: 0, 
    },

    "EPSG:9669" => CRS {
        coord_ref_sys_code: 9669,
        coord_ref_sys_name: "BGS2005 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1300, 
        base_crs_code: -1,
        remarks: "Adopted 2010-07-29 as official Bulgarian reference datum through decree 153, replacing Baltic 1982 system (CRS code 5786). ",
        information_source: "Geodesy, Cartography and Geoinformation Systems Directorate (GCGIS).",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:21413" => CRS {
        coord_ref_sys_code: 21413,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger zone 13",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / Gauss-Kruger CM 75E (code 24153). Replaced by Xian 1980 / Gauss-Kruger zone 13 (code 2327).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21414" => CRS {
        coord_ref_sys_code: 21414,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger zone 14",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / Gauss-Kruger CM 81E (code 24154). Replaced by Xian 1980 / Gauss-Kruger zone 14 (code 2328).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21415" => CRS {
        coord_ref_sys_code: 21415,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger zone 15",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / Gauss-Kruger CM 87E (code 24155). Replaced by Xian 1980 / Gauss-Kruger zone 15 (code 2329).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21416" => CRS {
        coord_ref_sys_code: 21416,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger zone 16",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / Gauss-Kruger CM 93E (code 24156). Replaced by Xian 1980 / Gauss-Kruger zone 16 (code 2330).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21417" => CRS {
        coord_ref_sys_code: 21417,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger zone 17",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / Gauss-Kruger CM 99E (code 24157). Replaced by Xian 1980 / Gauss-Kruger zone 17 (code 2331).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:6179" => CRS {
        coord_ref_sys_code: 6179,
        coord_ref_sys_name: "Cais da Vila - Porto Santo height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1102, 
        base_crs_code: -1,
        remarks: "",
        information_source: "General Directorate of Land, http://www.dgterritorio.pt/",
        data_source: "EPSG",
        revision_date: "2013-03-15",
        deprecated: 0, 
    },

    "EPSG:21418" => CRS {
        coord_ref_sys_code: 21418,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger zone 18",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / Gauss-Kruger CM 105E (code 24158). Replaced by Xian 1980 / Gauss-Kruger zone 18 (code 2332).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21419" => CRS {
        coord_ref_sys_code: 21419,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger zone 19",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / Gauss-Kruger CM 111E (code 24159). Replaced by Xian 1980 / Gauss-Kruger zone 19 (code 2333).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21420" => CRS {
        coord_ref_sys_code: 21420,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger zone 20",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / Gauss-Kruger CM 117E (code 24160). Replaced by Xian 1980 / Gauss-Kruger zone 20 (code 2334).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21421" => CRS {
        coord_ref_sys_code: 21421,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger zone 21",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / Gauss-Kruger CM 123E (code 24161). Replaced by Xian 1980 / Gauss-Kruger zone 21 (code 2335).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21422" => CRS {
        coord_ref_sys_code: 21422,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger zone 22",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / Gauss-Kruger CM 129E (code 24162). Replaced by Xian 1980 / Gauss-Kruger zone 22 (code 2336).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21423" => CRS {
        coord_ref_sys_code: 21423,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger zone 23",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Also found with truncated false easting - see Beijing 1954 / Gauss-Kruger CM 135E (code 24163). Replaced by Xian 1980 / Gauss-Kruger zone 23 (code 2337).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21453" => CRS {
        coord_ref_sys_code: 21453,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger CM 75E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / Gauss-Kruger zone 13 (code 21413). Replaced by Xian 1980 / Gauss-Kruger CM 75E (code 2338).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21454" => CRS {
        coord_ref_sys_code: 21454,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger CM 81E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / Gauss-Kruger zone 14 (code 21414). Replaced by Xian 1980 / Gauss-Kruger CM 81E (code 2339).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21455" => CRS {
        coord_ref_sys_code: 21455,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger CM 87E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / Gauss-Kruger zone 15 (code 21415). Replaced by Xian 1980 / Gauss-Kruger CM 87E (code 2340).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21456" => CRS {
        coord_ref_sys_code: 21456,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger CM 93E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / Gauss-Kruger zone 16 (code 21416). Replaced by Xian 1980 / Gauss-Kruger CM 93E (code 2341).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21457" => CRS {
        coord_ref_sys_code: 21457,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger CM 99E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / Gauss-Kruger zone 17 (code 21417). Replaced by Xian 1980 / Gauss-Kruger CM 99E (code 2342).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21458" => CRS {
        coord_ref_sys_code: 21458,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger CM 105E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / Gauss-Kruger zone 18 (code 21418). Replaced by Xian 1980 / Gauss-Kruger CM 105E (code 2343).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21459" => CRS {
        coord_ref_sys_code: 21459,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger CM 111E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / Gauss-Kruger zone 19 (code 21419). Replaced by Xian 1980 / Gauss-Kruger CM 111E (code 2344).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21460" => CRS {
        coord_ref_sys_code: 21460,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger CM 117E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / Gauss-Kruger zone 20 (code 21420). Replaced by Xian 1980 / Gauss-Kruger CM 117E (code 2345).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21461" => CRS {
        coord_ref_sys_code: 21461,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger CM 123E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / Gauss-Kruger zone 21 (code 21421). Replaced by Xian 1980 / Gauss-Kruger CM 123E (code 2346).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21462" => CRS {
        coord_ref_sys_code: 21462,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger CM 129E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / Gauss-Kruger zone 22 (code 21422). Replaced by Xian 1980 / Gauss-Kruger CM 129E (code 2347).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21463" => CRS {
        coord_ref_sys_code: 21463,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger CM 135E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "Truncated form of Beijing 1954 / Gauss-Kruger zone 33 (code 21423). Replaced by Xian 1980 / Gauss-Kruger CM 135E (code 2348).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:21473" => CRS {
        coord_ref_sys_code: 21473,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger 13N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:21474" => CRS {
        coord_ref_sys_code: 21474,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger 14N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:21475" => CRS {
        coord_ref_sys_code: 21475,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger 15N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:21476" => CRS {
        coord_ref_sys_code: 21476,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger 16N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:21477" => CRS {
        coord_ref_sys_code: 21477,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger 17N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:21478" => CRS {
        coord_ref_sys_code: 21478,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger 18N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:21479" => CRS {
        coord_ref_sys_code: 21479,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger 19N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:21480" => CRS {
        coord_ref_sys_code: 21480,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger 20N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:21481" => CRS {
        coord_ref_sys_code: 21481,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger 21N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:21482" => CRS {
        coord_ref_sys_code: 21482,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger 22N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:21483" => CRS {
        coord_ref_sys_code: 21483,
        coord_ref_sys_name: "Beijing 1954 / Gauss-Kruger 23N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4214,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:21780" => CRS {
        coord_ref_sys_code: 21780,
        coord_ref_sys_name: "Bern 1898 (Bern) / LV03C",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4498,
        datum_code: -1, 
        base_crs_code: 4801,
        remarks: "In Switzerland, replaced by CH1903 / LV03 (CRS code 21781). For use in Liechtenstein may be implemented through Greenwich based equivalent CRS code 21782.",
        information_source: "Bundesamt für Landestopografie swisstopo; www.swisstopo.ch",
        data_source: "EPSG",
        revision_date: "2015-11-25",
        deprecated: 0, 
    },

    "EPSG:61656405" => CRS {
        coord_ref_sys_code: 61656405,
        coord_ref_sys_name: "Bissau (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4165,
        remarks: "See CRS code 4165 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62156405" => CRS {
        coord_ref_sys_code: 62156405,
        coord_ref_sys_name: "Belge 1950 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4215,
        remarks: "See CRS code 4215 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62166405" => CRS {
        coord_ref_sys_code: 62166405,
        coord_ref_sys_name: "Bermuda 1957 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4216,
        remarks: "See CRS code 4216 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:63066405" => CRS {
        coord_ref_sys_code: 63066405,
        coord_ref_sys_name: "Bern 1938 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4306,
        remarks: "See CRS code 4306 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:63136405" => CRS {
        coord_ref_sys_code: 63136405,
        coord_ref_sys_name: "Belge 1972 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4313,
        remarks: "See CRS code 4313 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:68016405" => CRS {
        coord_ref_sys_code: 68016405,
        coord_ref_sys_name: "Bern 1898 (Bern) (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4801,
        remarks: "See CRS code 4801 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:68096405" => CRS {
        coord_ref_sys_code: 68096405,
        coord_ref_sys_name: "Belge 1950 (Brussels) (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4809,
        remarks: "See CRS code 4809 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:2088" => CRS {
        coord_ref_sys_code: 2088,
        coord_ref_sys_name: "Carthage / TM 11 NE",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4223,
        remarks: "",
        information_source: "Total-Fina",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2095" => CRS {
        coord_ref_sys_code: 2095,
        coord_ref_sys_name: "Bissau / UTM zone 28N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4165,
        remarks: "",
        information_source: "US National Imagery and Mapping Agency TR8350.2.",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2315" => CRS {
        coord_ref_sys_code: 2315,
        coord_ref_sys_name: "Campo Inchauspe / UTM zone 19S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4221,
        remarks: "",
        information_source: "TotalFinaElf",
        data_source: "EPSG",
        revision_date: "2016-12-15",
        deprecated: 0, 
    },

    "EPSG:2316" => CRS {
        coord_ref_sys_code: 2316,
        coord_ref_sys_name: "Campo Inchauspe / UTM zone 20S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4221,
        remarks: "",
        information_source: "TotalFinaElf",
        data_source: "EPSG",
        revision_date: "2016-12-15",
        deprecated: 0, 
    },

    "EPSG:4218" => CRS {
        coord_ref_sys_code: 4218,
        coord_ref_sys_name: "Bogota 1975",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6218, 
        base_crs_code: -1,
        remarks: "Replaces earlier 3 adjustments of 1951, 1944 and 1941. Replaced by MAGNA-SIRGAS (CRS code 4685).",
        information_source: "IGAC special publication no. 1, &quot;Geodesia&quot; 4th edition, 1975.",
        data_source: "EPSG",
        revision_date: "2010-03-30",
        deprecated: 0, 
    },

    "EPSG:4219" => CRS {
        coord_ref_sys_code: 4219,
        coord_ref_sys_name: "Bukit Rimpah",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6219, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4220" => CRS {
        coord_ref_sys_code: 4220,
        coord_ref_sys_name: "Camacupa 1948",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6220, 
        base_crs_code: -1,
        remarks: "Provisional adjustment. Coastal stations used for offshore radio-navigation positioning and determination of transformations to WGS. Differs from second adjustment (Camacupa 2015, CRS code 8694), which is not used for offshore E&amp;P, by up to 25m.",
        information_source: "IOGP, Portuguese Hydrographic Institute and Clifford J. Mugnier, PE&amp;RS journal, March 2001.",
        data_source: "EPSG",
        revision_date: "2020-04-30",
        deprecated: 0, 
    },

    "EPSG:4221" => CRS {
        coord_ref_sys_code: 4221,
        coord_ref_sys_name: "Campo Inchauspe",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6221, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2011-03-28",
        deprecated: 0, 
    },

    "EPSG:4222" => CRS {
        coord_ref_sys_code: 4222,
        coord_ref_sys_name: "Cape",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6222, 
        base_crs_code: -1,
        remarks: "Replaced by Hartbeesthoek94 from 1999.",
        information_source: "Directorate of Surveys and Mapping, Mowbray, South Africa.",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4223" => CRS {
        coord_ref_sys_code: 4223,
        coord_ref_sys_name: "Carthage",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6223, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4472" => CRS {
        coord_ref_sys_code: 4472,
        coord_ref_sys_name: "Cadastre 1997",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1037, 
        base_crs_code: 4473,
        remarks: "",
        information_source: "CERTU",
        data_source: "EPSG",
        revision_date: "2009-10-02",
        deprecated: 0, 
    },

    "EPSG:4473" => CRS {
        coord_ref_sys_code: 4473,
        coord_ref_sys_name: "Cadastre 1997",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1037, 
        base_crs_code: -1,
        remarks: "",
        information_source: "CERTU.",
        data_source: "EPSG",
        revision_date: "2009-10-02",
        deprecated: 0, 
    },

    "EPSG:4474" => CRS {
        coord_ref_sys_code: 4474,
        coord_ref_sys_name: "Cadastre 1997 / UTM zone 38S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4632,
        remarks: "Replaces Combani 1950 / UTM zone 38S (CRS code 2980) for cadastral purposes only. For other purposes see RGM04 / UTM zone 38S (CRS code 4471).",
        information_source: "CERTU",
        data_source: "EPSG",
        revision_date: "2009-10-02",
        deprecated: 1, 
    },

    "EPSG:4475" => CRS {
        coord_ref_sys_code: 4475,
        coord_ref_sys_name: "Cadastre 1997",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1037, 
        base_crs_code: 4472,
        remarks: "Replaces Combani 1950 (CRS code 4632) for cadastral purposes only. For other purposes see RGM04 (CRS code 4470).",
        information_source: "EPSG. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 0, 
    },

    "EPSG:4534" => CRS {
        coord_ref_sys_code: 4534,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 75E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 25 (code 4513).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4535" => CRS {
        coord_ref_sys_code: 4535,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 78E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 26 (code 4514).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4536" => CRS {
        coord_ref_sys_code: 4536,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 81E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 27 (code 4515).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4543" => CRS {
        coord_ref_sys_code: 4543,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 102E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 34 (code 4522).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4544" => CRS {
        coord_ref_sys_code: 4544,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 105E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 35 (code 4523).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4545" => CRS {
        coord_ref_sys_code: 4545,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 108E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 36 (code 4524).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4546" => CRS {
        coord_ref_sys_code: 4546,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 111E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 37 (code 4525).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4547" => CRS {
        coord_ref_sys_code: 4547,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 114E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 38 (code 4526).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4548" => CRS {
        coord_ref_sys_code: 4548,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 117E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 39 (code 4527).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4549" => CRS {
        coord_ref_sys_code: 4549,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 120E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 40 (code 4528).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4550" => CRS {
        coord_ref_sys_code: 4550,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 123E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 41 (code 4529).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4551" => CRS {
        coord_ref_sys_code: 4551,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 126E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 42 (code 4530).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-05",
        deprecated: 0, 
    },

    "EPSG:4552" => CRS {
        coord_ref_sys_code: 4552,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 129E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 43 (code 4531).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-05",
        deprecated: 0, 
    },

    "EPSG:4553" => CRS {
        coord_ref_sys_code: 4553,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 132E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 44 (code 4532).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-05",
        deprecated: 0, 
    },

    "EPSG:4554" => CRS {
        coord_ref_sys_code: 4554,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 135E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 45 (code 4533).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-05",
        deprecated: 0, 
    },

    "EPSG:4715" => CRS {
        coord_ref_sys_code: 4715,
        coord_ref_sys_name: "Camp Area Astro",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6715, 
        base_crs_code: -1,
        remarks: "Replaced by RSRGD2000 (CRS code 4764). The relationship to this is variable. See Land Information New Zealand LINZS25001.",
        information_source: "DMA / NIMA / NGA TR8350.2 (3rd edition, January 2000).",
        data_source: "EPSG",
        revision_date: "2008-04-04",
        deprecated: 0, 
    },

    "EPSG:4717" => CRS {
        coord_ref_sys_code: 4717,
        coord_ref_sys_name: "Cape Canaveral",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6717, 
        base_crs_code: -1,
        remarks: "",
        information_source: "DMA / NIMA / NGA TR8350.2 (3rd edition, January 2000).",
        data_source: "EPSG",
        revision_date: "2006-01-26",
        deprecated: 0, 
    },

    "EPSG:4802" => CRS {
        coord_ref_sys_code: 4802,
        coord_ref_sys_name: "Bogota 1975 (Bogota)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6802, 
        base_crs_code: -1,
        remarks: "Replaces earlier 3 adjustments of 1951, 1944 and 1941.",
        information_source: "IGAC special publication no. 1, &quot;Geodesia&quot; 4th edition, 1975.",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4816" => CRS {
        coord_ref_sys_code: 4816,
        coord_ref_sys_name: "Carthage (Paris)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6403,
        datum_code: 6816, 
        base_crs_code: -1,
        remarks: "Replaced by Greenwich-based Carthage geogCRS.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:5336" => CRS {
        coord_ref_sys_code: 5336,
        coord_ref_sys_name: "Black Sea depth",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6498,
        datum_code: -1, 
        base_crs_code: 5735,
        remarks: "Black Sea datum is 0.4m below Baltic datum.",
        information_source: "BP",
        data_source: "EPSG",
        revision_date: "2021-04-20",
        deprecated: 0, 
    },

    "EPSG:5607" => CRS {
        coord_ref_sys_code: 5607,
        coord_ref_sys_name: "Bora Bora SAU 2001 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5202, 
        base_crs_code: -1,
        remarks: "Part of NGPF (CRS code 5600).",
        information_source: "Gouvernement de la Polynésie Française, Service de l&#x27;Urbanisme, Section topographie.",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5611" => CRS {
        coord_ref_sys_code: 5611,
        coord_ref_sys_name: "Caspian height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5106, 
        base_crs_code: -1,
        remarks: "Caspian Sea water levels are now offset appreciably from this datum.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2020-06-17",
        deprecated: 0, 
    },

    "EPSG:5706" => CRS {
        coord_ref_sys_code: 5706,
        coord_ref_sys_name: "Caspian depth",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6498,
        datum_code: -1, 
        base_crs_code: 5611,
        remarks: "Caspian Sea water levels are now offset appreciably from this datum.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-04-20",
        deprecated: 0, 
    },

    "EPSG:5735" => CRS {
        coord_ref_sys_code: 5735,
        coord_ref_sys_name: "Black Sea height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5134, 
        base_crs_code: -1,
        remarks: "Black Sea datum is 0.4m below Baltic datum.",
        information_source: "BP",
        data_source: "EPSG",
        revision_date: "2011-01-25",
        deprecated: 0, 
    },

    "EPSG:5760" => CRS {
        coord_ref_sys_code: 5760,
        coord_ref_sys_name: "Bluff 1955 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5158, 
        base_crs_code: -1,
        remarks: "",
        information_source: "OSG Technical Report #10, February 2001.",
        data_source: "EPSG",
        revision_date: "2011-01-25",
        deprecated: 0, 
    },

    "EPSG:5780" => CRS {
        coord_ref_sys_code: 5780,
        coord_ref_sys_name: "Cascais height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5178, 
        base_crs_code: -1,
        remarks: "",
        information_source: "EuroGeographics, http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5879" => CRS {
        coord_ref_sys_code: 5879,
        coord_ref_sys_name: "Cadastre 1997 / UTM zone 38S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4475,
        remarks: "Replaces Combani 1950 / UTM zone 38S (CRS code 2980) for cadastral purposes only. For other purposes see RGM04 / UTM zone 38S (CRS code 4471).",
        information_source: "CERTU",
        data_source: "EPSG",
        revision_date: "2012-09-27",
        deprecated: 0, 
    },

    "EPSG:6132" => CRS {
        coord_ref_sys_code: 6132,
        coord_ref_sys_name: "CBVD61 height (ft)",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 1030,
        datum_code: 1099, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Lands and Survey Department, Cayman Islands Government.",
        data_source: "EPSG",
        revision_date: "2017-10-16",
        deprecated: 0, 
    },

    "EPSG:6141" => CRS {
        coord_ref_sys_code: 6141,
        coord_ref_sys_name: "Cayman Islands National Grid 2011",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 1039,
        datum_code: -1, 
        base_crs_code: 6135,
        remarks: "",
        information_source: "Lands and Surveys Department, Cayman Islands Government.",
        data_source: "EPSG",
        revision_date: "2013-01-29",
        deprecated: 1, 
    },

    "EPSG:22391" => CRS {
        coord_ref_sys_code: 22391,
        coord_ref_sys_name: "Carthage / Nord Tunisie",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4223,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:6180" => CRS {
        coord_ref_sys_code: 6180,
        coord_ref_sys_name: "Cais das Velas height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1103, 
        base_crs_code: -1,
        remarks: "",
        information_source: "General Directorate of Land, http://www.dgterritorio.pt/",
        data_source: "EPSG",
        revision_date: "2013-03-15",
        deprecated: 0, 
    },

    "EPSG:6182" => CRS {
        coord_ref_sys_code: 6182,
        coord_ref_sys_name: "Cais da Madalena height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1105, 
        base_crs_code: -1,
        remarks: "",
        information_source: "General Directorate of Land, http://www.dgterritorio.pt/",
        data_source: "EPSG",
        revision_date: "2013-03-15",
        deprecated: 0, 
    },

    "EPSG:6184" => CRS {
        coord_ref_sys_code: 6184,
        coord_ref_sys_name: "Cais da Figueirinha - Angra do Heroismo height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1107, 
        base_crs_code: -1,
        remarks: "",
        information_source: "General Directorate of Land, http://www.dgterritorio.pt/",
        data_source: "EPSG",
        revision_date: "2014-09-18",
        deprecated: 0, 
    },

    "EPSG:6186" => CRS {
        coord_ref_sys_code: 6186,
        coord_ref_sys_name: "Cais da Vila do Porto height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1109, 
        base_crs_code: -1,
        remarks: "",
        information_source: "General Directorate of Land, http://www.dgterritorio.pt/",
        data_source: "EPSG",
        revision_date: "2013-03-25",
        deprecated: 0, 
    },

    "EPSG:6391" => CRS {
        coord_ref_sys_code: 6391,
        coord_ref_sys_name: "Cayman Islands National Grid 2011",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 1039,
        datum_code: -1, 
        base_crs_code: 6135,
        remarks: "",
        information_source: "Lands and Surveys Department, Cayman Islands Government.",
        data_source: "EPSG",
        revision_date: "2013-09-01",
        deprecated: 0, 
    },

    "EPSG:8694" => CRS {
        coord_ref_sys_code: 8694,
        coord_ref_sys_name: "Camacupa 2015",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1217, 
        base_crs_code: -1,
        remarks: "Camacupa 1948 (CRS code 4220) is used for offshore oil and gas exploration and production. Camacupa 2015 differs from Camacupa 1948 by up to 25m.",
        information_source: "University of Lisbon and Comissão Interministerial para a Delimitação e Demarcação dos Espaços Marítimos de Angola (CIDDEMA).",
        data_source: "EPSG",
        revision_date: "2018-12-04",
        deprecated: 0, 
    },

    "EPSG:9402" => CRS {
        coord_ref_sys_code: 9402,
        coord_ref_sys_name: "Ceuta 2 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1285, 
        base_crs_code: -1,
        remarks: "",
        information_source: "National Geographic Institute of Spain (IGN), http://www.ign.es",
        data_source: "EPSG",
        revision_date: "2020-06-27",
        deprecated: 0, 
    },

    "EPSG:9672" => CRS {
        coord_ref_sys_code: 9672,
        coord_ref_sys_name: "CD Norway depth",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6498,
        datum_code: 1301, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority, Hydrographic Service.",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:9721" => CRS {
        coord_ref_sys_code: 9721,
        coord_ref_sys_name: "Catania 1965 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1306, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Istituto Geografico Militare (IGM), www.igmi.org",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:9722" => CRS {
        coord_ref_sys_code: 9722,
        coord_ref_sys_name: "Cagliari 1956 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1307, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Istituto Geografico Militare (IGM), www.igmi.org",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:21817" => CRS {
        coord_ref_sys_code: 21817,
        coord_ref_sys_name: "Bogota 1975 / UTM zone 17N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4218,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-10-19",
        deprecated: 1, 
    },

    "EPSG:21818" => CRS {
        coord_ref_sys_code: 21818,
        coord_ref_sys_name: "Bogota 1975 / UTM zone 18N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4218,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-10-19",
        deprecated: 0, 
    },

    "EPSG:21891" => CRS {
        coord_ref_sys_code: 21891,
        coord_ref_sys_name: "Bogota 1975 / Colombia West zone",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4218,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-10-19",
        deprecated: 1, 
    },

    "EPSG:21892" => CRS {
        coord_ref_sys_code: 21892,
        coord_ref_sys_name: "Bogota 1975 / Colombia Bogota zone",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4218,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-10-19",
        deprecated: 1, 
    },

    "EPSG:21893" => CRS {
        coord_ref_sys_code: 21893,
        coord_ref_sys_name: "Bogota 1975 / Colombia East Central zone",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4218,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-10-19",
        deprecated: 1, 
    },

    "EPSG:21894" => CRS {
        coord_ref_sys_code: 21894,
        coord_ref_sys_name: "Bogota 1975 / Colombia East",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4218,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-10-19",
        deprecated: 1, 
    },

    "EPSG:21896" => CRS {
        coord_ref_sys_code: 21896,
        coord_ref_sys_name: "Bogota 1975 / Colombia West zone",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4218,
        remarks: "Replaced by MAGNA-SIRGAS / Colombia Far West zone and MAGNA-SIRGAS / Colombia West zone (CRS codes 3114 and 3115).",
        information_source: "Instituto Geografico Agustin Codazzi (IGAC).",
        data_source: "EPSG",
        revision_date: "2018-06-12",
        deprecated: 0, 
    },

    "EPSG:21897" => CRS {
        coord_ref_sys_code: 21897,
        coord_ref_sys_name: "Bogota 1975 / Colombia Bogota zone",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4218,
        remarks: "Replaced by MAGNA-SIRGAS / Colombia Bogota zone (CRS code 3116).",
        information_source: "Instituto Geografico Agustin Codazzi (IGAC).",
        data_source: "EPSG",
        revision_date: "2018-06-12",
        deprecated: 0, 
    },

    "EPSG:21898" => CRS {
        coord_ref_sys_code: 21898,
        coord_ref_sys_name: "Bogota 1975 / Colombia East Central zone",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4218,
        remarks: "Replaced by MAGNA-SIRGAS / Colombia East Central zone (CRS code 3117).",
        information_source: "Instituto Geografico Agustin Codazzi (IGAC).",
        data_source: "EPSG",
        revision_date: "2018-06-12",
        deprecated: 0, 
    },

    "EPSG:21899" => CRS {
        coord_ref_sys_code: 21899,
        coord_ref_sys_name: "Bogota 1975 / Colombia East zone",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4218,
        remarks: "Replaced by MAGNA-SIRGAS / Colombia East zone (CRS code 3118).",
        information_source: "Instituto Geografico Agustin Codazzi (IGAC).",
        data_source: "EPSG",
        revision_date: "2018-06-12",
        deprecated: 0, 
    },

    "EPSG:22032" => CRS {
        coord_ref_sys_code: 22032,
        coord_ref_sys_name: "Camacupa 1948 / UTM zone 32S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4220,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2019-09-17",
        deprecated: 0, 
    },

    "EPSG:22033" => CRS {
        coord_ref_sys_code: 22033,
        coord_ref_sys_name: "Camacupa 1948 / UTM zone 33S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4220,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2019-09-17",
        deprecated: 0, 
    },

    "EPSG:22091" => CRS {
        coord_ref_sys_code: 22091,
        coord_ref_sys_name: "Camacupa 1948 / TM 11.30 SE",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4220,
        remarks: "Used by Esso Angola for offshore blocks 15 and 24.",
        information_source: "Esso Angola",
        data_source: "EPSG",
        revision_date: "2019-09-17",
        deprecated: 0, 
    },

    "EPSG:22092" => CRS {
        coord_ref_sys_code: 22092,
        coord_ref_sys_name: "Camacupa 1948 / TM 12 SE",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4220,
        remarks: "Used for exploration and production geoscience activity. Note: WGS 84 / TM 12 SE (CRS code 5842) used for Angola LNG project.",
        information_source: "Shell Angola",
        data_source: "EPSG",
        revision_date: "2019-09-17",
        deprecated: 0, 
    },

    "EPSG:22191" => CRS {
        coord_ref_sys_code: 22191,
        coord_ref_sys_name: "Campo Inchauspe / Argentina 1",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4221,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2016-12-15",
        deprecated: 0, 
    },

    "EPSG:22192" => CRS {
        coord_ref_sys_code: 22192,
        coord_ref_sys_name: "Campo Inchauspe / Argentina 2",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4221,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:22193" => CRS {
        coord_ref_sys_code: 22193,
        coord_ref_sys_name: "Campo Inchauspe / Argentina 3",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4221,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:22194" => CRS {
        coord_ref_sys_code: 22194,
        coord_ref_sys_name: "Campo Inchauspe / Argentina 4",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4221,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:22195" => CRS {
        coord_ref_sys_code: 22195,
        coord_ref_sys_name: "Campo Inchauspe / Argentina 5",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4221,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2016-12-15",
        deprecated: 0, 
    },

    "EPSG:22196" => CRS {
        coord_ref_sys_code: 22196,
        coord_ref_sys_name: "Campo Inchauspe / Argentina 6",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4221,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2016-12-15",
        deprecated: 0, 
    },

    "EPSG:22197" => CRS {
        coord_ref_sys_code: 22197,
        coord_ref_sys_name: "Campo Inchauspe / Argentina 7",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4221,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2016-12-15",
        deprecated: 0, 
    },

    "EPSG:22234" => CRS {
        coord_ref_sys_code: 22234,
        coord_ref_sys_name: "Cape / UTM zone 34S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4222,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-12-08",
        deprecated: 0, 
    },

    "EPSG:22235" => CRS {
        coord_ref_sys_code: 22235,
        coord_ref_sys_name: "Cape / UTM zone 35S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4222,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-12-08",
        deprecated: 0, 
    },

    "EPSG:22236" => CRS {
        coord_ref_sys_code: 22236,
        coord_ref_sys_name: "Cape / UTM zone 36S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4222,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-12-08",
        deprecated: 1, 
    },

    "EPSG:22275" => CRS {
        coord_ref_sys_code: 22275,
        coord_ref_sys_name: "Cape / Lo15",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4222,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping, Mowbray, South Africa.",
        data_source: "EPSG",
        revision_date: "2007-02-12",
        deprecated: 0, 
    },

    "EPSG:22277" => CRS {
        coord_ref_sys_code: 22277,
        coord_ref_sys_name: "Cape / Lo17",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4222,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping, Mowbray, South Africa.",
        data_source: "EPSG",
        revision_date: "2007-02-12",
        deprecated: 0, 
    },

    "EPSG:22279" => CRS {
        coord_ref_sys_code: 22279,
        coord_ref_sys_name: "Cape / Lo19",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4222,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping, Mowbray, South Africa.",
        data_source: "EPSG",
        revision_date: "2007-02-12",
        deprecated: 0, 
    },

    "EPSG:22281" => CRS {
        coord_ref_sys_code: 22281,
        coord_ref_sys_name: "Cape / Lo21",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4222,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping, Mowbray, South Africa.",
        data_source: "EPSG",
        revision_date: "2007-02-12",
        deprecated: 0, 
    },

    "EPSG:22283" => CRS {
        coord_ref_sys_code: 22283,
        coord_ref_sys_name: "Cape / Lo23",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4222,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping, Mowbray, South Africa.",
        data_source: "EPSG",
        revision_date: "2007-02-12",
        deprecated: 0, 
    },

    "EPSG:22285" => CRS {
        coord_ref_sys_code: 22285,
        coord_ref_sys_name: "Cape / Lo25",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4222,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping, Mowbray, South Africa.",
        data_source: "EPSG",
        revision_date: "2007-02-12",
        deprecated: 0, 
    },

    "EPSG:22287" => CRS {
        coord_ref_sys_code: 22287,
        coord_ref_sys_name: "Cape / Lo27",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4222,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping, Mowbray, South Africa.",
        data_source: "EPSG",
        revision_date: "2007-02-12",
        deprecated: 0, 
    },

    "EPSG:22289" => CRS {
        coord_ref_sys_code: 22289,
        coord_ref_sys_name: "Cape / Lo29",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4222,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping, Mowbray, South Africa.",
        data_source: "EPSG",
        revision_date: "2007-02-12",
        deprecated: 0, 
    },

    "EPSG:22291" => CRS {
        coord_ref_sys_code: 22291,
        coord_ref_sys_name: "Cape / Lo31",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4222,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping, Mowbray, South Africa.",
        data_source: "EPSG",
        revision_date: "2007-02-12",
        deprecated: 0, 
    },

    "EPSG:22293" => CRS {
        coord_ref_sys_code: 22293,
        coord_ref_sys_name: "Cape / Lo33",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4222,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping, Mowbray, South Africa.",
        data_source: "EPSG",
        revision_date: "2007-02-12",
        deprecated: 0, 
    },

    "EPSG:22300" => CRS {
        coord_ref_sys_code: 22300,
        coord_ref_sys_name: "Carthage (Paris) / Tunisia Mining Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4406,
        datum_code: -1, 
        base_crs_code: 4816,
        remarks: "CAUTION: Carthage datum did not exist when the 1953 decree was issued and an inference is that grid should be applied to the Voirol 1875 geogCRS. Common practice assumes that the current Tunisian geodetic datum of Carthage applies.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2013-06-12",
        deprecated: 0, 
    },

    "EPSG:22332" => CRS {
        coord_ref_sys_code: 22332,
        coord_ref_sys_name: "Carthage / UTM zone 32N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4223,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:22392" => CRS {
        coord_ref_sys_code: 22392,
        coord_ref_sys_name: "Carthage / Sud Tunisie",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4223,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:62186405" => CRS {
        coord_ref_sys_code: 62186405,
        coord_ref_sys_name: "Bogota 1975 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4218,
        remarks: "See CRS code 4218 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62196405" => CRS {
        coord_ref_sys_code: 62196405,
        coord_ref_sys_name: "Bukit Rimpah (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4219,
        remarks: "See CRS code 4219 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62206405" => CRS {
        coord_ref_sys_code: 62206405,
        coord_ref_sys_name: "Camacupa (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4220,
        remarks: "See CRS code 4220 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62216405" => CRS {
        coord_ref_sys_code: 62216405,
        coord_ref_sys_name: "Campo Inchauspe (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4221,
        remarks: "See CRS code 4221 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62226405" => CRS {
        coord_ref_sys_code: 62226405,
        coord_ref_sys_name: "Cape (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4222,
        remarks: "See CRS code 4222 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62236405" => CRS {
        coord_ref_sys_code: 62236405,
        coord_ref_sys_name: "Carthage (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4223,
        remarks: "See CRS code 4223 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:68026405" => CRS {
        coord_ref_sys_code: 68026405,
        coord_ref_sys_name: "Bogota 1975 (Bogota) (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4802,
        remarks: "See CRS code 4802 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:2056" => CRS {
        coord_ref_sys_code: 2056,
        coord_ref_sys_name: "CH1903+ / LV95",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4150,
        remarks: "Replaces CH1903/LV03 (code 21781).",
        information_source: "Bundesamt für Landestopografie swisstopo; www.swisstopo.ch",
        data_source: "EPSG",
        revision_date: "2015-11-25",
        deprecated: 0, 
    },

    "EPSG:2081" => CRS {
        coord_ref_sys_code: 2081,
        coord_ref_sys_name: "Chos Malal 1914 / Argentina 2",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4160,
        remarks: "Replaced by Campo Inchauspe / Argentina 2 (CRS code 22192) for topographic mapping, use for oil exploration and production continues.",
        information_source: "Various oil industry records.",
        data_source: "EPSG",
        revision_date: "2020-01-16",
        deprecated: 0, 
    },

    "EPSG:2980" => CRS {
        coord_ref_sys_code: 2980,
        coord_ref_sys_name: "Combani 1950 / UTM zone 38S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4632,
        remarks: "Replaced by Cadastre 1997 / UTM zone 38S (CRS code 5879) for cadastral purposes only and by RGM04 / UTM zone 38S (CRS code 4471) for all other purposes.",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2012-11-26",
        deprecated: 0, 
    },

    "EPSG:4071" => CRS {
        coord_ref_sys_code: 4071,
        coord_ref_sys_name: "Chua / UTM zone 23S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4224,
        remarks: "",
        information_source: "Cartography System of Distrito Federal (SICAD)",
        data_source: "EPSG",
        revision_date: "2009-05-14",
        deprecated: 0, 
    },

    "EPSG:4149" => CRS {
        coord_ref_sys_code: 4149,
        coord_ref_sys_name: "CH1903",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6149, 
        base_crs_code: -1,
        remarks: "Replaced by CH1903+.",
        information_source: "Bundesamt für Landestopografie swisstopo; www.swisstopo.ch",
        data_source: "EPSG",
        revision_date: "2015-11-25",
        deprecated: 0, 
    },

    "EPSG:4150" => CRS {
        coord_ref_sys_code: 4150,
        coord_ref_sys_name: "CH1903+",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6150, 
        base_crs_code: -1,
        remarks: "Replaces CH1903.",
        information_source: "Bundesamt für Landestopografie swisstopo; www.swisstopo.ch",
        data_source: "EPSG",
        revision_date: "2015-11-25",
        deprecated: 0, 
    },

    "EPSG:4151" => CRS {
        coord_ref_sys_code: 4151,
        coord_ref_sys_name: "CHTRS95",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6151, 
        base_crs_code: 4933,
        remarks: "",
        information_source: "EPSG. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2021-07-16",
        deprecated: 0, 
    },

    "EPSG:4160" => CRS {
        coord_ref_sys_code: 4160,
        coord_ref_sys_name: "Chos Malal 1914",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6160, 
        base_crs_code: -1,
        remarks: "Replaced by Campo Inchauspe (geogCRS code 4221) for topographic mapping, use for oil exploration and production continues.",
        information_source: "Various oil company records.",
        data_source: "EPSG",
        revision_date: "2020-01-16",
        deprecated: 0, 
    },

    "EPSG:4224" => CRS {
        coord_ref_sys_code: 4224,
        coord_ref_sys_name: "Chua",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6224, 
        base_crs_code: -1,
        remarks: "The Chua origin and associated network is in Brazil with a connecting traverse through northern Paraguay. In Brazil used only as input into the Corrego Allegre adjustment (CRS code 4225), except for government work including SICAD in Distrito Federal.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2009-06-02",
        deprecated: 0, 
    },

    "EPSG:4315" => CRS {
        coord_ref_sys_code: 4315,
        coord_ref_sys_name: "Conakry 1905",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6315, 
        base_crs_code: -1,
        remarks: "Replaced by Dabola 1981 (EPSG code 4155).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4343" => CRS {
        coord_ref_sys_code: 4343,
        coord_ref_sys_name: "CHTRF95 (3D)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6401,
        datum_code: 6151, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:4344" => CRS {
        coord_ref_sys_code: 4344,
        coord_ref_sys_name: "CHTRF95 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6151, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:4479" => CRS {
        coord_ref_sys_code: 4479,
        coord_ref_sys_name: "China Geodetic Coordinate System 2000",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1043, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Chinese Academy of Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2020-08-31",
        deprecated: 0, 
    },

    "EPSG:4480" => CRS {
        coord_ref_sys_code: 4480,
        coord_ref_sys_name: "China Geodetic Coordinate System 2000",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1043, 
        base_crs_code: 4479,
        remarks: "",
        information_source: "Chinese Academy of Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2020-08-31",
        deprecated: 0, 
    },

    "EPSG:4490" => CRS {
        coord_ref_sys_code: 4490,
        coord_ref_sys_name: "China Geodetic Coordinate System 2000",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1043, 
        base_crs_code: 4480,
        remarks: "Adopted July 2008. Replaces Xian 1980 (CRS code 4610).",
        information_source: "EPSG. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2020-08-31",
        deprecated: 0, 
    },

    "EPSG:4491" => CRS {
        coord_ref_sys_code: 4491,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger zone 13",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / [6-degree] Gauss-Kruger CM 75E (code 4502).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-12",
        deprecated: 0, 
    },

    "EPSG:4492" => CRS {
        coord_ref_sys_code: 4492,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger zone 14",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / [6-degree] Gauss-Kruger CM 81E (code 4503).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-10-31",
        deprecated: 0, 
    },

    "EPSG:4493" => CRS {
        coord_ref_sys_code: 4493,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger zone 15",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / [6-degree] Gauss-Kruger CM 87E (code 4504).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-10-31",
        deprecated: 0, 
    },

    "EPSG:4494" => CRS {
        coord_ref_sys_code: 4494,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger zone 16",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / [6-degree] Gauss-Kruger CM 93E (code 4505).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4495" => CRS {
        coord_ref_sys_code: 4495,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger zone 17",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / [6-degree] Gauss-Kruger CM 99E (code 4506).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-10-31",
        deprecated: 0, 
    },

    "EPSG:4496" => CRS {
        coord_ref_sys_code: 4496,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger zone 18",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / [6-degree] Gauss-Kruger CM 105E (code 4507).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-01-05",
        deprecated: 0, 
    },

    "EPSG:4497" => CRS {
        coord_ref_sys_code: 4497,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger zone 19",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / [6-degree] Gauss-Kruger CM 111E (code 4508).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-01-05",
        deprecated: 0, 
    },

    "EPSG:4498" => CRS {
        coord_ref_sys_code: 4498,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger zone 20",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / [6-degree] Gauss-Kruger CM 117E (code 4509).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-01-05",
        deprecated: 0, 
    },

    "EPSG:4499" => CRS {
        coord_ref_sys_code: 4499,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger zone 21",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / [6-degree] Gauss-Kruger CM 123E (code 4510).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-01-05",
        deprecated: 0, 
    },

    "EPSG:4500" => CRS {
        coord_ref_sys_code: 4500,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger zone 22",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / [6-degree] Gauss-Kruger CM 129E (code 4511).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-01-05",
        deprecated: 0, 
    },

    "EPSG:4501" => CRS {
        coord_ref_sys_code: 4501,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger zone 23",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / [6-degree] Gauss-Kruger CM 135E (code 4512).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-10-31",
        deprecated: 0, 
    },

    "EPSG:4502" => CRS {
        coord_ref_sys_code: 4502,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger CM 75E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / [6-degree] Gauss-Kruger zone 13 (code 4491).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-10-31",
        deprecated: 0, 
    },

    "EPSG:4503" => CRS {
        coord_ref_sys_code: 4503,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger CM 81E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / [6-degree] Gauss-Kruger zone 14 (code 4492).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-10-31",
        deprecated: 0, 
    },

    "EPSG:4504" => CRS {
        coord_ref_sys_code: 4504,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger CM 87E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / [6-degree] Gauss-Kruger zone 15 (code 4493).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-10-31",
        deprecated: 0, 
    },

    "EPSG:4505" => CRS {
        coord_ref_sys_code: 4505,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger CM 93E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / [6-degree] Gauss-Kruger zone 16 (code 4494).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-10-31",
        deprecated: 0, 
    },

    "EPSG:4506" => CRS {
        coord_ref_sys_code: 4506,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger CM 99E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / [6-degree] Gauss-Kruger zone 17 (code 4495).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-10-31",
        deprecated: 0, 
    },

    "EPSG:4507" => CRS {
        coord_ref_sys_code: 4507,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger CM 105E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / [6-degree] Gauss-Kruger zone 18 (code 4496).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-01-05",
        deprecated: 0, 
    },

    "EPSG:4508" => CRS {
        coord_ref_sys_code: 4508,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger CM 111E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / [6-degree] Gauss-Kruger zone 19 (code 4497).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-01-05",
        deprecated: 0, 
    },

    "EPSG:4509" => CRS {
        coord_ref_sys_code: 4509,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger CM 117E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / [6-degree] Gauss-Kruger zone 20 (code 4498).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-01-05",
        deprecated: 0, 
    },

    "EPSG:4510" => CRS {
        coord_ref_sys_code: 4510,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger CM 123E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / [6-degree] Gauss-Kruger zone 21 (code 4499).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-01-05",
        deprecated: 0, 
    },

    "EPSG:4511" => CRS {
        coord_ref_sys_code: 4511,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger CM 129E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / [6-degree] Gauss-Kruger zone 22 (code 4500).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-01-05",
        deprecated: 0, 
    },

    "EPSG:4512" => CRS {
        coord_ref_sys_code: 4512,
        coord_ref_sys_name: "CGCS2000 / Gauss-Kruger CM 135E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / [6-degree] Gauss-Kruger zone 23 (code 4501).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-10-31",
        deprecated: 0, 
    },

    "EPSG:4513" => CRS {
        coord_ref_sys_code: 4513,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 25",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 75E (code 4534).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4514" => CRS {
        coord_ref_sys_code: 4514,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 26",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 78E (code 4535).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4515" => CRS {
        coord_ref_sys_code: 4515,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 27",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 81E (code 4536).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4516" => CRS {
        coord_ref_sys_code: 4516,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 28",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 84E (code 4537).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4517" => CRS {
        coord_ref_sys_code: 4517,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 29",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 87E (code 4538).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4518" => CRS {
        coord_ref_sys_code: 4518,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 30",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 90E (code 4539).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4519" => CRS {
        coord_ref_sys_code: 4519,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 31",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 93E (code 4540).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4520" => CRS {
        coord_ref_sys_code: 4520,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 32",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 96E (code 4541).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4521" => CRS {
        coord_ref_sys_code: 4521,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 33",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 99E (code 4542).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4522" => CRS {
        coord_ref_sys_code: 4522,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 34",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 102E (code 4543).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4523" => CRS {
        coord_ref_sys_code: 4523,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 35",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 105E (code 4544).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4524" => CRS {
        coord_ref_sys_code: 4524,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 36",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 108E (code 4545).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4525" => CRS {
        coord_ref_sys_code: 4525,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 37",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 111E (code 4546).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4526" => CRS {
        coord_ref_sys_code: 4526,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 38",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 114E (code 4547).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4527" => CRS {
        coord_ref_sys_code: 4527,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 39",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 117E (code 4548).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4528" => CRS {
        coord_ref_sys_code: 4528,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 40",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 120E (code 4549).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4529" => CRS {
        coord_ref_sys_code: 4529,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 41",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 123E (code 4550).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-14",
        deprecated: 0, 
    },

    "EPSG:4530" => CRS {
        coord_ref_sys_code: 4530,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 42",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 126E (code 4551).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4531" => CRS {
        coord_ref_sys_code: 4531,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 43",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 129E (code 4552).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4532" => CRS {
        coord_ref_sys_code: 4532,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 44",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 132E (code 4553).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4533" => CRS {
        coord_ref_sys_code: 4533,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger zone 45",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Also found with truncated false easting - see CGCS2000 / 3-degree Gauss-Kruger CM 135E (code 4554).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-13",
        deprecated: 0, 
    },

    "EPSG:4537" => CRS {
        coord_ref_sys_code: 4537,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 84E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 28 (code 4516).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4538" => CRS {
        coord_ref_sys_code: 4538,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 87E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 29 (code 4517).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4539" => CRS {
        coord_ref_sys_code: 4539,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 90E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 30 (code 4518).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4540" => CRS {
        coord_ref_sys_code: 4540,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 93E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 31 (code 4519).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4541" => CRS {
        coord_ref_sys_code: 4541,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 96E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 32 (code 4520).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4542" => CRS {
        coord_ref_sys_code: 4542,
        coord_ref_sys_name: "CGCS2000 / 3-degree Gauss-Kruger CM 99E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4490,
        remarks: "Truncated form of CGCS2000 / 3-degree Gauss-Kruger zone 33 (code 4521).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-11-04",
        deprecated: 0, 
    },

    "EPSG:4632" => CRS {
        coord_ref_sys_code: 4632,
        coord_ref_sys_name: "Combani 1950",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6632, 
        base_crs_code: -1,
        remarks: "Replaced by Cadastre 1997 (CRS code 4475) for cadastral purposes only and by RGM04 (CRS code 4470) for all other purposes.",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2009-10-29",
        deprecated: 0, 
    },

    "EPSG:4672" => CRS {
        coord_ref_sys_code: 4672,
        coord_ref_sys_name: "Chatham Islands 1971",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6672, 
        base_crs_code: -1,
        remarks: "Replaced by CI1979.",
        information_source: "Office of Surveyor General (OSG) Technical Report 14, June 2001.",
        data_source: "EPSG",
        revision_date: "2011-06-20",
        deprecated: 0, 
    },

    "EPSG:4673" => CRS {
        coord_ref_sys_code: 4673,
        coord_ref_sys_name: "Chatham Islands 1979",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6673, 
        base_crs_code: -1,
        remarks: "Replaces CI1971. Replaced by NZGD2000 from March 2000.",
        information_source: "Office of Surveyor General (OSG) Technical Report 14, June 2001.",
        data_source: "EPSG",
        revision_date: "2008-04-04",
        deprecated: 0, 
    },

    "EPSG:4708" => CRS {
        coord_ref_sys_code: 4708,
        coord_ref_sys_name: "Cocos Islands 1965",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6708, 
        base_crs_code: -1,
        remarks: "",
        information_source: "DMA / NIMA / NGA TR8350.2 (3rd edition, January 2000).",
        data_source: "EPSG",
        revision_date: "2006-01-26",
        deprecated: 0, 
    },

    "EPSG:4932" => CRS {
        coord_ref_sys_code: 4932,
        coord_ref_sys_name: "CHTRS95",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6151, 
        base_crs_code: -1,
        remarks: "Referenced to ETRS89 at epoch 1993.0. For CRS used for topographic and cadastral purposes see CH1903+ (CRS code 4150).",
        information_source: "Bundesamt für Landestopografie swisstopo; www.swisstopo.ch",
        data_source: "EPSG",
        revision_date: "2021-07-16",
        deprecated: 0, 
    },

    "EPSG:4933" => CRS {
        coord_ref_sys_code: 4933,
        coord_ref_sys_name: "CHTRS95",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6151, 
        base_crs_code: 4932,
        remarks: "Referenced to ETRS89 at epoch 1993.0. For CRS used for topographic and cadastral purposes see CH1903+ (CRS code 4150).",
        information_source: "Bundesamt für Landestopografie swisstopo; www.swisstopo.ch",
        data_source: "EPSG",
        revision_date: "2021-07-16",
        deprecated: 0, 
    },

    "EPSG:5518" => CRS {
        coord_ref_sys_code: 5518,
        coord_ref_sys_name: "CI1971 / Chatham Islands Map Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4672,
        remarks: "Replaced by CI1979 / CIMG (CRS code 5519).",
        information_source: "Land Information New Zealand.",
        data_source: "EPSG",
        revision_date: "2011-05-12",
        deprecated: 0, 
    },

    "EPSG:5519" => CRS {
        coord_ref_sys_code: 5519,
        coord_ref_sys_name: "CI1979 / Chatham Islands Map Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4673,
        remarks: "Replaces CI1971 / CIMG (CRS code 5518). Replaced by NZGD2000 / Chatham Islands TM 2000 (CRS code 3793).",
        information_source: "Land Information New Zealand.",
        data_source: "EPSG",
        revision_date: "2011-05-31",
        deprecated: 0, 
    },

    "EPSG:5713" => CRS {
        coord_ref_sys_code: 5713,
        coord_ref_sys_name: "CGVD28 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5114, 
        base_crs_code: -1,
        remarks: "From November 2013 replaced by CGVD2013 height (CRS code 6647).",
        information_source: "Geodetic Survey Division of Natural Resources Canada.",
        data_source: "EPSG",
        revision_date: "2019-09-17",
        deprecated: 0, 
    },

    "EPSG:5771" => CRS {
        coord_ref_sys_code: 5771,
        coord_ref_sys_name: "Chatham Island 1959 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5169, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Interim Report GEO/T1/19/51 for LINZ by John Hannah, &quot;An Assessment of New Zealand&#x27;s Height Systems and Options for a Future Height System&quot;, January 2001.",
        data_source: "EPSG",
        revision_date: "2011-06-20",
        deprecated: 0, 
    },

    "EPSG:5781" => CRS {
        coord_ref_sys_code: 5781,
        coord_ref_sys_name: "Constanta height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5179, 
        base_crs_code: -1,
        remarks: "",
        information_source: "EuroGeographics, http://crs.bkg.bund.de/evrs/Relations.html",
        data_source: "EPSG",
        revision_date: "2010-07-13",
        deprecated: 0, 
    },

    "EPSG:6133" => CRS {
        coord_ref_sys_code: 6133,
        coord_ref_sys_name: "CIGD11",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1100, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Lands and Surveys Department, Cayman Islands Government.",
        data_source: "EPSG",
        revision_date: "2012-12-12",
        deprecated: 0, 
    },

    "EPSG:6134" => CRS {
        coord_ref_sys_code: 6134,
        coord_ref_sys_name: "CIGD11",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1100, 
        base_crs_code: 6133,
        remarks: "",
        information_source: "Lands and Surveys Department, Cayman Islands Government.",
        data_source: "EPSG",
        revision_date: "2012-12-12",
        deprecated: 0, 
    },

    "EPSG:6135" => CRS {
        coord_ref_sys_code: 6135,
        coord_ref_sys_name: "CIGD11",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1100, 
        base_crs_code: 6134,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-12",
        deprecated: 0, 
    },

    "EPSG:6309" => CRS {
        coord_ref_sys_code: 6309,
        coord_ref_sys_name: "CGRS93",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1112, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Cyprus Department of Lands and Surveys (DLS).",
        data_source: "EPSG",
        revision_date: "2015-10-14",
        deprecated: 0, 
    },

    "EPSG:6310" => CRS {
        coord_ref_sys_code: 6310,
        coord_ref_sys_name: "CGRS93",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1112, 
        base_crs_code: 6309,
        remarks: "",
        information_source: "Cyprus Department of Lands and Surveys (DLS).",
        data_source: "EPSG",
        revision_date: "2015-09-28",
        deprecated: 0, 
    },

    "EPSG:6311" => CRS {
        coord_ref_sys_code: 6311,
        coord_ref_sys_name: "CGRS93",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1112, 
        base_crs_code: 6310,
        remarks: "Adopted by DLS in 1993 for new survey plans and maps.",
        information_source: "IOGP. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2016-02-15",
        deprecated: 0, 
    },

    "EPSG:6312" => CRS {
        coord_ref_sys_code: 6312,
        coord_ref_sys_name: "CGRS93 / Cyprus Local Transverse Mercator",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 6311,
        remarks: "",
        information_source: "Cyprus Department of Lands and Surveys (DLS).",
        data_source: "EPSG",
        revision_date: "2015-10-17",
        deprecated: 0, 
    },

    "EPSG:6647" => CRS {
        coord_ref_sys_code: 6647,
        coord_ref_sys_name: "CGVD2013(CGG2013) height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1127, 
        base_crs_code: -1,
        remarks: "Replaces CGVD28 height (CRS code 5713). CGVD2013(CGG2013) height is realized by geoid model CGG2013 (CT code 9246). Replaced by CGVD2013(CGG2013a) height (CRS code 9245).",
        information_source: "Geodetic Survey Division, Natural Resources Canada.",
        data_source: "EPSG",
        revision_date: "2019-11-11",
        deprecated: 0, 
    },

    "EPSG:6715" => CRS {
        coord_ref_sys_code: 6715,
        coord_ref_sys_name: "Christmas Island Grid 1985",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 4400,
        datum_code: 1134, 
        base_crs_code: -1,
        remarks: "Replaced by GDA94 / CIG94 (CRS code 6721).",
        information_source: "Landgate (Government of Western Australia), Christmas Island Geographic Information System documentation, www.ga.gov.au/christmas.",
        data_source: "EPSG",
        revision_date: "2014-02-01",
        deprecated: 0, 
    },

    "EPSG:9245" => CRS {
        coord_ref_sys_code: 9245,
        coord_ref_sys_name: "CGVD2013(CGG2013a) height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1256, 
        base_crs_code: -1,
        remarks: "Replaces CGVD2013(CGG2013) height (CRS code 6647). CGG2013a is identical to CGG2013 except in the western sector of Lake of the Woods. CGVD2013(CGG2013a) height is realized by Canadian gravimetric geoid model CGG2013a (CT code 9247).",
        information_source: "Geodetic Survey Division, Natural Resources Canada.",
        data_source: "EPSG",
        revision_date: "2021-04-20",
        deprecated: 0, 
    },

    "EPSG:9502" => CRS {
        coord_ref_sys_code: 9502,
        coord_ref_sys_name: "CIGD11 + CBVD61 height (ft)",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:9503" => CRS {
        coord_ref_sys_code: 9503,
        coord_ref_sys_name: "CIGD11 + GCVD54 height (ft)",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:9504" => CRS {
        coord_ref_sys_code: 9504,
        coord_ref_sys_name: "CIGD11 + LCVD61 height (ft)",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:21781" => CRS {
        coord_ref_sys_code: 21781,
        coord_ref_sys_name: "CH1903 / LV03",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4498,
        datum_code: -1, 
        base_crs_code: 4149,
        remarks: "Replaces LV03C (CRS code 21780) in Switzerland, but used only occasionally in Liechtenstein. Replaced by CH1903+ / LV95 (CRS code 2056).",
        information_source: "Bundesamt für Landestopografie swisstopo; www.swisstopo.ch",
        data_source: "EPSG",
        revision_date: "2015-11-25",
        deprecated: 0, 
    },

    "EPSG:21782" => CRS {
        coord_ref_sys_code: 21782,
        coord_ref_sys_name: "CH1903 / LV03C-G",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4498,
        datum_code: -1, 
        base_crs_code: 4149,
        remarks: "Greenwich-based equivalent of LV03C (CRS code 21780).",
        information_source: "Bundesamt für Landestopografie swisstopo; www.swisstopo.ch",
        data_source: "EPSG",
        revision_date: "2015-11-25",
        deprecated: 0, 
    },

    "EPSG:31528" => CRS {
        coord_ref_sys_code: 31528,
        coord_ref_sys_name: "Conakry 1905 / UTM zone 28N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4315,
        remarks: "Replaced by Dabola 1981 / UTM zone 28 (EPSG code 3461).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2007-02-06",
        deprecated: 0, 
    },

    "EPSG:31529" => CRS {
        coord_ref_sys_code: 31529,
        coord_ref_sys_name: "Conakry 1905 / UTM zone 29N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4315,
        remarks: "Replaced by Dabola 1981 / UTM zone 29 (EPSG code 3462).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2007-02-06",
        deprecated: 0, 
    },

    "EPSG:61496405" => CRS {
        coord_ref_sys_code: 61496405,
        coord_ref_sys_name: "CH1903 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4149,
        remarks: "See CRS code 4149 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61506405" => CRS {
        coord_ref_sys_code: 61506405,
        coord_ref_sys_name: "CH1903+ (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4150,
        remarks: "See CRS code 4150 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61516405" => CRS {
        coord_ref_sys_code: 61516405,
        coord_ref_sys_name: "CHTRF95 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4151,
        remarks: "See CRS code 4151 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61516413" => CRS {
        coord_ref_sys_code: 61516413,
        coord_ref_sys_name: "CHTRF95 (3D deg)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6413,
        datum_code: -1, 
        base_crs_code: 4343,
        remarks: "See CRS code 4343 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:61606405" => CRS {
        coord_ref_sys_code: 61606405,
        coord_ref_sys_name: "Chos Malal 1914 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4160,
        remarks: "See CRS code 4160 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62246405" => CRS {
        coord_ref_sys_code: 62246405,
        coord_ref_sys_name: "Chua (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4224,
        remarks: "See CRS code 4224 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:63156405" => CRS {
        coord_ref_sys_code: 63156405,
        coord_ref_sys_name: "Conakry 1905 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4315,
        remarks: "See CRS code 4315 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:66326405" => CRS {
        coord_ref_sys_code: 66326405,
        coord_ref_sys_name: "Combani 1950 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4632,
        remarks: "See CRS code 4632 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:2063" => CRS {
        coord_ref_sys_code: 2063,
        coord_ref_sys_name: "Dabola 1981 / UTM zone 28N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4315,
        remarks: "Supersedes Conakry 1905 / UTM zone 28 (EPSG code 31528).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1999-12-09",
        deprecated: 1, 
    },

    "EPSG:2064" => CRS {
        coord_ref_sys_code: 2064,
        coord_ref_sys_name: "Dabola 1981 / UTM zone 29N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4315,
        remarks: "Supersedes Conakry 1905 / UTM zone 29 (EPSG code 31529).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1999-12-09",
        deprecated: 1, 
    },

    "EPSG:2971" => CRS {
        coord_ref_sys_code: 2971,
        coord_ref_sys_name: "CSG67 / UTM zone 22N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4623,
        remarks: "Replaced by RGFG95 / UTM zone 22N (CRS code 2972).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2010-11-02",
        deprecated: 0, 
    },

    "EPSG:3312" => CRS {
        coord_ref_sys_code: 3312,
        coord_ref_sys_name: "CSG67 / UTM zone 21N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4623,
        remarks: "Replaced by RGFG95 / UTM zone 21N (CRS code 3313).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2010-11-02",
        deprecated: 0, 
    },

    "EPSG:3461" => CRS {
        coord_ref_sys_code: 3461,
        coord_ref_sys_name: "Dabola 1981 / UTM zone 28N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4155,
        remarks: "Replaces Conakry 1905 / UTM zone 28 (EPSG code 31528).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2007-02-06",
        deprecated: 0, 
    },

    "EPSG:3462" => CRS {
        coord_ref_sys_code: 3462,
        coord_ref_sys_name: "Dabola 1981 / UTM zone 29N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4155,
        remarks: "Replaces Conakry 1905 / UTM zone 29 (EPSG code 31529).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2007-02-06",
        deprecated: 0, 
    },

    "EPSG:3854" => CRS {
        coord_ref_sys_code: 3854,
        coord_ref_sys_name: "County ST74",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4531,
        datum_code: -1, 
        base_crs_code: 4619,
        remarks: "Simulation of engineering (regional) CRS through a Sweref-related projected CRS. Accuracy better than 0.05m when used together with a Gtrans reprojection model otherwise not better than 0.1m. In Stockholm commune, replaces ST74 (CRS code 3152).",
        information_source: "Surveying Department, Stockholm City Planning Administration.",
        data_source: "EPSG",
        revision_date: "2009-01-30",
        deprecated: 0, 
    },

    "EPSG:4155" => CRS {
        coord_ref_sys_code: 4155,
        coord_ref_sys_name: "Dabola 1981",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6155, 
        base_crs_code: -1,
        remarks: "Replaces Conakry 1905 (EPSG code 4315).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4225" => CRS {
        coord_ref_sys_code: 4225,
        coord_ref_sys_name: "Corrego Alegre 1970-72",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6225, 
        base_crs_code: -1,
        remarks: "Replaces 1961 adjustment (CRS code 5524). Replaced by SAD69 (CRS code 4291).",
        information_source: "IBGE.",
        data_source: "EPSG",
        revision_date: "2011-07-27",
        deprecated: 0, 
    },

    "EPSG:4226" => CRS {
        coord_ref_sys_code: 4226,
        coord_ref_sys_name: "Cote d&#x27;Ivoire",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6402,
        datum_code: 6226, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:4227" => CRS {
        coord_ref_sys_code: 4227,
        coord_ref_sys_name: "Deir ez Zor",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6227, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4274" => CRS {
        coord_ref_sys_code: 4274,
        coord_ref_sys_name: "Datum 73",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6274, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Geodetic Services Directorate, Instituto Geografico e Cadastral, Lisbon; http://www.igeo.pt/",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4316" => CRS {
        coord_ref_sys_code: 4316,
        coord_ref_sys_name: "Dealul Piscului 1930",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6316, 
        base_crs_code: -1,
        remarks: "Replaced by Pulkovo 1942(58) (geogCRS code 4179).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2008-09-24",
        deprecated: 0, 
    },

    "EPSG:4317" => CRS {
        coord_ref_sys_code: 4317,
        coord_ref_sys_name: "Dealul Piscului 1970",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6317, 
        base_crs_code: -1,
        remarks: "Replaces 1933 system (geogCRS code 4316).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-04-07",
        deprecated: 1, 
    },

    "EPSG:4623" => CRS {
        coord_ref_sys_code: 4623,
        coord_ref_sys_name: "CSG67",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6623, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2005-04-24",
        deprecated: 0, 
    },

    "EPSG:4736" => CRS {
        coord_ref_sys_code: 4736,
        coord_ref_sys_name: "Deception Island",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6736, 
        base_crs_code: -1,
        remarks: "",
        information_source: "DMA / NIMA / NGA TR8350.2 (3rd edition, January 2000).",
        data_source: "EPSG",
        revision_date: "2006-01-26",
        deprecated: 0, 
    },

    "EPSG:4755" => CRS {
        coord_ref_sys_code: 4755,
        coord_ref_sys_name: "DGN95",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6755, 
        base_crs_code: 4898,
        remarks: "Replaces ID74.",
        information_source: "OGP. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2007-08-27",
        deprecated: 0, 
    },

    "EPSG:4897" => CRS {
        coord_ref_sys_code: 4897,
        coord_ref_sys_name: "DGN95",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6755, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:4898" => CRS {
        coord_ref_sys_code: 4898,
        coord_ref_sys_name: "DGN95",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6755, 
        base_crs_code: 4897,
        remarks: "",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2007-08-27",
        deprecated: 0, 
    },

    "EPSG:5363" => CRS {
        coord_ref_sys_code: 5363,
        coord_ref_sys_name: "CR05",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1065, 
        base_crs_code: -1,
        remarks: "Replaced by CR-SIRGAS (CRS code 8905) from April 2018.",
        information_source: "Instituto Geografico Nacional Costa Rica report: El Sistema de Referencia CR05 y la Proyeccion Transversal de Mercator para Costa Rica CRTM05. (Report available through www.sirgas.org website in national networks list).",
        data_source: "EPSG",
        revision_date: "2019-03-22",
        deprecated: 0, 
    },

    "EPSG:5364" => CRS {
        coord_ref_sys_code: 5364,
        coord_ref_sys_name: "CR05",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1065, 
        base_crs_code: 5363,
        remarks: "Replaced by CR-SIRGAS (CRS code 8906) from April 2018.",
        information_source: "Instituto Geografico Nacional Costa Rica report: El Sistema de Referencia CR05 y la Proyeccion Transversal de Mercator para Costa Rica CRTM05. (Report available from www.sirgas.org website in national networks list).",
        data_source: "EPSG",
        revision_date: "2019-03-22",
        deprecated: 0, 
    },

    "EPSG:5365" => CRS {
        coord_ref_sys_code: 5365,
        coord_ref_sys_name: "CR05",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1065, 
        base_crs_code: 5364,
        remarks: "Replaces Ocotepeque (CRS code 5451) in Costa Rica from March 2007. Replaced by CR-SIRGAS (CRS code 8907) from April 2018.",
        information_source: "OGP. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2019-03-22",
        deprecated: 0, 
    },

    "EPSG:5367" => CRS {
        coord_ref_sys_code: 5367,
        coord_ref_sys_name: "CR05 / CRTM05",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 5365,
        remarks: "Replaces Ocotepeque / Costa Rica Norte (CRS code 5456) and Ocotepeque / Costa Rica Sur (CRS code 5457) from March 2007. Replaced by CR-SIRGAS / CRTM05 (CRS code 8908) from April 2018.",
        information_source: "Instituto Geografico Nacional Costa Rica report: El Sistema de Referencia CR05 y la Proyeccion Transversal de Mercator para Costa Rica CRTM05. (Report available from www.sirgas.org website).",
        data_source: "EPSG",
        revision_date: "2019-03-22",
        deprecated: 0, 
    },

    "EPSG:5524" => CRS {
        coord_ref_sys_code: 5524,
        coord_ref_sys_name: "Corrego Alegre 1961",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1074, 
        base_crs_code: -1,
        remarks: "Replaced by Corrego Alegre 1970-72 (CRS code 4225).",
        information_source: "IBGE.",
        data_source: "EPSG",
        revision_date: "2011-07-19",
        deprecated: 0, 
    },

    "EPSG:5536" => CRS {
        coord_ref_sys_code: 5536,
        coord_ref_sys_name: "Corrego Alegre 1961 / UTM zone 21S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5524,
        remarks: "Replaced by Corrego Alegre 1970-72 / UTM zone 21S (CRS code 22521).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5537" => CRS {
        coord_ref_sys_code: 5537,
        coord_ref_sys_name: "Corrego Alegre 1961 / UTM zone 22S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5524,
        remarks: "Replaced by Corrego Alegre 1970-72 / UTM zone 22S (CRS code 22522).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5538" => CRS {
        coord_ref_sys_code: 5538,
        coord_ref_sys_name: "Corrego Alegre 1961 / UTM zone 23S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5524,
        remarks: "Replaced by Corrego Alegre 1970-72 / UTM zone 23S (CRS code 22523).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2011-07-14",
        deprecated: 0, 
    },

    "EPSG:5539" => CRS {
        coord_ref_sys_code: 5539,
        coord_ref_sys_name: "Corrego Alegre 1961 / UTM zone 24S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5524,
        remarks: "Replaced by Corrego Alegre 1970-72 / UTM zone 24S (CRS code 22524).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2011-07-14",
        deprecated: 0, 
    },

    "EPSG:5681" => CRS {
        coord_ref_sys_code: 5681,
        coord_ref_sys_name: "DB_REF",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1081, 
        base_crs_code: 5830,
        remarks: "Differs from DHDN by 0.5-1m in former West Germany and by a maximum of 3m in former East Germany.",
        information_source: "Deutsche Bahn AG via Geo++, www.geopp.de",
        data_source: "EPSG",
        revision_date: "2012-03-25",
        deprecated: 0, 
    },

    "EPSG:5682" => CRS {
        coord_ref_sys_code: 5682,
        coord_ref_sys_name: "DB_REF / 3-degree Gauss-Kruger zone 2 (E-N)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5681,
        remarks: "",
        information_source: "Deutsche Bahn AG via Geo++, www.geopp.de",
        data_source: "EPSG",
        revision_date: "2012-03-25",
        deprecated: 0, 
    },

    "EPSG:5683" => CRS {
        coord_ref_sys_code: 5683,
        coord_ref_sys_name: "DB_REF / 3-degree Gauss-Kruger zone 3 (E-N)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5681,
        remarks: "",
        information_source: "Deutsche Bahn AG via Geo++, www.geopp.de",
        data_source: "EPSG",
        revision_date: "2012-03-25",
        deprecated: 0, 
    },

    "EPSG:5684" => CRS {
        coord_ref_sys_code: 5684,
        coord_ref_sys_name: "DB_REF / 3-degree Gauss-Kruger zone 4 (E-N)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5681,
        remarks: "",
        information_source: "Deutsche Bahn AG via Geo++, www.geopp.de",
        data_source: "EPSG",
        revision_date: "2012-03-25",
        deprecated: 0, 
    },

    "EPSG:5685" => CRS {
        coord_ref_sys_code: 5685,
        coord_ref_sys_name: "DB_REF / 3-degree Gauss-Kruger zone 5 (E-N)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5681,
        remarks: "",
        information_source: "Deutsche Bahn AG via Geo++, www.geopp.de",
        data_source: "EPSG",
        revision_date: "2012-03-25",
        deprecated: 0, 
    },

    "EPSG:5792" => CRS {
        coord_ref_sys_code: 5792,
        coord_ref_sys_name: "Danger 1950 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5190, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IGN Paris.",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5813" => CRS {
        coord_ref_sys_code: 5813,
        coord_ref_sys_name: "Dabajuro",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 4500,
        datum_code: 9307, 
        base_crs_code: -1,
        remarks: "Grid coordinates 200000N; 200000E at the centre of the tower of the church at Dabajuro. The tower is ascribed coordinates of 11°01&#x27;19&quot;N, 70°40&#x27;40&quot;W.",
        information_source: "Ministry of Mines standards manual of 1974.",
        data_source: "EPSG",
        revision_date: "2008-06-23",
        deprecated: 0, 
    },

    "EPSG:5828" => CRS {
        coord_ref_sys_code: 5828,
        coord_ref_sys_name: "DB_REF",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1081, 
        base_crs_code: -1,
        remarks: "Derived from ETRS89 through transformation code 5826.",
        information_source: "Deutsche Bahn AG via Geo++, www.geopp.de",
        data_source: "EPSG",
        revision_date: "2012-03-25",
        deprecated: 0, 
    },

    "EPSG:5830" => CRS {
        coord_ref_sys_code: 5830,
        coord_ref_sys_name: "DB_REF",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1081, 
        base_crs_code: 5828,
        remarks: "",
        information_source: "Deutsche Bahn AG via Geo++, www.geopp.de",
        data_source: "EPSG",
        revision_date: "2012-03-25",
        deprecated: 0, 
    },

    "EPSG:5832" => CRS {
        coord_ref_sys_code: 5832,
        coord_ref_sys_name: "DB_REF / 3-degree Gauss-Kruger zone 2 (E-N) + DHHN92 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Deutsche Bahn AG via Geo++, www.geopp.de",
        data_source: "EPSG",
        revision_date: "2012-03-25",
        deprecated: 0, 
    },

    "EPSG:5833" => CRS {
        coord_ref_sys_code: 5833,
        coord_ref_sys_name: "DB_REF / 3-degree Gauss-Kruger zone 3 (E-N) + DHHN92 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Deutsche Bahn AG via Geo++, www.geopp.de",
        data_source: "EPSG",
        revision_date: "2012-03-25",
        deprecated: 0, 
    },

    "EPSG:5834" => CRS {
        coord_ref_sys_code: 5834,
        coord_ref_sys_name: "DB_REF / 3-degree Gauss-Kruger zone 4 (E-N) + DHHN92 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Deutsche Bahn AG via Geo++, www.geopp.de",
        data_source: "EPSG",
        revision_date: "2012-03-25",
        deprecated: 0, 
    },

    "EPSG:5835" => CRS {
        coord_ref_sys_code: 5835,
        coord_ref_sys_name: "DB_REF / 3-degree Gauss-Kruger zone 5 (E-N) + DHHN92 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Deutsche Bahn AG via Geo++, www.geopp.de",
        data_source: "EPSG",
        revision_date: "2012-03-25",
        deprecated: 0, 
    },

    "EPSG:8905" => CRS {
        coord_ref_sys_code: 8905,
        coord_ref_sys_name: "CR-SIRGAS",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1225, 
        base_crs_code: -1,
        remarks: "Replaces CR05 (CRS code 5363) from April 2018.",
        information_source: "Instituto Geografico Nacional Costa Rica, executive decree number 40692-MJP of 2018-04-17.",
        data_source: "EPSG",
        revision_date: "2019-03-09",
        deprecated: 0, 
    },

    "EPSG:8906" => CRS {
        coord_ref_sys_code: 8906,
        coord_ref_sys_name: "CR-SIRGAS",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1225, 
        base_crs_code: 8905,
        remarks: "Replaces CR05 (CRS code 5364) from April 2018.",
        information_source: "Instituto Geografico Nacional Costa Rica, executive decree number 40692-MJP of 2018-04-17.",
        data_source: "EPSG",
        revision_date: "2019-03-09",
        deprecated: 0, 
    },

    "EPSG:23877" => CRS {
        coord_ref_sys_code: 23877,
        coord_ref_sys_name: "DGN95 / UTM zone 47S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Replaces ID74 / UTM zone 47S.",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:8907" => CRS {
        coord_ref_sys_code: 8907,
        coord_ref_sys_name: "CR-SIRGAS",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1225, 
        base_crs_code: 8906,
        remarks: "Replaces CR05 (CRS code 5365) from April 2018.",
        information_source: "IOGP. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2019-03-09",
        deprecated: 0, 
    },

    "EPSG:8908" => CRS {
        coord_ref_sys_code: 8908,
        coord_ref_sys_name: "CR-SIRGAS / CRTM05",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 8907,
        remarks: "Replaces CR05 / CRTM05 (CRS code 5367) from April 2018.",
        information_source: "Instituto Geografico Nacional Costa Rica, executive decree number 40692-MJP of 2018-04-17.",
        data_source: "EPSG",
        revision_date: "2019-03-09",
        deprecated: 0, 
    },

    "EPSG:8909" => CRS {
        coord_ref_sys_code: 8909,
        coord_ref_sys_name: "CR-SIRGAS / UTM zone 16N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 8907,
        remarks: "Replaces CR05 / CRTM05 (CRS code 5367) from April 2018.",
        information_source: "Instituto Geografico Nacional Costa Rica, executive decree number 40692-MJP of 2018-04-17.",
        data_source: "EPSG",
        revision_date: "2019-03-09",
        deprecated: 0, 
    },

    "EPSG:8910" => CRS {
        coord_ref_sys_code: 8910,
        coord_ref_sys_name: "CR-SIRGAS / UTM zone 17N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 8907,
        remarks: "Replaces CR05 / CRTM05 (CRS code 5367) from April 2018.",
        information_source: "Instituto Geografico Nacional Costa Rica, executive decree number 40692-MJP of 2018-04-17.",
        data_source: "EPSG",
        revision_date: "2019-03-09",
        deprecated: 0, 
    },

    "EPSG:8911" => CRS {
        coord_ref_sys_code: 8911,
        coord_ref_sys_name: "DACR52 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1226, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Instituto Geografico Nacional Costa Rica, executive decree number 40692-MJP of 2018-04-17.",
        data_source: "EPSG",
        revision_date: "2019-03-09",
        deprecated: 0, 
    },

    "EPSG:8912" => CRS {
        coord_ref_sys_code: 8912,
        coord_ref_sys_name: "CR-SIRGAS / CRTM05 + DACR52 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "With geoid model and gravity, part of official national dynamic geodetic framework from April 2018.",
        information_source: "Instituto Geografico Nacional Costa Rica, executive decree number 40692-MJP of 2018-04-17.",
        data_source: "EPSG",
        revision_date: "2019-03-09",
        deprecated: 0, 
    },

    "EPSG:22521" => CRS {
        coord_ref_sys_code: 22521,
        coord_ref_sys_name: "Corrego Alegre 1970-72 / UTM zone 21S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4225,
        remarks: "Replaces Corrego Alegre 1961 / UTM zone 21S (CRS code 5536). Replaced by SAD69 / UTM zone 21S (CRS code 29191).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2011-07-27",
        deprecated: 0, 
    },

    "EPSG:22522" => CRS {
        coord_ref_sys_code: 22522,
        coord_ref_sys_name: "Corrego Alegre 1970-72 / UTM zone 22S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4225,
        remarks: "Replaces Corrego Alegre 1961 / UTM zone 22S (CRS code 5537). Replaced by SAD69 / UTM zone 22S (CRS code 29192).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2011-07-27",
        deprecated: 0, 
    },

    "EPSG:22523" => CRS {
        coord_ref_sys_code: 22523,
        coord_ref_sys_name: "Corrego Alegre 1970-72 / UTM zone 23S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4225,
        remarks: "Replaces Corrego Alegre 1961 / UTM zone 23S (CRS code 5538). Replaced by SAD69 / UTM zone 23S (CRS code 29193).",
        information_source: "IBGE",
        data_source: "EPSG",
        revision_date: "2011-07-27",
        deprecated: 0, 
    },

    "EPSG:22524" => CRS {
        coord_ref_sys_code: 22524,
        coord_ref_sys_name: "Corrego Alegre 1970-72 / UTM zone 24S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4225,
        remarks: "Replaces Corrego Alegre 1961 / UTM zone 24S (CRS code 5539). Replaced by SAD69 / UTM zone 24S (CRS code 29194).",
        information_source: "IBGE",
        data_source: "EPSG",
        revision_date: "2011-07-27",
        deprecated: 0, 
    },

    "EPSG:22525" => CRS {
        coord_ref_sys_code: 22525,
        coord_ref_sys_name: "Corrego Alegre 1970-72 / UTM zone 25S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4225,
        remarks: "Replaced by SAD69 / UTM zone 25S (CRS code 29195).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2011-07-27",
        deprecated: 0, 
    },

    "EPSG:22700" => CRS {
        coord_ref_sys_code: 22700,
        coord_ref_sys_name: "Deir ez Zor / Levant Zone",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4227,
        remarks: "Replaced by Deir ez Zor / Syria Lambert (EPSG code 22770) from 1973.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1999-04-22",
        deprecated: 0, 
    },

    "EPSG:22770" => CRS {
        coord_ref_sys_code: 22770,
        coord_ref_sys_name: "Deir ez Zor / Syria Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4227,
        remarks: "Replaces Deir ez Zor / Levant zone (EPSG code 22700) from 1973.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1999-10-20",
        deprecated: 0, 
    },

    "EPSG:22780" => CRS {
        coord_ref_sys_code: 22780,
        coord_ref_sys_name: "Deir ez Zor / Levant Stereographic",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4227,
        remarks: "Used prior to World War II.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-10-19",
        deprecated: 0, 
    },

    "EPSG:23830" => CRS {
        coord_ref_sys_code: 23830,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 46.2",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23831" => CRS {
        coord_ref_sys_code: 23831,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 47.1",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2007-05-10",
        deprecated: 0, 
    },

    "EPSG:23832" => CRS {
        coord_ref_sys_code: 23832,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 47.2",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2007-05-10",
        deprecated: 0, 
    },

    "EPSG:23833" => CRS {
        coord_ref_sys_code: 23833,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 48.1",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2007-05-10",
        deprecated: 0, 
    },

    "EPSG:23834" => CRS {
        coord_ref_sys_code: 23834,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 48.2",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2007-05-10",
        deprecated: 0, 
    },

    "EPSG:23835" => CRS {
        coord_ref_sys_code: 23835,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 49.1",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2007-05-10",
        deprecated: 0, 
    },

    "EPSG:23836" => CRS {
        coord_ref_sys_code: 23836,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 49.2",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2007-05-10",
        deprecated: 0, 
    },

    "EPSG:23837" => CRS {
        coord_ref_sys_code: 23837,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 50.1",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2007-05-10",
        deprecated: 0, 
    },

    "EPSG:23838" => CRS {
        coord_ref_sys_code: 23838,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 50.2",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2007-05-10",
        deprecated: 0, 
    },

    "EPSG:23839" => CRS {
        coord_ref_sys_code: 23839,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 51.1",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2007-05-10",
        deprecated: 0, 
    },

    "EPSG:23840" => CRS {
        coord_ref_sys_code: 23840,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 51.2",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2007-05-10",
        deprecated: 0, 
    },

    "EPSG:23841" => CRS {
        coord_ref_sys_code: 23841,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 52.1",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2007-05-10",
        deprecated: 0, 
    },

    "EPSG:23842" => CRS {
        coord_ref_sys_code: 23842,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 52.2",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2007-05-10",
        deprecated: 0, 
    },

    "EPSG:23843" => CRS {
        coord_ref_sys_code: 23843,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 53.1",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2007-05-10",
        deprecated: 0, 
    },

    "EPSG:23844" => CRS {
        coord_ref_sys_code: 23844,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 53.2",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2007-05-10",
        deprecated: 0, 
    },

    "EPSG:23845" => CRS {
        coord_ref_sys_code: 23845,
        coord_ref_sys_name: "DGN95 / Indonesia TM-3 zone 54.1",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Used by BPN for large scale (1:10,000 and greater) land use mapping.",
        information_source: "National Land Agency (Badan Pertanahan Nasional, BPN)",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23866" => CRS {
        coord_ref_sys_code: 23866,
        coord_ref_sys_name: "DGN95 / UTM zone 46N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Replaces ID74 / UTM zone 46N.",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:23867" => CRS {
        coord_ref_sys_code: 23867,
        coord_ref_sys_name: "DGN95 / UTM zone 47N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Replaces ID74 / UTM zone 47N.",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:23868" => CRS {
        coord_ref_sys_code: 23868,
        coord_ref_sys_name: "DGN95 / UTM zone 48N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Replaces ID74 / UTM zone 48N.",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:23869" => CRS {
        coord_ref_sys_code: 23869,
        coord_ref_sys_name: "DGN95 / UTM zone 49N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Replaces ID74 / UTM zone 49N.",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:23870" => CRS {
        coord_ref_sys_code: 23870,
        coord_ref_sys_name: "DGN95 / UTM zone 50N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Replaces ID74 / UTM zone 50N.",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:23871" => CRS {
        coord_ref_sys_code: 23871,
        coord_ref_sys_name: "DGN95 / UTM zone 51N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Replaces ID74 / UTM zone 51N.",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:23872" => CRS {
        coord_ref_sys_code: 23872,
        coord_ref_sys_name: "DGN95 / UTM zone 52N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Replaces ID74 / UTM zone 52N.",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:23878" => CRS {
        coord_ref_sys_code: 23878,
        coord_ref_sys_name: "DGN95 / UTM zone 48S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Replaces ID74 / UTM zone 48S.",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:23879" => CRS {
        coord_ref_sys_code: 23879,
        coord_ref_sys_name: "DGN95 / UTM zone 49S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Replaces ID74 / UTM zone 49S.",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:23880" => CRS {
        coord_ref_sys_code: 23880,
        coord_ref_sys_name: "DGN95 / UTM zone 50S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Replaces ID74 / UTM zone 50S.",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:23881" => CRS {
        coord_ref_sys_code: 23881,
        coord_ref_sys_name: "DGN95 / UTM zone 51S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Replaces ID74 / UTM zone 51S.",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:23882" => CRS {
        coord_ref_sys_code: 23882,
        coord_ref_sys_name: "DGN95 / UTM zone 52S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Replaces ID74 / UTM zone 52S.",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:23883" => CRS {
        coord_ref_sys_code: 23883,
        coord_ref_sys_name: "DGN95 / UTM zone 53S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Replaces ID74 / UTM zone 53S.",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:23884" => CRS {
        coord_ref_sys_code: 23884,
        coord_ref_sys_name: "DGN95 / UTM zone 54S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4755,
        remarks: "Replaces ID74 / UTM zone 54S.",
        information_source: "Bakosurtanal.",
        data_source: "EPSG",
        revision_date: "2006-08-25",
        deprecated: 0, 
    },

    "EPSG:27429" => CRS {
        coord_ref_sys_code: 27429,
        coord_ref_sys_name: "Datum 73 / UTM zone 29N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4274,
        remarks: "",
        information_source: "Geodetic Services Directorate, Instituto Geografico e Cadastral, Lisbon; http://www.igeo.pt/",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:27492" => CRS {
        coord_ref_sys_code: 27492,
        coord_ref_sys_name: "Datum 73 / Modified Portuguese Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4274,
        remarks: "",
        information_source: "Instituto Geografico e Cadastral Lisbon via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 1, 
    },

    "EPSG:27493" => CRS {
        coord_ref_sys_code: 27493,
        coord_ref_sys_name: "Datum 73 / Modified Portuguese Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4274,
        remarks: "The projection parameters have been designed such that this CRS grid is almost the same as the Lisbon (Lisbon)/Portuguese Grid (CRS code 20791).",
        information_source: "Geodetic Services Directorate, Instituto Geografico e Cadastral, Lisbon; http://www.igeo.pt/",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:31600" => CRS {
        coord_ref_sys_code: 31600,
        coord_ref_sys_name: "Dealul Piscului 1930 / Stereo 33",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4316,
        remarks: "Replaced by Stereo 70 system (ProjCRS code 3844).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2008-09-24",
        deprecated: 0, 
    },

    "EPSG:31700" => CRS {
        coord_ref_sys_code: 31700,
        coord_ref_sys_name: "Dealul Piscului 1970/ Stereo 70",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4317,
        remarks: "Replaces Stereo 33 system.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1996-04-12",
        deprecated: 1, 
    },

    "EPSG:61556405" => CRS {
        coord_ref_sys_code: 61556405,
        coord_ref_sys_name: "Dabola 1981 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4155,
        remarks: "See CRS code 4155 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62256405" => CRS {
        coord_ref_sys_code: 62256405,
        coord_ref_sys_name: "Corrego Alegre (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4225,
        remarks: "See CRS code 4225 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62276405" => CRS {
        coord_ref_sys_code: 62276405,
        coord_ref_sys_name: "Deir ez Zor (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4227,
        remarks: "See CRS code 4227 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62746405" => CRS {
        coord_ref_sys_code: 62746405,
        coord_ref_sys_name: "Datum 73 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4274,
        remarks: "See CRS code 4274 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:63166405" => CRS {
        coord_ref_sys_code: 63166405,
        coord_ref_sys_name: "Dealul Piscului 1933 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4316,
        remarks: "See CRS code 4316 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:63176405" => CRS {
        coord_ref_sys_code: 63176405,
        coord_ref_sys_name: "Dealul Piscului 1970 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4317,
        remarks: "See CRS code 4317 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:66236405" => CRS {
        coord_ref_sys_code: 66236405,
        coord_ref_sys_name: "CSG67 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4623,
        remarks: "See CRS code 4623 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:2002" => CRS {
        coord_ref_sys_code: 2002,
        coord_ref_sys_name: "Dominica 1945 / British West Indies Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4602,
        remarks: "",
        information_source: "Ordnance Survey of Great Britain",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2192" => CRS {
        coord_ref_sys_code: 2192,
        coord_ref_sys_name: "ED50 / France EuroLambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "IGN Paris via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2001-06-05",
        deprecated: 1, 
    },

    "EPSG:2206" => CRS {
        coord_ref_sys_code: 2206,
        coord_ref_sys_name: "ED50 / 3-degree Gauss-Kruger zone 9",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Also found with truncated false easting - see ED50 / TM27 (code 2319).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2001-11-06",
        deprecated: 0, 
    },

    "EPSG:2207" => CRS {
        coord_ref_sys_code: 2207,
        coord_ref_sys_name: "ED50 / 3-degree Gauss-Kruger zone 10",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Also found with truncated false easting - see ED50 / TM30 (code 2320).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2001-11-06",
        deprecated: 0, 
    },

    "EPSG:2208" => CRS {
        coord_ref_sys_code: 2208,
        coord_ref_sys_name: "ED50 / 3-degree Gauss-Kruger zone 11",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Also found with truncated false easting - see ED50 / TM33 (code 2321).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2001-11-06",
        deprecated: 0, 
    },

    "EPSG:2209" => CRS {
        coord_ref_sys_code: 2209,
        coord_ref_sys_name: "ED50 / 3-degree Gauss-Kruger zone 12",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Also found with truncated false easting - see ED50 / TM36 (code 2322).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2001-11-06",
        deprecated: 0, 
    },

    "EPSG:2210" => CRS {
        coord_ref_sys_code: 2210,
        coord_ref_sys_name: "ED50 / 3-degree Gauss-Kruger zone 13",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Also found with truncated false easting - see ED50 / TM39 (code 2323).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2001-11-06",
        deprecated: 0, 
    },

    "EPSG:2211" => CRS {
        coord_ref_sys_code: 2211,
        coord_ref_sys_name: "ED50 / 3-degree Gauss-Kruger zone 14",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Also found with truncated false easting - see ED50 / TM42 (code 2324).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2001-11-06",
        deprecated: 0, 
    },

    "EPSG:2212" => CRS {
        coord_ref_sys_code: 2212,
        coord_ref_sys_name: "ED50 / 3-degree Gauss-Kruger zone 15",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Also found with truncated false easting - see ED50 / TM45 (code 2325).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2001-11-06",
        deprecated: 0, 
    },

    "EPSG:2214" => CRS {
        coord_ref_sys_code: 2214,
        coord_ref_sys_name: "Douala 1948 / AOF west",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4192,
        remarks: "Superseded by Manoca 1962 / UTM zone 32N (code 2215).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-01-18",
        deprecated: 1, 
    },

    "EPSG:2319" => CRS {
        coord_ref_sys_code: 2319,
        coord_ref_sys_name: "ED50 / TM27",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Also found with zone number prefix to false easting - see ED50 / 3-degree Gauss-Kruger zone 9 (code 2206).",
        information_source: "General Command of Mapping via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2006-07-22",
        deprecated: 0, 
    },

    "EPSG:2320" => CRS {
        coord_ref_sys_code: 2320,
        coord_ref_sys_name: "ED50 / TM30",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Also found with zone number prefix to false easting - see ED50 / 3-degree Gauss-Kruger zone 10 (code 2207).",
        information_source: "General Command of Mapping via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 0, 
    },

    "EPSG:2321" => CRS {
        coord_ref_sys_code: 2321,
        coord_ref_sys_name: "ED50 / TM33",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Also found with zone number prefix to false easting - see ED50 / 3-degree Gauss-Kruger zone 11 (code 2208).",
        information_source: "General Command of Mapping via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2006-07-22",
        deprecated: 0, 
    },

    "EPSG:2322" => CRS {
        coord_ref_sys_code: 2322,
        coord_ref_sys_name: "ED50 / TM36",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Also found with zone number prefix to false easting - see ED50 / 3-degree Gauss-Kruger zone 12 (code 2209).",
        information_source: "General Command of Mapping via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 0, 
    },

    "EPSG:2323" => CRS {
        coord_ref_sys_code: 2323,
        coord_ref_sys_name: "ED50 / TM39",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Also found with zone number prefix to false easting - see ED50 / 3-degree Gauss-Kruger zone 13 (code 2210).",
        information_source: "General Command of Mapping via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2006-07-22",
        deprecated: 0, 
    },

    "EPSG:2324" => CRS {
        coord_ref_sys_code: 2324,
        coord_ref_sys_name: "ED50 / TM42",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Also found with zone number prefix to false easting - see ED50 / 3-degree Gauss-Kruger zone 14 (code 2211).",
        information_source: "General Command of Mapping via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 0, 
    },

    "EPSG:2325" => CRS {
        coord_ref_sys_code: 2325,
        coord_ref_sys_name: "ED50 / TM45",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Also found with zone number prefix to false easting - see ED50 / 3-degree Gauss-Kruger zone 15 (code 2212).",
        information_source: "General Command of Mapping via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2006-07-22",
        deprecated: 0, 
    },

    "EPSG:3066" => CRS {
        coord_ref_sys_code: 3066,
        coord_ref_sys_name: "ED50 / Jordan TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Information has not been confirmed by National Mapping Agency.",
        information_source: "Various industry sources",
        data_source: "EPSG",
        revision_date: "2004-01-29",
        deprecated: 0, 
    },

    "EPSG:3068" => CRS {
        coord_ref_sys_code: 3068,
        coord_ref_sys_name: "DHDN / Soldner Berlin",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4531,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "",
        information_source: "Berlin state statistical office. Also at
http://www.kulturbuch-verlag.de/online/brv/D0026/F01293.pdf",
        data_source: "EPSG",
        revision_date: "2004-04-22",
        deprecated: 0, 
    },

    "EPSG:3119" => CRS {
        coord_ref_sys_code: 3119,
        coord_ref_sys_name: "Douala 1948 / AEF west",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4192,
        remarks: "Replaced by Manoca 1962 / UTM zone 32N (code 2215).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2005-06-17",
        deprecated: 0, 
    },

    "EPSG:3893" => CRS {
        coord_ref_sys_code: 3893,
        coord_ref_sys_name: "ED50 / Iraq National Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "Survey Division, Ministry of Water Resources (MoWR).",
        data_source: "EPSG",
        revision_date: "2009-03-16",
        deprecated: 0, 
    },

    "EPSG:4192" => CRS {
        coord_ref_sys_code: 4192,
        coord_ref_sys_name: "Douala 1948",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6192, 
        base_crs_code: -1,
        remarks: "Replaced by Manoca 1962 (code 4193).",
        information_source: "TotalFinaElf",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4228" => CRS {
        coord_ref_sys_code: 4228,
        coord_ref_sys_name: "Douala",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6402,
        datum_code: 6228, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:4230" => CRS {
        coord_ref_sys_code: 4230,
        coord_ref_sys_name: "ED50",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6230, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-10-23",
        deprecated: 0, 
    },

    "EPSG:4314" => CRS {
        coord_ref_sys_code: 4314,
        coord_ref_sys_name: "DHDN",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6314, 
        base_crs_code: -1,
        remarks: "See also RD/83 for Saxony and PD/83 for Thuringen. For national digital cartographic purposes used across all German states.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2006-06-12",
        deprecated: 0, 
    },

    "EPSG:4458" => CRS {
        coord_ref_sys_code: 4458,
        coord_ref_sys_name: "Dunedin-Bluff 1960 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1040, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Land Information New Zealand (LINZ) standard S25004.",
        data_source: "EPSG",
        revision_date: "2011-01-25",
        deprecated: 0, 
    },

    "EPSG:4602" => CRS {
        coord_ref_sys_code: 4602,
        coord_ref_sys_name: "Dominica 1945",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6602, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4719" => CRS {
        coord_ref_sys_code: 4719,
        coord_ref_sys_name: "Easter Island 1967",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6719, 
        base_crs_code: -1,
        remarks: "",
        information_source: "DMA / NIMA / NGA TR8350.2 (3rd edition, January 2000).",
        data_source: "EPSG",
        revision_date: "2006-01-26",
        deprecated: 0, 
    },

    "EPSG:4724" => CRS {
        coord_ref_sys_code: 4724,
        coord_ref_sys_name: "Diego Garcia 1969",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6724, 
        base_crs_code: -1,
        remarks: "",
        information_source: "DMA / NIMA / NGA TR8350.2 (3rd edition, January 2000).",
        data_source: "EPSG",
        revision_date: "2006-01-26",
        deprecated: 0, 
    },

    "EPSG:5262" => CRS {
        coord_ref_sys_code: 5262,
        coord_ref_sys_name: "DRUKREF 03",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1058, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-09-06",
        deprecated: 0, 
    },

    "EPSG:5263" => CRS {
        coord_ref_sys_code: 5263,
        coord_ref_sys_name: "DRUKREF 03",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1058, 
        base_crs_code: 5262,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-09-06",
        deprecated: 0, 
    },

    "EPSG:5264" => CRS {
        coord_ref_sys_code: 5264,
        coord_ref_sys_name: "DRUKREF 03",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1058, 
        base_crs_code: 5263,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2010-09-06",
        deprecated: 0, 
    },

    "EPSG:5266" => CRS {
        coord_ref_sys_code: 5266,
        coord_ref_sys_name: "DRUKREF 03 / Bhutan National Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-10-26",
        deprecated: 0, 
    },

    "EPSG:5292" => CRS {
        coord_ref_sys_code: 5292,
        coord_ref_sys_name: "DRUKREF 03 / Bumthang TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-09-24",
        deprecated: 0, 
    },

    "EPSG:5293" => CRS {
        coord_ref_sys_code: 5293,
        coord_ref_sys_name: "DRUKREF 03 / Chhukha TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-09-24",
        deprecated: 0, 
    },

    "EPSG:5294" => CRS {
        coord_ref_sys_code: 5294,
        coord_ref_sys_name: "DRUKREF 03 / Dagana TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-09-24",
        deprecated: 0, 
    },

    "EPSG:5295" => CRS {
        coord_ref_sys_code: 5295,
        coord_ref_sys_name: "DRUKREF 03 / Gasa TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-09-24",
        deprecated: 0, 
    },

    "EPSG:5296" => CRS {
        coord_ref_sys_code: 5296,
        coord_ref_sys_name: "DRUKREF 03 / Ha TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-09-24",
        deprecated: 0, 
    },

    "EPSG:5297" => CRS {
        coord_ref_sys_code: 5297,
        coord_ref_sys_name: "DRUKREF 03 / Lhuentse TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-09-24",
        deprecated: 0, 
    },

    "EPSG:5298" => CRS {
        coord_ref_sys_code: 5298,
        coord_ref_sys_name: "DRUKREF 03 / Mongar TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-09-24",
        deprecated: 0, 
    },

    "EPSG:5299" => CRS {
        coord_ref_sys_code: 5299,
        coord_ref_sys_name: "DRUKREF 03 / Paro TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-09-24",
        deprecated: 0, 
    },

    "EPSG:5300" => CRS {
        coord_ref_sys_code: 5300,
        coord_ref_sys_name: "DRUKREF 03 / Pemagatshel TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-10-26",
        deprecated: 0, 
    },

    "EPSG:5301" => CRS {
        coord_ref_sys_code: 5301,
        coord_ref_sys_name: "DRUKREF 03 / Punakha TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-10-26",
        deprecated: 0, 
    },

    "EPSG:5302" => CRS {
        coord_ref_sys_code: 5302,
        coord_ref_sys_name: "DRUKREF 03 / Samdrup Jongkhar TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-10-26",
        deprecated: 0, 
    },

    "EPSG:5303" => CRS {
        coord_ref_sys_code: 5303,
        coord_ref_sys_name: "DRUKREF 03 / Samtse TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-09-24",
        deprecated: 0, 
    },

    "EPSG:5304" => CRS {
        coord_ref_sys_code: 5304,
        coord_ref_sys_name: "DRUKREF 03 / Sarpang TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-09-24",
        deprecated: 0, 
    },

    "EPSG:5305" => CRS {
        coord_ref_sys_code: 5305,
        coord_ref_sys_name: "DRUKREF 03 / Thimphu TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-10-26",
        deprecated: 0, 
    },

    "EPSG:5306" => CRS {
        coord_ref_sys_code: 5306,
        coord_ref_sys_name: "DRUKREF 03 / Trashigang TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-10-26",
        deprecated: 0, 
    },

    "EPSG:5307" => CRS {
        coord_ref_sys_code: 5307,
        coord_ref_sys_name: "DRUKREF 03 / Trongsa TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-09-24",
        deprecated: 0, 
    },

    "EPSG:5308" => CRS {
        coord_ref_sys_code: 5308,
        coord_ref_sys_name: "DRUKREF 03 / Tsirang TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-09-24",
        deprecated: 0, 
    },

    "EPSG:5309" => CRS {
        coord_ref_sys_code: 5309,
        coord_ref_sys_name: "DRUKREF 03 / Wangdue Phodrang TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-10-26",
        deprecated: 0, 
    },

    "EPSG:5310" => CRS {
        coord_ref_sys_code: 5310,
        coord_ref_sys_name: "DRUKREF 03 / Yangtse TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-10-26",
        deprecated: 0, 
    },

    "EPSG:5311" => CRS {
        coord_ref_sys_code: 5311,
        coord_ref_sys_name: "DRUKREF 03 / Zhemgang TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5264,
        remarks: "",
        information_source: "Department of Survey and Land Records (DSLR), National Land Commission of Bhutan (NLC) via Lantmäteriet Sweden.",
        data_source: "EPSG",
        revision_date: "2010-09-24",
        deprecated: 0, 
    },

    "EPSG:5520" => CRS {
        coord_ref_sys_code: 5520,
        coord_ref_sys_name: "DHDN / 3-degree Gauss-Kruger zone 1",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "See CRS code 5680 for variant with axes order reversed to easting before northing for use in GIS applications.",
        information_source: "Geological Survey of Lower Saxony via Bundesamt für Kartographie und Geodäsie (BKG).",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:5627" => CRS {
        coord_ref_sys_code: 5627,
        coord_ref_sys_name: "ED50 / TM 6 NE",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "Used by Total.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2011-12-01",
        deprecated: 0, 
    },

    "EPSG:5643" => CRS {
        coord_ref_sys_code: 5643,
        coord_ref_sys_name: "ED50 / SPBA LCC",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "TNO - Geological Survey of The Netherlands",
        data_source: "EPSG",
        revision_date: "2012-01-27",
        deprecated: 0, 
    },

    "EPSG:5676" => CRS {
        coord_ref_sys_code: 5676,
        coord_ref_sys_name: "DHDN / 3-degree Gauss-Kruger zone 2 (E-N)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "DHDN / 3-degree Gauss-Kruger zone 2 (CRS code 31466) but with axis order reversed (and axis abbreviations changed) for GIS applications.",
        information_source: "Bundesamt für Kartographie und Geodäsie, http://crs.bkg.bund.de",
        data_source: "EPSG",
        revision_date: "2012-02-13",
        deprecated: 0, 
    },

    "EPSG:5677" => CRS {
        coord_ref_sys_code: 5677,
        coord_ref_sys_name: "DHDN / 3-degree Gauss-Kruger zone 3 (E-N)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "DHDN / 3-degree Gauss-Kruger zone 3 (CRS code 31467) but with axis order reversed (and axis abbreviations changed) for GIS applications.",
        information_source: "Bundesamt für Kartographie und Geodäsie, http://crs.bkg.bund.de",
        data_source: "EPSG",
        revision_date: "2012-02-13",
        deprecated: 0, 
    },

    "EPSG:5678" => CRS {
        coord_ref_sys_code: 5678,
        coord_ref_sys_name: "DHDN / 3-degree Gauss-Kruger zone 4 (E-N)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "DHDN / 3-degree Gauss-Kruger zone 4 (CRS code 31468) but with axis order reversed (and axis abbreviations changed) for GIS applications.",
        information_source: "Bundesamt für Kartographie und Geodäsie, http://crs.bkg.bund.de",
        data_source: "EPSG",
        revision_date: "2012-02-13",
        deprecated: 0, 
    },

    "EPSG:5679" => CRS {
        coord_ref_sys_code: 5679,
        coord_ref_sys_name: "DHDN / 3-degree Gauss-Kruger zone 5 (E-N)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "DHDN / 3-degree Gauss-Kruger zone 5 (CRS code 31469) but with axis order reversed (and axis abbreviations changed) for GIS applications.",
        information_source: "Bundesamt für Kartographie und Geodäsie, http://crs.bkg.bund.de",
        data_source: "EPSG",
        revision_date: "2012-02-13",
        deprecated: 0, 
    },

    "EPSG:5680" => CRS {
        coord_ref_sys_code: 5680,
        coord_ref_sys_name: "DHDN / 3-degree Gauss-Kruger zone 1 (E-N)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "DHDN / 3-degree Gauss-Kruger zone 1 (CRS code 5520) but with axis order reversed (and axis abbreviations changed) for GIS applications.",
        information_source: "Bundesamt für Kartographie und Geodäsie (BKG), http://www.bkg.bund.de",
        data_source: "EPSG",
        revision_date: "2012-02-13",
        deprecated: 0, 
    },

    "EPSG:5733" => CRS {
        coord_ref_sys_code: 5733,
        coord_ref_sys_name: "DNN height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5132, 
        base_crs_code: -1,
        remarks: "Replaced by DVR90 height (CRS code 5799).",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5750" => CRS {
        coord_ref_sys_code: 5750,
        coord_ref_sys_name: "Douglas height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5148, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Ordnance Survey of Great Britain",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5761" => CRS {
        coord_ref_sys_code: 5761,
        coord_ref_sys_name: "Dunedin 1958 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5159, 
        base_crs_code: -1,
        remarks: "",
        information_source: "OSG Technical Report #10, February 2001.",
        data_source: "EPSG",
        revision_date: "2011-01-25",
        deprecated: 0, 
    },

    "EPSG:5777" => CRS {
        coord_ref_sys_code: 5777,
        coord_ref_sys_name: "Durres height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5175, 
        base_crs_code: -1,
        remarks: "",
        information_source: "EuroGeographics, http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5783" => CRS {
        coord_ref_sys_code: 5783,
        coord_ref_sys_name: "DHHN92 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5181, 
        base_crs_code: -1,
        remarks: "Replaces SNN76 height (CRS code 5785) and DHHN85 height (CRS code 5784). Replaced by DHHN2016 height (CRS code 7837).",
        information_source: "Bundesamt für Kartographie und Geodäsie (BKG) http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2016-09-16",
        deprecated: 0, 
    },

    "EPSG:5784" => CRS {
        coord_ref_sys_code: 5784,
        coord_ref_sys_name: "DHHN85 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5182, 
        base_crs_code: -1,
        remarks: "Replaced by DNNH92 height (CRS code 5783).",
        information_source: "Bundesamt für Kartographie und Geodäsie (BKG) http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5799" => CRS {
        coord_ref_sys_code: 5799,
        coord_ref_sys_name: "DVR90 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5206, 
        base_crs_code: -1,
        remarks: "Replaces Dansk Normal Null height (CRS code 5733).",
        information_source: "Kort &amp; Matrikelstyrelsen: The Danish height system DVR90, Publ. 4.series, vol. 8, 2000.",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:7699" => CRS {
        coord_ref_sys_code: 7699,
        coord_ref_sys_name: "DHHN12 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1161, 
        base_crs_code: -1,
        remarks: "Replaced by SNN56 and then SNN76 in East Germany and by DHHN85 in West Germany.",
        information_source: "Landesamt für Digitalisierung, Breitband und Vermessung of Bavaria.",
        data_source: "EPSG",
        revision_date: "2015-12-20",
        deprecated: 0, 
    },

    "EPSG:7837" => CRS {
        coord_ref_sys_code: 7837,
        coord_ref_sys_name: "DHHN2016 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1170, 
        base_crs_code: -1,
        remarks: "Replaces DHHN92 height (CRS code 5783).",
        information_source: "Bundesamt für Kartographie und Geodäsie (BKG) via http://www.crs-geo.eu",
        data_source: "EPSG",
        revision_date: "2016-09-04",
        deprecated: 0, 
    },

    "EPSG:9758" => CRS {
        coord_ref_sys_code: 9758,
        coord_ref_sys_name: "ECML14_NB-IRF",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1310, 
        base_crs_code: -1,
        remarks: "Intermediate CRS created in 2021 to assist the emulation of the ETRS89 / ECML14_NB SnakeGrid projected CRS through transformation ETRS89 to ECML14_NB-IRF (1) (code 9759) used in conjunction with the ECML14_NB-TM map projection (code 9760).",
        information_source: "Network Rail.",
        data_source: "EPSG",
        revision_date: "2021-07-16",
        deprecated: 0, 
    },

    "EPSG:9761" => CRS {
        coord_ref_sys_code: 9761,
        coord_ref_sys_name: "ECML14_NB Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 9758,
        remarks: "The CRS&#x27;s definition through transformation ETRS89 to ECML14_NB-IRF (1) (code 9759) and map projection ECML14_NB-TM (code 9760) emulates the ECML14_NB Snake projection applied to ETRS89 as realised through OSNet 2009 CORS.",
        information_source: "Network Rail.",
        data_source: "EPSG",
        revision_date: "2021-07-16",
        deprecated: 0, 
    },

    "EPSG:9762" => CRS {
        coord_ref_sys_code: 9762,
        coord_ref_sys_name: "ECML14_NB Grid + ODN height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Network Rail.",
        data_source: "EPSG",
        revision_date: "2021-07-16",
        deprecated: 0, 
    },

    "EPSG:22832" => CRS {
        coord_ref_sys_code: 22832,
        coord_ref_sys_name: "Douala / UTM zone 32N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4228,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:23028" => CRS {
        coord_ref_sys_code: 23028,
        coord_ref_sys_name: "ED50 / UTM zone 28N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:23029" => CRS {
        coord_ref_sys_code: 23029,
        coord_ref_sys_name: "ED50 / UTM zone 29N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:23030" => CRS {
        coord_ref_sys_code: 23030,
        coord_ref_sys_name: "ED50 / UTM zone 30N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 0, 
    },

    "EPSG:23031" => CRS {
        coord_ref_sys_code: 23031,
        coord_ref_sys_name: "ED50 / UTM zone 31N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 0, 
    },

    "EPSG:23032" => CRS {
        coord_ref_sys_code: 23032,
        coord_ref_sys_name: "ED50 / UTM zone 32N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 0, 
    },

    "EPSG:23033" => CRS {
        coord_ref_sys_code: 23033,
        coord_ref_sys_name: "ED50 / UTM zone 33N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 0, 
    },

    "EPSG:23034" => CRS {
        coord_ref_sys_code: 23034,
        coord_ref_sys_name: "ED50 / UTM zone 34N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:23035" => CRS {
        coord_ref_sys_code: 23035,
        coord_ref_sys_name: "ED50 / UTM zone 35N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 0, 
    },

    "EPSG:23036" => CRS {
        coord_ref_sys_code: 23036,
        coord_ref_sys_name: "ED50 / UTM zone 36N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 0, 
    },

    "EPSG:23037" => CRS {
        coord_ref_sys_code: 23037,
        coord_ref_sys_name: "ED50 / UTM zone 37N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 0, 
    },

    "EPSG:23038" => CRS {
        coord_ref_sys_code: 23038,
        coord_ref_sys_name: "ED50 / UTM zone 38N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 0, 
    },

    "EPSG:23090" => CRS {
        coord_ref_sys_code: 23090,
        coord_ref_sys_name: "ED50 / TM 0 N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "Shell UK",
        data_source: "EPSG",
        revision_date: "1998-11-11",
        deprecated: 0, 
    },

    "EPSG:23095" => CRS {
        coord_ref_sys_code: 23095,
        coord_ref_sys_name: "ED50 / TM 5 NE",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "",
        information_source: "NAM",
        data_source: "EPSG",
        revision_date: "1998-11-11",
        deprecated: 0, 
    },

    "EPSG:31461" => CRS {
        coord_ref_sys_code: 31461,
        coord_ref_sys_name: "DHDN / 3-degree Gauss zone 1",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 1, 
    },

    "EPSG:31462" => CRS {
        coord_ref_sys_code: 31462,
        coord_ref_sys_name: "DHDN / 3-degree Gauss zone 2",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 1, 
    },

    "EPSG:31463" => CRS {
        coord_ref_sys_code: 31463,
        coord_ref_sys_name: "DHDN / 3-degree Gauss zone 3",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 1, 
    },

    "EPSG:31464" => CRS {
        coord_ref_sys_code: 31464,
        coord_ref_sys_name: "DHDN / 3-degree Gauss zone 4",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 1, 
    },

    "EPSG:31465" => CRS {
        coord_ref_sys_code: 31465,
        coord_ref_sys_name: "DHDN / 3-degree Gauss zone 5",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 1, 
    },

    "EPSG:31466" => CRS {
        coord_ref_sys_code: 31466,
        coord_ref_sys_name: "DHDN / 3-degree Gauss-Kruger zone 2",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "Zone width 3 degrees. Also used offshore between 4.5°E and 7.5°E by State Geological Surveys. Not used offshore in oil industry. See CRS code 5676 for variant with axes order reversed to easting before northing for use in GIS applications.",
        information_source: "Bundesamt für Kartographie und Geodäsie via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:31467" => CRS {
        coord_ref_sys_code: 31467,
        coord_ref_sys_name: "DHDN / 3-degree Gauss-Kruger zone 3",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "Zone width 3 degrees. Also used offshore by State Geological Surveys. Not used offshore in oil industry. See CRS code 5677 for variant with axes order reversed to easting before northing for use in GIS applications.",
        information_source: "Bundesamt für Kartographie und Geodäsie via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:31468" => CRS {
        coord_ref_sys_code: 31468,
        coord_ref_sys_name: "DHDN / 3-degree Gauss-Kruger zone 4",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "Zone width 3 degrees. Also used offshore by State Geological Surveys. Not used offshore in oil industry. See CRS code 5678 for variant with axes order reversed to easting before northing for use in GIS applications.",
        information_source: "Bundesamt für Kartographie und Geodäsie via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:31469" => CRS {
        coord_ref_sys_code: 31469,
        coord_ref_sys_name: "DHDN / 3-degree Gauss-Kruger zone 5",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "Zone width 3 degrees. Also used offshore by State Geological Surveys. Not used offshore in oil industry. See CRS code 5679 for variant with axes order reversed to easting before northing for use in GIS applications.",
        information_source: "Bundesamt für Kartographie und Geodäsie via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:61926405" => CRS {
        coord_ref_sys_code: 61926405,
        coord_ref_sys_name: "Douala 1948 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4192,
        remarks: "See CRS code 4192 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62306405" => CRS {
        coord_ref_sys_code: 62306405,
        coord_ref_sys_name: "ED50 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4230,
        remarks: "See CRS code 4230 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:63146405" => CRS {
        coord_ref_sys_code: 63146405,
        coord_ref_sys_name: "DHDN (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4314,
        remarks: "See CRS code 4314 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:66026405" => CRS {
        coord_ref_sys_code: 66026405,
        coord_ref_sys_name: "Dominica 1945 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4602,
        remarks: "See CRS code 4602 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:2058" => CRS {
        coord_ref_sys_code: 2058,
        coord_ref_sys_name: "ED50(ED77) / UTM zone 38N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4154,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2059" => CRS {
        coord_ref_sys_code: 2059,
        coord_ref_sys_name: "ED50(ED77) / UTM zone 39N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4154,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2060" => CRS {
        coord_ref_sys_code: 2060,
        coord_ref_sys_name: "ED50(ED77) / UTM zone 40N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4154,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2061" => CRS {
        coord_ref_sys_code: 2061,
        coord_ref_sys_name: "ED50(ED77) / UTM zone 41N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4154,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2068" => CRS {
        coord_ref_sys_code: 2068,
        coord_ref_sys_name: "ELD79 / Libya zone 5",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4159,
        remarks: "",
        information_source: "Brown and Root",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2069" => CRS {
        coord_ref_sys_code: 2069,
        coord_ref_sys_name: "ELD79 / Libya zone 6",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4159,
        remarks: "",
        information_source: "Brown and Root",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2070" => CRS {
        coord_ref_sys_code: 2070,
        coord_ref_sys_name: "ELD79 / Libya zone 7",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4159,
        remarks: "",
        information_source: "Brown and Root",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2071" => CRS {
        coord_ref_sys_code: 2071,
        coord_ref_sys_name: "ELD79 / Libya zone 8",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4159,
        remarks: "",
        information_source: "Brown and Root",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2072" => CRS {
        coord_ref_sys_code: 2072,
        coord_ref_sys_name: "ELD79 / Libya zone 9",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4159,
        remarks: "",
        information_source: "Brown and Root",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2073" => CRS {
        coord_ref_sys_code: 2073,
        coord_ref_sys_name: "ELD79 / Libya zone 10",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4159,
        remarks: "",
        information_source: "Brown and Root",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2074" => CRS {
        coord_ref_sys_code: 2074,
        coord_ref_sys_name: "ELD79 / Libya zone 11",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4159,
        remarks: "",
        information_source: "Brown and Root",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2075" => CRS {
        coord_ref_sys_code: 2075,
        coord_ref_sys_name: "ELD79 / Libya zone 12",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4159,
        remarks: "",
        information_source: "Brown and Root",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2076" => CRS {
        coord_ref_sys_code: 2076,
        coord_ref_sys_name: "ELD79 / Libya zone 13",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4159,
        remarks: "",
        information_source: "Brown and Root",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2077" => CRS {
        coord_ref_sys_code: 2077,
        coord_ref_sys_name: "ELD79 / UTM zone 32N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4159,
        remarks: "",
        information_source: "Brown and Root",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2078" => CRS {
        coord_ref_sys_code: 2078,
        coord_ref_sys_name: "ELD79 / UTM zone 33N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4159,
        remarks: "",
        information_source: "Brown and Root",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2079" => CRS {
        coord_ref_sys_code: 2079,
        coord_ref_sys_name: "ELD79 / UTM zone 34N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4159,
        remarks: "",
        information_source: "Brown and Root",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2080" => CRS {
        coord_ref_sys_code: 2080,
        coord_ref_sys_name: "ELD79 / UTM zone 35N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4159,
        remarks: "",
        information_source: "Brown and Root",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2087" => CRS {
        coord_ref_sys_code: 2087,
        coord_ref_sys_name: "ELD79 / TM 12 NE",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4159,
        remarks: "",
        information_source: "Total-Fina",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2176" => CRS {
        coord_ref_sys_code: 2176,
        coord_ref_sys_name: "ETRF2000-PL / CS2000/15",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4531,
        datum_code: -1, 
        base_crs_code: 9702,
        remarks: "See ETRF2000-PL / CS92 (code 2180) for smaller scale mapping.",
        information_source: "Ordinance of the Council of Ministers of 15th November 2012 on the state system of spatial reference system, Glowny Urzad Geodezji i Kartografii (GUGiK), http://www.gugik.gov.pl",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:2177" => CRS {
        coord_ref_sys_code: 2177,
        coord_ref_sys_name: "ETRF2000-PL / CS2000/18",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4531,
        datum_code: -1, 
        base_crs_code: 9702,
        remarks: "See ETRF2000-PL / CS92 (code 2180) for smaller scale mapping.",
        information_source: "Ordinance of the Council of Ministers of 15th November 2012 on the state system of spatial reference system, Glowny Urzad Geodezji i Kartografii (GUGiK), http://www.gugik.gov.pl",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:2178" => CRS {
        coord_ref_sys_code: 2178,
        coord_ref_sys_name: "ETRF2000-PL / CS2000/21",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4531,
        datum_code: -1, 
        base_crs_code: 9702,
        remarks: "See ETRF2000-PL / CS92 (code 2180) for smaller scale mapping.",
        information_source: "Ordinance of the Council of Ministers of 15th November 2012 on the state system of spatial reference system, Glowny Urzad Geodezji i Kartografii (GUGiK), http://www.gugik.gov.pl",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:2179" => CRS {
        coord_ref_sys_code: 2179,
        coord_ref_sys_name: "ETRF2000-PL / CS2000/24",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4531,
        datum_code: -1, 
        base_crs_code: 9702,
        remarks: "See ETRF2000-PL / CS92 (code 2180) for smaller scale mapping.",
        information_source: "Ordinance of the Council of Ministers of 15th November 2012 on the state system of spatial reference system, Glowny Urzad Geodezji i Kartografii (GUGiK), http://www.gugik.gov.pl",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:2180" => CRS {
        coord_ref_sys_code: 2180,
        coord_ref_sys_name: "ETRF2000-PL / CS92",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4531,
        datum_code: -1, 
        base_crs_code: 9702,
        remarks: "See ETRF2000-PL / CS2000 zones 2000/15, 2000/18, 2000/21 and 2000/24 (codes 2176-79) for large scale purposes.",
        information_source: "Ordinance of the Council of Ministers of 15th November 2012 on the state system of spatial reference system, Glowny Urzad Geodezji i Kartografii (GUGiK), http://www.gugik.gov.pl",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:3300" => CRS {
        coord_ref_sys_code: 3300,
        coord_ref_sys_name: "Estonian Coordinate System of 1992",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4133,
        remarks: "EST97 / TM Baltic (CRS code 25884) used for medium and small scale applications. Replaced by Estonian Coordinate System of 1997 (code 3301).",
        information_source: "http://www.geo.ut.ee/",
        data_source: "EPSG",
        revision_date: "2011-06-30",
        deprecated: 0, 
    },

    "EPSG:3301" => CRS {
        coord_ref_sys_code: 3301,
        coord_ref_sys_name: "Estonian Coordinate System of 1997",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4180,
        remarks: "Replaces Estonian Coordinate System of 1992 (code 3300). EST97 / TM Baltic (CRS code 25884) used for medium and small scale applications.",
        information_source: "Estonian National Land Board via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 0, 
    },

    "EPSG:3355" => CRS {
        coord_ref_sys_code: 3355,
        coord_ref_sys_name: "Egypt Gulf of Suez S-650 TL / Red Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4706,
        remarks: "Differs from Egypt 1907 / Red Belt (CRS code 22992) by approximately 20m.",
        information_source: "Various industry sources.",
        data_source: "EPSG",
        revision_date: "2006-01-31",
        deprecated: 0, 
    },

    "EPSG:3855" => CRS {
        coord_ref_sys_code: 3855,
        coord_ref_sys_name: "EGM2008 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1027, 
        base_crs_code: -1,
        remarks: "Zero-height surface resulting from the application of the EGM2008 geoid model to the WGS 84 ellipsoid. Replaces EGM96 height (CRS code 5773).",
        information_source: "http://earth-info.nga.mil/GandG/wgs84/gravitymod/egm2008/egm08_wgs84.html",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:4133" => CRS {
        coord_ref_sys_code: 4133,
        coord_ref_sys_name: "EST92",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6133, 
        base_crs_code: -1,
        remarks: "This name is also used for a projected CRS (see projCRS code 3300). Replaced by EST97 (code 4180).",
        information_source: "http://www.geo.ut.ee/",
        data_source: "EPSG",
        revision_date: "2011-06-30",
        deprecated: 0, 
    },

    "EPSG:4154" => CRS {
        coord_ref_sys_code: 4154,
        coord_ref_sys_name: "ED50(ED77)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6154, 
        base_crs_code: -1,
        remarks: "",
        information_source: "National Cartographic Centre of Iran",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4159" => CRS {
        coord_ref_sys_code: 4159,
        coord_ref_sys_name: "ELD79",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6159, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Various oil company records.",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4180" => CRS {
        coord_ref_sys_code: 4180,
        coord_ref_sys_name: "EST97",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6180, 
        base_crs_code: 4935,
        remarks: "This name is also used for a projected CRS (see projCRS code 3301). Replaces EST92 (code 4133).",
        information_source: "EPSG. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2007-08-27",
        deprecated: 0, 
    },

    "EPSG:4199" => CRS {
        coord_ref_sys_code: 4199,
        coord_ref_sys_name: "Egypt 1930",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6199, 
        base_crs_code: -1,
        remarks: "Note that Egypt 1930 uses the International 1924 ellipsoid, unlike the Egypt 1907 CRS (code 4229) which uses the Helmert ellipsoid. Oil industry references to the Egypt 1930 name and the Helmert ellipsoid probably mean Egypt 1907.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4229" => CRS {
        coord_ref_sys_code: 4229,
        coord_ref_sys_name: "Egypt 1907",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6229, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4231" => CRS {
        coord_ref_sys_code: 4231,
        coord_ref_sys_name: "ED87",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6231, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4341" => CRS {
        coord_ref_sys_code: 4341,
        coord_ref_sys_name: "EST97 (3D)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6401,
        datum_code: 6180, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Estonian National Land Board via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:4342" => CRS {
        coord_ref_sys_code: 4342,
        coord_ref_sys_name: "EST97 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6180, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Estonian National Land Board via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:4668" => CRS {
        coord_ref_sys_code: 4668,
        coord_ref_sys_name: "ED79",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6668, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2005-05-27",
        deprecated: 0, 
    },

    "EPSG:4706" => CRS {
        coord_ref_sys_code: 4706,
        coord_ref_sys_name: "Egypt Gulf of Suez S-650 TL",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6706, 
        base_crs_code: -1,
        remarks: "Differs from Egypt 1907 (CRS code 4229) by approximately 20m.",
        information_source: "Various industry sources.",
        data_source: "EPSG",
        revision_date: "2006-01-31",
        deprecated: 0, 
    },

    "EPSG:9452" => CRS {
        coord_ref_sys_code: 9452,
        coord_ref_sys_name: "ETRS89 + BI height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Ordnance Survey Limited",
        data_source: "EPSG",
        revision_date: "2020-07-24",
        deprecated: 0, 
    },

    "EPSG:4934" => CRS {
        coord_ref_sys_code: 4934,
        coord_ref_sys_name: "EST97",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6180, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Estonian National Land Board via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2004-01-05",
        deprecated: 0, 
    },

    "EPSG:4935" => CRS {
        coord_ref_sys_code: 4935,
        coord_ref_sys_name: "EST97",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6180, 
        base_crs_code: 4934,
        remarks: "",
        information_source: "Estonian National Land Board via EuroGeographics; http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2007-08-27",
        deprecated: 0, 
    },

    "EPSG:5773" => CRS {
        coord_ref_sys_code: 5773,
        coord_ref_sys_name: "EGM96 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5171, 
        base_crs_code: -1,
        remarks: "Zero-height surface resulting from the application of the EGM96 geoid model to the WGS 84 ellipsoid. Replaces EGM84 height (CRS code 5798). Replaced by EGM2008 height (CRS code 3855).",
        information_source: "NASA http://cddis.gsfc.nasa.gov/926/egm96/egm96.html",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:5787" => CRS {
        coord_ref_sys_code: 5787,
        coord_ref_sys_name: "EOMA 1980 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5185, 
        base_crs_code: -1,
        remarks: "",
        information_source: "EuroGeographics, http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5798" => CRS {
        coord_ref_sys_code: 5798,
        coord_ref_sys_name: "EGM84 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5203, 
        base_crs_code: -1,
        remarks: "Zero-height surface resulting from the application of the EGM84 geoid model to the WGS 84 ellipsoid. Replaced by EGM96 height (CRS code 5773).",
        information_source: "US National Geospatial-Intelligence Agency (NGA); https://earth-info.nga.mil/GandG/update/index.php",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:5804" => CRS {
        coord_ref_sys_code: 5804,
        coord_ref_sys_name: "EPSG seismic bin grid example A",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 6508,
        datum_code: 9312, 
        base_crs_code: -1,
        remarks: "Bin grid I&#x3D;J&#x3D;1 at WGS 84 / UTM zone 31N 456781E, 5836723N.
Bin grid orientation &#x3D; 20 degrees grid. Bin width I&#x3D;25m, J&#x3D;12.5m. Bin increment I&#x3D;1, J&#x3D;1. Scale factor at origin 0.99984",
        information_source: "UKOOA P6/98 seismic bin grid guideline.",
        data_source: "EPSG",
        revision_date: "2000-06-23",
        deprecated: 1, 
    },

    "EPSG:5805" => CRS {
        coord_ref_sys_code: 5805,
        coord_ref_sys_name: "EPSG seismic bin grid example B",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 6506,
        datum_code: 9312, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-06-23",
        deprecated: 1, 
    },

    "EPSG:5806" => CRS {
        coord_ref_sys_code: 5806,
        coord_ref_sys_name: "EPSG local engineering grid example A",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 6505,
        datum_code: 9313, 
        base_crs_code: -1,
        remarks: "Coordinates are 0,0 at conductor slot A1.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2016-07-13",
        deprecated: 1, 
    },

    "EPSG:5807" => CRS {
        coord_ref_sys_code: 5807,
        coord_ref_sys_name: "EPSG local engineering grid example B",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 6504,
        datum_code: 9313, 
        base_crs_code: -1,
        remarks: "Shows alternative use of coordinate axis name.
Coordinates are 0,0 at conductor slot A1.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2016-07-13",
        deprecated: 1, 
    },

    "EPSG:5812" => CRS {
        coord_ref_sys_code: 5812,
        coord_ref_sys_name: "El Cubo",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 4500,
        datum_code: 9306, 
        base_crs_code: -1,
        remarks: "Grid coordinates (0; 0) at El Cubo (8°44&#x27;17.258&quot;N, 72°30&#x27;09.01&quot;W).",
        information_source: "Ministry of Mines standards manual of 1974.",
        data_source: "EPSG",
        revision_date: "2008-06-23",
        deprecated: 0, 
    },

    "EPSG:5818" => CRS {
        coord_ref_sys_code: 5818,
        coord_ref_sys_name: "enter here name of I&#x3D;J+90 bin grid",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 1033,
        datum_code: 9315, 
        base_crs_code: -1,
        remarks: "This CRS offers a template for bin grid definition when I-axis direction &#x3D; J+90°. Enter here &quot;The bin grid is defined through coordinate operation code ... &quot; [enter code for coordinate operation defining bin grid, in this example template code 6918].",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2020-01-16",
        deprecated: 1, 
    },

    "EPSG:5859" => CRS {
        coord_ref_sys_code: 5859,
        coord_ref_sys_name: "enter here name of I&#x3D;J-90 bin grid",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 1034,
        datum_code: 9315, 
        base_crs_code: -1,
        remarks: "This CRS offers a template for bin grid definition when I-axis direction &#x3D; J-90°. Enter here &quot;The bin grid is defined through coordinate operation code ... &quot; [enter code for coordinate operation defining bin grid, in this example template code 6919].",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2020-01-16",
        deprecated: 1, 
    },

    "EPSG:7706" => CRS {
        coord_ref_sys_code: 7706,
        coord_ref_sys_name: "EPSG moving platform example",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 1042,
        datum_code: 1163, 
        base_crs_code: -1,
        remarks: "Right-handed local 3D CRS fixed to a moving platform. Other permutations of axis order or positive direction may be encountered for which different CSs should be used.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2016-02-09",
        deprecated: 0, 
    },

    "EPSG:7914" => CRS {
        coord_ref_sys_code: 7914,
        coord_ref_sys_name: "ETRF89",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1178, 
        base_crs_code: -1,
        remarks: "Replaced by ETRF90 (code 7916).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7915" => CRS {
        coord_ref_sys_code: 7915,
        coord_ref_sys_name: "ETRF89",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1178, 
        base_crs_code: 7914,
        remarks: "Replaced by ETRF90 (code 7917).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7916" => CRS {
        coord_ref_sys_code: 7916,
        coord_ref_sys_name: "ETRF90",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1179, 
        base_crs_code: -1,
        remarks: "Replaces ETRF89 (code 7914). Replaced by ETRF91 (code 7918).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7930" => CRS {
        coord_ref_sys_code: 7930,
        coord_ref_sys_name: "ETRF2000",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1186, 
        base_crs_code: -1,
        remarks: "Replaces ETRF97 (code 7928). On the publication of ETRF2005 (code 8397),  the EUREF Technical Working Group recommended that ETRF2000 be the realization of ETRS89. ETRF2014 (code 8401) is technically superior to all earlier realizations of ETRS89.",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7931" => CRS {
        coord_ref_sys_code: 7931,
        coord_ref_sys_name: "ETRF2000",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1186, 
        base_crs_code: 7930,
        remarks: "Replaces ETRF97 (code 7929). On the publication of ETRF2005 (code 8399),  the EUREF Technical Working Group recommended that ETRF2000 be the realization of ETRS89. ETRF2014 (code 8403) is technically superior to all earlier realizations of ETRS89.",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:8377" => CRS {
        coord_ref_sys_code: 8377,
        coord_ref_sys_name: "EPSG example wellbore local horizontal CRS - true north",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 1047,
        datum_code: 1203, 
        base_crs_code: -1,
        remarks: "Horizontal component of wellbore local cubic CRS (CRS code 8820). Local coordinates are (0,0) at well reference point, with n-axis aligned to true north.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2019-02-19",
        deprecated: 0, 
    },

    "EPSG:8378" => CRS {
        coord_ref_sys_code: 8378,
        coord_ref_sys_name: "EPSG example wellbore local vertical CRS",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 1049,
        datum_code: 1205, 
        base_crs_code: -1,
        remarks: "Vertical component of wellbore cubic CRS (CRS code 8820). Depth is relative to the zero depth point used during wellbore survey (e.g. derrick floor).",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2019-02-16",
        deprecated: 0, 
    },

    "EPSG:8397" => CRS {
        coord_ref_sys_code: 8397,
        coord_ref_sys_name: "ETRF2005",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1204, 
        base_crs_code: -1,
        remarks: "On publication in 2007 of this CRS, the EUREF Technical Working Group recommended that ETRF2000 (EPSG code 7930) remained as the preferred realization of ETRS89. Replaced by ETRF2014 (code 8401).",
        information_source: "Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing&quot; v8 and EUREF Technical Note 1: Relationship and Transformation between the International and European Terrestrial Reference Systems, Nov. 16, 2017.",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:8399" => CRS {
        coord_ref_sys_code: 8399,
        coord_ref_sys_name: "ETRF2005",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1204, 
        base_crs_code: 8397,
        remarks: "On publication in 2007 of this CRS, the EUREF Technical Working Group recommended that ETRF2000 (EPSG code 7931) remained as the preferred realization of ETRS89. Replaced by ETRF2014 (code 8403).",
        information_source: "Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing&quot; v8 and EUREF Technical Note 1: Relationship and Transformation between the International and European Terrestrial Reference Systems, Nov. 16, 2017.",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:8401" => CRS {
        coord_ref_sys_code: 8401,
        coord_ref_sys_name: "ETRF2014",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1206, 
        base_crs_code: -1,
        remarks: "Replaces ETRF2005 (code 8397). ETRF2014 is technically superior to ETRF2000 but ETRF2000 and other previous realizations may be preferred for backward compatibility reasons. Differences between ETRF2014 and ETRF2000 can reach 7cm.",
        information_source: "EUREF Technical Note 1: Relationship and Transformation between the International and European Terrestrial Reference Systems, November 16, 2017.",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:8403" => CRS {
        coord_ref_sys_code: 8403,
        coord_ref_sys_name: "ETRF2014",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1206, 
        base_crs_code: 8401,
        remarks: "Replaces ETRF2005 (code 8399). ETRF2014 is technically superior to ETRF2000 but ETRF2000 and other previous realizations may be preferred for backward compatibility reasons. Differences between ETRF2014 and ETRF2000 can reach 7cm.",
        information_source: "EUREF Technical Note 1: Relationship and Transformation between the International and European Terrestrial Reference Systems, November 16, 2017.",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:8820" => CRS {
        coord_ref_sys_code: 8820,
        coord_ref_sys_name: "EPSG example wellbore local cubic CRS - true north",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Derived from wellbore measured depth, azimuth and inclination measurements where facility reference direction is true north. Further processing is through the individual horizontal and vertical CRSs (CRS codes 8377 and 8378).",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2019-02-17",
        deprecated: 0, 
    },

    "EPSG:8821" => CRS {
        coord_ref_sys_code: 8821,
        coord_ref_sys_name: "EPSG example wellbore local cubic CRS - grid north (ft)",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Derived from wellbore measured depth, azimuth and inclination measurements where facility reference direction is grid north. Further processing is through the individual horizontal and vertical CRSs (CRS codes 8896 and 8897).",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2019-02-17",
        deprecated: 0, 
    },

    "EPSG:8896" => CRS {
        coord_ref_sys_code: 8896,
        coord_ref_sys_name: "EPSG example wellbore local horizontal CRS - grid north (ft)",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 1048,
        datum_code: 1222, 
        base_crs_code: -1,
        remarks: "Horizontal component of wellbore local cubic CRS (CRS code 8821). Local coordinates are (0,0) at well reference point, with n-axis aligned to grid north in the projected CRS specified in the wellbore local horizontal datum origin definition.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2019-02-19",
        deprecated: 0, 
    },

    "EPSG:8897" => CRS {
        coord_ref_sys_code: 8897,
        coord_ref_sys_name: "EPSG example wellbore local vertical CRS (ft)",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 1050,
        datum_code: 1205, 
        base_crs_code: -1,
        remarks: "Vertical component of wellbore cubic CRS (CRS code 8821). Depth is relative to the zero depth point used during wellbore survey (e.g. derrick floor).",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2019-02-17",
        deprecated: 0, 
    },

    "EPSG:9059" => CRS {
        coord_ref_sys_code: 9059,
        coord_ref_sys_name: "ETRF89",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1178, 
        base_crs_code: 7915,
        remarks: "Replaced by ETRF90 (code 9060).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-05-23",
        deprecated: 0, 
    },

    "EPSG:9067" => CRS {
        coord_ref_sys_code: 9067,
        coord_ref_sys_name: "ETRF2000",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1186, 
        base_crs_code: 7931,
        remarks: "Replaces ETRF97 (code 9066). On the publication of ETRF2005 (code 9068),  the EUREF Technical Working Group recommended that ETRF2000 be the realization of ETRS89. ETRF2014 (code 9069) is technically superior to all earlier realizations of ETRS89.",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-05-23",
        deprecated: 0, 
    },

    "EPSG:9068" => CRS {
        coord_ref_sys_code: 9068,
        coord_ref_sys_name: "ETRF2005",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1204, 
        base_crs_code: 8399,
        remarks: "On publication in 2007 of this CRS, the EUREF Technical Working Group recommended that ETRF2000 (EPSG code 9067) remained as the preferred realization of ETRS89. Replaced by ETRF2014 (code 9069).",
        information_source: "Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing&quot; v8 and EUREF Technical Note 1: Relationship and Transformation between the International and European Terrestrial Reference Systems, Nov. 16, 2017.",
        data_source: "EPSG",
        revision_date: "2019-05-23",
        deprecated: 0, 
    },

    "EPSG:9069" => CRS {
        coord_ref_sys_code: 9069,
        coord_ref_sys_name: "ETRF2014",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1206, 
        base_crs_code: 8403,
        remarks: "Replaces ETRF2005 (code 9068). ETRF2014 is technically superior to ETRF2000 but ETRF2000 and other previous realizations may be preferred for backward compatibility reasons. Differences between ETRF2014 and ETRF2000 can reach 7cm.",
        information_source: "EUREF Technical Note 1: Relationship and Transformation between the International and European Terrestrial Reference Systems, November 16, 2017.",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:9401" => CRS {
        coord_ref_sys_code: 9401,
        coord_ref_sys_name: "El Hierro height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1284, 
        base_crs_code: -1,
        remarks: "",
        information_source: "National Geographic Institute of Spain (IGN), http://www.ign.es",
        data_source: "EPSG",
        revision_date: "2020-06-27",
        deprecated: 0, 
    },

    "EPSG:9656" => CRS {
        coord_ref_sys_code: 9656,
        coord_ref_sys_name: "ETRF2000-PL + Baltic 1986 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:9657" => CRS {
        coord_ref_sys_code: 9657,
        coord_ref_sys_name: "ETRF2000-PL + EVRF2007-PL height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:9663" => CRS {
        coord_ref_sys_code: 9663,
        coord_ref_sys_name: "EH2000 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1298, 
        base_crs_code: -1,
        remarks: "In Estonia replaces Baltic 1977 system (CRS code 5705) from January 2018.",
        information_source: "Estonian National Land Board, http://www.maaamet.ee, via Eurogeographics, http://www.crs-geo.eu/",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:9700" => CRS {
        coord_ref_sys_code: 9700,
        coord_ref_sys_name: "ETRF2000-PL",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1305, 
        base_crs_code: -1,
        remarks: "Adopted from 2012-12-01.",
        information_source: "Ordinance of the Council of Ministers of 15th November 2012 on the state system of spatial reference system, Glowny Urzad Geodezji i Kartografii (GUGiK), http://www.gugik.gov.pl",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:9701" => CRS {
        coord_ref_sys_code: 9701,
        coord_ref_sys_name: "ETRF2000-PL",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1305, 
        base_crs_code: 9700,
        remarks: "Adopted from 2012-12-01.",
        information_source: "Ordinance of the Council of Ministers of 15th November 2012 on the state system of spatial reference system, Glowny Urzad Geodezji i Kartografii (GUGiK), http://www.gugik.gov.pl",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:9702" => CRS {
        coord_ref_sys_code: 9702,
        coord_ref_sys_name: "ETRF2000-PL",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1305, 
        base_crs_code: 9701,
        remarks: "",
        information_source: "IOGP. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:9739" => CRS {
        coord_ref_sys_code: 9739,
        coord_ref_sys_name: "EOS21-IRF",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1308, 
        base_crs_code: -1,
        remarks: "Intermediate CRS created in 2021 to assist the emulation of the ETRS89 / EOS21 SnakeGrid projected CRS through transformation ETRS89 to EOS21-IRF (1) (code 9740) used in conjunction with the EOS21-TM map projection (code 9738).",
        information_source: "Network Rail.",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:9741" => CRS {
        coord_ref_sys_code: 9741,
        coord_ref_sys_name: "EOS21 Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 9739,
        remarks: "The CRS&#x27;s definition through transformation ETRS89 to EOS21-IRF (1) (code 9740) and map projection EOS21-TM (code 9738) emulates the EOS21 Snake projection applied to ETRS89 as realised through OSNet 2009 CORS.",
        information_source: "Network Rail.",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:9742" => CRS {
        coord_ref_sys_code: 9742,
        coord_ref_sys_name: "EOS21 Grid + ODN height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Network Rail.",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:22991" => CRS {
        coord_ref_sys_code: 22991,
        coord_ref_sys_name: "Egypt 1907 / Blue Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4229,
        remarks: "Also known as Egypt 1907 / Green Belt. Oil industry usually uses Egypt 1907 / Red Belt rather than this projected CRS.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2002-07-16",
        deprecated: 0, 
    },

    "EPSG:22992" => CRS {
        coord_ref_sys_code: 22992,
        coord_ref_sys_name: "Egypt 1907 / Red Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4229,
        remarks: "See also Egypt 1907 / Blue Belt for non oil industry usage in Sinai peninsula.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:22993" => CRS {
        coord_ref_sys_code: 22993,
        coord_ref_sys_name: "Egypt 1907 / Purple Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4229,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:22994" => CRS {
        coord_ref_sys_code: 22994,
        coord_ref_sys_name: "Egypt 1907 / Extended Purple Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4229,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:32596" => CRS {
        coord_ref_sys_code: 32596,
        coord_ref_sys_name: "EPSG example I&#x3D;J+90 bin grid (real values)",
        coord_ref_sys_kind: "derived",
        coord_sys_code: 1051,
        datum_code: -1, 
        base_crs_code: 32631,
        remarks: "Example of a bin grid defined with respect to a map grid when I-axis direction &#x3D; J+90° and bin values may be non-integer. If bin node coordinates values and node distances are integer see CRS 32762. If bin grid is not georeferenced see CRS 32598.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:32597" => CRS {
        coord_ref_sys_code: 32597,
        coord_ref_sys_name: "EPSG example I&#x3D;J-90 bin grid (real values)",
        coord_ref_sys_kind: "derived",
        coord_sys_code: 1052,
        datum_code: -1, 
        base_crs_code: 32066,
        remarks: "Example of a bin grid defined with respect to a map grid when I-axis direction &#x3D; J-90° and bin values may be non-integer. If bin node coordinates values and node distances are integer see CRS 32763. If bin grid is not georeferenced see CRS 32599.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:32598" => CRS {
        coord_ref_sys_code: 32598,
        coord_ref_sys_name: "EPSG example I&#x3D;J+90 local bin grid (real values)",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 1051,
        datum_code: 9315, 
        base_crs_code: -1,
        remarks: "Example of a non-georeferenced bin grid definition when I-axis direction &#x3D; J+90° and bin values may be non-integer. To georeference this grid see CT 6918. If bin node coordinates values and node distances are integer see CRS 32764.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:32599" => CRS {
        coord_ref_sys_code: 32599,
        coord_ref_sys_name: "EPSG example I&#x3D;J-90 local bin grid (real values)",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 1052,
        datum_code: 9315, 
        base_crs_code: -1,
        remarks: "Example of a non-georeferenced bin grid definition when I-axis direction &#x3D; J-90° and bin values may be non-integer. To georeference this grid see CT 6919. If bin node coordinates values and node distances are integer see CRS 32765.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:32762" => CRS {
        coord_ref_sys_code: 32762,
        coord_ref_sys_name: "EPSG example I&#x3D;J+90 bin grid (integer values)",
        coord_ref_sys_kind: "derived",
        coord_sys_code: 32760,
        datum_code: -1, 
        base_crs_code: 32631,
        remarks: "Example of a bin grid defined with respect to a map grid when I-axis direction &#x3D; J+90° and bin values are integer. If bin node coordinates values and node distances are non-integer see CRS 32596. If bin grid is not georeferenced see CRS 32764.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:32763" => CRS {
        coord_ref_sys_code: 32763,
        coord_ref_sys_name: "EPSG example I&#x3D;J-90 bin grid (integer values)",
        coord_ref_sys_kind: "derived",
        coord_sys_code: 32761,
        datum_code: -1, 
        base_crs_code: 32066,
        remarks: "Example of a bin grid defined with respect to a map grid when I-axis direction &#x3D; J-90° and bin values are integer. If bin node coordinates values and node distances are non-integer see CRS 32597. If bin grid is not georeferenced see CRS 32765.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:32764" => CRS {
        coord_ref_sys_code: 32764,
        coord_ref_sys_name: "EPSG example I&#x3D;J+90 local bin grid (integer values)",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 32760,
        datum_code: 9315, 
        base_crs_code: -1,
        remarks: "Example of a non-georeferenced bin grid definition when I-axis direction &#x3D; J+90° and bin values are integer. To georeference this grid see CT 6918. If bin node coordinates values and node distances may be non-integer see CRS 32598.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:32765" => CRS {
        coord_ref_sys_code: 32765,
        coord_ref_sys_name: "EPSG example I&#x3D;J-90 local bin grid (integer values)",
        coord_ref_sys_kind: "engineering",
        coord_sys_code: 32761,
        datum_code: 9315, 
        base_crs_code: -1,
        remarks: "Example of a non-georeferenced bin grid definition when I-axis direction &#x3D; J-90° and bin values are integer. To georeference this grid see CT 6919. If bin node coordinates values and node distances may be non-integer see CRS 32599.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:61336405" => CRS {
        coord_ref_sys_code: 61336405,
        coord_ref_sys_name: "EST92 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4133,
        remarks: "See CRS code 4133 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61546405" => CRS {
        coord_ref_sys_code: 61546405,
        coord_ref_sys_name: "ED50(ED77) (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4154,
        remarks: "See CRS code 4154 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61596405" => CRS {
        coord_ref_sys_code: 61596405,
        coord_ref_sys_name: "ELD79 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4159,
        remarks: "See CRS code 4159 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61806405" => CRS {
        coord_ref_sys_code: 61806405,
        coord_ref_sys_name: "EST97 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4180,
        remarks: "See CRS code 4180 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61806413" => CRS {
        coord_ref_sys_code: 61806413,
        coord_ref_sys_name: "EST97 (3D deg)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6413,
        datum_code: -1, 
        base_crs_code: 4341,
        remarks: "See CRS code 4341 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:61996405" => CRS {
        coord_ref_sys_code: 61996405,
        coord_ref_sys_name: "Egypt 1930 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4199,
        remarks: "See CRS code 4199 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62296405" => CRS {
        coord_ref_sys_code: 62296405,
        coord_ref_sys_name: "Egypt 1907 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4229,
        remarks: "See CRS code 4229 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62316405" => CRS {
        coord_ref_sys_code: 62316405,
        coord_ref_sys_name: "ED87 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4231,
        remarks: "See CRS code 4231 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:2196" => CRS {
        coord_ref_sys_code: 2196,
        coord_ref_sys_name: "ETRS89 / Kp2000 Jutland",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces System 34 Jutland zone. For engineering survey and construction ETRS89 / DKTM (CRS codes 4093-94) are also used. For medium and small scale topographic mapping and cadastre ETRS89 / UTM zone 32N (code 25832) is used.",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:2197" => CRS {
        coord_ref_sys_code: 2197,
        coord_ref_sys_name: "ETRS89 / Kp2000 Zealand",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces System 34 Zealand zone. For engineering survey and construction ETRS89 / DKTM3 (CRS code 4095) is also used. For medium and small scale topographic mapping and cadastre ETRS89 / UTM zones 32N and 33N (CRS codes 25832-33) are used.",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:2198" => CRS {
        coord_ref_sys_code: 2198,
        coord_ref_sys_name: "ETRS89 / Kp2000 Bornholm",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces System 45. For engineering survey and construction ETRS89 / DKTM4 (code 4096) is also used. For medium and small scale topographic mapping and cadastre ETRS89 / UTM zone 33N (code 25833) is used.",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3108" => CRS {
        coord_ref_sys_code: 3108,
        coord_ref_sys_name: "ETRS89 / Guernsey Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaced ED50 / UTM zone 30N with effect from 1996.",
        information_source: "States of Guernsey and Digimap Ltd, Guernsey.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3109" => CRS {
        coord_ref_sys_code: 3109,
        coord_ref_sys_name: "ETRS89 / Jersey Transverse Mercator",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Effective from January 1st 2005.",
        information_source: "Jersey Planning &amp; Environment Department.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3126" => CRS {
        coord_ref_sys_code: 3126,
        coord_ref_sys_name: "ETRS89 / ETRS-GK19FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Withdrawn and replaced in 2009 by ETRS89 / GK19FIN (CRS code 3873).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3127" => CRS {
        coord_ref_sys_code: 3127,
        coord_ref_sys_name: "ETRS89 / ETRS-GK20FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Withdrawn and replaced in 2009 by ETRS89 / GK20FIN (CRS code 3874).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3128" => CRS {
        coord_ref_sys_code: 3128,
        coord_ref_sys_name: "ETRS89 / ETRS-GK21FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Withdrawn and replaced in 2009 by ETRS89 / GK21FIN (CRS code 3875).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3129" => CRS {
        coord_ref_sys_code: 3129,
        coord_ref_sys_name: "ETRS89 / ETRS-GK22FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Withdrawn and replaced in 2009 by ETRS89 / GK22FIN (CRS code 3876).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3130" => CRS {
        coord_ref_sys_code: 3130,
        coord_ref_sys_name: "ETRS89 / ETRS-GK23FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Withdrawn and replaced in 2009 by ETRS89 / GK23FIN (CRS code 3877).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3131" => CRS {
        coord_ref_sys_code: 3131,
        coord_ref_sys_name: "ETRS89 / ETRS-GK24FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Withdrawn and replaced in 2009 by ETRS89 / GK24FIN (CRS code 3878).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3132" => CRS {
        coord_ref_sys_code: 3132,
        coord_ref_sys_name: "ETRS89 / ETRS-GK25FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Withdrawn and replaced in 2009 by ETRS89 / GK25FIN (CRS code 3879).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3133" => CRS {
        coord_ref_sys_code: 3133,
        coord_ref_sys_name: "ETRS89 / ETRS-GK26FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Withdrawn and replaced in 2009 by ETRS89 / GK26FIN (CRS code 3880).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3134" => CRS {
        coord_ref_sys_code: 3134,
        coord_ref_sys_name: "ETRS89 / ETRS-GK27FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Withdrawn and replaced in 2009 by ETRS89 / GK27FIN (CRS code 3881).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3135" => CRS {
        coord_ref_sys_code: 3135,
        coord_ref_sys_name: "ETRS89 / ETRS-GK28FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Withdrawn and replaced in 2009 by ETRS89 / GK28FIN (CRS code 3882).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3136" => CRS {
        coord_ref_sys_code: 3136,
        coord_ref_sys_name: "ETRS89 / ETRS-GK29FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Withdrawn and replaced in 2009 by ETRS89 / GK29FIN (CRS code 3883).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3137" => CRS {
        coord_ref_sys_code: 3137,
        coord_ref_sys_name: "ETRS89 / ETRS-GK30FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Withdrawn and replaced in 2009 by ETRS89 / GK30FIN (CRS code 3884).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3138" => CRS {
        coord_ref_sys_code: 3138,
        coord_ref_sys_name: "ETRS89 / ETRS-GK31FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Withdrawn and replaced in 2009 by ETRS89 / GK31FIN (CRS code 3885).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3145" => CRS {
        coord_ref_sys_code: 3145,
        coord_ref_sys_name: "ETRS89 / Faroe Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 1031,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces ED50 / UTM zone 29N for topographic mapping. For cadastral survey see fk89 (CRS code 3173).",
        information_source: "KMS",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3416" => CRS {
        coord_ref_sys_code: 3416,
        coord_ref_sys_name: "ETRS89 / Austria Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces MGI / Austria Lambert (CRS code 31287).",
        information_source: "Bundesamt fur Eich- und Vermessungswesen (BEV); Wien. www.bev.gv.at",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3447" => CRS {
        coord_ref_sys_code: 3447,
        coord_ref_sys_name: "ETRS89 / Belgian Lambert 2005",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Coordinates referenced to ETRS89 / Lambert 2005 differ from coordinates referenced to BD72 / Lambert 1972 (CRS code 31370) by approximately 1km in easting and northing. Replaced in 2008 by ETRS89 / Lambert 2008 (CRS code 3812).",
        information_source: "IGN Brussels; www.ngi.be/",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3812" => CRS {
        coord_ref_sys_code: 3812,
        coord_ref_sys_name: "ETRS89 / Belgian Lambert 2008",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces ETRS89 / Lambert 2005 (code 3447). Coordinates referenced to ETRS89 / Lambert 2008 differ from coordinates referenced to BD72 / Lambert 1972 (CRS code 31370) by approximately 500km in easting and northing.",
        information_source: "IGN Brussels; www.ngi.be/",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3873" => CRS {
        coord_ref_sys_code: 3873,
        coord_ref_sys_name: "ETRS89 / GK19FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces ETRS89 / ETRS-GK19FIN (CRS code 3126). (For medium scale mapping and country-wide work use ETRS89 / TM35FIN(N,E), code 5048).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3874" => CRS {
        coord_ref_sys_code: 3874,
        coord_ref_sys_name: "ETRS89 / GK20FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces ETRS89 / ETRS-GK20FIN (CRS code 3127). (For medium scale mapping and country-wide work use ETRS89 / TM35FIN(N,E), code 5048).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3875" => CRS {
        coord_ref_sys_code: 3875,
        coord_ref_sys_name: "ETRS89 / GK21FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces ETRS89 / ETRS-GK21FIN (CRS code 3128). (For medium scale mapping and country-wide work use ETRS89 / TM35FIN(N,E), code 5048).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3876" => CRS {
        coord_ref_sys_code: 3876,
        coord_ref_sys_name: "ETRS89 / GK22FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces ETRS89 / ETRS-GK22FIN (CRS code 3129). (For medium scale mapping and country-wide work use ETRS89 / TM35FIN(N,E), code 5048).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3877" => CRS {
        coord_ref_sys_code: 3877,
        coord_ref_sys_name: "ETRS89 / GK23FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces ETRS89 / ETRS-GK23FIN (CRS code 3130). (For medium scale mapping and country-wide work use ETRS89 / TM35FIN(N,E), code 5048).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3878" => CRS {
        coord_ref_sys_code: 3878,
        coord_ref_sys_name: "ETRS89 / GK24FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces ETRS89 / ETRS-GK24FIN (CRS code 3131). (For medium scale mapping and country-wide work use ETRS89 / TM35FIN(N,E), code 5048).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3879" => CRS {
        coord_ref_sys_code: 3879,
        coord_ref_sys_name: "ETRS89 / GK25FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces ETRS89 / ETRS-GK25FIN (CRS code 3132). (For medium scale mapping and country-wide work use ETRS89 / TM35FIN(N,E), code 5048).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3880" => CRS {
        coord_ref_sys_code: 3880,
        coord_ref_sys_name: "ETRS89 / GK26FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces ETRS89 / ETRS-GK26FIN (CRS code 3133). (For medium scale mapping and country-wide work use ETRS89 / TM35FIN(N,E), code 5048).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3881" => CRS {
        coord_ref_sys_code: 3881,
        coord_ref_sys_name: "ETRS89 / GK27FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces ETRS89 / ETRS-GK27FIN (CRS code 3134). (For medium scale mapping and country-wide work use ETRS89 / TM35FIN(N,E), code 5048).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3882" => CRS {
        coord_ref_sys_code: 3882,
        coord_ref_sys_name: "ETRS89 / GK28FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces ETRS89 / ETRS-GK28FIN (CRS code 3135). (For medium scale mapping and country-wide work use ETRS89 / TM35FIN(N,E), code 5048).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3883" => CRS {
        coord_ref_sys_code: 3883,
        coord_ref_sys_name: "ETRS89 / GK29FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces ETRS89 / ETRS-GK29FIN (CRS code 3136). (For medium scale mapping and country-wide work use ETRS89 / TM35FIN(N,E), code 5048).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3884" => CRS {
        coord_ref_sys_code: 3884,
        coord_ref_sys_name: "ETRS89 / GK30FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces ETRS89 / ETRS-GK30FIN (CRS code 3137). (For medium scale mapping and country-wide work use ETRS89 / TM35FIN(N,E), code 5048).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3885" => CRS {
        coord_ref_sys_code: 3885,
        coord_ref_sys_name: "ETRS89 / GK31FIN",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Replaces ETRS89 / ETRS-GK31FIN (CRS code 3138). (For medium scale mapping and country-wide work use ETRS89 / TM35FIN(N,E), code 5048).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:4093" => CRS {
        coord_ref_sys_code: 4093,
        coord_ref_sys_name: "ETRS89 / DKTM1",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "For large scale topographic mapping and cadastral survey ETRS89 / Kp2000 Jutland (CRS code 2196) is used. For medium and small scale topographic mapping ETRS89 / UTM zone 32N (CRS code 25832) is used.",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:4094" => CRS {
        coord_ref_sys_code: 4094,
        coord_ref_sys_name: "ETRS89 / DKTM2",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "For large scale topographic mapping and cadastral survey ETRS89 / Kp2000 Jutland (CRS code 2196) is used. For medium and small scale topographic mapping ETRS89 / UTM zone 32N (CRS code 25832) is used.",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:4095" => CRS {
        coord_ref_sys_code: 4095,
        coord_ref_sys_name: "ETRS89 / DKTM3",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "For large scale topographic mapping and cadastral survey ETRS89 / Kp2000 Zealand (CRS code 2197) is used. For medium and small scale topographic mapping ETRS89 / UTM (CRS codes 25832 and 25833) are used.",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:4096" => CRS {
        coord_ref_sys_code: 4096,
        coord_ref_sys_name: "ETRS89 / DKTM4",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "For large scale topographic mapping and cadastral survey ETRS89 / Kp2000 Bornholm (CRS code 2198) is used. For medium and small scale topographic mapping ETRS89 / UTM zone 33N (CRS code 25833) is used.",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-14",
        deprecated: 0, 
    },

    "EPSG:4097" => CRS {
        coord_ref_sys_code: 4097,
        coord_ref_sys_name: "ETRS89 / DKTM1 + DVR90 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:4098" => CRS {
        coord_ref_sys_code: 4098,
        coord_ref_sys_name: "ETRS89 / DKTM2 + DVR90 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:4099" => CRS {
        coord_ref_sys_code: 4099,
        coord_ref_sys_name: "ETRS89 / DKTM3 + DVR90 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:9500" => CRS {
        coord_ref_sys_code: 9500,
        coord_ref_sys_name: "ETRS89 + EVRF2000 Austria height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:4100" => CRS {
        coord_ref_sys_code: 4100,
        coord_ref_sys_name: "ETRS89 / DKTM4 + DVR90 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:4258" => CRS {
        coord_ref_sys_code: 4258,
        coord_ref_sys_name: "ETRS89",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6258, 
        base_crs_code: 4937,
        remarks: "Has been realized through ETRF89, ETRF90, ETRF91, ETRF92, ETRF93, ETRF94, ETRF96, ETRF97, ETRF2000, ETRF2005 and ETRF2014. This &#x27;ensemble&#x27; covers any or all of these realizations without distinction.",
        information_source: "EPSG. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2020-03-14",
        deprecated: 0, 
    },

    "EPSG:4345" => CRS {
        coord_ref_sys_code: 4345,
        coord_ref_sys_name: "ETRS89 (3D)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6401,
        datum_code: 6258, 
        base_crs_code: -1,
        remarks: "The distinction in usage between ETRF89 and ETRS89 is confused: although in principle conceptually different in practice both are used as synonyms.",
        information_source: "Norwegian Geodetic Institute geodetic publication 1990:1",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:4346" => CRS {
        coord_ref_sys_code: 4346,
        coord_ref_sys_name: "ETRS89 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6258, 
        base_crs_code: -1,
        remarks: "The distinction in usage between ETRF89 and ETRS89 is confused: although in principle conceptually different in practice both are used as synonyms.",
        information_source: "Norwegian Geodetic Institute geodetic publication 1990:1",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:4839" => CRS {
        coord_ref_sys_code: 4839,
        coord_ref_sys_name: "ETRS89 / LCC Germany (N-E)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Variant for Germany of ETRS89 / LCC Europe (CRS code 3034). See ETRS89 / LCC Germany (E-N) (CRS code 5243) for similar CRS but with alternative axis order.",
        information_source: "Geodatenzentrum (GDZ) am Bundesamt fuer Kartographie und Geodaesie (BKG).",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:4860" => CRS {
        coord_ref_sys_code: 4860,
        coord_ref_sys_name: "ETRS89 / NTM zone 10",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4861" => CRS {
        coord_ref_sys_code: 4861,
        coord_ref_sys_name: "ETRS89 / NTM zone 11",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4862" => CRS {
        coord_ref_sys_code: 4862,
        coord_ref_sys_name: "ETRS89 / NTM zone 12",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4936" => CRS {
        coord_ref_sys_code: 4936,
        coord_ref_sys_name: "ETRS89",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6258, 
        base_crs_code: -1,
        remarks: "Has been realized through ETRF89, ETRF90, ETRF91, ETRF92, ETRF93, ETRF94, ETRF96, ETRF97, ETRF2000, ETRF2005 and ETRF2014. This &#x27;ensemble&#x27; covers any or all of these realizations without distinction.",
        information_source: "EPSG",
        data_source: "EPSG",
        revision_date: "2020-03-14",
        deprecated: 0, 
    },

    "EPSG:4937" => CRS {
        coord_ref_sys_code: 4937,
        coord_ref_sys_name: "ETRS89",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6258, 
        base_crs_code: 4936,
        remarks: "Has been realized through ETRF89, ETRF90, ETRF91, ETRF92, ETRF93, ETRF94, ETRF96, ETRF97, ETRF2000, ETRF2005 and ETRF2014. This &#x27;ensemble&#x27; covers any or all of these realizations without distinction.",
        information_source: "EPSG",
        data_source: "EPSG",
        revision_date: "2020-03-14",
        deprecated: 0, 
    },

    "EPSG:5110" => CRS {
        coord_ref_sys_code: 5110,
        coord_ref_sys_name: "ETRS89 / NTM zone 10",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5111" => CRS {
        coord_ref_sys_code: 5111,
        coord_ref_sys_name: "ETRS89 / NTM zone 11",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5112" => CRS {
        coord_ref_sys_code: 5112,
        coord_ref_sys_name: "ETRS89 / NTM zone 12",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5243" => CRS {
        coord_ref_sys_code: 5243,
        coord_ref_sys_name: "ETRS89 / LCC Germany (E-N)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "See ETRS89 / LCC Germany (N-E) (CRS code 4839) for similar CRS but with alternative axis order.",
        information_source: "Geodatenzentrum (GDZ) am Bundesamt fuer Kartographie und Geodaesie (BKG).",
        data_source: "EPSG",
        revision_date: "2020-03-14",
        deprecated: 0, 
    },

    "EPSG:5316" => CRS {
        coord_ref_sys_code: 5316,
        coord_ref_sys_name: "ETRS89 / Faroe TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Introduced in 2010. For topographic mapping see ETRS89 / Faroe Lambert (CRS code 3145). For cadastral survey see fk89 (CRS code 3173). In due course this CRS (5316) is expected to replace these other CRSs.",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2020-04-30",
        deprecated: 0, 
    },

    "EPSG:5318" => CRS {
        coord_ref_sys_code: 5318,
        coord_ref_sys_name: "ETRS89 / Faroe TM + FVR09 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Introduced in 2010.",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2020-04-30",
        deprecated: 0, 
    },

    "EPSG:5950" => CRS {
        coord_ref_sys_code: 5950,
        coord_ref_sys_name: "ETRS89 / NTM zone 10 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5951" => CRS {
        coord_ref_sys_code: 5951,
        coord_ref_sys_name: "ETRS89 / NTM zone 11 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5952" => CRS {
        coord_ref_sys_code: 5952,
        coord_ref_sys_name: "ETRS89 / NTM zone 12 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6069" => CRS {
        coord_ref_sys_code: 6069,
        coord_ref_sys_name: "ETRS89 / EPSG Arctic zone 2-22",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6070" => CRS {
        coord_ref_sys_code: 6070,
        coord_ref_sys_name: "ETRS89 / EPSG Arctic zone 3-11",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6071" => CRS {
        coord_ref_sys_code: 6071,
        coord_ref_sys_name: "ETRS89 / EPSG Arctic zone 4-26",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6072" => CRS {
        coord_ref_sys_code: 6072,
        coord_ref_sys_name: "ETRS89 / EPSG Arctic zone 4-28",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6073" => CRS {
        coord_ref_sys_code: 6073,
        coord_ref_sys_name: "ETRS89 / EPSG Arctic zone 5-11",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6074" => CRS {
        coord_ref_sys_code: 6074,
        coord_ref_sys_name: "ETRS89 / EPSG Arctic zone 5-13",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6125" => CRS {
        coord_ref_sys_code: 6125,
        coord_ref_sys_name: "ETRS89 / EPSG Arctic zone 5-47",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6150" => CRS {
        coord_ref_sys_code: 6150,
        coord_ref_sys_name: "ETRS89 / NTM zone 10 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 10 + NN2000 height (compound CRS code 5950).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6151" => CRS {
        coord_ref_sys_code: 6151,
        coord_ref_sys_name: "ETRS89 / NTM zone 11 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 11 + NN2000 height (compound CRS code 5951).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6152" => CRS {
        coord_ref_sys_code: 6152,
        coord_ref_sys_name: "ETRS89 / NTM zone 12 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 12 + NN2000 height (compound CRS code 5952).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6870" => CRS {
        coord_ref_sys_code: 6870,
        coord_ref_sys_name: "ETRS89 / Albania TM 2010",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Planimetric component of Albanian Geodetic Reference Framework (KRGJSH) 2010. ETRS89 / Albania LCC 2010 (CRS code 6962) used for mapping at scales of 1/500,000 and smaller.",
        information_source: "State Authority for Geospatial Information (ASIG).",
        data_source: "EPSG",
        revision_date: "2020-03-14",
        deprecated: 0, 
    },

    "EPSG:6962" => CRS {
        coord_ref_sys_code: 6962,
        coord_ref_sys_name: "ETRS89 / Albania LCC 2010",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / Albania TM 2010 (CRS code 6870) used for larger scales.",
        information_source: "State Authority for Geospatial Information (ASIG).",
        data_source: "EPSG",
        revision_date: "2020-03-14",
        deprecated: 0, 
    },

    "EPSG:9505" => CRS {
        coord_ref_sys_code: 9505,
        coord_ref_sys_name: "ETRS89 + Alicante height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:7418" => CRS {
        coord_ref_sys_code: 7418,
        coord_ref_sys_name: "ETRS89 / Kp2000 Jutland + DVR90 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:7419" => CRS {
        coord_ref_sys_code: 7419,
        coord_ref_sys_name: "ETRS89 / Kp2000 Zealand + DVR90 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:7420" => CRS {
        coord_ref_sys_code: 7420,
        coord_ref_sys_name: "ETRS89 / Kp2000 Bornholm + DVR90 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:7917" => CRS {
        coord_ref_sys_code: 7917,
        coord_ref_sys_name: "ETRF90",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1179, 
        base_crs_code: 7916,
        remarks: "Replaces ETRF89 (code 7915). Replaced by ETRF91 (code 7919).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7918" => CRS {
        coord_ref_sys_code: 7918,
        coord_ref_sys_name: "ETRF91",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1180, 
        base_crs_code: -1,
        remarks: "Replaces ETRF90 (code 7916). Replaced by ETRF92 (code 7920).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7919" => CRS {
        coord_ref_sys_code: 7919,
        coord_ref_sys_name: "ETRF91",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1180, 
        base_crs_code: 7918,
        remarks: "Replaces ETRF90 (code 7917). Replaced by ETRF92 (code 7921).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7920" => CRS {
        coord_ref_sys_code: 7920,
        coord_ref_sys_name: "ETRF92",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1181, 
        base_crs_code: -1,
        remarks: "Replaces ETRF91 (code 7918). Replaced by ETRF93 (code 7922).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7921" => CRS {
        coord_ref_sys_code: 7921,
        coord_ref_sys_name: "ETRF92",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1181, 
        base_crs_code: 7920,
        remarks: "Replaces ETRF91 (code 7919). Replaced by ETRF93 (code 7923).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7922" => CRS {
        coord_ref_sys_code: 7922,
        coord_ref_sys_name: "ETRF93",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1182, 
        base_crs_code: -1,
        remarks: "Replaces ETRF92 (code 7920). Replaced by ETRF94 (code 7924).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7923" => CRS {
        coord_ref_sys_code: 7923,
        coord_ref_sys_name: "ETRF93",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1182, 
        base_crs_code: 7922,
        remarks: "Replaces ETRF92 (code 7921). Replaced by ETRF94 (code 7925).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7924" => CRS {
        coord_ref_sys_code: 7924,
        coord_ref_sys_name: "ETRF94",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1183, 
        base_crs_code: -1,
        remarks: "Replaces ETRF93 (code 7922). Replaced by ETRF96 (code 7926).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7925" => CRS {
        coord_ref_sys_code: 7925,
        coord_ref_sys_name: "ETRF94",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1183, 
        base_crs_code: 7924,
        remarks: "Replaces ETRF93 (code 7923). Replaced by ETRF96 (code 7927).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7926" => CRS {
        coord_ref_sys_code: 7926,
        coord_ref_sys_name: "ETRF96",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1184, 
        base_crs_code: -1,
        remarks: "Replaces ETRF94 (code 7924). Replaced by ETRF97 (code 7928).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7927" => CRS {
        coord_ref_sys_code: 7927,
        coord_ref_sys_name: "ETRF96",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1184, 
        base_crs_code: 7926,
        remarks: "Replaces ETRF94 (code 7925). Replaced by ETRF97 (code 7929).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7928" => CRS {
        coord_ref_sys_code: 7928,
        coord_ref_sys_name: "ETRF97",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1185, 
        base_crs_code: -1,
        remarks: "Replaces ETRF96 (code 7926). Replaced by ETRF2000 (code 7930).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:7929" => CRS {
        coord_ref_sys_code: 7929,
        coord_ref_sys_name: "ETRF97",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1185, 
        base_crs_code: 7928,
        remarks: "Replaces ETRF96 (code 7927). Replaced by ETRF2000 (code 7931).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:8370" => CRS {
        coord_ref_sys_code: 8370,
        coord_ref_sys_name: "ETRS89 / Belgian Lambert 2008 + Ostend height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2020-03-14",
        deprecated: 0, 
    },

    "EPSG:8395" => CRS {
        coord_ref_sys_code: 8395,
        coord_ref_sys_name: "ETRS89 / Gauss-Kruger CM 9E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "",
        information_source: "Landesbetrieb Geoinformation und Vermessung Hamburg.",
        data_source: "EPSG",
        revision_date: "2020-03-14",
        deprecated: 0, 
    },

    "EPSG:9060" => CRS {
        coord_ref_sys_code: 9060,
        coord_ref_sys_name: "ETRF90",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1179, 
        base_crs_code: 7917,
        remarks: "Replaces ETRF89 (code 9059). Replaced by ETRF91 (code 9061).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-05-23",
        deprecated: 0, 
    },

    "EPSG:9061" => CRS {
        coord_ref_sys_code: 9061,
        coord_ref_sys_name: "ETRF91",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1180, 
        base_crs_code: 7919,
        remarks: "Replaces ETRF90 (code 9060). Replaced by ETRF92 (code 9062).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-05-23",
        deprecated: 0, 
    },

    "EPSG:9062" => CRS {
        coord_ref_sys_code: 9062,
        coord_ref_sys_name: "ETRF92",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1181, 
        base_crs_code: 7921,
        remarks: "Replaces ETRF91 (code 9061). Replaced by ETRF93 (code 9063).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-05-23",
        deprecated: 0, 
    },

    "EPSG:9063" => CRS {
        coord_ref_sys_code: 9063,
        coord_ref_sys_name: "ETRF93",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1182, 
        base_crs_code: 7923,
        remarks: "Replaces ETRF92 (code 9062). Replaced by ETRF94 (code 9064).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-05-23",
        deprecated: 0, 
    },

    "EPSG:9506" => CRS {
        coord_ref_sys_code: 9506,
        coord_ref_sys_name: "ETRS89 + Ceuta 2 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:9064" => CRS {
        coord_ref_sys_code: 9064,
        coord_ref_sys_name: "ETRF94",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1183, 
        base_crs_code: 7925,
        remarks: "Replaces ETRF93 (code 9063). Replaced by ETRF96 (code 9065).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-05-23",
        deprecated: 0, 
    },

    "EPSG:9065" => CRS {
        coord_ref_sys_code: 9065,
        coord_ref_sys_name: "ETRF96",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1184, 
        base_crs_code: 7927,
        remarks: "Replaces ETRF94 (code 9064). Replaced by ETRF97 (code 9066).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-05-23",
        deprecated: 0, 
    },

    "EPSG:9066" => CRS {
        coord_ref_sys_code: 9066,
        coord_ref_sys_name: "ETRF97",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1185, 
        base_crs_code: 7929,
        remarks: "Replaces ETRF96 (code 9065). Replaced by ETRF2000 (code 9067).",
        information_source: "IAG Subcommision for the European Reference Frame (EUREF, http://etrs89.ensg.ign.fr/) and Boucher and Altamimi &quot;Memo : Specifications for reference frame fixing in the analysis of a EUREF GPS campaign&quot; (http://etrs89.ensg.ign.fr/memo-V8.pdf)",
        data_source: "EPSG",
        revision_date: "2019-05-23",
        deprecated: 0, 
    },

    "EPSG:62586405" => CRS {
        coord_ref_sys_code: 62586405,
        coord_ref_sys_name: "ETRS89 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "See CRS code 4258 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62586413" => CRS {
        coord_ref_sys_code: 62586413,
        coord_ref_sys_name: "ETRS89 (3D deg)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6413,
        datum_code: -1, 
        base_crs_code: 4345,
        remarks: "See CRS code 4345 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:2213" => CRS {
        coord_ref_sys_code: 2213,
        coord_ref_sys_name: "ETRS89 / TM 30 NE",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3038" => CRS {
        coord_ref_sys_code: 3038,
        coord_ref_sys_name: "ETRS89 / TM26",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89-LCC (CRS code 3034) used for conformal mapping at 1:500,000 and smaller scales. ETRS89-LAEA (CRS code 3035) used for statistical applications at all scales.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:3039" => CRS {
        coord_ref_sys_code: 3039,
        coord_ref_sys_name: "ETRS89 / TM27",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89-LCC (CRS code 3034) used for conformal mapping at 1:500,000 and smaller scales. ETRS89-LAEA (CRS code 3035) used for statistical applications at all scales.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:3067" => CRS {
        coord_ref_sys_code: 3067,
        coord_ref_sys_name: "ETRS89 / TM35FIN(E,N)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Identical to ETRS89 / UTM zone 35N (code 25835) except for area of use. See ETRS89 / TM35FIN(N,E) (code 5048) for more usually used alternative with axis order reversed.",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3763" => CRS {
        coord_ref_sys_code: 3763,
        coord_ref_sys_name: "ETRS89 / Portugal TM06",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "",
        information_source: "Instituto Geográfico Português (IGP).",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3903" => CRS {
        coord_ref_sys_code: 3903,
        coord_ref_sys_name: "ETRS89 / TM35FIN(N,E) + N2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaces ETRS89 / TM35FIN(N,E) + N60 height (CRS code 3902).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-14",
        deprecated: 0, 
    },

    "EPSG:4855" => CRS {
        coord_ref_sys_code: 4855,
        coord_ref_sys_name: "ETRS89 / NTM zone 5",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4856" => CRS {
        coord_ref_sys_code: 4856,
        coord_ref_sys_name: "ETRS89 / NTM zone 6",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4857" => CRS {
        coord_ref_sys_code: 4857,
        coord_ref_sys_name: "ETRS89 / NTM zone 7",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4858" => CRS {
        coord_ref_sys_code: 4858,
        coord_ref_sys_name: "ETRS89 / NTM zone 8",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4859" => CRS {
        coord_ref_sys_code: 4859,
        coord_ref_sys_name: "ETRS89 / NTM zone 9",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4863" => CRS {
        coord_ref_sys_code: 4863,
        coord_ref_sys_name: "ETRS89 / NTM zone 13",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4864" => CRS {
        coord_ref_sys_code: 4864,
        coord_ref_sys_name: "ETRS89 / NTM zone 14",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4865" => CRS {
        coord_ref_sys_code: 4865,
        coord_ref_sys_name: "ETRS89 / NTM zone 15",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4866" => CRS {
        coord_ref_sys_code: 4866,
        coord_ref_sys_name: "ETRS89 / NTM zone 16",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4867" => CRS {
        coord_ref_sys_code: 4867,
        coord_ref_sys_name: "ETRS89 / NTM zone 17",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4868" => CRS {
        coord_ref_sys_code: 4868,
        coord_ref_sys_name: "ETRS89 / NTM zone 18",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4869" => CRS {
        coord_ref_sys_code: 4869,
        coord_ref_sys_name: "ETRS89 / NTM zone 19",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4870" => CRS {
        coord_ref_sys_code: 4870,
        coord_ref_sys_name: "ETRS89 / NTM zone 20",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4871" => CRS {
        coord_ref_sys_code: 4871,
        coord_ref_sys_name: "ETRS89 / NTM zone 21",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4872" => CRS {
        coord_ref_sys_code: 4872,
        coord_ref_sys_name: "ETRS89 / NTM zone 22",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4873" => CRS {
        coord_ref_sys_code: 4873,
        coord_ref_sys_name: "ETRS89 / NTM zone 23",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4874" => CRS {
        coord_ref_sys_code: 4874,
        coord_ref_sys_name: "ETRS89 / NTM zone 24",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4875" => CRS {
        coord_ref_sys_code: 4875,
        coord_ref_sys_name: "ETRS89 / NTM zone 25",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:5725" => CRS {
        coord_ref_sys_code: 5725,
        coord_ref_sys_name: "Fahud HD height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5124, 
        base_crs_code: -1,
        remarks: "Replaced by PHD93 height (CRS code 5724) from 1993.",
        information_source: "Petroleum Development Oman",
        data_source: "EPSG",
        revision_date: "2012-02-13",
        deprecated: 0, 
    },

    "EPSG:4876" => CRS {
        coord_ref_sys_code: 4876,
        coord_ref_sys_name: "ETRS89 / NTM zone 26",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4877" => CRS {
        coord_ref_sys_code: 4877,
        coord_ref_sys_name: "ETRS89 / NTM zone 27",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4878" => CRS {
        coord_ref_sys_code: 4878,
        coord_ref_sys_name: "ETRS89 / NTM zone 28",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4879" => CRS {
        coord_ref_sys_code: 4879,
        coord_ref_sys_name: "ETRS89 / NTM zone 29",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:4880" => CRS {
        coord_ref_sys_code: 4880,
        coord_ref_sys_name: "ETRS89 / NTM zone 30",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM used as official map projection. This is an official alternative for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:5048" => CRS {
        coord_ref_sys_code: 5048,
        coord_ref_sys_name: "ETRS89 / TM35FIN(N,E)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Identical to ETRS89 / UTM zone 35N (N,E) (code 3047) except for area of use. Used in preference to ETRS89 / TM35FIN(E,N) (CRS code 3067) in which coordinate axes are in order easting-northing.",
        information_source: "National Land Survey of Finland; http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5105" => CRS {
        coord_ref_sys_code: 5105,
        coord_ref_sys_name: "ETRS89 / NTM zone 5",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5106" => CRS {
        coord_ref_sys_code: 5106,
        coord_ref_sys_name: "ETRS89 / NTM zone 6",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5107" => CRS {
        coord_ref_sys_code: 5107,
        coord_ref_sys_name: "ETRS89 / NTM zone 7",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5108" => CRS {
        coord_ref_sys_code: 5108,
        coord_ref_sys_name: "ETRS89 / NTM zone 8",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5109" => CRS {
        coord_ref_sys_code: 5109,
        coord_ref_sys_name: "ETRS89 / NTM zone 9",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5113" => CRS {
        coord_ref_sys_code: 5113,
        coord_ref_sys_name: "ETRS89 / NTM zone 13",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5114" => CRS {
        coord_ref_sys_code: 5114,
        coord_ref_sys_name: "ETRS89 / NTM zone 14",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5115" => CRS {
        coord_ref_sys_code: 5115,
        coord_ref_sys_name: "ETRS89 / NTM zone 15",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5116" => CRS {
        coord_ref_sys_code: 5116,
        coord_ref_sys_name: "ETRS89 / NTM zone 16",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5117" => CRS {
        coord_ref_sys_code: 5117,
        coord_ref_sys_name: "ETRS89 / NTM zone 17",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5118" => CRS {
        coord_ref_sys_code: 5118,
        coord_ref_sys_name: "ETRS89 / NTM zone 18",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5119" => CRS {
        coord_ref_sys_code: 5119,
        coord_ref_sys_name: "ETRS89 / NTM zone 19",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5120" => CRS {
        coord_ref_sys_code: 5120,
        coord_ref_sys_name: "ETRS89 / NTM zone 20",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5121" => CRS {
        coord_ref_sys_code: 5121,
        coord_ref_sys_name: "ETRS89 / NTM zone 21",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5122" => CRS {
        coord_ref_sys_code: 5122,
        coord_ref_sys_name: "ETRS89 / NTM zone 22",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5123" => CRS {
        coord_ref_sys_code: 5123,
        coord_ref_sys_name: "ETRS89 / NTM zone 23",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5124" => CRS {
        coord_ref_sys_code: 5124,
        coord_ref_sys_name: "ETRS89 / NTM zone 24",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:9430" => CRS {
        coord_ref_sys_code: 9430,
        coord_ref_sys_name: "ETRS89 + St. Marys height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Ordnance Survey Ltd., Great Britain.",
        data_source: "EPSG",
        revision_date: "2020-07-01",
        deprecated: 0, 
    },

    "EPSG:5125" => CRS {
        coord_ref_sys_code: 5125,
        coord_ref_sys_name: "ETRS89 / NTM zone 25",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5126" => CRS {
        coord_ref_sys_code: 5126,
        coord_ref_sys_name: "ETRS89 / NTM zone 26",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5127" => CRS {
        coord_ref_sys_code: 5127,
        coord_ref_sys_name: "ETRS89 / NTM zone 27",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5128" => CRS {
        coord_ref_sys_code: 5128,
        coord_ref_sys_name: "ETRS89 / NTM zone 28",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5129" => CRS {
        coord_ref_sys_code: 5129,
        coord_ref_sys_name: "ETRS89 / NTM zone 29",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This CRS using NTM is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5130" => CRS {
        coord_ref_sys_code: 5130,
        coord_ref_sys_name: "ETRS89 / NTM zone 30",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89 / UTM is the projected CRS for Norway recommended by the Norwegian Mapping Authority. This is an alternative when UTM scale is too far from unity, in particular for construction projects.",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5945" => CRS {
        coord_ref_sys_code: 5945,
        coord_ref_sys_name: "ETRS89 / NTM zone 5 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5946" => CRS {
        coord_ref_sys_code: 5946,
        coord_ref_sys_name: "ETRS89 / NTM zone 6 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5947" => CRS {
        coord_ref_sys_code: 5947,
        coord_ref_sys_name: "ETRS89 / NTM zone 7 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5948" => CRS {
        coord_ref_sys_code: 5948,
        coord_ref_sys_name: "ETRS89 / NTM zone 8 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5949" => CRS {
        coord_ref_sys_code: 5949,
        coord_ref_sys_name: "ETRS89 / NTM zone 9 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5953" => CRS {
        coord_ref_sys_code: 5953,
        coord_ref_sys_name: "ETRS89 / NTM zone 13 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5954" => CRS {
        coord_ref_sys_code: 5954,
        coord_ref_sys_name: "ETRS89 / NTM zone 14 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5955" => CRS {
        coord_ref_sys_code: 5955,
        coord_ref_sys_name: "ETRS89 / NTM zone 15 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5956" => CRS {
        coord_ref_sys_code: 5956,
        coord_ref_sys_name: "ETRS89 / NTM zone 16 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5957" => CRS {
        coord_ref_sys_code: 5957,
        coord_ref_sys_name: "ETRS89 / NTM zone 17 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5958" => CRS {
        coord_ref_sys_code: 5958,
        coord_ref_sys_name: "ETRS89 / NTM zone 18 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5959" => CRS {
        coord_ref_sys_code: 5959,
        coord_ref_sys_name: "ETRS89 / NTM zone 19 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5960" => CRS {
        coord_ref_sys_code: 5960,
        coord_ref_sys_name: "ETRS89 / NTM zone 20 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5961" => CRS {
        coord_ref_sys_code: 5961,
        coord_ref_sys_name: "ETRS89 / NTM zone 21 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5962" => CRS {
        coord_ref_sys_code: 5962,
        coord_ref_sys_name: "ETRS89 / NTM zone 22 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5963" => CRS {
        coord_ref_sys_code: 5963,
        coord_ref_sys_name: "ETRS89 / NTM zone 23 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5964" => CRS {
        coord_ref_sys_code: 5964,
        coord_ref_sys_name: "ETRS89 / NTM zone 24 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5965" => CRS {
        coord_ref_sys_code: 5965,
        coord_ref_sys_name: "ETRS89 / NTM zone 25 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5966" => CRS {
        coord_ref_sys_code: 5966,
        coord_ref_sys_name: "ETRS89 / NTM zone 26 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5967" => CRS {
        coord_ref_sys_code: 5967,
        coord_ref_sys_name: "ETRS89 / NTM zone 27 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5968" => CRS {
        coord_ref_sys_code: 5968,
        coord_ref_sys_name: "ETRS89 / NTM zone 28 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5969" => CRS {
        coord_ref_sys_code: 5969,
        coord_ref_sys_name: "ETRS89 / NTM zone 29 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5970" => CRS {
        coord_ref_sys_code: 5970,
        coord_ref_sys_name: "ETRS89 / NTM zone 30 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6145" => CRS {
        coord_ref_sys_code: 6145,
        coord_ref_sys_name: "ETRS89 / NTM zone 5 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 5 + NN2000 height (compound CRS code 5945).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6146" => CRS {
        coord_ref_sys_code: 6146,
        coord_ref_sys_name: "ETRS89 / NTM zone 6 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 6 + NN2000 height (compound CRS code 5946).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6147" => CRS {
        coord_ref_sys_code: 6147,
        coord_ref_sys_name: "ETRS89 / NTM zone 7 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 7 + NN2000 height (compound CRS code 5947).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6148" => CRS {
        coord_ref_sys_code: 6148,
        coord_ref_sys_name: "ETRS89 / NTM zone 8 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 8 + NN2000 height (compound CRS code 5948).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6149" => CRS {
        coord_ref_sys_code: 6149,
        coord_ref_sys_name: "ETRS89 / NTM zone 9 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 9 + NN2000 height (compound CRS code 5949).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6153" => CRS {
        coord_ref_sys_code: 6153,
        coord_ref_sys_name: "ETRS89 / NTM zone 13 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 13 + NN2000 height (compound CRS code 5953).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6154" => CRS {
        coord_ref_sys_code: 6154,
        coord_ref_sys_name: "ETRS89 / NTM zone 14 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 14 + NN2000 height (compound CRS code 5954).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6155" => CRS {
        coord_ref_sys_code: 6155,
        coord_ref_sys_name: "ETRS89 / NTM zone 15 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 15 + NN2000 height (compound CRS code 5955).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6156" => CRS {
        coord_ref_sys_code: 6156,
        coord_ref_sys_name: "ETRS89 / NTM zone 16 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 16 + NN2000 height (compound CRS code 5956).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6157" => CRS {
        coord_ref_sys_code: 6157,
        coord_ref_sys_name: "ETRS89 / NTM zone 17 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 17 + NN2000 height (compound CRS code 5957).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6158" => CRS {
        coord_ref_sys_code: 6158,
        coord_ref_sys_name: "ETRS89 / NTM zone 18 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 18 + NN2000 height (compound CRS code 5958).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6159" => CRS {
        coord_ref_sys_code: 6159,
        coord_ref_sys_name: "ETRS89 / NTM zone 19 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 19 + NN2000 height (compound CRS code 5959).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6160" => CRS {
        coord_ref_sys_code: 6160,
        coord_ref_sys_name: "ETRS89 / NTM zone 20 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 20 + NN2000 height (compound CRS code 5960).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6161" => CRS {
        coord_ref_sys_code: 6161,
        coord_ref_sys_name: "ETRS89 / NTM zone 21 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 21 + NN2000 height (compound CRS code 5961).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6162" => CRS {
        coord_ref_sys_code: 6162,
        coord_ref_sys_name: "ETRS89 / NTM zone 22 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 22 + NN2000 height (compound CRS code 5962).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6163" => CRS {
        coord_ref_sys_code: 6163,
        coord_ref_sys_name: "ETRS89 / NTM zone 23 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 23 + NN2000 height (compound CRS code 5963).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6164" => CRS {
        coord_ref_sys_code: 6164,
        coord_ref_sys_name: "ETRS89 / NTM zone 24 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 24 + NN2000 height (compound CRS code 5964).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6165" => CRS {
        coord_ref_sys_code: 6165,
        coord_ref_sys_name: "ETRS89 / NTM zone 25 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 25 + NN2000 height (compound CRS code 5965).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6166" => CRS {
        coord_ref_sys_code: 6166,
        coord_ref_sys_name: "ETRS89 / NTM zone 26 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 26 + NN2000 height (compound CRS code 5966).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6167" => CRS {
        coord_ref_sys_code: 6167,
        coord_ref_sys_name: "ETRS89 / NTM zone 27 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 27 + NN2000 height (compound CRS code 5967).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6168" => CRS {
        coord_ref_sys_code: 6168,
        coord_ref_sys_name: "ETRS89 / NTM zone 28 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 28 + NN2000 height (compound CRS code 5968).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6169" => CRS {
        coord_ref_sys_code: 6169,
        coord_ref_sys_name: "ETRS89 / NTM zone 29 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 29 + NN2000 height (compound CRS code 5969).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6170" => CRS {
        coord_ref_sys_code: 6170,
        coord_ref_sys_name: "ETRS89 / NTM zone 30 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / NTM zone 30 + NN2000 height (compound CRS code 5970).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:25884" => CRS {
        coord_ref_sys_code: 25884,
        coord_ref_sys_name: "ETRS89 / TM Baltic93",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Used as a common CRS for the Baltic states and for medium and small scale mapping since 1993. For large scale applications see CRS codes 3300-01 (Estonia), 3059 (Latvia) and 2600 (Lituania).",
        information_source: "http://www.geo.ut.ee/",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3034" => CRS {
        coord_ref_sys_code: 3034,
        coord_ref_sys_name: "ETRS89-extended / LCC Europe",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Use formally limited to ETRS89 area of continental Europe but extended for pan-European analysis at small scales. ETRS89 / UTM used for conformal mapping at scales larger than 1:500,000. ETRS89 / LAEA used for statistical applications at any scale.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3035" => CRS {
        coord_ref_sys_code: 3035,
        coord_ref_sys_name: "ETRS89-extended / LAEA Europe",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4532,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Use ETRS89 / LCC for conformal mapping at 1:500,000 scale or smaller or ETRS89 / UTM for conformal mapping at scales larger than 1:500,000. Use formally limited to ETRS89 area of continental Europe but extended for pan-European analysis at small scales.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3040" => CRS {
        coord_ref_sys_code: 3040,
        coord_ref_sys_name: "ETRS89 / UTM zone 28N (N-E)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89-LCC (CRS code 3034) used for conformal mapping at 1:500,000 and smaller scales. ETRS89-LAEA (CRS code 3035) used for statistical applications at all scales. See ETRS89 / UTM zone 28N (CRS code 25828) for CRS with preferred east-north axis order.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3041" => CRS {
        coord_ref_sys_code: 3041,
        coord_ref_sys_name: "ETRS89 / UTM zone 29N (N-E)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89-LCC (CRS code 3034) used for conformal mapping at 1:500,000 and smaller scales. ETRS89-LAEA (CRS code 3035) used for statistical applications at all scales. See ETRS89 / UTM zone 29N (CRS code 25829) for CRS with preferred east-north axis order.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3042" => CRS {
        coord_ref_sys_code: 3042,
        coord_ref_sys_name: "ETRS89 / UTM zone 30N (N-E)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89-LCC (CRS code 3034) used for conformal mapping at 1:500,000 and smaller scales. ETRS89-LAEA (CRS code 3035) used for statistical applications at all scales. See ETRS89 / UTM zone 30N (CRS code 25830) for CRS with preferred east-north axis order.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3043" => CRS {
        coord_ref_sys_code: 3043,
        coord_ref_sys_name: "ETRS89 / UTM zone 31N (N-E)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89-LCC (CRS code 3034) used for conformal mapping at 1:500,000 and smaller scales. ETRS89-LAEA (CRS code 3035) used for statistical applications at all scales. See ETRS89 / UTM zone 31N (CRS code 25831) for CRS with preferred east-north axis order.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3044" => CRS {
        coord_ref_sys_code: 3044,
        coord_ref_sys_name: "ETRS89 / UTM zone 32N (N-E)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89-LCC (CRS code 3034) used for conformal mapping at 1:500,000 and smaller scales. ETRS89-LAEA (CRS code 3035) used for statistical applications at all scales. See ETRS89 / UTM zone 32N (CRS code 25832) for CRS with preferred east-north axis order.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:9449" => CRS {
        coord_ref_sys_code: 9449,
        coord_ref_sys_name: "ETRS89 + Malin Head height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Ordnance Survey Ireland (OSi) and Land and Property Services Northern Ireland (LPS).",
        data_source: "EPSG",
        revision_date: "2020-07-10",
        deprecated: 0, 
    },

    "EPSG:9450" => CRS {
        coord_ref_sys_code: 9450,
        coord_ref_sys_name: "ETRS89 + Belfast height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Land and Property Services Northern Ireland.",
        data_source: "EPSG",
        revision_date: "2020-07-10",
        deprecated: 0, 
    },

    "EPSG:3045" => CRS {
        coord_ref_sys_code: 3045,
        coord_ref_sys_name: "ETRS89 / UTM zone 33N (N-E)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89-LCC (CRS code 3034) used for conformal mapping at 1:500,000 and smaller scales. ETRS89-LAEA (CRS code 3035) used for statistical applications at all scales. See ETRS89 / UTM zone 33N (CRS code 25833) for CRS with preferred east-north axis order.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3046" => CRS {
        coord_ref_sys_code: 3046,
        coord_ref_sys_name: "ETRS89 / UTM zone 34N (N-E)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89-LCC (CRS code 3034) used for conformal mapping at 1:500,000 and smaller scales. ETRS89-LAEA (CRS code 3035) used for statistical applications at all scales. See ETRS89 / UTM zone 34N (CRS code 25834) for CRS with preferred east-north axis order.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3047" => CRS {
        coord_ref_sys_code: 3047,
        coord_ref_sys_name: "ETRS89 / UTM zone 35N (N-E)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89-LCC (CRS code 3034) used for conformal mapping at 1:500,000 and smaller scales. ETRS89-LAEA (CRS code 3035) used for statistical applications at all scales. See ETRS89 / UTM zone 35N (CRS code 25835) for CRS with preferred east-north axis order.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3048" => CRS {
        coord_ref_sys_code: 3048,
        coord_ref_sys_name: "ETRS89 / UTM zone 36N (N-E)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89-LCC (CRS code 3034) used for conformal mapping at 1:500,000 and smaller scales. ETRS89-LAEA (CRS code 3035) used for statistical applications at all scales. See ETRS89 / UTM zone 36N (CRS code 25836) for CRS with preferred east-north axis order.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3049" => CRS {
        coord_ref_sys_code: 3049,
        coord_ref_sys_name: "ETRS89 / UTM zone 37N (N-E)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89-LCC (CRS code 3034) used for conformal mapping at 1:500,000 and smaller scales. ETRS89-LAEA (CRS code 3035) used for statistical applications at all scales. See ETRS89 / UTM zone 37N (CRS code 25837) for CRS with preferred east-north axis order.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3050" => CRS {
        coord_ref_sys_code: 3050,
        coord_ref_sys_name: "ETRS89 / TM38",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89-LCC (CRS code 3034) used for conformal mapping at 1:500,000 and smaller scales. ETRS89-LAEA (CRS code 3035) used for statistical applications at all scales.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:3051" => CRS {
        coord_ref_sys_code: 3051,
        coord_ref_sys_name: "ETRS89 / TM39",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "ETRS89-LCC (CRS code 3034) used for conformal mapping at 1:500,000 and smaller scales. ETRS89-LAEA (CRS code 3035) used for statistical applications at all scales.",
        information_source: "European Commission Joint Research Centre &quot;Map Projections for Europe&quot;. http://www.ec-gis.org",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 1, 
    },

    "EPSG:3303" => CRS {
        coord_ref_sys_code: 3303,
        coord_ref_sys_name: "Fatu Iva 72 / UTM zone 7S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4688,
        remarks: "Replaced by RGPF / UTM zone 7S, CRS code 3298.",
        information_source: "Gouvernement de la Polynésie Française, Service de l&#x27;Urbanisme, Section topographie.",
        data_source: "EPSG",
        revision_date: "2005-08-12",
        deprecated: 0, 
    },

    "EPSG:3886" => CRS {
        coord_ref_sys_code: 3886,
        coord_ref_sys_name: "Fao 1979 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1028, 
        base_crs_code: -1,
        remarks: "Replaces Fao height (CRS code 5751) for national map agency work in Iraq. At time of record creation some irrigation project data still referenced to Fao. Usage in oil industry is uncertain.",
        information_source: "Survey Department, Ministry of Water Resources (MoWR).",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 0, 
    },

    "EPSG:3902" => CRS {
        coord_ref_sys_code: 3902,
        coord_ref_sys_name: "ETRS89 / TM35FIN(N,E) + N60 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaces YKJ/N60 (CRS code 3901). Replaced by ETRS89-TM35FIN(N,E)/N2000 (CRS code 3903).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2020-03-14",
        deprecated: 0, 
    },

    "EPSG:4232" => CRS {
        coord_ref_sys_code: 4232,
        coord_ref_sys_name: "Fahud",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6232, 
        base_crs_code: -1,
        remarks: "Since 1993 replaced by PSD93 geogCRS (code 4134). Maximum differences to Fahud adjustment are 20 metres.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-02-13",
        deprecated: 0, 
    },

    "EPSG:4647" => CRS {
        coord_ref_sys_code: 4647,
        coord_ref_sys_name: "ETRS89 / UTM zone 32N (zE-N)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Variant of ETRS89 / UTM zone 32N (CRS code 25832) in which easting has zone prefix.",
        information_source: "Landesvermessungsamt Schleswig-Holstein.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:4688" => CRS {
        coord_ref_sys_code: 4688,
        coord_ref_sys_name: "Fatu Iva 72",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6688, 
        base_crs_code: -1,
        remarks: "Recomputed by IGN in 1972 using origin and observations of 1953-1955 Mission Hydrographique des Etablissement Francais d&#x27;Oceanie (MHEFO 55). Replaced by RGPF, CRS code 4687.",
        information_source: "Gouvernement de la Polynésie Française, Service de l&#x27;Urbanisme, Section topographie.",
        data_source: "EPSG",
        revision_date: "2005-08-12",
        deprecated: 0, 
    },

    "EPSG:4741" => CRS {
        coord_ref_sys_code: 4741,
        coord_ref_sys_name: "FD54",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6741, 
        base_crs_code: -1,
        remarks: "Except for cadastral survey, replaced by ED50 in the late 1970&#x27;s. For cadastral survey, replaced by fk89 (CRS code 4753).",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-03-16",
        deprecated: 0, 
    },

    "EPSG:5554" => CRS {
        coord_ref_sys_code: 5554,
        coord_ref_sys_name: "ETRS89 / UTM zone 31N + DHHN92 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Bundesanstalt für Geowissenschaften und Rohstoffe (BGR), Hannover.",
        data_source: "EPSG",
        revision_date: "2020-03-14",
        deprecated: 0, 
    },

    "EPSG:5555" => CRS {
        coord_ref_sys_code: 5555,
        coord_ref_sys_name: "ETRS89 / UTM zone 32N + DHHN92 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Bundesanstalt für Geowissenschaften und Rohstoffe (BGR), Hannover.",
        data_source: "EPSG",
        revision_date: "2020-03-14",
        deprecated: 0, 
    },

    "EPSG:5556" => CRS {
        coord_ref_sys_code: 5556,
        coord_ref_sys_name: "ETRS89 / UTM zone 33N + DHHN92 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Bundesanstalt für Geowissenschaften und Rohstoffe (BGR), Hannover.",
        data_source: "EPSG",
        revision_date: "2020-03-14",
        deprecated: 0, 
    },

    "EPSG:5597" => CRS {
        coord_ref_sys_code: 5597,
        coord_ref_sys_name: "FCSVR10 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1079, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Femern A/S",
        data_source: "EPSG",
        revision_date: "2011-09-24",
        deprecated: 0, 
    },

    "EPSG:5621" => CRS {
        coord_ref_sys_code: 5621,
        coord_ref_sys_name: "EVRF2007 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5215, 
        base_crs_code: -1,
        remarks: "Uses Normal heights. Replaces EVRF2000 height (CRS code 5730). Replaced by EVRF2019 height (CRS code 9389).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2020-07-24",
        deprecated: 0, 
    },

    "EPSG:5649" => CRS {
        coord_ref_sys_code: 5649,
        coord_ref_sys_name: "ETRS89 / UTM zone 31N (zE-N)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Variant of ETRS89 / UTM zone 31N (CRS code 25831) in which easting has zone prefix.",
        information_source: "Landesamt für innere Verwaltung Mecklenburg-Vorpommern.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5650" => CRS {
        coord_ref_sys_code: 5650,
        coord_ref_sys_name: "ETRS89 / UTM zone 33N (zE-N)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Variant of ETRS89 / UTM zone 33N (CRS code 25833) in which easting has zone prefix.",
        information_source: "Landesamt für innere Verwaltung Mecklenburg-Vorpommern.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5651" => CRS {
        coord_ref_sys_code: 5651,
        coord_ref_sys_name: "ETRS89 / UTM zone 31N (N-zE)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Variant of ETRS89 / UTM zone 31N (CRS code 25831) in which axis order is reversed and easting has zone prefix.",
        information_source: "Landesamt für innere Verwaltung Mecklenburg-Vorpommern.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5652" => CRS {
        coord_ref_sys_code: 5652,
        coord_ref_sys_name: "ETRS89 / UTM zone 32N (N-zE)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Variant of ETRS89 / UTM zone 32N (CRS code 25832) in which axis order is reversed and easting has zone prefix.",
        information_source: "Landesamt für innere Verwaltung Mecklenburg-Vorpommern.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5653" => CRS {
        coord_ref_sys_code: 5653,
        coord_ref_sys_name: "ETRS89 / UTM zone 33N (N-zE)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "Variant of ETRS89 / UTM zone 33N (CRS code 25833) in which axis order is reversed and easting has zone prefix.",
        information_source: "Landesamt für innere Verwaltung Mecklenburg-Vorpommern.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5730" => CRS {
        coord_ref_sys_code: 5730,
        coord_ref_sys_name: "EVRF2000 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5129, 
        base_crs_code: -1,
        remarks: "Uses Normal heights. Replaced by EVRF2007 height (CRS code 5621). In Austria, orthometric heights used instead - see CRS code 9274.",
        information_source: "IAG subcommission for Europe (EUREF). Also at EuroGeographics, http://crs.bkg.bund.de/crs-eu/",
        data_source: "EPSG",
        revision_date: "2020-01-16",
        deprecated: 0, 
    },

    "EPSG:5741" => CRS {
        coord_ref_sys_code: 5741,
        coord_ref_sys_name: "Fair Isle height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5139, 
        base_crs_code: -1,
        remarks: "Replaced by ODN (Offshore) height (CRS code 7707) in 2016.",
        information_source: "Ordnance Survey of Great Britain.",
        data_source: "EPSG",
        revision_date: "2020-06-17",
        deprecated: 0, 
    },

    "EPSG:5751" => CRS {
        coord_ref_sys_code: 5751,
        coord_ref_sys_name: "Fao height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5149, 
        base_crs_code: -1,
        remarks: "Replaced by Bandar Abbas (CRS code 5752) in Iran and Fao 1979 (code 3886) in Iraq. At time of record creation NIOC data in Ahwaz area of Iran and some irrigation project data in Iraq still usually referenced to Fao. Usage in Iraqi oil industry uncertain.",
        information_source: "National Iranian Oil Company (NIOC).",
        data_source: "EPSG",
        revision_date: "2010-03-01",
        deprecated: 0, 
    },

    "EPSG:5942" => CRS {
        coord_ref_sys_code: 5942,
        coord_ref_sys_name: "ETRS89 + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5971" => CRS {
        coord_ref_sys_code: 5971,
        coord_ref_sys_name: "ETRS89 / UTM zone 31N + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5972" => CRS {
        coord_ref_sys_code: 5972,
        coord_ref_sys_name: "ETRS89 / UTM zone 32N + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5973" => CRS {
        coord_ref_sys_code: 5973,
        coord_ref_sys_name: "ETRS89 / UTM zone 33N + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5974" => CRS {
        coord_ref_sys_code: 5974,
        coord_ref_sys_name: "ETRS89 / UTM zone 34N + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5975" => CRS {
        coord_ref_sys_code: 5975,
        coord_ref_sys_name: "ETRS89 / UTM zone 35N + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:5976" => CRS {
        coord_ref_sys_code: 5976,
        coord_ref_sys_name: "ETRS89 / UTM zone 36N + NN2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6144" => CRS {
        coord_ref_sys_code: 6144,
        coord_ref_sys_name: "ETRS89 + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 + NN2000 height (compound CRS code 5942).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6171" => CRS {
        coord_ref_sys_code: 6171,
        coord_ref_sys_name: "ETRS89 / UTM zone 31N + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / UTM zone 31N + NN2000 height (compound CRS code 5971).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6172" => CRS {
        coord_ref_sys_code: 6172,
        coord_ref_sys_name: "ETRS89 / UTM zone 32N + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / UTM zone 32N + NN2000 height (compound CRS code 5972).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6173" => CRS {
        coord_ref_sys_code: 6173,
        coord_ref_sys_name: "ETRS89 / UTM zone 33N + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / UTM zone 33N + NN2000 height (compound CRS code 5973).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6174" => CRS {
        coord_ref_sys_code: 6174,
        coord_ref_sys_name: "ETRS89 / UTM zone 34N + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / UTM zone 34N + NN2000 height (compound CRS code 5974).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6175" => CRS {
        coord_ref_sys_code: 6175,
        coord_ref_sys_name: "ETRS89 / UTM zone 35N + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / UTM zone 35N + NN2000 height (compound CRS code 5975).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:6176" => CRS {
        coord_ref_sys_code: 6176,
        coord_ref_sys_name: "ETRS89 / UTM zone 36N + NN54 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / UTM zone 36N + NN2000 height (compound CRS code 5976).",
        information_source: "Norwegian Mapping Authority.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:7409" => CRS {
        coord_ref_sys_code: 7409,
        coord_ref_sys_name: "ETRS89 + EVRF2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 + EVRF2007 height (CRS code 7423).",
        information_source: "IAG subcommission for Europe",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:7416" => CRS {
        coord_ref_sys_code: 7416,
        coord_ref_sys_name: "ETRS89 / UTM zone 32N + DVR90 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:7417" => CRS {
        coord_ref_sys_code: 7417,
        coord_ref_sys_name: "ETRS89 / UTM zone 33N + DVR90 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:7423" => CRS {
        coord_ref_sys_code: 7423,
        coord_ref_sys_name: "ETRS89 + EVRF2007 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaces ETRS89 + EVRF2000 height (CRS code 7409).",
        information_source: "IAG subcommission for Europe",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:7446" => CRS {
        coord_ref_sys_code: 7446,
        coord_ref_sys_name: "Famagusta 1960 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1148, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Cyprus Department of Lands and Surveys (DLS).",
        data_source: "EPSG",
        revision_date: "2015-11-03",
        deprecated: 0, 
    },

    "EPSG:8360" => CRS {
        coord_ref_sys_code: 8360,
        coord_ref_sys_name: "ETRS89 + Baltic 1957 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Land Survey Office (ZU), Prague; and Geodesy, Cartography and Cadastre Authority of Slovak Republic (UGKK SR).",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:9274" => CRS {
        coord_ref_sys_code: 9274,
        coord_ref_sys_name: "EVRF2000 Austria height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1261, 
        base_crs_code: -1,
        remarks: "Austria-specific version of EVRF using orthometric heights instead of the Normal heights used in EVRF2000 (CRS code 5730). Used for scientific purposes. See GHA height (CRS code 5778) for cadastral and other land survey purposes.",
        information_source: "Bundesamt für Eich- und Vermessungswesen (BEV).",
        data_source: "EPSG",
        revision_date: "2019-12-27",
        deprecated: 0, 
    },

    "EPSG:9286" => CRS {
        coord_ref_sys_code: 9286,
        coord_ref_sys_name: "ETRS89 + NAP height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2020-02-19",
        deprecated: 0, 
    },

    "EPSG:9289" => CRS {
        coord_ref_sys_code: 9289,
        coord_ref_sys_name: "ETRS89 + LAT NL depth",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2020-08-21",
        deprecated: 0, 
    },

    "EPSG:9290" => CRS {
        coord_ref_sys_code: 9290,
        coord_ref_sys_name: "ETRS89 + MSL NL depth",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2020-08-21",
        deprecated: 0, 
    },

    "EPSG:9389" => CRS {
        coord_ref_sys_code: 9389,
        coord_ref_sys_name: "EVRF2019 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1274, 
        base_crs_code: -1,
        remarks: "September 2020 adjustment. Replaces 2019 adjustment and EVRF2007 height (CRS code 5621). Uses Normal heights. Zero-tide solution. See EVRF2019 mean-tide height (CRS code 9390) for solution more appropriate for oceanography and GNSS-related activities.",
        information_source: "European vertical data centre at Bundesamt für Kartographie und Geodäsie (BKG), Leipzig branch. https://evrs.bkg.bund.de/Subsites/EVRS",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:9390" => CRS {
        coord_ref_sys_code: 9390,
        coord_ref_sys_name: "EVRF2019 mean-tide height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1287, 
        base_crs_code: -1,
        remarks: "September 2020 adjustment. Replaces 2019 adjustment. Uses Normal heights. Mean-tide solution. See EVRF2019 height (CRS code 9389) for zero-tide solution more appropriate for gravity-related activities.",
        information_source: "European vertical data centre at Bundesamt für Kartographie und Geodäsie (BKG), Leipzig branch. https://evrs.bkg.bund.de/Subsites/EVRS",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:9422" => CRS {
        coord_ref_sys_code: 9422,
        coord_ref_sys_name: "ETRS89 + EVRF2019 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaces ETRS89 + EVRF2007 height (CRS code 7423). See also ETRS89 + EVRF2019 mean-tide height (CRS code 9423).",
        information_source: "IAG subcommission for Europe.",
        data_source: "EPSG",
        revision_date: "2020-07-04",
        deprecated: 0, 
    },

    "EPSG:9423" => CRS {
        coord_ref_sys_code: 9423,
        coord_ref_sys_name: "ETRS89 + EVRF2019 mean-tide height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "See also ETRS89 + EVRF2019 (CRS code 9422).",
        information_source: "IAG subcommission for Europe.",
        data_source: "EPSG",
        revision_date: "2020-07-04",
        deprecated: 0, 
    },

    "EPSG:9424" => CRS {
        coord_ref_sys_code: 9424,
        coord_ref_sys_name: "ETRS89 + ODN height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Ordnance Survey Ltd., Great Britain.",
        data_source: "EPSG",
        revision_date: "2020-07-01",
        deprecated: 0, 
    },

    "EPSG:9425" => CRS {
        coord_ref_sys_code: 9425,
        coord_ref_sys_name: "ETRS89 + ODN (Offshore) height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Ordnance Survey Ltd., Great Britain.",
        data_source: "EPSG",
        revision_date: "2020-07-01",
        deprecated: 0, 
    },

    "EPSG:9426" => CRS {
        coord_ref_sys_code: 9426,
        coord_ref_sys_name: "ETRS89 + ODN Orkney height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Ordnance Survey Ltd., Great Britain.",
        data_source: "EPSG",
        revision_date: "2020-07-01",
        deprecated: 0, 
    },

    "EPSG:9427" => CRS {
        coord_ref_sys_code: 9427,
        coord_ref_sys_name: "ETRS89 + Lerwick height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Ordnance Survey Ltd., Great Britain.",
        data_source: "EPSG",
        revision_date: "2020-07-01",
        deprecated: 0, 
    },

    "EPSG:9428" => CRS {
        coord_ref_sys_code: 9428,
        coord_ref_sys_name: "ETRS89 + Stornoway height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Ordnance Survey Ltd., Great Britain.",
        data_source: "EPSG",
        revision_date: "2020-07-01",
        deprecated: 0, 
    },

    "EPSG:9429" => CRS {
        coord_ref_sys_code: 9429,
        coord_ref_sys_name: "ETRS89 + Douglas height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Ordnance Survey Ltd., Great Britain.",
        data_source: "EPSG",
        revision_date: "2020-07-01",
        deprecated: 0, 
    },

    "EPSG:9507" => CRS {
        coord_ref_sys_code: 9507,
        coord_ref_sys_name: "ETRS89 + Ibiza height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:9508" => CRS {
        coord_ref_sys_code: 9508,
        coord_ref_sys_name: "ETRS89 + Mallorca height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:9509" => CRS {
        coord_ref_sys_code: 9509,
        coord_ref_sys_name: "ETRS89 + Menorca height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:9651" => CRS {
        coord_ref_sys_code: 9651,
        coord_ref_sys_name: "EVRF2007-PL height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1297, 
        base_crs_code: -1,
        remarks: "Replaces Baltic 1986 height (CRS 9650) after 2023-12-31.",
        information_source: "Glowny Urzad Geodezji i Kartografii (GUGiK), http://www.gugik.gov.pl",
        data_source: "EPSG",
        revision_date: "2021-03-18",
        deprecated: 0, 
    },

    "EPSG:9723" => CRS {
        coord_ref_sys_code: 9723,
        coord_ref_sys_name: "ETRS89 + Genoa 1942 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:9724" => CRS {
        coord_ref_sys_code: 9724,
        coord_ref_sys_name: "ETRS89 + Catania 1965 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:9725" => CRS {
        coord_ref_sys_code: 9725,
        coord_ref_sys_name: "ETRS89 + Cagliari 1956 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:9763" => CRS {
        coord_ref_sys_code: 9763,
        coord_ref_sys_name: "EWR2-IRF",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1311, 
        base_crs_code: -1,
        remarks: "Intermediate CRS created in 2021 to assist the emulation of the ETRS89 / EWR2 SnakeGrid projected CRS through transformation ETRS89 to EWR2-IRF (1) (code 9764) used in conjunction with the EWR2-TM map projection (code 9765).",
        information_source: "Network Rail.",
        data_source: "EPSG",
        revision_date: "2021-07-16",
        deprecated: 0, 
    },

    "EPSG:9766" => CRS {
        coord_ref_sys_code: 9766,
        coord_ref_sys_name: "EWR2 Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 9763,
        remarks: "The CRS&#x27;s definition through transformation ETRS89 to EWR2-IRF (1) (code 9764) and map projection EWR2-TM (code 9765) emulates the EWR2 Snake projection applied to ETRS89 as realised through OSNet 2009 CORS.",
        information_source: "Network Rail.",
        data_source: "EPSG",
        revision_date: "2021-07-16",
        deprecated: 0, 
    },

    "EPSG:9767" => CRS {
        coord_ref_sys_code: 9767,
        coord_ref_sys_name: "EWR2 Grid + ODN height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Network Rail.",
        data_source: "EPSG",
        revision_date: "2021-07-16",
        deprecated: 0, 
    },

    "EPSG:23239" => CRS {
        coord_ref_sys_code: 23239,
        coord_ref_sys_name: "Fahud / UTM zone 39N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4232,
        remarks: "Since 1993 replaced by PSD93 / UTM zone 39N projCRS (code 3439). Maximum differences to Fahud adjustment are 20 metres.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-04-07",
        deprecated: 0, 
    },

    "EPSG:23240" => CRS {
        coord_ref_sys_code: 23240,
        coord_ref_sys_name: "Fahud / UTM zone 40N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4232,
        remarks: "Since 1993 replaced by PSD93 / UTM zone 40N projCRS (code 3440). Maximum differences to Fahud adjustment are 20 metres.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-02-13",
        deprecated: 0, 
    },

    "EPSG:25828" => CRS {
        coord_ref_sys_code: 25828,
        coord_ref_sys_name: "ETRS89 / UTM zone 28N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "The distinction in usage between ETRF89 and ETRS89 is confused: although in principle conceptually different in practice both are used as synonyms.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:25829" => CRS {
        coord_ref_sys_code: 25829,
        coord_ref_sys_name: "ETRS89 / UTM zone 29N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "The distinction in usage between ETRF89 and ETRS89 is confused: although in principle conceptually different in practice both are used as synonyms.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:25830" => CRS {
        coord_ref_sys_code: 25830,
        coord_ref_sys_name: "ETRS89 / UTM zone 30N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "The distinction in usage between ETRF89 and ETRS89 is confused: although in principle conceptually different in practice both are used as synonyms.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:25831" => CRS {
        coord_ref_sys_code: 25831,
        coord_ref_sys_name: "ETRS89 / UTM zone 31N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "The distinction in usage between ETRF89 and ETRS89 is confused: although in principle conceptually different in practice both are used as synonyms.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:25832" => CRS {
        coord_ref_sys_code: 25832,
        coord_ref_sys_name: "ETRS89 / UTM zone 32N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "The distinction in usage between ETRF89 and ETRS89 is confused: although in principle conceptually different in practice both are used as synonyms.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:25833" => CRS {
        coord_ref_sys_code: 25833,
        coord_ref_sys_name: "ETRS89 / UTM zone 33N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "The distinction in usage between ETRF89 and ETRS89 is confused: although in principle conceptually different in practice both are used as synonyms.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:25834" => CRS {
        coord_ref_sys_code: 25834,
        coord_ref_sys_name: "ETRS89 / UTM zone 34N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "The distinction in usage between ETRF89 and ETRS89 is confused: although in principle conceptually different in practice both are used as synonyms.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:25835" => CRS {
        coord_ref_sys_code: 25835,
        coord_ref_sys_name: "ETRS89 / UTM zone 35N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "The distinction in usage between ETRF89 and ETRS89 is confused: although in principle conceptually different in practice both are used as synonyms.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:25836" => CRS {
        coord_ref_sys_code: 25836,
        coord_ref_sys_name: "ETRS89 / UTM zone 36N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "The distinction in usage between ETRF89 and ETRS89 is confused: although in principle conceptually different in practice both are used as synonyms.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:25837" => CRS {
        coord_ref_sys_code: 25837,
        coord_ref_sys_name: "ETRS89 / UTM zone 37N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "The distinction in usage between ETRF89 and ETRS89 is confused: although in principle conceptually different in practice both are used as synonyms.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:25838" => CRS {
        coord_ref_sys_code: 25838,
        coord_ref_sys_name: "ETRS89 / UTM zone 38N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4258,
        remarks: "The distinction in usage between ETRF89 and ETRS89 is confused: although in principle conceptually different in practice both are used as synonyms.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2000-10-19",
        deprecated: 1, 
    },

    "EPSG:62326405" => CRS {
        coord_ref_sys_code: 62326405,
        coord_ref_sys_name: "Fahud (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4232,
        remarks: "See CRS code 4232 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:2312" => CRS {
        coord_ref_sys_code: 2312,
        coord_ref_sys_name: "Garoua / UTM zone 33N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4197,
        remarks: "",
        information_source: "TotalFinaElf",
        data_source: "EPSG",
        revision_date: "2002-02-12",
        deprecated: 0, 
    },

    "EPSG:2969" => CRS {
        coord_ref_sys_code: 2969,
        coord_ref_sys_name: "Fort Marigot / UTM zone 20N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4621,
        remarks: "Replaced by RRAF 1991 / UTM zone 20N (CRS code 4559).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2009-11-24",
        deprecated: 0, 
    },

    "EPSG:3112" => CRS {
        coord_ref_sys_code: 3112,
        coord_ref_sys_name: "GDA94 / Geoscience Australia Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "See also WGS 84 / ACRESLC (CRS code 4462).",
        information_source: "Geoscience Australia; http://www.ga.gov.au/map/broker/wms_info.php",
        data_source: "EPSG",
        revision_date: "2016-09-29",
        deprecated: 0, 
    },

    "EPSG:3113" => CRS {
        coord_ref_sys_code: 3113,
        coord_ref_sys_name: "GDA94 / BCSG02",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "Brisbane City Council",
        data_source: "EPSG",
        revision_date: "2005-02-21",
        deprecated: 0, 
    },

    "EPSG:3141" => CRS {
        coord_ref_sys_code: 3141,
        coord_ref_sys_name: "Fiji 1956 / UTM zone 60S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4721,
        remarks: "For topographic mapping, replaces Vanua Levu / Vanua Levu Grid (CRS code 3140). Replaced by Fiji 1986 / Fiji Map Grid (CRS code 3460).",
        information_source: "Clifford J. Mugnier in Photogrammetric Engineering and Remote Sensing, October 2000, www.asprs.org.",
        data_source: "EPSG",
        revision_date: "2007-02-06",
        deprecated: 0, 
    },

    "EPSG:3142" => CRS {
        coord_ref_sys_code: 3142,
        coord_ref_sys_name: "Fiji 1956 / UTM zone 1S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4721,
        remarks: "For topographic mapping, replaces Viti Levu / Viti Levu Grid (CRS code 3139) and Vanua Levu / Vanua Levu Grid (CRS code 3140). Replaced by Fiji 1986 / Fiji Map Grid (CRS code 3460).",
        information_source: "Clifford J. Mugnier in Photogrammetric Engineering and Remote Sensing, October 2000, www.asprs.org.",
        data_source: "EPSG",
        revision_date: "2007-02-06",
        deprecated: 0, 
    },

    "EPSG:3143" => CRS {
        coord_ref_sys_code: 3143,
        coord_ref_sys_name: "Fiji 1986 / Fiji Map Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4720,
        remarks: "Supersedes Viti Levu 1912 / Viti Levu Grid, Vanua Levu 1915 / Vanua Levu Grid, Fiji 1956 / UTM zone 60S and Fiji 1956 / UTM zone 1S (CRS codes 3139-42).",
        information_source: "Department of Lands and Survey, Fiji.",
        data_source: "EPSG",
        revision_date: "2006-07-19",
        deprecated: 1, 
    },

    "EPSG:3144" => CRS {
        coord_ref_sys_code: 3144,
        coord_ref_sys_name: "FD54 / Faroe Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 1031,
        datum_code: -1, 
        base_crs_code: 4741,
        remarks: "Historically also found with coordinate system axis abbreviations N/E (CS code 4501); second axis has abbreviation E but is positive to the west. Replaced by fk89 (CRS code 3173).",
        information_source: "KMS",
        data_source: "EPSG",
        revision_date: "2012-02-13",
        deprecated: 0, 
    },

    "EPSG:3173" => CRS {
        coord_ref_sys_code: 3173,
        coord_ref_sys_name: "fk89 / Faroe Lambert FK89",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 1031,
        datum_code: -1, 
        base_crs_code: 4753,
        remarks: "Replaces FD54 / Faroe Lambert (fk54) (CRS code 3144) for cadastral survey.",
        information_source: "KMS",
        data_source: "EPSG",
        revision_date: "2012-07-19",
        deprecated: 0, 
    },

    "EPSG:3200" => CRS {
        coord_ref_sys_code: 3200,
        coord_ref_sys_name: "FD58 / Iraq zone",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4132,
        remarks: "",
        information_source: "IOEPC records.",
        data_source: "EPSG",
        revision_date: "1998-11-11",
        deprecated: 0, 
    },

    "EPSG:3829" => CRS {
        coord_ref_sys_code: 3829,
        coord_ref_sys_name: "Hu Tzu Shan 1950 / UTM zone 51N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4236,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2010-03-30",
        deprecated: 0, 
    },

    "EPSG:3374" => CRS {
        coord_ref_sys_code: 3374,
        coord_ref_sys_name: "FD54 / UTM zone 29N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4741,
        remarks: "Replaced by ED50 / UTM zone 29N in late 1970&#x27;s.",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-08-04",
        deprecated: 0, 
    },

    "EPSG:3460" => CRS {
        coord_ref_sys_code: 3460,
        coord_ref_sys_name: "Fiji 1986 / Fiji Map Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4720,
        remarks: "Replaces Viti Levu 1912 / Viti Levu Grid, Vanua Levu 1915 / Vanua Levu Grid, Fiji 1956 / UTM zone 60S and Fiji 1956 / UTM zone 1S (CRS codes 3139-42).",
        information_source: "Department of Lands and Survey, Fiji.",
        data_source: "EPSG",
        revision_date: "2007-02-06",
        deprecated: 0, 
    },

    "EPSG:3577" => CRS {
        coord_ref_sys_code: 3577,
        coord_ref_sys_name: "GDA94 / Australian Albers",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "Australian Government Department of Agriculture, Fisheries and Forestry, Bureau of Rural Sciences. http://www.daff.gov.au/brs",
        data_source: "EPSG",
        revision_date: "2007-01-22",
        deprecated: 0, 
    },

    "EPSG:4132" => CRS {
        coord_ref_sys_code: 4132,
        coord_ref_sys_name: "FD58",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6132, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOEPC records.",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4197" => CRS {
        coord_ref_sys_code: 4197,
        coord_ref_sys_name: "Garoua",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6197, 
        base_crs_code: -1,
        remarks: "",
        information_source: "TotalFinaElf",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4233" => CRS {
        coord_ref_sys_code: 4233,
        coord_ref_sys_name: "Gandajika 1970",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6233, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 1, 
    },

    "EPSG:4234" => CRS {
        coord_ref_sys_code: 4234,
        coord_ref_sys_name: "Garoua",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6402,
        datum_code: 6234, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:4283" => CRS {
        coord_ref_sys_code: 4283,
        coord_ref_sys_name: "GDA94",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6283, 
        base_crs_code: 4939,
        remarks: "",
        information_source: "EPSG. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2019-09-17",
        deprecated: 0, 
    },

    "EPSG:4347" => CRS {
        coord_ref_sys_code: 4347,
        coord_ref_sys_name: "GDA94 (3D)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6283, 
        base_crs_code: -1,
        remarks: "Horizontal coordinates referenced to this CRS are in degrees. Any degree representation (e.g. DMSH, decimal, etc.) may be used but that used must be declared for the user.",
        information_source: "Australian Surveying and Land Information Group Internet WWW page.",
        data_source: "EPSG",
        revision_date: "2003-12-31",
        deprecated: 1, 
    },

    "EPSG:4348" => CRS {
        coord_ref_sys_code: 4348,
        coord_ref_sys_name: "GDA94 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6283, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Australian Surveying and Land Information Group Internet WWW page.",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:4621" => CRS {
        coord_ref_sys_code: 4621,
        coord_ref_sys_name: "Fort Marigot",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6621, 
        base_crs_code: -1,
        remarks: "Replaced by RRAF 1991 (CRS code 4558).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2009-11-24",
        deprecated: 0, 
    },

    "EPSG:4684" => CRS {
        coord_ref_sys_code: 4684,
        coord_ref_sys_name: "Gan 1970",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6684, 
        base_crs_code: -1,
        remarks: "In some references incorrectly named &quot;Gandajika 1970&quot;. See CRS &quot;Gandajika&quot;, code 4685, from the Democratic Republic of the Congo (Zaire).",
        information_source: "Various industry sources",
        data_source: "EPSG",
        revision_date: "2005-04-14",
        deprecated: 0, 
    },

    "EPSG:4685" => CRS {
        coord_ref_sys_code: 4685,
        coord_ref_sys_name: "Gandajika",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6685, 
        base_crs_code: -1,
        remarks: "In some references incorrectly attributed to the Maldives. See CRS &quot;Gan 1970&quot;, code 4684.",
        information_source: "Various industry sources",
        data_source: "EPSG",
        revision_date: "2005-04-14",
        deprecated: 1, 
    },

    "EPSG:4720" => CRS {
        coord_ref_sys_code: 4720,
        coord_ref_sys_name: "Fiji 1986",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6720, 
        base_crs_code: -1,
        remarks: "Replaces Viti Levu 1912 (CRS code 4752), Vanua Levu 1915 (CRS code 4748) and Fiji 1956 (CRS code 4721).",
        information_source: "Department of Lands and Survey, Fiji.",
        data_source: "EPSG",
        revision_date: "2006-07-14",
        deprecated: 0, 
    },

    "EPSG:4721" => CRS {
        coord_ref_sys_code: 4721,
        coord_ref_sys_name: "Fiji 1956",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6721, 
        base_crs_code: -1,
        remarks: "For topographic mapping replaces Viti Levu 1912 (CRS code 4752) and Vanua Levu 1915 (CRS code 4748). Replaced by Fiji 1986 (CRS code 4720).",
        information_source: "SOPAC, www.sopac.org, and Clifford J. Mugnier in Photogrammetric Engineering and Remote Sensing, October 2000, www.asprs.org.",
        data_source: "EPSG",
        revision_date: "2006-07-14",
        deprecated: 0, 
    },

    "EPSG:4723" => CRS {
        coord_ref_sys_code: 4723,
        coord_ref_sys_name: "GCGD59",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6723, 
        base_crs_code: -1,
        remarks: "Superseded by CIGD11 (CRS code 6135).",
        information_source: "Lands and Surveys Department, Cayman Islands Government.",
        data_source: "EPSG",
        revision_date: "2013-01-17",
        deprecated: 0, 
    },

    "EPSG:4753" => CRS {
        coord_ref_sys_code: 4753,
        coord_ref_sys_name: "fk89",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6753, 
        base_crs_code: -1,
        remarks: "Replaces FD54 (CRS code 4741). Coordinate differences are less than 0.05 seconds (2m). The name of this system is also used for the dependent projected CRS - see CRS code 3173.",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-08-04",
        deprecated: 0, 
    },

    "EPSG:4938" => CRS {
        coord_ref_sys_code: 4938,
        coord_ref_sys_name: "GDA94",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6283, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Australian Surveying and Land Information Group Internet WWW page.",
        data_source: "EPSG",
        revision_date: "2019-09-17",
        deprecated: 0, 
    },

    "EPSG:4939" => CRS {
        coord_ref_sys_code: 4939,
        coord_ref_sys_name: "GDA94",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6283, 
        base_crs_code: 4938,
        remarks: "",
        information_source: "Australian Surveying and Land Information Group Internet WWW page.",
        data_source: "EPSG",
        revision_date: "2019-09-17",
        deprecated: 0, 
    },

    "EPSG:5317" => CRS {
        coord_ref_sys_code: 5317,
        coord_ref_sys_name: "FVR09 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1059, 
        base_crs_code: -1,
        remarks: "Introduced in 2010.",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2010-10-27",
        deprecated: 0, 
    },

    "EPSG:5591" => CRS {
        coord_ref_sys_code: 5591,
        coord_ref_sys_name: "FEH2010",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1078, 
        base_crs_code: -1,
        remarks: "Trans-national system created due to small (but unacceptable for engineering tolerance) differences between the German and Danish realisations of ETRS89.",
        information_source: "Femern A/S.",
        data_source: "EPSG",
        revision_date: "2011-09-23",
        deprecated: 0, 
    },

    "EPSG:5592" => CRS {
        coord_ref_sys_code: 5592,
        coord_ref_sys_name: "FEH2010",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1078, 
        base_crs_code: 5591,
        remarks: "",
        information_source: "Femern A/S.",
        data_source: "EPSG",
        revision_date: "2011-09-23",
        deprecated: 0, 
    },

    "EPSG:5593" => CRS {
        coord_ref_sys_code: 5593,
        coord_ref_sys_name: "FEH2010",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1078, 
        base_crs_code: 5592,
        remarks: "Created for engineering survey and construction of Fehmarnbelt tunnel.",
        information_source: "OGP. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2011-09-23",
        deprecated: 0, 
    },

    "EPSG:5596" => CRS {
        coord_ref_sys_code: 5596,
        coord_ref_sys_name: "FEH2010 / Fehmarnbelt TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5593,
        remarks: "",
        information_source: "Femern A/S",
        data_source: "EPSG",
        revision_date: "2011-09-24",
        deprecated: 0, 
    },

    "EPSG:5598" => CRS {
        coord_ref_sys_code: 5598,
        coord_ref_sys_name: "FEH2010 / Fehmarnbelt TM + FCSVR10 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Femern A/S",
        data_source: "EPSG",
        revision_date: "2011-09-24",
        deprecated: 0, 
    },

    "EPSG:5743" => CRS {
        coord_ref_sys_code: 5743,
        coord_ref_sys_name: "Foula height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5141, 
        base_crs_code: -1,
        remarks: "Replaced by ODN (Offshore) height (CRS code 7707) in 2016.",
        information_source: "Ordnance Survey of Great Britain.",
        data_source: "EPSG",
        revision_date: "2020-06-17",
        deprecated: 0, 
    },

    "EPSG:5748" => CRS {
        coord_ref_sys_code: 5748,
        coord_ref_sys_name: "Flannan Isles height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5146, 
        base_crs_code: -1,
        remarks: "Replaced by ODN (Offshore) height (CRS code 7707) in 2016.",
        information_source: "Ordnance Survey of Great Britain.",
        data_source: "EPSG",
        revision_date: "2020-06-17",
        deprecated: 0, 
    },

    "EPSG:6130" => CRS {
        coord_ref_sys_code: 6130,
        coord_ref_sys_name: "GCVD54 height (ft)",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 1030,
        datum_code: 1097, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Lands and Survey Department, Cayman Islands Government.",
        data_source: "EPSG",
        revision_date: "2017-10-16",
        deprecated: 0, 
    },

    "EPSG:6721" => CRS {
        coord_ref_sys_code: 6721,
        coord_ref_sys_name: "GDA94 / CIG94",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "Usage restricted to areas below 290m above sea level. Replaces CIG85 and WGS 84 / CIG92 (CRS codes 6715 and 6721). Replaced by GDA2020 / CIG2020 (CRS code 8018).",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2017-03-19",
        deprecated: 0, 
    },

    "EPSG:6723" => CRS {
        coord_ref_sys_code: 6723,
        coord_ref_sys_name: "GDA94 / CKIG94",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "Replaces WGS 84 / CKIG92 (CRS code 6722). Replaced by GDA2020 / CKIG2020 (CRS code 8019).",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2017-03-19",
        deprecated: 0, 
    },

    "EPSG:6732" => CRS {
        coord_ref_sys_code: 6732,
        coord_ref_sys_name: "GDA94 / MGA zone 41",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "Geocentric Datum of Australia Technical Manual, Intergovernmental Committee on Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2014-02-06",
        deprecated: 1, 
    },

    "EPSG:6733" => CRS {
        coord_ref_sys_code: 6733,
        coord_ref_sys_name: "GDA94 / MGA zone 42",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "Geocentric Datum of Australia Technical Manual, Intergovernmental Committee on Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2014-02-06",
        deprecated: 1, 
    },

    "EPSG:6894" => CRS {
        coord_ref_sys_code: 6894,
        coord_ref_sys_name: "Gambia",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1139, 
        base_crs_code: -1,
        remarks: "",
        information_source: "US NGA GeoTrans software v3.4.",
        data_source: "EPSG",
        revision_date: "2014-07-30",
        deprecated: 0, 
    },

    "EPSG:7842" => CRS {
        coord_ref_sys_code: 7842,
        coord_ref_sys_name: "GDA2020",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1168, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2019-09-17",
        deprecated: 0, 
    },

    "EPSG:7843" => CRS {
        coord_ref_sys_code: 7843,
        coord_ref_sys_name: "GDA2020",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1168, 
        base_crs_code: 7842,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2019-09-17",
        deprecated: 0, 
    },

    "EPSG:7844" => CRS {
        coord_ref_sys_code: 7844,
        coord_ref_sys_name: "GDA2020",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1168, 
        base_crs_code: 7843,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2019-09-17",
        deprecated: 0, 
    },

    "EPSG:7845" => CRS {
        coord_ref_sys_code: 7845,
        coord_ref_sys_name: "GDA2020 / GA LCC",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2016-09-29",
        deprecated: 0, 
    },

    "EPSG:7846" => CRS {
        coord_ref_sys_code: 7846,
        coord_ref_sys_name: "GDA2020 / MGA zone 46",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2016-09-07",
        deprecated: 0, 
    },

    "EPSG:7847" => CRS {
        coord_ref_sys_code: 7847,
        coord_ref_sys_name: "GDA2020 / MGA zone 47",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2016-09-07",
        deprecated: 0, 
    },

    "EPSG:7848" => CRS {
        coord_ref_sys_code: 7848,
        coord_ref_sys_name: "GDA2020 / MGA zone 48",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2016-09-07",
        deprecated: 0, 
    },

    "EPSG:7849" => CRS {
        coord_ref_sys_code: 7849,
        coord_ref_sys_name: "GDA2020 / MGA zone 49",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2016-09-07",
        deprecated: 0, 
    },

    "EPSG:7850" => CRS {
        coord_ref_sys_code: 7850,
        coord_ref_sys_name: "GDA2020 / MGA zone 50",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2016-09-07",
        deprecated: 0, 
    },

    "EPSG:7851" => CRS {
        coord_ref_sys_code: 7851,
        coord_ref_sys_name: "GDA2020 / MGA zone 51",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2016-09-07",
        deprecated: 0, 
    },

    "EPSG:7852" => CRS {
        coord_ref_sys_code: 7852,
        coord_ref_sys_name: "GDA2020 / MGA zone 52",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2016-09-07",
        deprecated: 0, 
    },

    "EPSG:7853" => CRS {
        coord_ref_sys_code: 7853,
        coord_ref_sys_name: "GDA2020 / MGA zone 53",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2016-09-07",
        deprecated: 0, 
    },

    "EPSG:7854" => CRS {
        coord_ref_sys_code: 7854,
        coord_ref_sys_name: "GDA2020 / MGA zone 54",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2016-09-07",
        deprecated: 0, 
    },

    "EPSG:7855" => CRS {
        coord_ref_sys_code: 7855,
        coord_ref_sys_name: "GDA2020 / MGA zone 55",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2016-09-07",
        deprecated: 0, 
    },

    "EPSG:7856" => CRS {
        coord_ref_sys_code: 7856,
        coord_ref_sys_name: "GDA2020 / MGA zone 56",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2016-09-07",
        deprecated: 0, 
    },

    "EPSG:7857" => CRS {
        coord_ref_sys_code: 7857,
        coord_ref_sys_name: "GDA2020 / MGA zone 57",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2016-09-07",
        deprecated: 0, 
    },

    "EPSG:7858" => CRS {
        coord_ref_sys_code: 7858,
        coord_ref_sys_name: "GDA2020 / MGA zone 58",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2016-09-07",
        deprecated: 0, 
    },

    "EPSG:7859" => CRS {
        coord_ref_sys_code: 7859,
        coord_ref_sys_name: "GDA2020 / MGA zone 59",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2016-09-07",
        deprecated: 0, 
    },

    "EPSG:7899" => CRS {
        coord_ref_sys_code: 7899,
        coord_ref_sys_name: "GDA2020 / Vicgrid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "May also be used to display Victoria, New South Wales, Tasmania and South Australia as a single entity.",
        information_source: "Office of Surveyor-General Victoria.",
        data_source: "EPSG",
        revision_date: "2016-11-04",
        deprecated: 0, 
    },

    "EPSG:8013" => CRS {
        coord_ref_sys_code: 8013,
        coord_ref_sys_name: "GDA2020 / ALB2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8014" => CRS {
        coord_ref_sys_code: 8014,
        coord_ref_sys_name: "GDA2020 / BIO2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8015" => CRS {
        coord_ref_sys_code: 8015,
        coord_ref_sys_name: "GDA2020 / BRO2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8016" => CRS {
        coord_ref_sys_code: 8016,
        coord_ref_sys_name: "GDA2020 / BCG2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8017" => CRS {
        coord_ref_sys_code: 8017,
        coord_ref_sys_name: "GDA2020 / CARN2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8018" => CRS {
        coord_ref_sys_code: 8018,
        coord_ref_sys_name: "GDA2020 / CIG2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "Usage restricted to areas below 290m above sea level.",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8019" => CRS {
        coord_ref_sys_code: 8019,
        coord_ref_sys_name: "GDA2020 / CKIG2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8020" => CRS {
        coord_ref_sys_code: 8020,
        coord_ref_sys_name: "GDA2020 / COL2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8021" => CRS {
        coord_ref_sys_code: 8021,
        coord_ref_sys_name: "GDA2020 / ESP2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8022" => CRS {
        coord_ref_sys_code: 8022,
        coord_ref_sys_name: "GDA2020 / EXM2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8023" => CRS {
        coord_ref_sys_code: 8023,
        coord_ref_sys_name: "GDA2020 / GCG2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8024" => CRS {
        coord_ref_sys_code: 8024,
        coord_ref_sys_name: "GDA2020 / GOLD2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8025" => CRS {
        coord_ref_sys_code: 8025,
        coord_ref_sys_name: "GDA2020 / JCG2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8026" => CRS {
        coord_ref_sys_code: 8026,
        coord_ref_sys_name: "GDA2020 / KALB2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8027" => CRS {
        coord_ref_sys_code: 8027,
        coord_ref_sys_name: "GDA2020 / KAR2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8028" => CRS {
        coord_ref_sys_code: 8028,
        coord_ref_sys_name: "GDA2020 / KUN2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8029" => CRS {
        coord_ref_sys_code: 8029,
        coord_ref_sys_name: "GDA2020 / LCG2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8030" => CRS {
        coord_ref_sys_code: 8030,
        coord_ref_sys_name: "GDA2020 / MRCG2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8031" => CRS {
        coord_ref_sys_code: 8031,
        coord_ref_sys_name: "GDA2020 / PCG2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8032" => CRS {
        coord_ref_sys_code: 8032,
        coord_ref_sys_name: "GDA2020 / PHG2020",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Landgate (Government of Western Australia), www.landgate.wa.gov.au.",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:8058" => CRS {
        coord_ref_sys_code: 8058,
        coord_ref_sys_name: "GDA2020 / NSW Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Spatial Services, NSW Department of Finance, Services and Innovation.",
        data_source: "EPSG",
        revision_date: "2017-05-15",
        deprecated: 0, 
    },

    "EPSG:8059" => CRS {
        coord_ref_sys_code: 8059,
        coord_ref_sys_name: "GDA2020 / SA Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "Development Division, South Australia Department of Planning, Transport and Infrastructure.",
        data_source: "EPSG",
        revision_date: "2017-05-15",
        deprecated: 0, 
    },

    "EPSG:9396" => CRS {
        coord_ref_sys_code: 9396,
        coord_ref_sys_name: "Fuerteventura height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1279, 
        base_crs_code: -1,
        remarks: "",
        information_source: "National Geographic Institute of Spain (IGN), http://www.ign.es",
        data_source: "EPSG",
        revision_date: "2020-06-27",
        deprecated: 0, 
    },

    "EPSG:9453" => CRS {
        coord_ref_sys_code: 9453,
        coord_ref_sys_name: "GBK19-IRF",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1289, 
        base_crs_code: -1,
        remarks: "Intermediate CRS created in 2020 to assist the emulation of the ETRS89 / GBK19 SnakeGrid projected CRS through transformation ETRS89 to GBK19-IRF (1) (code 9454) used in conjunction with the GBK19-TM map projection (code 9455).",
        information_source: "Network Rail.",
        data_source: "EPSG",
        revision_date: "2020-08-05",
        deprecated: 0, 
    },

    "EPSG:9456" => CRS {
        coord_ref_sys_code: 9456,
        coord_ref_sys_name: "GBK19 Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 9453,
        remarks: "The CRS&#x27;s definition through transformation ETRS89 to GBK19-IRF (1) (code 9454) and map projection GBK19-TM (code 9455) emulates the GBK19 Snake projection applied to ETRS89 as realised through OSNet 2009 CORS.",
        information_source: "Network Rail.",
        data_source: "EPSG",
        revision_date: "2020-08-05",
        deprecated: 0, 
    },

    "EPSG:9457" => CRS {
        coord_ref_sys_code: 9457,
        coord_ref_sys_name: "GBK19 Grid + ODN height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Network Rail.",
        data_source: "EPSG",
        revision_date: "2020-08-05",
        deprecated: 0, 
    },

    "EPSG:9462" => CRS {
        coord_ref_sys_code: 9462,
        coord_ref_sys_name: "GDA2020 + AVWS height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "National Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2020-08-29",
        deprecated: 0, 
    },

    "EPSG:9463" => CRS {
        coord_ref_sys_code: 9463,
        coord_ref_sys_name: "GDA2020 + AHD height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "National Geodesy Section, Geoscience Australia, www.ga.gov.au, geodesy/ahd pages.",
        data_source: "EPSG",
        revision_date: "2020-08-29",
        deprecated: 0, 
    },

    "EPSG:9473" => CRS {
        coord_ref_sys_code: 9473,
        coord_ref_sys_name: "GDA2020 / Australian Albers",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 7844,
        remarks: "",
        information_source: "National Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2020-08-29",
        deprecated: 0, 
    },

    "EPSG:9519" => CRS {
        coord_ref_sys_code: 9519,
        coord_ref_sys_name: "FEH2010 + FCSVR10 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:23433" => CRS {
        coord_ref_sys_code: 23433,
        coord_ref_sys_name: "Garoua / UTM zone 33N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4234,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:61326405" => CRS {
        coord_ref_sys_code: 61326405,
        coord_ref_sys_name: "FD58 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4132,
        remarks: "See CRS code 4132 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61976405" => CRS {
        coord_ref_sys_code: 61976405,
        coord_ref_sys_name: "Garoua (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4197,
        remarks: "See CRS code 4197 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62336405" => CRS {
        coord_ref_sys_code: 62336405,
        coord_ref_sys_name: "Gandajika 1970 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4233,
        remarks: "See CRS code 4233 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62836405" => CRS {
        coord_ref_sys_code: 62836405,
        coord_ref_sys_name: "GDA94 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "See CRS code 4283 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62836413" => CRS {
        coord_ref_sys_code: 62836413,
        coord_ref_sys_name: "GDA94 (3D deg)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6413,
        datum_code: -1, 
        base_crs_code: 4347,
        remarks: "See CRS code 4347 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:66216405" => CRS {
        coord_ref_sys_code: 66216405,
        coord_ref_sys_name: "Fort Marigot (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4621,
        remarks: "See CRS code 4621 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:66256405" => CRS {
        coord_ref_sys_code: 66256405,
        coord_ref_sys_name: "Fort Desaix (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4625,
        remarks: "See CRS code 4625 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:2003" => CRS {
        coord_ref_sys_code: 2003,
        coord_ref_sys_name: "Grenada 1953 / British West Indies Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4603,
        remarks: "",
        information_source: "Ordnance Survey of Great Britain",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2100" => CRS {
        coord_ref_sys_code: 2100,
        coord_ref_sys_name: "GGRS87 / Greek Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4121,
        remarks: "Oil industry uses ED50 / UTM zone 34N and ED50 / UTM zone 35N.",
        information_source: "Geodesy Department; Public Petroleum Corporation of Greece.",
        data_source: "EPSG",
        revision_date: "2011-07-20",
        deprecated: 0, 
    },

    "EPSG:2999" => CRS {
        coord_ref_sys_code: 2999,
        coord_ref_sys_name: "Grand Comoros / UTM zone 38S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4646,
        remarks: "",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 0, 
    },

    "EPSG:3107" => CRS {
        coord_ref_sys_code: 3107,
        coord_ref_sys_name: "GDA94 / SA Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "South Australia Department for Environment and Heritage, Spatial Information Committee (SICOM); http://www.environment.sa.gov.au/mapland/sicom/sicom/lambert.html",
        data_source: "EPSG",
        revision_date: "2004-10-20",
        deprecated: 0, 
    },

    "EPSG:3111" => CRS {
        coord_ref_sys_code: 3111,
        coord_ref_sys_name: "GDA94 / Vicgrid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "May also be used to display Victoria, New South Wales, Tasmania and South Australia as a single entity. Replaces Vicgrid66 (CRS code 3110) with effect from 2nd February 2000.",
        information_source: "Victoria Land Registry; http://www.land.vic.gov.au",
        data_source: "EPSG",
        revision_date: "2016-12-15",
        deprecated: 0, 
    },

    "EPSG:3178" => CRS {
        coord_ref_sys_code: 3178,
        coord_ref_sys_name: "GR96 / UTM zone 18N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-07-30",
        deprecated: 0, 
    },

    "EPSG:3179" => CRS {
        coord_ref_sys_code: 3179,
        coord_ref_sys_name: "GR96 / UTM zone 19N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-07-30",
        deprecated: 0, 
    },

    "EPSG:3180" => CRS {
        coord_ref_sys_code: 3180,
        coord_ref_sys_name: "GR96 / UTM zone 20N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-07-30",
        deprecated: 0, 
    },

    "EPSG:3181" => CRS {
        coord_ref_sys_code: 3181,
        coord_ref_sys_name: "GR96 / UTM zone 21N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-07-30",
        deprecated: 0, 
    },

    "EPSG:3182" => CRS {
        coord_ref_sys_code: 3182,
        coord_ref_sys_name: "GR96 / UTM zone 22N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-07-30",
        deprecated: 0, 
    },

    "EPSG:3183" => CRS {
        coord_ref_sys_code: 3183,
        coord_ref_sys_name: "GR96 / UTM zone 23N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-07-30",
        deprecated: 0, 
    },

    "EPSG:3184" => CRS {
        coord_ref_sys_code: 3184,
        coord_ref_sys_name: "GR96 / UTM zone 24N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-07-30",
        deprecated: 0, 
    },

    "EPSG:3185" => CRS {
        coord_ref_sys_code: 3185,
        coord_ref_sys_name: "GR96 / UTM zone 25N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-07-30",
        deprecated: 0, 
    },

    "EPSG:3186" => CRS {
        coord_ref_sys_code: 3186,
        coord_ref_sys_name: "GR96 / UTM zone 26N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-07-30",
        deprecated: 0, 
    },

    "EPSG:3187" => CRS {
        coord_ref_sys_code: 3187,
        coord_ref_sys_name: "GR96 / UTM zone 27N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-07-30",
        deprecated: 0, 
    },

    "EPSG:3188" => CRS {
        coord_ref_sys_code: 3188,
        coord_ref_sys_name: "GR96 / UTM zone 28N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-07-30",
        deprecated: 0, 
    },

    "EPSG:3189" => CRS {
        coord_ref_sys_code: 3189,
        coord_ref_sys_name: "GR96 / UTM zone 29N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-07-30",
        deprecated: 0, 
    },

    "EPSG:3308" => CRS {
        coord_ref_sys_code: 3308,
        coord_ref_sys_name: "GDA94 / NSW Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "NSW Department of Lands.",
        data_source: "EPSG",
        revision_date: "2005-08-18",
        deprecated: 0, 
    },

    "EPSG:3356" => CRS {
        coord_ref_sys_code: 3356,
        coord_ref_sys_name: "Grand Cayman 1959 / UTM zone 17N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4723,
        remarks: "",
        information_source: "Clifford J. Mugnier; in Photogrammetric Engineering &amp; Remote Sensing November 1998. http://www.asprs.org/",
        data_source: "EPSG",
        revision_date: "2006-01-26",
        deprecated: 1, 
    },

    "EPSG:3375" => CRS {
        coord_ref_sys_code: 3375,
        coord_ref_sys_name: "GDM2000 / Peninsula RSO",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4742,
        remarks: "Replaces Kertau (RSO) / RSO Malaya (CRS codes 3167-68).",
        information_source: "GDM2000 Technical Manual; Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:3376" => CRS {
        coord_ref_sys_code: 3376,
        coord_ref_sys_name: "GDM2000 / East Malaysia BRSO",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4742,
        remarks: "Replaces Timbalai 1948 / RSO Borneo (CRS codes 29871-73).",
        information_source: "GDM2000 Technical Manual; Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:3377" => CRS {
        coord_ref_sys_code: 3377,
        coord_ref_sys_name: "GDM2000 / Johor Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4742,
        remarks: "Replaces Kertau 1968 / Johor Grid (CRS code 4390).",
        information_source: "GDM2000 Technical Manual; Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-11-21",
        deprecated: 0, 
    },

    "EPSG:3378" => CRS {
        coord_ref_sys_code: 3378,
        coord_ref_sys_name: "GDM2000 / Sembilan and Melaka Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4742,
        remarks: "Replaces Kertau 1968 / Sembilan and Melaka Grid (CRS code 4391).",
        information_source: "GDM2000 Technical Manual; Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-11-21",
        deprecated: 0, 
    },

    "EPSG:3379" => CRS {
        coord_ref_sys_code: 3379,
        coord_ref_sys_name: "GDM2000 / Pahang Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4742,
        remarks: "Replaces Kertau 1968 / Pahang Grid (CRS code 4392).",
        information_source: "GDM2000 Technical Manual; Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-11-21",
        deprecated: 0, 
    },

    "EPSG:3380" => CRS {
        coord_ref_sys_code: 3380,
        coord_ref_sys_name: "GDM2000 / Selangor Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4742,
        remarks: "Replaces Kertau 1968 / Selangor Grid (CRS code 4393).",
        information_source: "GDM2000 Technical Manual; Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-11-21",
        deprecated: 0, 
    },

    "EPSG:3381" => CRS {
        coord_ref_sys_code: 3381,
        coord_ref_sys_name: "GDM2000 / Terengganu Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4742,
        remarks: "Replaces Kertau 1968 / Terengganu Grid (CRS code 4394).",
        information_source: "GDM2000 Technical Manual; Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-11-21",
        deprecated: 0, 
    },

    "EPSG:3382" => CRS {
        coord_ref_sys_code: 3382,
        coord_ref_sys_name: "GDM2000 / Pinang Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4742,
        remarks: "Replaces Kertau 1968 / Pinang Grid (CRS code 4395).",
        information_source: "GDM2000 Technical Manual; Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-11-21",
        deprecated: 0, 
    },

    "EPSG:3383" => CRS {
        coord_ref_sys_code: 3383,
        coord_ref_sys_name: "GDM2000 / Kedah and Perlis Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4742,
        remarks: "Replaces Kertau 1968 / Kedah and Perlis Grid (CRS code 4396).",
        information_source: "GDM2000 Technical Manual; Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-11-21",
        deprecated: 0, 
    },

    "EPSG:3384" => CRS {
        coord_ref_sys_code: 3384,
        coord_ref_sys_name: "GDM2000 / Perak Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4742,
        remarks: "Replaces Kertau 1968 / Perak Revised Grid (CRS code 4397).",
        information_source: "GDM2000 Technical Manual; Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-11-21",
        deprecated: 0, 
    },

    "EPSG:3385" => CRS {
        coord_ref_sys_code: 3385,
        coord_ref_sys_name: "GDM2000 / Kelantan Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4742,
        remarks: "Replaces Kertau 1968 / Kelantan Grid (CRS code 4398).",
        information_source: "GDM2000 Technical Manual; Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-11-21",
        deprecated: 0, 
    },

    "EPSG:4120" => CRS {
        coord_ref_sys_code: 4120,
        coord_ref_sys_name: "Greek",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6120, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Topography Department; National Technical University of Greece.",
        data_source: "EPSG",
        revision_date: "2011-07-20",
        deprecated: 0, 
    },

    "EPSG:4121" => CRS {
        coord_ref_sys_code: 4121,
        coord_ref_sys_name: "GGRS87",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6121, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Geodesy Department; Public Petroleum Corporation of Greece.",
        data_source: "EPSG",
        revision_date: "2011-07-20",
        deprecated: 0, 
    },

    "EPSG:4603" => CRS {
        coord_ref_sys_code: 4603,
        coord_ref_sys_name: "Grenada 1953",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6603, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4646" => CRS {
        coord_ref_sys_code: 4646,
        coord_ref_sys_name: "Grand Comoros",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6646, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2011-06-30",
        deprecated: 0, 
    },

    "EPSG:4742" => CRS {
        coord_ref_sys_code: 4742,
        coord_ref_sys_name: "GDM2000",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6742, 
        base_crs_code: 4921,
        remarks: "Replaces all earlier Malaysian geographic CRSs.",
        information_source: "OGP. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2007-08-27",
        deprecated: 0, 
    },

    "EPSG:4236" => CRS {
        coord_ref_sys_code: 4236,
        coord_ref_sys_name: "Hu Tzu Shan 1950",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6236, 
        base_crs_code: -1,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2008-08-12",
        deprecated: 0, 
    },

    "EPSG:4747" => CRS {
        coord_ref_sys_code: 4747,
        coord_ref_sys_name: "GR96",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6747, 
        base_crs_code: 4909,
        remarks: "Replaces all earlier Greenland geographic CRSs.",
        information_source: "OGP. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2007-08-27",
        deprecated: 0, 
    },

    "EPSG:4815" => CRS {
        coord_ref_sys_code: 4815,
        coord_ref_sys_name: "Greek (Athens)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6815, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Topography Department; National Technical University of Greece.",
        data_source: "EPSG",
        revision_date: "2011-07-20",
        deprecated: 0, 
    },

    "EPSG:4908" => CRS {
        coord_ref_sys_code: 4908,
        coord_ref_sys_name: "GR96",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6747, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2006-06-12",
        deprecated: 0, 
    },

    "EPSG:4909" => CRS {
        coord_ref_sys_code: 4909,
        coord_ref_sys_name: "GR96",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6747, 
        base_crs_code: 4908,
        remarks: "",
        information_source: "Kort &amp; Matrikelstyrelsen (KMS), Copenhagen.",
        data_source: "EPSG",
        revision_date: "2007-08-27",
        deprecated: 0, 
    },

    "EPSG:4920" => CRS {
        coord_ref_sys_code: 4920,
        coord_ref_sys_name: "GDM2000",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6742, 
        base_crs_code: -1,
        remarks: "",
        information_source: "GDM2000 Technical Manual; Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2006-03-16",
        deprecated: 0, 
    },

    "EPSG:4921" => CRS {
        coord_ref_sys_code: 4921,
        coord_ref_sys_name: "GDM2000",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6742, 
        base_crs_code: 4920,
        remarks: "",
        information_source: "GDM2000 Technical Manual; Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2007-08-27",
        deprecated: 0, 
    },

    "EPSG:5214" => CRS {
        coord_ref_sys_code: 5214,
        coord_ref_sys_name: "Genoa 1942 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1051, 
        base_crs_code: -1,
        remarks: "",
        information_source: "EuroGeographics, http://www.crs-geo.eu",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:5244" => CRS {
        coord_ref_sys_code: 5244,
        coord_ref_sys_name: "GDBD2009",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1056, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Survey Department, Brunei.",
        data_source: "EPSG",
        revision_date: "2010-08-31",
        deprecated: 0, 
    },

    "EPSG:5245" => CRS {
        coord_ref_sys_code: 5245,
        coord_ref_sys_name: "GDBD2009",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1056, 
        base_crs_code: 5244,
        remarks: "",
        information_source: "Survey Department, Brunei.",
        data_source: "EPSG",
        revision_date: "2010-08-31",
        deprecated: 0, 
    },

    "EPSG:5246" => CRS {
        coord_ref_sys_code: 5246,
        coord_ref_sys_name: "GDBD2009",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1056, 
        base_crs_code: 5245,
        remarks: "Introduced from July 2009 to replace Timbalai 1948 (CRS code 4298) for government purposes.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2010-08-31",
        deprecated: 0, 
    },

    "EPSG:5247" => CRS {
        coord_ref_sys_code: 5247,
        coord_ref_sys_name: "GDBD2009 / Brunei BRSO",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 5246,
        remarks: "Replaces Timbalai 1948 / RSO Borneo (CRS codes 29871 and 29873).",
        information_source: "GDM2000 Technical Manual; Department of Survey and Mapping Malaysia, www.jupem.gov.my, and Survey Department, Brunei.",
        data_source: "EPSG",
        revision_date: "2010-09-08",
        deprecated: 0, 
    },

    "EPSG:5762" => CRS {
        coord_ref_sys_code: 5762,
        coord_ref_sys_name: "Gisborne 1926 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5160, 
        base_crs_code: -1,
        remarks: "",
        information_source: "OSG Technical Report #10, February 2001.",
        data_source: "EPSG",
        revision_date: "2011-01-25",
        deprecated: 0, 
    },

    "EPSG:5778" => CRS {
        coord_ref_sys_code: 5778,
        coord_ref_sys_name: "GHA height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5176, 
        base_crs_code: -1,
        remarks: "For scientific purposes see EVRF2000 Austria height (CRS code 9274).",
        information_source: "Bundesamt für Eich- und Vermessungswesen (BEV).",
        data_source: "EPSG",
        revision_date: "2020-01-16",
        deprecated: 0, 
    },

    "EPSG:6050" => CRS {
        coord_ref_sys_code: 6050,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 1-25",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-08",
        deprecated: 0, 
    },

    "EPSG:6051" => CRS {
        coord_ref_sys_code: 6051,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 2-18",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-08",
        deprecated: 0, 
    },

    "EPSG:6052" => CRS {
        coord_ref_sys_code: 6052,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 2-20",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-08",
        deprecated: 0, 
    },

    "EPSG:6053" => CRS {
        coord_ref_sys_code: 6053,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 3-29",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-08",
        deprecated: 0, 
    },

    "EPSG:6054" => CRS {
        coord_ref_sys_code: 6054,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 3-31",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-08",
        deprecated: 0, 
    },

    "EPSG:6055" => CRS {
        coord_ref_sys_code: 6055,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 3-33",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-08",
        deprecated: 0, 
    },

    "EPSG:6056" => CRS {
        coord_ref_sys_code: 6056,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 4-20",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-08",
        deprecated: 0, 
    },

    "EPSG:6057" => CRS {
        coord_ref_sys_code: 6057,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 4-22",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2013-03-30",
        deprecated: 0, 
    },

    "EPSG:6058" => CRS {
        coord_ref_sys_code: 6058,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 4-24",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2013-03-30",
        deprecated: 0, 
    },

    "EPSG:6059" => CRS {
        coord_ref_sys_code: 6059,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 5-41",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-08",
        deprecated: 0, 
    },

    "EPSG:6060" => CRS {
        coord_ref_sys_code: 6060,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 5-43",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-08",
        deprecated: 0, 
    },

    "EPSG:6061" => CRS {
        coord_ref_sys_code: 6061,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 5-45",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-08",
        deprecated: 0, 
    },

    "EPSG:6062" => CRS {
        coord_ref_sys_code: 6062,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 6-26",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-08",
        deprecated: 0, 
    },

    "EPSG:6063" => CRS {
        coord_ref_sys_code: 6063,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 6-28",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-08",
        deprecated: 0, 
    },

    "EPSG:6064" => CRS {
        coord_ref_sys_code: 6064,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 6-30",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-12",
        deprecated: 0, 
    },

    "EPSG:6065" => CRS {
        coord_ref_sys_code: 6065,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 7-11",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-12",
        deprecated: 0, 
    },

    "EPSG:6066" => CRS {
        coord_ref_sys_code: 6066,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 7-13",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-12",
        deprecated: 0, 
    },

    "EPSG:6067" => CRS {
        coord_ref_sys_code: 6067,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 8-20",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-08",
        deprecated: 0, 
    },

    "EPSG:6068" => CRS {
        coord_ref_sys_code: 6068,
        coord_ref_sys_name: "GR96 / EPSG Arctic zone 8-22",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4747,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-12-08",
        deprecated: 0, 
    },

    "EPSG:6128" => CRS {
        coord_ref_sys_code: 6128,
        coord_ref_sys_name: "Grand Cayman National Grid 1959",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 1039,
        datum_code: -1, 
        base_crs_code: 4723,
        remarks: "",
        information_source: "Lands and Surveys Department, Cayman Islands Government.",
        data_source: "EPSG",
        revision_date: "2012-12-12",
        deprecated: 0, 
    },

    "EPSG:6734" => CRS {
        coord_ref_sys_code: 6734,
        coord_ref_sys_name: "GDA94 / MGA zone 43",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "Geocentric Datum of Australia Technical Manual, Intergovernmental Committee on Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2014-02-06",
        deprecated: 1, 
    },

    "EPSG:6735" => CRS {
        coord_ref_sys_code: 6735,
        coord_ref_sys_name: "GDA94 / MGA zone 44",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "Geocentric Datum of Australia Technical Manual, Intergovernmental Committee on Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2014-02-06",
        deprecated: 1, 
    },

    "EPSG:6736" => CRS {
        coord_ref_sys_code: 6736,
        coord_ref_sys_name: "GDA94 / MGA zone 46",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "Geocentric Datum of Australia Technical Manual, Intergovernmental Committee on Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2014-02-06",
        deprecated: 0, 
    },

    "EPSG:6737" => CRS {
        coord_ref_sys_code: 6737,
        coord_ref_sys_name: "GDA94 / MGA zone 47",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "Geocentric Datum of Australia Technical Manual, Intergovernmental Committee on Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2014-02-06",
        deprecated: 0, 
    },

    "EPSG:6738" => CRS {
        coord_ref_sys_code: 6738,
        coord_ref_sys_name: "GDA94 / MGA zone 59",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "Geocentric Datum of Australia Technical Manual, Intergovernmental Committee on Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2014-02-06",
        deprecated: 0, 
    },

    "EPSG:7681" => CRS {
        coord_ref_sys_code: 7681,
        coord_ref_sys_name: "GSK-2011",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1159, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Parametry Zemli 1990 Reference Document, Military Topgraphic Department (VTU) of the General Staff of Armed Forces of the Russian Federation.",
        data_source: "EPSG",
        revision_date: "2016-01-25",
        deprecated: 0, 
    },

    "EPSG:7682" => CRS {
        coord_ref_sys_code: 7682,
        coord_ref_sys_name: "GSK-2011",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1159, 
        base_crs_code: 7681,
        remarks: "",
        information_source: "Parametry Zemli 1990 Reference Document, Military Topgraphic Department (VTU) of the General Staff of Armed Forces of the Russian Federation.",
        data_source: "EPSG",
        revision_date: "2016-01-25",
        deprecated: 0, 
    },

    "EPSG:7683" => CRS {
        coord_ref_sys_code: 7683,
        coord_ref_sys_name: "GSK-2011",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1159, 
        base_crs_code: 7682,
        remarks: "Replaces Pulkovo 1995 (CRS code 4200) with effect from 21st October 2011.",
        information_source: "EPSG. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2015-11-21",
        deprecated: 0, 
    },

    "EPSG:8349" => CRS {
        coord_ref_sys_code: 8349,
        coord_ref_sys_name: "GR96 + GVR2000 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by GR96 + GVR2016 height (CRS code 8350).",
        information_source: "Agency for Data Supply and Efficiency (SDFE), Denmark.",
        data_source: "EPSG",
        revision_date: "2017-09-22",
        deprecated: 0, 
    },

    "EPSG:8350" => CRS {
        coord_ref_sys_code: 8350,
        coord_ref_sys_name: "GR96 + GVR2016 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaces GR96 + GVR2000 height (CRS code 8349).",
        information_source: "Agency for Data Supply and Efficiency (SDFE), Denmark.",
        data_source: "EPSG",
        revision_date: "2017-09-22",
        deprecated: 0, 
    },

    "EPSG:8391" => CRS {
        coord_ref_sys_code: 8391,
        coord_ref_sys_name: "GDA94 / WEIPA94",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "Rio Tinto Weipa.",
        data_source: "EPSG",
        revision_date: "2018-01-12",
        deprecated: 0, 
    },

    "EPSG:9397" => CRS {
        coord_ref_sys_code: 9397,
        coord_ref_sys_name: "Gran Canaria height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1280, 
        base_crs_code: -1,
        remarks: "",
        information_source: "National Geographic Institute of Spain (IGN), http://www.ign.es",
        data_source: "EPSG",
        revision_date: "2020-06-27",
        deprecated: 0, 
    },

    "EPSG:9464" => CRS {
        coord_ref_sys_code: 9464,
        coord_ref_sys_name: "GDA94 + AHD height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "National Geodesy Section, Geoscience Australia.",
        data_source: "EPSG",
        revision_date: "2020-08-29",
        deprecated: 0, 
    },

    "EPSG:21018" => CRS {
        coord_ref_sys_code: 21018,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 105E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 18 (code 20918).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21019" => CRS {
        coord_ref_sys_code: 21019,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 111E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 19 (code 20919).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21020" => CRS {
        coord_ref_sys_code: 21020,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 117E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 20 (code 20920).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:28348" => CRS {
        coord_ref_sys_code: 28348,
        coord_ref_sys_name: "GDA94 / MGA zone 48",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "Geocentric Datum of Australia Technical Manual, Intergovernmental Committee on Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2017-09-22",
        deprecated: 0, 
    },

    "EPSG:28349" => CRS {
        coord_ref_sys_code: 28349,
        coord_ref_sys_name: "GDA94 / MGA zone 49",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "Geocentric Datum of Australia Technical Manual, Intergovernmental Committee on Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2014-02-15",
        deprecated: 0, 
    },

    "EPSG:28350" => CRS {
        coord_ref_sys_code: 28350,
        coord_ref_sys_name: "GDA94 / MGA zone 50",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "Geocentric Datum of Australia Technical Manual, Intergovernmental Committee on Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2014-02-15",
        deprecated: 0, 
    },

    "EPSG:28351" => CRS {
        coord_ref_sys_code: 28351,
        coord_ref_sys_name: "GDA94 / MGA zone 51",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:28352" => CRS {
        coord_ref_sys_code: 28352,
        coord_ref_sys_name: "GDA94 / MGA zone 52",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:28353" => CRS {
        coord_ref_sys_code: 28353,
        coord_ref_sys_name: "GDA94 / MGA zone 53",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:28354" => CRS {
        coord_ref_sys_code: 28354,
        coord_ref_sys_name: "GDA94 / MGA zone 54",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:28355" => CRS {
        coord_ref_sys_code: 28355,
        coord_ref_sys_name: "GDA94 / MGA zone 55",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:28356" => CRS {
        coord_ref_sys_code: 28356,
        coord_ref_sys_name: "GDA94 / MGA zone 56",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:28357" => CRS {
        coord_ref_sys_code: 28357,
        coord_ref_sys_name: "GDA94 / MGA zone 57",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "Geocentric Datum of Australia Technical Manual, Intergovernmental Committee on Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2014-02-15",
        deprecated: 0, 
    },

    "EPSG:28358" => CRS {
        coord_ref_sys_code: 28358,
        coord_ref_sys_name: "GDA94 / MGA zone 58",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4283,
        remarks: "",
        information_source: "Geocentric Datum of Australia Technical Manual, Intergovernmental Committee on Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2014-02-15",
        deprecated: 0, 
    },

    "EPSG:61206405" => CRS {
        coord_ref_sys_code: 61206405,
        coord_ref_sys_name: "Greek (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4120,
        remarks: "See CRS code 4120 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61216405" => CRS {
        coord_ref_sys_code: 61216405,
        coord_ref_sys_name: "GGRS87 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4121,
        remarks: "See CRS code 4121 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:66036405" => CRS {
        coord_ref_sys_code: 66036405,
        coord_ref_sys_name: "Grenada 1953 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4603,
        remarks: "See CRS code 4603 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:66466405" => CRS {
        coord_ref_sys_code: 66466405,
        coord_ref_sys_name: "Grand Comoros (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4646,
        remarks: "See CRS code 4640 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:68156405" => CRS {
        coord_ref_sys_code: 68156405,
        coord_ref_sys_name: "Greek (Athens) (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4815,
        remarks: "See CRS code 4815 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:20904" => CRS {
        coord_ref_sys_code: 20904,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 4",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 21E (code 21004).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20905" => CRS {
        coord_ref_sys_code: 20905,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 5",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 27E (code 21005).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20906" => CRS {
        coord_ref_sys_code: 20906,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 6",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 33E (code 21006).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20907" => CRS {
        coord_ref_sys_code: 20907,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 7",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 39E (code 21007).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20908" => CRS {
        coord_ref_sys_code: 20908,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 8",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 45E (code 21008).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20909" => CRS {
        coord_ref_sys_code: 20909,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 9",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 51E (code 21009).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20910" => CRS {
        coord_ref_sys_code: 20910,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 10",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 57E (code 21010).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20911" => CRS {
        coord_ref_sys_code: 20911,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 11",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 63E (code 21011).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20912" => CRS {
        coord_ref_sys_code: 20912,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 12",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 69E (code 21012).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20913" => CRS {
        coord_ref_sys_code: 20913,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 13",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 75E (code 21013).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20914" => CRS {
        coord_ref_sys_code: 20914,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 14",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 81E (code 21014).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20915" => CRS {
        coord_ref_sys_code: 20915,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 15",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 87E (code 21015).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20916" => CRS {
        coord_ref_sys_code: 20916,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 16",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 93E (code 21016).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20917" => CRS {
        coord_ref_sys_code: 20917,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 17",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 99E (code 21017).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20918" => CRS {
        coord_ref_sys_code: 20918,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 18",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 105E (code 21018).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20919" => CRS {
        coord_ref_sys_code: 20919,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 19",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 111E (code 21019).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:4237" => CRS {
        coord_ref_sys_code: 4237,
        coord_ref_sys_name: "HD72",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6237, 
        base_crs_code: -1,
        remarks: "Replaced HD1909 (EPSG CRS code 3819).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2008-08-02",
        deprecated: 0, 
    },

    "EPSG:20920" => CRS {
        coord_ref_sys_code: 20920,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 20",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 117E (code 21020).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20921" => CRS {
        coord_ref_sys_code: 20921,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 21",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 123E (code 21021).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20922" => CRS {
        coord_ref_sys_code: 20922,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 22",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 129E (code 21022).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20923" => CRS {
        coord_ref_sys_code: 20923,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 23",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 135E (code 21023).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20924" => CRS {
        coord_ref_sys_code: 20924,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 24",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 141E (code 21024).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20925" => CRS {
        coord_ref_sys_code: 20925,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 25",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 147E (code 21025).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20926" => CRS {
        coord_ref_sys_code: 20926,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 26",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 153E (code 21026).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20927" => CRS {
        coord_ref_sys_code: 20927,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 27",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 159E (code 21027).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20928" => CRS {
        coord_ref_sys_code: 20928,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 28",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 165E (code 21028).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20929" => CRS {
        coord_ref_sys_code: 20929,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 29",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 171E (code 21029).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20930" => CRS {
        coord_ref_sys_code: 20930,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 30",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 177E (code 21030).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20931" => CRS {
        coord_ref_sys_code: 20931,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 31",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 177W (code 21031).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:20932" => CRS {
        coord_ref_sys_code: 20932,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger zone 32",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Effective from 2021-01-01. Also found with truncated false easting - see GSK-2011 / Gauss-Kruger CM 171W (code 21032).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21004" => CRS {
        coord_ref_sys_code: 21004,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 21E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 4 (code 20904).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21005" => CRS {
        coord_ref_sys_code: 21005,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 27E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 5 (code 20905).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21006" => CRS {
        coord_ref_sys_code: 21006,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 33E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 6 (code 20906).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21007" => CRS {
        coord_ref_sys_code: 21007,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 39E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 7 (code 20907).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21008" => CRS {
        coord_ref_sys_code: 21008,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 45E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 8 (code 20908).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21009" => CRS {
        coord_ref_sys_code: 21009,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 51E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 9 (code 20909).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21010" => CRS {
        coord_ref_sys_code: 21010,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 57E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 10 (code 20910).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21011" => CRS {
        coord_ref_sys_code: 21011,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 63E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 11 (code 20911).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21012" => CRS {
        coord_ref_sys_code: 21012,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 69E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 12 (code 20912).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21013" => CRS {
        coord_ref_sys_code: 21013,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 75E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 13 (code 20913).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21014" => CRS {
        coord_ref_sys_code: 21014,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 81E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 14 (code 20914).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21015" => CRS {
        coord_ref_sys_code: 21015,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 87E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 15 (code 20915).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21016" => CRS {
        coord_ref_sys_code: 21016,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 93E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 16 (code 20916).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21017" => CRS {
        coord_ref_sys_code: 21017,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 99E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 17 (code 20917).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21021" => CRS {
        coord_ref_sys_code: 21021,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 123E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 21 (code 20921).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21022" => CRS {
        coord_ref_sys_code: 21022,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 129E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 22 (code 20922).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:4238" => CRS {
        coord_ref_sys_code: 4238,
        coord_ref_sys_name: "ID74",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6238, 
        base_crs_code: -1,
        remarks: "Replaced by DGN95.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-03-25",
        deprecated: 0, 
    },

    "EPSG:21023" => CRS {
        coord_ref_sys_code: 21023,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 135E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 23 (code 20923).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21024" => CRS {
        coord_ref_sys_code: 21024,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 141E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 24 (code 20924).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21025" => CRS {
        coord_ref_sys_code: 21025,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 147E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 25 (code 20925).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21026" => CRS {
        coord_ref_sys_code: 21026,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 153E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 26 (code 20926).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21027" => CRS {
        coord_ref_sys_code: 21027,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 159E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 27 (code 20927).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21028" => CRS {
        coord_ref_sys_code: 21028,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 165E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 28 (code 20928).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21029" => CRS {
        coord_ref_sys_code: 21029,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 171E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 29 (code 20929).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21030" => CRS {
        coord_ref_sys_code: 21030,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 177E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 30 (code 20930).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21031" => CRS {
        coord_ref_sys_code: 21031,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 177W",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 31 (code 20931).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21032" => CRS {
        coord_ref_sys_code: 21032,
        coord_ref_sys_name: "GSK-2011 / Gauss-Kruger CM 171W",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / Gauss-Kruger zone 32 (code 20932).",
        information_source: "Ministry of Economic Development order N271 of 6th June 2017.",
        data_source: "EPSG",
        revision_date: "2021-06-08",
        deprecated: 0, 
    },

    "EPSG:21307" => CRS {
        coord_ref_sys_code: 21307,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 21E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 7 (code 21207). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21308" => CRS {
        coord_ref_sys_code: 21308,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 24E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 8 (code 21208). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21309" => CRS {
        coord_ref_sys_code: 21309,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 27E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 9 (code 21209). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21310" => CRS {
        coord_ref_sys_code: 21310,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 30E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 10 (code 21210). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21311" => CRS {
        coord_ref_sys_code: 21311,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 33E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 11 (code 21211). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21312" => CRS {
        coord_ref_sys_code: 21312,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 36E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 12 (code 21212). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21313" => CRS {
        coord_ref_sys_code: 21313,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 39E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 13 (code 21213). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21314" => CRS {
        coord_ref_sys_code: 21314,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 42E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 14 (code 21214). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21315" => CRS {
        coord_ref_sys_code: 21315,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 45E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 15 (code 21215). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21316" => CRS {
        coord_ref_sys_code: 21316,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 48E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 16 (code 21216). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21317" => CRS {
        coord_ref_sys_code: 21317,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 51E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 17 (code 21217). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21318" => CRS {
        coord_ref_sys_code: 21318,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 54E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 18 (code 21218). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21319" => CRS {
        coord_ref_sys_code: 21319,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 57E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 19 (code 21219). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21320" => CRS {
        coord_ref_sys_code: 21320,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 60E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 20 (code 21220). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21334" => CRS {
        coord_ref_sys_code: 21334,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 102E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 34 (code 21234). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21335" => CRS {
        coord_ref_sys_code: 21335,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 105E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 35 (code 21235). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:4254" => CRS {
        coord_ref_sys_code: 4254,
        coord_ref_sys_name: "Hito XVIII 1963",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6254, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2010-03-31",
        deprecated: 0, 
    },

    "EPSG:21336" => CRS {
        coord_ref_sys_code: 21336,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 108E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 36 (code 21226). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21337" => CRS {
        coord_ref_sys_code: 21337,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 111E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 37 (code 21237). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21338" => CRS {
        coord_ref_sys_code: 21338,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 114E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 38 (code 21238). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21339" => CRS {
        coord_ref_sys_code: 21339,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 117E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 39 (code 21239). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21340" => CRS {
        coord_ref_sys_code: 21340,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 120E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 40 (code 21240). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21341" => CRS {
        coord_ref_sys_code: 21341,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 123E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 41 (code 21241). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21342" => CRS {
        coord_ref_sys_code: 21342,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 126E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 42 (code 21242). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21343" => CRS {
        coord_ref_sys_code: 21343,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 129E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 43 (code 21243). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21344" => CRS {
        coord_ref_sys_code: 21344,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 132E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 44 (code 21244). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21345" => CRS {
        coord_ref_sys_code: 21345,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 135E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 45 (code 21245). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21346" => CRS {
        coord_ref_sys_code: 21346,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 138E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 46 (code 21246). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21347" => CRS {
        coord_ref_sys_code: 21347,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 141E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 47 (code 21247). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21348" => CRS {
        coord_ref_sys_code: 21348,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 144E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 48 (code 21248). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21349" => CRS {
        coord_ref_sys_code: 21349,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 147E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 49 (code 21249). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21350" => CRS {
        coord_ref_sys_code: 21350,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 150E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 50 (code 21250). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21351" => CRS {
        coord_ref_sys_code: 21351,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 153E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 51 (code 21251). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21352" => CRS {
        coord_ref_sys_code: 21352,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 156E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 52 (code 21252). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21353" => CRS {
        coord_ref_sys_code: 21353,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 159E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 53 (code 21253). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21354" => CRS {
        coord_ref_sys_code: 21354,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 162E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 54 (code 21254). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21355" => CRS {
        coord_ref_sys_code: 21355,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 165E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 55 (code 21255). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21356" => CRS {
        coord_ref_sys_code: 21356,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 168E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 56 (code 21256). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21357" => CRS {
        coord_ref_sys_code: 21357,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 171E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 57 (code 21257). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21358" => CRS {
        coord_ref_sys_code: 21358,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 174E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 58 (code 21258). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:4255" => CRS {
        coord_ref_sys_code: 4255,
        coord_ref_sys_name: "Herat North",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6255, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:21359" => CRS {
        coord_ref_sys_code: 21359,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 177E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 59 (code 21259). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21360" => CRS {
        coord_ref_sys_code: 21360,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 180E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 60 (code 21260). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21361" => CRS {
        coord_ref_sys_code: 21361,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 177W",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 61 (code 21261). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21362" => CRS {
        coord_ref_sys_code: 21362,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 174W",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 62 (code 21262). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21363" => CRS {
        coord_ref_sys_code: 21363,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 171W",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 63 (code 21263). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21364" => CRS {
        coord_ref_sys_code: 21364,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 168W",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 64 (code 21264). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:2044" => CRS {
        coord_ref_sys_code: 2044,
        coord_ref_sys_name: "Hanoi 1972 / Gauss-Kruger zone 18",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4147,
        remarks: "Replaces use of Indian 1960 / UTM zone 48 after 1988. Replaced by VN-2000 / UTM zone 48N (CRS code 3405).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2006-09-13",
        deprecated: 0, 
    },

    "EPSG:2045" => CRS {
        coord_ref_sys_code: 2045,
        coord_ref_sys_name: "Hanoi 1972 / Gauss-Kruger zone 19",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4147,
        remarks: "Replaces use of Indian 1960 / UTM zone 49 after 1988. Replaced by VN-2000 / UTM zone 48N (CRS code 3406).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2006-09-13",
        deprecated: 0, 
    },

    "EPSG:2093" => CRS {
        coord_ref_sys_code: 2093,
        coord_ref_sys_name: "Hanoi 1972 / GK 106 NE",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4147,
        remarks: "",
        information_source: "BP Amoco",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2970" => CRS {
        coord_ref_sys_code: 2970,
        coord_ref_sys_name: "Guadeloupe 1948 / UTM zone 20N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4622,
        remarks: "Replaced by RRAF 1991 / UTM zone 20N (CRS code 4559).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2009-11-24",
        deprecated: 0, 
    },

    "EPSG:3106" => CRS {
        coord_ref_sys_code: 3106,
        coord_ref_sys_name: "Gulshan 303 / TM 90 NE",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4682,
        remarks: "Used by Survey of Bangladesh from 1995 to 2009, after which replaced by WGS 84 / TM 90 NE (CRS code 9680). See Gulshan 303 / BTM (code 9678) for other uses including water resource management.",
        information_source: "Survey of Bangladesh via IGN Paris and Tullow Oil.",
        data_source: "EPSG",
        revision_date: "2021-01-13",
        deprecated: 0, 
    },

    "EPSG:3295" => CRS {
        coord_ref_sys_code: 3295,
        coord_ref_sys_name: "Guam 1963 / Yap Islands",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4675,
        remarks: "",
        information_source: "Yap State Code, Title 20, Chapter 5. http://www.fsmlaw.org/yap/code/title20/T20_Ch05.htm",
        data_source: "EPSG",
        revision_date: "2005-05-21",
        deprecated: 0, 
    },

    "EPSG:3993" => CRS {
        coord_ref_sys_code: 3993,
        coord_ref_sys_name: "Guam 1963 / Guam SPCS",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4675,
        remarks: "Replaced by NAD83(HARN) / Guam Map Grid (CRS code 4414) from 1995.",
        information_source: "US National Geodetic Survey",
        data_source: "EPSG",
        revision_date: "2009-07-29",
        deprecated: 0, 
    },

    "EPSG:4147" => CRS {
        coord_ref_sys_code: 4147,
        coord_ref_sys_name: "Hanoi 1972",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6147, 
        base_crs_code: -1,
        remarks: "Replaces use of Indian 1960. Replaced by VN-2000 (CRS code 4756).",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2006-09-13",
        deprecated: 0, 
    },

    "EPSG:4148" => CRS {
        coord_ref_sys_code: 4148,
        coord_ref_sys_name: "Hartebeesthoek94",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6148, 
        base_crs_code: 4941,
        remarks: "Replaces Cape (code 4222) from 1999.",
        information_source: "EPSG. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2019-01-14",
        deprecated: 0, 
    },

    "EPSG:4235" => CRS {
        coord_ref_sys_code: 4235,
        coord_ref_sys_name: "Guyane Francaise",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6402,
        datum_code: 6235, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 1, 
    },

    "EPSG:4349" => CRS {
        coord_ref_sys_code: 4349,
        coord_ref_sys_name: "Hartebeesthoek94 (3D)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6401,
        datum_code: 6148, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping; http://w3sli.wcape.gov.za/surveys/mapping/wgs84.htm",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:4622" => CRS {
        coord_ref_sys_code: 4622,
        coord_ref_sys_name: "Guadeloupe 1948",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6622, 
        base_crs_code: -1,
        remarks: "Replaced by RRAF 1991 (CRS code 4558).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2009-11-24",
        deprecated: 0, 
    },

    "EPSG:4675" => CRS {
        coord_ref_sys_code: 4675,
        coord_ref_sys_name: "Guam 1963",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6675, 
        base_crs_code: -1,
        remarks: "Replaced by NAD83(HARN) alias Guam Geodetic Network 1993 (CRS code 4152) from 1995.",
        information_source: "National Geodetic Survey",
        data_source: "EPSG",
        revision_date: "2018-08-29",
        deprecated: 0, 
    },

    "EPSG:4682" => CRS {
        coord_ref_sys_code: 4682,
        coord_ref_sys_name: "Gulshan 303",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6682, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Survey of Bangladesh via IGN Paris and Tullow Oil.",
        data_source: "EPSG",
        revision_date: "2006-06-22",
        deprecated: 0, 
    },

    "EPSG:4940" => CRS {
        coord_ref_sys_code: 4940,
        coord_ref_sys_name: "Hartebeesthoek94",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6148, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping; http://w3sli.wcape.gov.za/surveys/mapping/wgs84.htm",
        data_source: "EPSG",
        revision_date: "2019-01-14",
        deprecated: 0, 
    },

    "EPSG:4941" => CRS {
        coord_ref_sys_code: 4941,
        coord_ref_sys_name: "Hartebeesthoek94",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6148, 
        base_crs_code: 4940,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping; http://w3sli.wcape.gov.za/surveys/mapping/wgs84.htm",
        data_source: "EPSG",
        revision_date: "2019-01-14",
        deprecated: 0, 
    },

    "EPSG:5726" => CRS {
        coord_ref_sys_code: 5726,
        coord_ref_sys_name: "Ha Tien 1960 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5125, 
        base_crs_code: -1,
        remarks: "In Vietnam replaced by Hon Dau height (CRS code 5727) from 1992.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5757" => CRS {
        coord_ref_sys_code: 5757,
        coord_ref_sys_name: "Guadeloupe 1988 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5155, 
        base_crs_code: -1,
        remarks: "Replaces Guadeloupe 1951 height (CRS code 5795).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5795" => CRS {
        coord_ref_sys_code: 5795,
        coord_ref_sys_name: "Guadeloupe 1951 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5193, 
        base_crs_code: -1,
        remarks: "Replaced by Guadeloupe 1988 height (CRS code 5757).",
        information_source: "IGN Paris.",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:6639" => CRS {
        coord_ref_sys_code: 6639,
        coord_ref_sys_name: "Guam 1963 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1122, 
        base_crs_code: -1,
        remarks: "Replaced by GUVD04 height (CRS code 6644).",
        information_source: "National Geodetic Survey, http://www.ngs.noaa.gov",
        data_source: "EPSG",
        revision_date: "2013-10-25",
        deprecated: 0, 
    },

    "EPSG:6644" => CRS {
        coord_ref_sys_code: 6644,
        coord_ref_sys_name: "GUVD04 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1126, 
        base_crs_code: -1,
        remarks: "Replaces Guam 1963 height (CRS code 6639).",
        information_source: "National Geodetic Survey, http://www.ngs.noaa.gov",
        data_source: "EPSG",
        revision_date: "2019-09-17",
        deprecated: 0, 
    },

    "EPSG:8042" => CRS {
        coord_ref_sys_code: 8042,
        coord_ref_sys_name: "Gusterberg (Ferro)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1188, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Land Survey Office (ZU), Prague, www.cuzk.cz/zu",
        data_source: "EPSG",
        revision_date: "2017-03-03",
        deprecated: 0, 
    },

    "EPSG:8044" => CRS {
        coord_ref_sys_code: 8044,
        coord_ref_sys_name: "Gusterberg Grid (Ferro)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6501,
        datum_code: -1, 
        base_crs_code: 8042,
        remarks: "",
        information_source: "Land Survey Office (ZU), Prague, www.cuzk.cz/zu",
        data_source: "EPSG",
        revision_date: "2017-03-03",
        deprecated: 0, 
    },

    "EPSG:8266" => CRS {
        coord_ref_sys_code: 8266,
        coord_ref_sys_name: "GVR2000 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1199, 
        base_crs_code: -1,
        remarks: "Replaced by GVR2016 height (CRS code 8267). GVR2000 is realized by gravimetric geoid model 2000 (transformation code 8268) applied to GR96 (CRS code 4909).",
        information_source: "Agency for Data Supply and Efficiency (SDFE), Denmark.",
        data_source: "EPSG",
        revision_date: "2017-09-05",
        deprecated: 0, 
    },

    "EPSG:8267" => CRS {
        coord_ref_sys_code: 8267,
        coord_ref_sys_name: "GVR2016 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1200, 
        base_crs_code: -1,
        remarks: "Replaces GVR2000 height (CRS code 8266). GVR2016 is realized by gravimetric geoid model 2016 (transformation code 8269) applied to GR96 (CRS code 4909).",
        information_source: "Agency for Data Supply and Efficiency (SDFE), Denmark.",
        data_source: "EPSG",
        revision_date: "2017-09-05",
        deprecated: 0, 
    },

    "EPSG:9678" => CRS {
        coord_ref_sys_code: 9678,
        coord_ref_sys_name: "Gulshan 303 / Bangladesh Transverse Mercator",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4682,
        remarks: "See Gulshan 303 / TM 90 NE (CRS code 3106) for topographic mapping and offshore oil and gas activities.",
        information_source: "Survey of Bangladesh.",
        data_source: "EPSG",
        revision_date: "2021-01-13",
        deprecated: 0, 
    },

    "EPSG:21207" => CRS {
        coord_ref_sys_code: 21207,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 7",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 21E (code 21307). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:4350" => CRS {
        coord_ref_sys_code: 4350,
        coord_ref_sys_name: "Hartebeesthoek94 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6148, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping; http://w3sli.wcape.gov.za/surveys/mapping/wgs84.htm",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:21208" => CRS {
        coord_ref_sys_code: 21208,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 8",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 24E (code 21308). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21209" => CRS {
        coord_ref_sys_code: 21209,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 9",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 27E (code 21309). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21210" => CRS {
        coord_ref_sys_code: 21210,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 10",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 30E (code 21310). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21211" => CRS {
        coord_ref_sys_code: 21211,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 11",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 33E (code 21311). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21212" => CRS {
        coord_ref_sys_code: 21212,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 12",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 36E (code 21312). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21213" => CRS {
        coord_ref_sys_code: 21213,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 13",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 39E (code 21313). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21214" => CRS {
        coord_ref_sys_code: 21214,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 14",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 42E (code 21314). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21215" => CRS {
        coord_ref_sys_code: 21215,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 15",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 45E (code 21315). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21216" => CRS {
        coord_ref_sys_code: 21216,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 16",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 48E (code 21316). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21217" => CRS {
        coord_ref_sys_code: 21217,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 17",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 51E (code 21317). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21218" => CRS {
        coord_ref_sys_code: 21218,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 18",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 54E (code 21318). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21219" => CRS {
        coord_ref_sys_code: 21219,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 19",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 57E (code 21319). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21220" => CRS {
        coord_ref_sys_code: 21220,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 20",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 60E (code 21320). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21221" => CRS {
        coord_ref_sys_code: 21221,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 21",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 63E (code 21321). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21222" => CRS {
        coord_ref_sys_code: 21222,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 22",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 66E (code 21322). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21223" => CRS {
        coord_ref_sys_code: 21223,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 23",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 69E (code 21323). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21224" => CRS {
        coord_ref_sys_code: 21224,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 24",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 72E (code 21324). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21225" => CRS {
        coord_ref_sys_code: 21225,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 25",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 75E (code 21325). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21226" => CRS {
        coord_ref_sys_code: 21226,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 26",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 78E (code 21326). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21227" => CRS {
        coord_ref_sys_code: 21227,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 27",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 81E (code 21327). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21228" => CRS {
        coord_ref_sys_code: 21228,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 28",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 84E (code 21328). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:9222" => CRS {
        coord_ref_sys_code: 9222,
        coord_ref_sys_name: "Hartebeesthoek94 / ZAF BSU Albers 44E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4148,
        remarks: "",
        information_source: "Statistics South Africa.",
        data_source: "EPSG",
        revision_date: "2019-09-06",
        deprecated: 0, 
    },

    "EPSG:21229" => CRS {
        coord_ref_sys_code: 21229,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 29",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 87E (code 21329). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21230" => CRS {
        coord_ref_sys_code: 21230,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 30",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 90E (code 21330). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21231" => CRS {
        coord_ref_sys_code: 21231,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 31",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 93E (code 21331). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21232" => CRS {
        coord_ref_sys_code: 21232,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 32",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 96E (code 21332). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21233" => CRS {
        coord_ref_sys_code: 21233,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 33",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 99E (code 21333). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21234" => CRS {
        coord_ref_sys_code: 21234,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 34",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 102E (code 21334). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21235" => CRS {
        coord_ref_sys_code: 21235,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 35",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 105E (code 21335). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21236" => CRS {
        coord_ref_sys_code: 21236,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 36",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 108E (code 21336). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21237" => CRS {
        coord_ref_sys_code: 21237,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 37",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 111E (code 21337). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21238" => CRS {
        coord_ref_sys_code: 21238,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 38",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 114E (code 21338). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21239" => CRS {
        coord_ref_sys_code: 21239,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 39",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 117E (code 21339). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21240" => CRS {
        coord_ref_sys_code: 21240,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 40",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 120E (code 21340). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21241" => CRS {
        coord_ref_sys_code: 21241,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 41",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 123E (code 21341). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21242" => CRS {
        coord_ref_sys_code: 21242,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 42",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 126E (code 21342). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21243" => CRS {
        coord_ref_sys_code: 21243,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 43",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 129E (code 21343). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21244" => CRS {
        coord_ref_sys_code: 21244,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 44",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 132E (code 21344). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21245" => CRS {
        coord_ref_sys_code: 21245,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 45",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 135E (code 21345). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21246" => CRS {
        coord_ref_sys_code: 21246,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 46",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 138E (code 21346). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21247" => CRS {
        coord_ref_sys_code: 21247,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 47",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 141E (code 21347). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21248" => CRS {
        coord_ref_sys_code: 21248,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 48",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 144E (code 21348). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21249" => CRS {
        coord_ref_sys_code: 21249,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 49",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 147E (code 21349). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:24048" => CRS {
        coord_ref_sys_code: 24048,
        coord_ref_sys_name: "Indian 1975 / UTM zone 48N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4240,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:21250" => CRS {
        coord_ref_sys_code: 21250,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 50",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 150E (code 21350). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21251" => CRS {
        coord_ref_sys_code: 21251,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 51",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 153E (code 21351). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21252" => CRS {
        coord_ref_sys_code: 21252,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 52",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 156E (code 21352). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21253" => CRS {
        coord_ref_sys_code: 21253,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 53",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 159E (code 21353). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21254" => CRS {
        coord_ref_sys_code: 21254,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 54",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 162E (code 21354). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21255" => CRS {
        coord_ref_sys_code: 21255,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 55",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 165E (code 21355). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21256" => CRS {
        coord_ref_sys_code: 21256,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 56",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 168E (code 21356). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21257" => CRS {
        coord_ref_sys_code: 21257,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 57",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 171E (code 21357). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21258" => CRS {
        coord_ref_sys_code: 21258,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 58",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 174E (code 21358). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21259" => CRS {
        coord_ref_sys_code: 21259,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 59",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 177E (code 21359). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21260" => CRS {
        coord_ref_sys_code: 21260,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 60",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 180E (code 21360). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21261" => CRS {
        coord_ref_sys_code: 21261,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 61",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 177W (code 21361). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21262" => CRS {
        coord_ref_sys_code: 21262,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 62",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 174W (code 21362). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21263" => CRS {
        coord_ref_sys_code: 21263,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 63",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 171W (code 21363). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21264" => CRS {
        coord_ref_sys_code: 21264,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK zone 64",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Also found with truncated false easting - see GSK-2011 / GSK 3GK CM 168W (code 21364). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21321" => CRS {
        coord_ref_sys_code: 21321,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 63E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 21 (code 21221). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21322" => CRS {
        coord_ref_sys_code: 21322,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 66E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 22 (code 21222). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21323" => CRS {
        coord_ref_sys_code: 21323,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 69E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 23 (code 21223). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21324" => CRS {
        coord_ref_sys_code: 21324,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 72E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 24 (code 21224). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21325" => CRS {
        coord_ref_sys_code: 21325,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 75E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 25 (code 21225). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21326" => CRS {
        coord_ref_sys_code: 21326,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 78E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 26 (code 21226). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21327" => CRS {
        coord_ref_sys_code: 21327,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 81E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 27 (code 21227). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21328" => CRS {
        coord_ref_sys_code: 21328,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 84E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 28 (code 21228). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21329" => CRS {
        coord_ref_sys_code: 21329,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 87E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 29 (code 21229). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21330" => CRS {
        coord_ref_sys_code: 21330,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 90E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 30 (code 21230). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21331" => CRS {
        coord_ref_sys_code: 21331,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 93E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 31 (code 21231). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21332" => CRS {
        coord_ref_sys_code: 21332,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 96E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 32 (code 21232). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:21333" => CRS {
        coord_ref_sys_code: 21333,
        coord_ref_sys_name: "GSK-2011 / GSK 3GK CM 99E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 7683,
        remarks: "Truncated form of GSK-2011 / GSK 3GK zone 33 (code 21233). Replaces Pulkovo-based CRSs from 2021-01-01.",
        information_source: "Ministry of Economic Development of the Russian Federation, order N271 of 6th June 2017 as amended 11th December 2017.",
        data_source: "EPSG",
        revision_date: "2021-07-06",
        deprecated: 0, 
    },

    "EPSG:61476405" => CRS {
        coord_ref_sys_code: 61476405,
        coord_ref_sys_name: "Hanoi 1972 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4147,
        remarks: "See CRS code 4147 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61486405" => CRS {
        coord_ref_sys_code: 61486405,
        coord_ref_sys_name: "Hartebeesthoek94 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4148,
        remarks: "See CRS code 4148 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61486413" => CRS {
        coord_ref_sys_code: 61486413,
        coord_ref_sys_name: "Hartebeesthoek94 (3D deg)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6413,
        datum_code: -1, 
        base_crs_code: 4349,
        remarks: "See CRS code 4349 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:2046" => CRS {
        coord_ref_sys_code: 2046,
        coord_ref_sys_name: "Hartebeesthoek94 / Lo15",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4148,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping; http://w3sli.wcape.gov.za/surveys/mapping/wgs84.htm",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2047" => CRS {
        coord_ref_sys_code: 2047,
        coord_ref_sys_name: "Hartebeesthoek94 / Lo17",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4148,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping; http://w3sli.wcape.gov.za/surveys/mapping/wgs84.htm",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2048" => CRS {
        coord_ref_sys_code: 2048,
        coord_ref_sys_name: "Hartebeesthoek94 / Lo19",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4148,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping; http://w3sli.wcape.gov.za/surveys/mapping/wgs84.htm",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2049" => CRS {
        coord_ref_sys_code: 2049,
        coord_ref_sys_name: "Hartebeesthoek94 / Lo21",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4148,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping; http://w3sli.wcape.gov.za/surveys/mapping/wgs84.htm",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2050" => CRS {
        coord_ref_sys_code: 2050,
        coord_ref_sys_name: "Hartebeesthoek94 / Lo23",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4148,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping; http://w3sli.wcape.gov.za/surveys/mapping/wgs84.htm",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2051" => CRS {
        coord_ref_sys_code: 2051,
        coord_ref_sys_name: "Hartebeesthoek94 / Lo25",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4148,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping; http://w3sli.wcape.gov.za/surveys/mapping/wgs84.htm",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2052" => CRS {
        coord_ref_sys_code: 2052,
        coord_ref_sys_name: "Hartebeesthoek94 / Lo27",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4148,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping; http://w3sli.wcape.gov.za/surveys/mapping/wgs84.htm",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2053" => CRS {
        coord_ref_sys_code: 2053,
        coord_ref_sys_name: "Hartebeesthoek94 / Lo29",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4148,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping; http://w3sli.wcape.gov.za/surveys/mapping/wgs84.htm",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2054" => CRS {
        coord_ref_sys_code: 2054,
        coord_ref_sys_name: "Hartebeesthoek94 / Lo31",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4148,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping; http://w3sli.wcape.gov.za/surveys/mapping/wgs84.htm",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2055" => CRS {
        coord_ref_sys_code: 2055,
        coord_ref_sys_name: "Hartebeesthoek94 / Lo33",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 6503,
        datum_code: -1, 
        base_crs_code: 4148,
        remarks: "",
        information_source: "Directorate of Surveys and Mapping; http://w3sli.wcape.gov.za/surveys/mapping/wgs84.htm",
        data_source: "EPSG",
        revision_date: "2000-03-07",
        deprecated: 0, 
    },

    "EPSG:2083" => CRS {
        coord_ref_sys_code: 2083,
        coord_ref_sys_name: "Hito XVIII 1963 / Argentina 2",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4254,
        remarks: "",
        information_source: "Total-Fina",
        data_source: "EPSG",
        revision_date: "2010-03-31",
        deprecated: 0, 
    },

    "EPSG:2084" => CRS {
        coord_ref_sys_code: 2084,
        coord_ref_sys_name: "Hito XVIII 1963 / UTM zone 19S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4254,
        remarks: "",
        information_source: "Total-Fina",
        data_source: "EPSG",
        revision_date: "2010-11-02",
        deprecated: 0, 
    },

    "EPSG:2326" => CRS {
        coord_ref_sys_code: 2326,
        coord_ref_sys_name: "Hong Kong 1980 Grid System",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4611,
        remarks: "Replaces Hong Kong 1963 Grid System.",
        information_source: "Survey and Mapping Office, Lands Department. Http://www.info.gov.hk/landsd/mapping/tindex.htm",
        data_source: "EPSG",
        revision_date: "2006-06-22",
        deprecated: 0, 
    },

    "EPSG:3053" => CRS {
        coord_ref_sys_code: 3053,
        coord_ref_sys_name: "Hjorsey 1955 / Lambert 1955",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4491,
        datum_code: -1, 
        base_crs_code: 4658,
        remarks: "Replaces Reykjavik 1900 / Lambert 1900 (CRS code 3052). Replaced by ISN93 / Lambert 1993 (CRS code 3057).",
        information_source: "Landmaelingar Islands (National Survey of Iceland).",
        data_source: "EPSG",
        revision_date: "2003-06-27",
        deprecated: 0, 
    },

    "EPSG:3054" => CRS {
        coord_ref_sys_code: 3054,
        coord_ref_sys_name: "Hjorsey 1955 / UTM zone 26N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4658,
        remarks: "Replaced by ISN93 / Lambert 1993 (CRS code 3057).",
        information_source: "Landmaelingar Islands (National Survey of Iceland).",
        data_source: "EPSG",
        revision_date: "2003-06-27",
        deprecated: 0, 
    },

    "EPSG:3055" => CRS {
        coord_ref_sys_code: 3055,
        coord_ref_sys_name: "Hjorsey 1955 / UTM zone 27N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4658,
        remarks: "Replaced by ISN93 / Lambert 1993 (CRS code 3057).",
        information_source: "Landmaelingar Islands (National Survey of Iceland).",
        data_source: "EPSG",
        revision_date: "2003-06-27",
        deprecated: 0, 
    },

    "EPSG:3056" => CRS {
        coord_ref_sys_code: 3056,
        coord_ref_sys_name: "Hjorsey 1955 / UTM zone 28N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4658,
        remarks: "Replaced by ISN93 / Lambert 1993 (CRS code 3057).",
        information_source: "Landmaelingar Islands (National Survey of Iceland).",
        data_source: "EPSG",
        revision_date: "2003-06-27",
        deprecated: 0, 
    },

    "EPSG:3058" => CRS {
        coord_ref_sys_code: 3058,
        coord_ref_sys_name: "Helle 1954 / Jan Mayen Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4531,
        datum_code: -1, 
        base_crs_code: 4660,
        remarks: "",
        information_source: "Statens kartverk",
        data_source: "EPSG",
        revision_date: "2003-06-27",
        deprecated: 0, 
    },

    "EPSG:3366" => CRS {
        coord_ref_sys_code: 3366,
        coord_ref_sys_name: "Hong Kong 1963 Grid System",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4738,
        remarks: "Superseded by Hong Kong 1980 Grid System (CRS code 2326).",
        information_source: "Survey and Mapping Office, Lands Department. Http://www.info.gov.hk/landsd",
        data_source: "EPSG",
        revision_date: "2006-02-06",
        deprecated: 1, 
    },

    "EPSG:3407" => CRS {
        coord_ref_sys_code: 3407,
        coord_ref_sys_name: "Hong Kong 1963 Grid System",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4502,
        datum_code: -1, 
        base_crs_code: 4738,
        remarks: "Replaced by Hong Kong 1980 Grid System (CRS code 2326).",
        information_source: "Survey and Mapping Office, Lands Department. Http://www.info.gov.hk/landsd",
        data_source: "EPSG",
        revision_date: "2006-09-25",
        deprecated: 0, 
    },

    "EPSG:3765" => CRS {
        coord_ref_sys_code: 3765,
        coord_ref_sys_name: "HTRS96 / Croatia TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4761,
        remarks: "",
        information_source: "State Geodetic Administration of the Republic of Croatia.",
        data_source: "EPSG",
        revision_date: "2007-09-29",
        deprecated: 0, 
    },

    "EPSG:3766" => CRS {
        coord_ref_sys_code: 3766,
        coord_ref_sys_name: "HTRS96 / Croatia LCC",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4761,
        remarks: "",
        information_source: "State Geodetic Administration of the Republic of Croatia.",
        data_source: "EPSG",
        revision_date: "2007-09-29",
        deprecated: 0, 
    },

    "EPSG:3767" => CRS {
        coord_ref_sys_code: 3767,
        coord_ref_sys_name: "HTRS96 / UTM zone 33N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4761,
        remarks: "",
        information_source: "State Geodetic Administration of the Republic of Croatia.",
        data_source: "EPSG",
        revision_date: "2007-09-29",
        deprecated: 0, 
    },

    "EPSG:3768" => CRS {
        coord_ref_sys_code: 3768,
        coord_ref_sys_name: "HTRS96 / UTM zone 34N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4761,
        remarks: "",
        information_source: "State Geodetic Administration of the Republic of Croatia.",
        data_source: "EPSG",
        revision_date: "2007-09-29",
        deprecated: 0, 
    },

    "EPSG:3819" => CRS {
        coord_ref_sys_code: 3819,
        coord_ref_sys_name: "HD1909",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1024, 
        base_crs_code: -1,
        remarks: "Replaced earlier HD1863 adjustment also on Bessel ellipsoid. Both HD1863 and HD1909 were originally on Ferro Prime Meridian but subsequently converted to Greenwich. Replaced by HD72 (CRS code 4237).",
        information_source: "Timár, Molnár and Márta in Geodezia es Kartografia 55(3) pp16-21. www.fomi.hu/internet/magyar/szaklap/geodkart.htm",
        data_source: "EPSG",
        revision_date: "2008-08-02",
        deprecated: 0, 
    },

    "EPSG:4611" => CRS {
        coord_ref_sys_code: 4611,
        coord_ref_sys_name: "Hong Kong 1980",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6611, 
        base_crs_code: -1,
        remarks: "Replaces Hong Kong 1963 and Hong Kong 1963(67).",
        information_source: "Survey and Mapping Office, Lands Department. Http://www.info.gov.hk/landsd/mapping/tindex.htm",
        data_source: "EPSG",
        revision_date: "2006-02-06",
        deprecated: 0, 
    },

    "EPSG:4658" => CRS {
        coord_ref_sys_code: 4658,
        coord_ref_sys_name: "Hjorsey 1955",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6658, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Landmaelingar Islands (National Survey of Iceland).",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4660" => CRS {
        coord_ref_sys_code: 4660,
        coord_ref_sys_name: "Helle 1954",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6660, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Statens kartverk",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4738" => CRS {
        coord_ref_sys_code: 4738,
        coord_ref_sys_name: "Hong Kong 1963",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6738, 
        base_crs_code: -1,
        remarks: "Replaced by Hong Kong 1963(67) (CRS code 4839) for military purposes only. For all purposes, replaced by Hong Kong 1980 (CRS code 4611).",
        information_source: "Survey and Mapping Office, Lands Department. Http://www.info.gov.hk/landsd/",
        data_source: "EPSG",
        revision_date: "2006-02-03",
        deprecated: 0, 
    },

    "EPSG:4739" => CRS {
        coord_ref_sys_code: 4739,
        coord_ref_sys_name: "Hong Kong 1963(67)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6739, 
        base_crs_code: -1,
        remarks: "For military purposes only, replaces Hong Kong 1963. Replaced by Hong Kong 1980 (CRS code 4611).",
        information_source: "UK Hydrographic Office.",
        data_source: "EPSG",
        revision_date: "2006-02-03",
        deprecated: 0, 
    },

    "EPSG:4761" => CRS {
        coord_ref_sys_code: 4761,
        coord_ref_sys_name: "HTRS96",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6761, 
        base_crs_code: 4889,
        remarks: "",
        information_source: "OGP. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2007-09-29",
        deprecated: 0, 
    },

    "EPSG:4888" => CRS {
        coord_ref_sys_code: 4888,
        coord_ref_sys_name: "HTRS96",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6761, 
        base_crs_code: -1,
        remarks: "",
        information_source: "State Geodetic Administration of the Republic of Croatia.",
        data_source: "EPSG",
        revision_date: "2007-09-29",
        deprecated: 0, 
    },

    "EPSG:4889" => CRS {
        coord_ref_sys_code: 4889,
        coord_ref_sys_name: "HTRS96",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6761, 
        base_crs_code: 4888,
        remarks: "",
        information_source: "State Geodetic Administration of the Republic of Croatia.",
        data_source: "EPSG",
        revision_date: "2007-09-29",
        deprecated: 0, 
    },

    "EPSG:5605" => CRS {
        coord_ref_sys_code: 5605,
        coord_ref_sys_name: "Huahine SAU 2001 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5200, 
        base_crs_code: -1,
        remarks: "Part of NGPF (CRS code 5600).",
        information_source: "Gouvernement de la Polynésie Française, Service de l&#x27;Urbanisme, Section topographie.",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5610" => CRS {
        coord_ref_sys_code: 5610,
        coord_ref_sys_name: "HVRS71 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5207, 
        base_crs_code: -1,
        remarks: "Replaces Trieste height (CRS code 5195).",
        information_source: "State Geodetic Administration of the Republic of Croatia.",
        data_source: "EPSG",
        revision_date: "2020-07-24",
        deprecated: 0, 
    },

    "EPSG:5727" => CRS {
        coord_ref_sys_code: 5727,
        coord_ref_sys_name: "Hon Dau 1992 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5126, 
        base_crs_code: -1,
        remarks: "In Vietnam replaces Ha Tien height (CRS code 5726) from 1992.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-05-08",
        deprecated: 0, 
    },

    "EPSG:5738" => CRS {
        coord_ref_sys_code: 5738,
        coord_ref_sys_name: "HKPD height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5135, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Survey and Mapping Office, Lands Department. Http://www.info.gov.hk/landsd/mapping/tindex.htm",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5739" => CRS {
        coord_ref_sys_code: 5739,
        coord_ref_sys_name: "HKCD depth",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6498,
        datum_code: 5136, 
        base_crs_code: -1,
        remarks: "Chart datum is 0.15 metres below Hong Kong Principal Datum (CRS code 5738) and 1.38m below MSL at Quarry Bay.",
        information_source: "Survey and Mapping Office, Lands Department. Http://www.info.gov.hk/landsd/mapping/tindex.htm",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5871" => CRS {
        coord_ref_sys_code: 5871,
        coord_ref_sys_name: "HHWLT height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1084, 
        base_crs_code: -1,
        remarks: "Not specific to any location or epoch.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-08-05",
        deprecated: 0, 
    },

    "EPSG:5872" => CRS {
        coord_ref_sys_code: 5872,
        coord_ref_sys_name: "HAT height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1082, 
        base_crs_code: -1,
        remarks: "Not specific to any location or epoch.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2020-08-31",
        deprecated: 0, 
    },

    "EPSG:5874" => CRS {
        coord_ref_sys_code: 5874,
        coord_ref_sys_name: "High Water height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1094, 
        base_crs_code: -1,
        remarks: "Not specific to any location or epoch.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-08-10",
        deprecated: 0, 
    },

    "EPSG:6181" => CRS {
        coord_ref_sys_code: 6181,
        coord_ref_sys_name: "Horta height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1104, 
        base_crs_code: -1,
        remarks: "",
        information_source: "General Directorate of Land, http://www.dgterritorio.pt/",
        data_source: "EPSG",
        revision_date: "2013-03-15",
        deprecated: 0, 
    },

    "EPSG:6981" => CRS {
        coord_ref_sys_code: 6981,
        coord_ref_sys_name: "IG05 Intermediate CRS",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1142, 
        base_crs_code: -1,
        remarks: "Intermediate system not used for spatial referencing - use IGD05 (CRS code 6978).",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2014-11-19",
        deprecated: 0, 
    },

    "EPSG:6982" => CRS {
        coord_ref_sys_code: 6982,
        coord_ref_sys_name: "IG05 Intermediate CRS",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1142, 
        base_crs_code: 6981,
        remarks: "Intermediate system not used for spatial referencing - use IGD05 (CRS code 6979).",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2014-11-19",
        deprecated: 0, 
    },

    "EPSG:6983" => CRS {
        coord_ref_sys_code: 6983,
        coord_ref_sys_name: "IG05 Intermediate CRS",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1142, 
        base_crs_code: 6982,
        remarks: "Intermediate system not used for spatial referencing - use IGD05 (CRS code 6980). Referred to in Israeli documentation as &quot;in GRS80&quot;.",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2014-11-19",
        deprecated: 0, 
    },

    "EPSG:6988" => CRS {
        coord_ref_sys_code: 6988,
        coord_ref_sys_name: "IG05/12 Intermediate CRS",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1144, 
        base_crs_code: -1,
        remarks: "Intermediate system not used for spatial referencing - use IGD05/12 (CRS code 6985).",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2014-11-19",
        deprecated: 0, 
    },

    "EPSG:6989" => CRS {
        coord_ref_sys_code: 6989,
        coord_ref_sys_name: "IG05/12 Intermediate CRS",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1144, 
        base_crs_code: 6988,
        remarks: "Intermediate system not used for spatial referencing - use IGD05/12 (CRS code 6986).",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2014-11-19",
        deprecated: 0, 
    },

    "EPSG:6990" => CRS {
        coord_ref_sys_code: 6990,
        coord_ref_sys_name: "IG05/12 Intermediate CRS",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1144, 
        base_crs_code: 6989,
        remarks: "Intermediate system not used for spatial referencing - use IGD05/12 (CRS code 6987). Referred to in Israeli documentation as &quot;in GRS80&quot;.",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2014-11-19",
        deprecated: 0, 
    },

    "EPSG:7976" => CRS {
        coord_ref_sys_code: 7976,
        coord_ref_sys_name: "HKPD depth",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6498,
        datum_code: -1, 
        base_crs_code: 5738,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-04-20",
        deprecated: 0, 
    },

    "EPSG:8425" => CRS {
        coord_ref_sys_code: 8425,
        coord_ref_sys_name: "Hong Kong Geodetic CS",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1209, 
        base_crs_code: -1,
        remarks: "Locally sometimes referred to as ITRF96 or WGS 84, these are not strictly correct.",
        information_source: "Survey and Mapping Office, Lands Department.",
        data_source: "EPSG",
        revision_date: "2018-02-15",
        deprecated: 0, 
    },

    "EPSG:8426" => CRS {
        coord_ref_sys_code: 8426,
        coord_ref_sys_name: "Hong Kong Geodetic CS",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1209, 
        base_crs_code: 8425,
        remarks: "Locally sometimes referred to as ITRF96 or WGS 84, these are not strictly correct.",
        information_source: "Survey and Mapping Office, Lands Department.",
        data_source: "EPSG",
        revision_date: "2018-02-08",
        deprecated: 0, 
    },

    "EPSG:8427" => CRS {
        coord_ref_sys_code: 8427,
        coord_ref_sys_name: "Hong Kong Geodetic CS",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1209, 
        base_crs_code: 8426,
        remarks: "Locally sometimes referred to as ITRF96 or WGS 84, these are not strictly correct.",
        information_source: "EPSG. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2018-02-08",
        deprecated: 0, 
    },

    "EPSG:9007" => CRS {
        coord_ref_sys_code: 9007,
        coord_ref_sys_name: "IGb00",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1246, 
        base_crs_code: -1,
        remarks: "Adopted by the International GNSS Service (IGS) from 2004-01-11 through 2006-11-04. Replaces IGS00, replaced by IGS05 (CRS codes 9004 and 9010). For all practical purposes IGb00 is equivalent to ITRF2000.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9008" => CRS {
        coord_ref_sys_code: 9008,
        coord_ref_sys_name: "IGb00",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1246, 
        base_crs_code: 9007,
        remarks: "Adopted by the International GNSS Service (IGS) from 2004-01-11 through 2006-11-04. Replaces IGS00, replaced by IGS05 (CRS codes 9005 and 9011). For all practical purposes IGb00 is equivalent to ITRF2000.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9009" => CRS {
        coord_ref_sys_code: 9009,
        coord_ref_sys_name: "IGb00",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1246, 
        base_crs_code: 9008,
        remarks: "Adopted by the International GNSS Service (IGS) from 2004-01-11 through 2006-11-04. Replaces IGS00, replaced by IGS05 (CRS codes 9006 and 9012). For all practical purposes IGb00 is equivalent to ITRF2000.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9015" => CRS {
        coord_ref_sys_code: 9015,
        coord_ref_sys_name: "IGb08",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1248, 
        base_crs_code: -1,
        remarks: "Adopted by the International GNSS Service (IGS) from 2012-10-07 through 2017-01-28. Replaces IGS08, replaced by IGS14 (CRS codes 6934 and 8227). For all practical purposes IGb08 is equivalent to ITRF2008.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9016" => CRS {
        coord_ref_sys_code: 9016,
        coord_ref_sys_name: "IGb08",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1248, 
        base_crs_code: 9015,
        remarks: "Adopted by the International GNSS Service (IGS) from 2012-10-07 through 2017-01-28. Replaces IGS08, replaced by IGS14 (CRS codes 9013 and 9018). For all practical purposes IGb08 is equivalent to ITRF2008.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9017" => CRS {
        coord_ref_sys_code: 9017,
        coord_ref_sys_name: "IGb08",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1248, 
        base_crs_code: 9016,
        remarks: "Adopted by the International GNSS Service (IGS) from 2012-10-07 through 2017-01-28. Replaces IGS08, replaced by IGS14 (CRS codes 9014 and 9019). For all practical purposes IGb08 is equivalent to ITRF2008.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9221" => CRS {
        coord_ref_sys_code: 9221,
        coord_ref_sys_name: "Hartebeesthoek94 / ZAF BSU Albers 25E",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 4148,
        remarks: "Used for construction of the Basic Spatial Unit (BSU) grid. Came into use from 2019-01-01. Use is extended northwards into neighbouring countries to include the river catchment areas of the Orange/Gariep and Limpopo rivers.",
        information_source: "Statistics South Africa.",
        data_source: "EPSG",
        revision_date: "2019-09-06",
        deprecated: 0, 
    },

    "EPSG:9299" => CRS {
        coord_ref_sys_code: 9299,
        coord_ref_sys_name: "HS2-IRF",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1264, 
        base_crs_code: -1,
        remarks: "Intermediate CRS created to assist the emulation of the ETRS89 / HS2P1+14 SnakeGrid projected CRS through transformation HS2-IRF to ETRS89 (1) (code 9302) used in conjunction with the HS2-TM map projection (code 9301).",
        information_source: "HS2 Limited.",
        data_source: "EPSG",
        revision_date: "2020-07-01",
        deprecated: 0, 
    },

    "EPSG:9300" => CRS {
        coord_ref_sys_code: 9300,
        coord_ref_sys_name: "HS2 Survey Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 9299,
        remarks: "HS2 Survey Grid is equivalent to the tailored zero-distortion ETRS89-based SnakeGrid HS2P1+14, the Snake projection emulated using an NTv2 transformation (code 9302) and a Transverse Mercator map projection (code 9301).",
        information_source: "HS2 Limited.",
        data_source: "EPSG",
        revision_date: "2020-07-01",
        deprecated: 0, 
    },

    "EPSG:9303" => CRS {
        coord_ref_sys_code: 9303,
        coord_ref_sys_name: "HS2-VRF height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1265, 
        base_crs_code: -1,
        remarks: "HS2-VRF height is equivalent to ODN height as historically realised through OSNet v2001 and OSGM02. After the ODN realization was updated to be through OSNet v2009 and OSGM15, HS2-VRF height was introduced for backward consistency.",
        information_source: "HS2 Limited.",
        data_source: "EPSG",
        revision_date: "2020-07-01",
        deprecated: 0, 
    },

    "EPSG:9306" => CRS {
        coord_ref_sys_code: 9306,
        coord_ref_sys_name: "HS2 Survey Grid + HS2-VRF height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Realized by use of HS2TN15 transformation and HS2GM15 geoid model from ETRS89 OSNet v2009.",
        information_source: "HS2 Limited.",
        data_source: "EPSG",
        revision_date: "2020-07-01",
        deprecated: 0, 
    },

    "EPSG:9378" => CRS {
        coord_ref_sys_code: 9378,
        coord_ref_sys_name: "IGb14",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1272, 
        base_crs_code: -1,
        remarks: "Used for products from the International GNSS Service (IGS) from 2020-05-17. Replaces IGS14 (code 8227). For most practical purposes IGb14 is equivalent to ITRF2014.",
        information_source: "International GNSS Service (IGS), &quot;Switch to IGb14 reference frame&quot;, IGSMAIL-7921.",
        data_source: "EPSG",
        revision_date: "2020-06-10",
        deprecated: 0, 
    },

    "EPSG:9379" => CRS {
        coord_ref_sys_code: 9379,
        coord_ref_sys_name: "IGb14",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1272, 
        base_crs_code: 9378,
        remarks: "Used for products from the International GNSS Service (IGS) from 2020-05-17. Replaces IGS14 (code 9018). For most practical purposes IGb14 is equivalent to ITRF2014.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2020-06-10",
        deprecated: 0, 
    },

    "EPSG:9394" => CRS {
        coord_ref_sys_code: 9394,
        coord_ref_sys_name: "Ibiza height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1277, 
        base_crs_code: -1,
        remarks: "",
        information_source: "National Geographic Institute of Spain (IGN), http://www.ign.es",
        data_source: "EPSG",
        revision_date: "2020-06-27",
        deprecated: 0, 
    },

    "EPSG:23700" => CRS {
        coord_ref_sys_code: 23700,
        coord_ref_sys_name: "HD72 / EOV",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4498,
        datum_code: -1, 
        base_crs_code: 4237,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1997-07-22",
        deprecated: 0, 
    },

    "EPSG:23846" => CRS {
        coord_ref_sys_code: 23846,
        coord_ref_sys_name: "ID74 / UTM zone 46N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "Replaced by DGN95 / UTM zone 46N.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23847" => CRS {
        coord_ref_sys_code: 23847,
        coord_ref_sys_name: "ID74 / UTM zone 47N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "Replaced by DGN95 / UTM zone 47N.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23848" => CRS {
        coord_ref_sys_code: 23848,
        coord_ref_sys_name: "ID74 / UTM zone 48N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "Replaced by DGN95 / UTM zone 48N.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23849" => CRS {
        coord_ref_sys_code: 23849,
        coord_ref_sys_name: "ID74 / UTM zone 49N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "Replaced by DGN95 / UTM zone 49N.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23850" => CRS {
        coord_ref_sys_code: 23850,
        coord_ref_sys_name: "ID74 / UTM zone 50N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "Replaced by DGN95 / UTM zone 50N.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23851" => CRS {
        coord_ref_sys_code: 23851,
        coord_ref_sys_name: "ID74 / UTM zone 51N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "Replaced by DGN95 / UTM zone 51N.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23852" => CRS {
        coord_ref_sys_code: 23852,
        coord_ref_sys_name: "ID74 / UTM zone 52N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "Replaced by DGN95 / UTM zone 52N.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23853" => CRS {
        coord_ref_sys_code: 23853,
        coord_ref_sys_name: "ID74 / UTM zone 53N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-07-21",
        deprecated: 1, 
    },

    "EPSG:23886" => CRS {
        coord_ref_sys_code: 23886,
        coord_ref_sys_name: "ID74 / UTM zone 46S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-08-08",
        deprecated: 1, 
    },

    "EPSG:23887" => CRS {
        coord_ref_sys_code: 23887,
        coord_ref_sys_name: "ID74 / UTM zone 47S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "Replaced by DGN95 / UTM zone 47S.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23888" => CRS {
        coord_ref_sys_code: 23888,
        coord_ref_sys_name: "ID74 / UTM zone 48S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "Replaced by DGN95 / UTM zone 48S.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23889" => CRS {
        coord_ref_sys_code: 23889,
        coord_ref_sys_name: "ID74 / UTM zone 49S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "Replaced by DGN95 / UTM zone 49S.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23890" => CRS {
        coord_ref_sys_code: 23890,
        coord_ref_sys_name: "ID74 / UTM zone 50S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "Replaced by DGN95 / UTM zone 50S.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23891" => CRS {
        coord_ref_sys_code: 23891,
        coord_ref_sys_name: "ID74 / UTM zone 51S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "Replaced by DGN95 / UTM zone 51S.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23892" => CRS {
        coord_ref_sys_code: 23892,
        coord_ref_sys_name: "ID74 / UTM zone 52S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "Replaced by DGN95 / UTM zone 52S.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23893" => CRS {
        coord_ref_sys_code: 23893,
        coord_ref_sys_name: "ID74 / UTM zone 53S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "Replaced by DGN95 / UTM zone 53S.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:23894" => CRS {
        coord_ref_sys_code: 23894,
        coord_ref_sys_name: "ID74 / UTM zone 54S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "Replaced by DGN95 / UTM zone 54S.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:62366405" => CRS {
        coord_ref_sys_code: 62366405,
        coord_ref_sys_name: "Hu Tzu Shan (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4236,
        remarks: "See CRS code 4236 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62376405" => CRS {
        coord_ref_sys_code: 62376405,
        coord_ref_sys_name: "HD72 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4237,
        remarks: "See CRS code 4237 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62386405" => CRS {
        coord_ref_sys_code: 62386405,
        coord_ref_sys_name: "ID74 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4238,
        remarks: "See CRS code 4238 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62546405" => CRS {
        coord_ref_sys_code: 62546405,
        coord_ref_sys_name: "Hito XVIII 1963 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4254,
        remarks: "See CRS code 4254 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62556405" => CRS {
        coord_ref_sys_code: 62556405,
        coord_ref_sys_name: "Herat North (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4255,
        remarks: "See CRS code 4255 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:66116405" => CRS {
        coord_ref_sys_code: 66116405,
        coord_ref_sys_name: "Hong Kong 1980 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4611,
        remarks: "See CRS code 4611 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:66586405" => CRS {
        coord_ref_sys_code: 66586405,
        coord_ref_sys_name: "Hjorsey 1955 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4658,
        remarks: "See CRS code 4658 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2003-06-27",
        deprecated: 1, 
    },

    "EPSG:66606405" => CRS {
        coord_ref_sys_code: 66606405,
        coord_ref_sys_name: "Helle 1954 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4660,
        remarks: "See CRS code 4660 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2003-06-27",
        deprecated: 1, 
    },

    "EPSG:2978" => CRS {
        coord_ref_sys_code: 2978,
        coord_ref_sys_name: "IGN72 Nuku Hiva / UTM zone 7S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4630,
        remarks: "Replaced by RGPF / UTM zone 7S, CRS code 3298.",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2005-08-12",
        deprecated: 0, 
    },

    "EPSG:2981" => CRS {
        coord_ref_sys_code: 2981,
        coord_ref_sys_name: "IGN56 Lifou / UTM zone 58S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4633,
        remarks: "Replaced by RGNC91-93 / Lambert Caledonie (CRS code 3163).",
        information_source: "Service Topographique de la Nouvelle Caledonie, Direction des Infrastructures, de la Topographie et des Transports Terrestres. www.dittt.gouv.nc",
        data_source: "EPSG",
        revision_date: "2006-07-21",
        deprecated: 0, 
    },

    "EPSG:2982" => CRS {
        coord_ref_sys_code: 2982,
        coord_ref_sys_name: "IGN72 Grand Terre / UTM zone 58S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4634,
        remarks: "Superseded by RGNC 1991 / Lambert Caledonie (CRS code 2984).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:2995" => CRS {
        coord_ref_sys_code: 2995,
        coord_ref_sys_name: "IGN53 Mare / UTM zone 58S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4641,
        remarks: "Replaced by RGNC91-93 / Lambert Caledonie (CRS code 3163).",
        information_source: "Service Topographique de la Nouvelle Caledonie, Direction des Infrastructures, de la Topographie et des Transports Terrestres. www.dittt.gouv.nc",
        data_source: "EPSG",
        revision_date: "2006-07-21",
        deprecated: 0, 
    },

    "EPSG:3060" => CRS {
        coord_ref_sys_code: 3060,
        coord_ref_sys_name: "IGN72 Grande Terre / UTM zone 58S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4662,
        remarks: "Replaced by RGNC91-93 / Lambert Caledonie (CRS code 3163).",
        information_source: "Service Topographique de la Nouvelle Caledonie, Direction des Infrastructures, de la Topographie et des Transports Terrestres. www.dittt.gouv.nc",
        data_source: "EPSG",
        revision_date: "2006-07-21",
        deprecated: 0, 
    },

    "EPSG:3064" => CRS {
        coord_ref_sys_code: 3064,
        coord_ref_sys_name: "IGM95 / UTM zone 32N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4670,
        remarks: "Replaced by RDN2008 / UTM zone 32N (CRS codes 6707 and 7791) from 2011-11-10.",
        information_source: "ENI",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:3065" => CRS {
        coord_ref_sys_code: 3065,
        coord_ref_sys_name: "IGM95 / UTM zone 33N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4670,
        remarks: "Replaced by RDN2008 / UTM zone 33N and RDN2008 / UTM zone 34N (CRS codes 6708-09 and 7792-93) from 2011-11-10.",
        information_source: "ENI",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:3172" => CRS {
        coord_ref_sys_code: 3172,
        coord_ref_sys_name: "IGN53 Mare / UTM zone 59S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4641,
        remarks: "Replaced by RGNC91-93 / Lambert Caledonie (CRS code 3163).",
        information_source: "Service Topographique de la Nouvelle Caledonie, Direction des Infrastructures, de la Topographie et des Transports Terrestres. www.dittt.gouv.nc",
        data_source: "EPSG",
        revision_date: "2006-07-25",
        deprecated: 0, 
    },

    "EPSG:3176" => CRS {
        coord_ref_sys_code: 3176,
        coord_ref_sys_name: "Indian 1960 / TM 106 NE",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4131,
        remarks: "",
        information_source: "Petrovietnam",
        data_source: "EPSG",
        revision_date: "1998-11-11",
        deprecated: 0, 
    },

    "EPSG:3302" => CRS {
        coord_ref_sys_code: 3302,
        coord_ref_sys_name: "IGN63 Hiva Oa / UTM zone 7S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4689,
        remarks: "Replaced by RGPF / UTM zone 7S, CRS code 3298.",
        information_source: "Gouvernement de la Polynésie Française, Service de l&#x27;Urbanisme, Section topographie.",
        data_source: "EPSG",
        revision_date: "2005-08-12",
        deprecated: 0, 
    },

    "EPSG:3318" => CRS {
        coord_ref_sys_code: 3318,
        coord_ref_sys_name: "IGC 1962 / Congo TM zone 12",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4697,
        remarks: "",
        information_source: "Institut Geographique du Congo.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 0, 
    },

    "EPSG:3319" => CRS {
        coord_ref_sys_code: 3319,
        coord_ref_sys_name: "IGC 1962 / Congo TM zone 14",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4697,
        remarks: "",
        information_source: "Institut Geographique du Congo.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 0, 
    },

    "EPSG:3320" => CRS {
        coord_ref_sys_code: 3320,
        coord_ref_sys_name: "IGC 1962 / Congo TM zone 16",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4697,
        remarks: "",
        information_source: "Institut Geographique du Congo.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 0, 
    },

    "EPSG:3321" => CRS {
        coord_ref_sys_code: 3321,
        coord_ref_sys_name: "IGC 1962 / Congo TM zone 18",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4697,
        remarks: "",
        information_source: "Institut Geographique du Congo.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 0, 
    },

    "EPSG:3322" => CRS {
        coord_ref_sys_code: 3322,
        coord_ref_sys_name: "IGC 1962 / Congo TM zone 20",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4697,
        remarks: "",
        information_source: "Institut Geographique du Congo.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 0, 
    },

    "EPSG:3323" => CRS {
        coord_ref_sys_code: 3323,
        coord_ref_sys_name: "IGC 1962 / Congo TM zone 22",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4697,
        remarks: "",
        information_source: "Institut Geographique du Congo.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 0, 
    },

    "EPSG:3324" => CRS {
        coord_ref_sys_code: 3324,
        coord_ref_sys_name: "IGC 1962 / Congo TM zone 24",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4697,
        remarks: "",
        information_source: "Institut Geographique du Congo.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 0, 
    },

    "EPSG:3325" => CRS {
        coord_ref_sys_code: 3325,
        coord_ref_sys_name: "IGC 1962 / Congo TM zone 26",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4697,
        remarks: "",
        information_source: "Institut Geographique du Congo.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 0, 
    },

    "EPSG:3326" => CRS {
        coord_ref_sys_code: 3326,
        coord_ref_sys_name: "IGC 1962 / Congo TM zone 28",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4697,
        remarks: "",
        information_source: "Institut Geographique du Congo.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 0, 
    },

    "EPSG:3327" => CRS {
        coord_ref_sys_code: 3327,
        coord_ref_sys_name: "IGC 1962 / Congo TM zone 30",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4697,
        remarks: "",
        information_source: "Institut Geographique du Congo.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 0, 
    },

    "EPSG:3336" => CRS {
        coord_ref_sys_code: 3336,
        coord_ref_sys_name: "IGN 1962 Kerguelen / UTM zone 42S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4698,
        remarks: "Replaced by RGTAAF07 / UTM zone 42S (CRS code 7079).",
        information_source: "IGN Paris.",
        data_source: "EPSG",
        revision_date: "2015-05-22",
        deprecated: 0, 
    },

    "EPSG:3339" => CRS {
        coord_ref_sys_code: 3339,
        coord_ref_sys_name: "IGCB 1955 / Congo TM zone 12",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4701,
        remarks: "Oil industry operations used IGBC 1955 / UTM zone 33S (code 3342).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2005-11-28",
        deprecated: 0, 
    },

    "EPSG:3340" => CRS {
        coord_ref_sys_code: 3340,
        coord_ref_sys_name: "IGCB 1955 / Congo TM zone 14",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4701,
        remarks: "Oil industry operations used IGBC 1955 / UTM zone 33S (code 3342).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2005-11-28",
        deprecated: 0, 
    },

    "EPSG:3341" => CRS {
        coord_ref_sys_code: 3341,
        coord_ref_sys_name: "IGCB 1955 / Congo TM zone 16",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4701,
        remarks: "Oil industry operations used IGBC 1955 / UTM zone 33S (code 3342).",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2012-05-08",
        deprecated: 0, 
    },

    "EPSG:3342" => CRS {
        coord_ref_sys_code: 3342,
        coord_ref_sys_name: "IGCB 1955 / UTM zone 33S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4701,
        remarks: "",
        information_source: "Petrofina.",
        data_source: "EPSG",
        revision_date: "2005-11-28",
        deprecated: 0, 
    },

    "EPSG:3367" => CRS {
        coord_ref_sys_code: 3367,
        coord_ref_sys_name: "IGN Astro 1960 / UTM zone 28N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4700,
        remarks: "Mining title descriptions referring only to &quot;Clarke 1880 ellipsoid, UTM zone 28&quot; should be assumed to be referenced to this CRS. Oil industry considers &quot;Mauritanian Mining Cadastre 1999 / UTM zone 28N&quot; to be exactly defined through tfm code 15857.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2006-03-16",
        deprecated: 0, 
    },

    "EPSG:3368" => CRS {
        coord_ref_sys_code: 3368,
        coord_ref_sys_name: "IGN Astro 1960 / UTM zone 29N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4700,
        remarks: "Mining title descriptions referring only to &quot;Clarke 1880 ellipsoid, UTM zone 29&quot; should be assumed to be referenced to this CRS. Oil industry considers &quot;Mauritanian Mining Cadastre 1999 / UTM zone 29N&quot; to be exactly defined through tfm code 15858.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2006-03-16",
        deprecated: 0, 
    },

    "EPSG:3369" => CRS {
        coord_ref_sys_code: 3369,
        coord_ref_sys_name: "IGN Astro 1960 / UTM zone 30N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4700,
        remarks: "Mining title descriptions referring only to &quot;Clarke 1880 ellipsoid, UTM zone 30&quot; should be assumed to be referenced to this CRS. Oil industry considers &quot;Mauritanian Mining Cadastre 1999 / UTM zone 30N&quot; to be exactly defined through tfm code 15859.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2006-03-16",
        deprecated: 0, 
    },

    "EPSG:3887" => CRS {
        coord_ref_sys_code: 3887,
        coord_ref_sys_name: "IGRS",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1029, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Survey Division, Ministry of Water Resources (MoWR).",
        data_source: "EPSG",
        revision_date: "2009-02-03",
        deprecated: 0, 
    },

    "EPSG:3888" => CRS {
        coord_ref_sys_code: 3888,
        coord_ref_sys_name: "IGRS",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1029, 
        base_crs_code: 3887,
        remarks: "",
        information_source: "Survey Division, Ministry of Water Resources (MoWR).",
        data_source: "EPSG",
        revision_date: "2009-02-03",
        deprecated: 0, 
    },

    "EPSG:3889" => CRS {
        coord_ref_sys_code: 3889,
        coord_ref_sys_name: "IGRS",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1029, 
        base_crs_code: 3888,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2009-02-03",
        deprecated: 0, 
    },

    "EPSG:3890" => CRS {
        coord_ref_sys_code: 3890,
        coord_ref_sys_name: "IGRS / UTM zone 37N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 3889,
        remarks: "",
        information_source: "Survey Division, Ministry of Water Resources (MoWR).",
        data_source: "EPSG",
        revision_date: "2009-02-03",
        deprecated: 0, 
    },

    "EPSG:3891" => CRS {
        coord_ref_sys_code: 3891,
        coord_ref_sys_name: "IGRS / UTM zone 38N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 3889,
        remarks: "",
        information_source: "Survey Division, Ministry of Water Resources (MoWR).",
        data_source: "EPSG",
        revision_date: "2009-02-03",
        deprecated: 0, 
    },

    "EPSG:3892" => CRS {
        coord_ref_sys_code: 3892,
        coord_ref_sys_name: "IGRS / UTM zone 39N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 3889,
        remarks: "",
        information_source: "Survey Division, Ministry of Water Resources (MoWR).",
        data_source: "EPSG",
        revision_date: "2012-01-21",
        deprecated: 0, 
    },

    "EPSG:4131" => CRS {
        coord_ref_sys_code: 4131,
        coord_ref_sys_name: "Indian 1960",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6131, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-03-25",
        deprecated: 0, 
    },

    "EPSG:4239" => CRS {
        coord_ref_sys_code: 4239,
        coord_ref_sys_name: "Indian 1954",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6239, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4630" => CRS {
        coord_ref_sys_code: 4630,
        coord_ref_sys_name: "IGN72 Nuku Hiva",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6630, 
        base_crs_code: -1,
        remarks: "Replaced by RGPF, CRS code 4687.",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2005-08-12",
        deprecated: 0, 
    },

    "EPSG:4633" => CRS {
        coord_ref_sys_code: 4633,
        coord_ref_sys_name: "IGN56 Lifou",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6633, 
        base_crs_code: -1,
        remarks: "Replaced by RGNC91-93 (CRS code 4749).",
        information_source: "Service Topographique de la Nouvelle Caledonie, Direction des Infrastructures, de la Topographie et des Transports Terrestres. www.dittt.gouv.nc",
        data_source: "EPSG",
        revision_date: "2006-07-21",
        deprecated: 0, 
    },

    "EPSG:4634" => CRS {
        coord_ref_sys_code: 4634,
        coord_ref_sys_name: "IGN72 Grand Terre",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6402,
        datum_code: 6634, 
        base_crs_code: -1,
        remarks: "Superseded by RGNC 1991 (CRS code 4645).",
        information_source: "Service Topographique de la Nouvelle Caledonie",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:4641" => CRS {
        coord_ref_sys_code: 4641,
        coord_ref_sys_name: "IGN53 Mare",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6641, 
        base_crs_code: -1,
        remarks: "Replaced by RGNC91-93 (CRS code 4749).",
        information_source: "Service Topographique de la Nouvelle Caledonie, Direction des Infrastructures, de la Topographie et des Transports Terrestres. www.dittt.gouv.nc",
        data_source: "EPSG",
        revision_date: "2006-07-21",
        deprecated: 0, 
    },

    "EPSG:4662" => CRS {
        coord_ref_sys_code: 4662,
        coord_ref_sys_name: "IGN72 Grande Terre",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6634, 
        base_crs_code: -1,
        remarks: "Replaced by RGNC91-93 (CRS code 4749).",
        information_source: "Service Topographique de la Nouvelle Caledonie, Direction des Infrastructures, de la Topographie et des Transports Terrestres. www.dittt.gouv.nc",
        data_source: "EPSG",
        revision_date: "2006-07-21",
        deprecated: 0, 
    },

    "EPSG:4667" => CRS {
        coord_ref_sys_code: 4667,
        coord_ref_sys_name: "IKBD-92",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6667, 
        base_crs_code: -1,
        remarks: "",
        information_source: "United Nations Iraq-Kuwait Boundary Demarcation Commission, Final Report, May 1993.",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4670" => CRS {
        coord_ref_sys_code: 4670,
        coord_ref_sys_name: "IGM95",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6670, 
        base_crs_code: 4983,
        remarks: "Replaced by RDN2008 (CRS code 6706) from 2011-11-10.",
        information_source: "EPSG. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:4689" => CRS {
        coord_ref_sys_code: 4689,
        coord_ref_sys_name: "IGN63 Hiva Oa",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6689, 
        base_crs_code: -1,
        remarks: "Replaced by RGPF, CRS code 4687.",
        information_source: "Gouvernement de la Polynésie Française, Service de l&#x27;Urbanisme, Section topographie.",
        data_source: "EPSG",
        revision_date: "2005-08-12",
        deprecated: 0, 
    },

    "EPSG:4697" => CRS {
        coord_ref_sys_code: 4697,
        coord_ref_sys_name: "IGC 1962 6th Parallel South",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6697, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Institute Geographique du Congo.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 0, 
    },

    "EPSG:4698" => CRS {
        coord_ref_sys_code: 4698,
        coord_ref_sys_name: "IGN 1962 Kerguelen",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6698, 
        base_crs_code: -1,
        remarks: "Replaced by RGTAAF07 (CRS code 7073).",
        information_source: "IGN Paris.",
        data_source: "EPSG",
        revision_date: "2015-05-22",
        deprecated: 0, 
    },

    "EPSG:4700" => CRS {
        coord_ref_sys_code: 4700,
        coord_ref_sys_name: "IGN Astro 1960",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6700, 
        base_crs_code: -1,
        remarks: "Mining title descriptions referring only to &quot;Clarke 1880 ellipsoid&quot; should be assumed to be referenced to this CRS. Oil industry considers Mining Cadastre 1999 to be exactly defined through tfm codes 15857-9. Replaced by Mauritania 1999 (code 4702).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-01-05",
        deprecated: 0, 
    },

    "EPSG:4701" => CRS {
        coord_ref_sys_code: 4701,
        coord_ref_sys_name: "IGCB 1955",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6701, 
        base_crs_code: -1,
        remarks: "Replaced by IGC 1962 Arc of the 6th Parallel South, except for oil industry activities.",
        information_source: "Institut Geographique du Bas Congo.",
        data_source: "EPSG",
        revision_date: "2005-11-28",
        deprecated: 0, 
    },

    "EPSG:4982" => CRS {
        coord_ref_sys_code: 4982,
        coord_ref_sys_name: "IGM95",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6670, 
        base_crs_code: -1,
        remarks: "Replaced by RDN2008 (CRS code 6704) from 2011-11-10.",
        information_source: "ENI",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:4983" => CRS {
        coord_ref_sys_code: 4983,
        coord_ref_sys_name: "IGM95",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6670, 
        base_crs_code: 4982,
        remarks: "Replaced by RDN2008 (CRS code 6705) from 2011-11-10.",
        information_source: "ENI",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:5193" => CRS {
        coord_ref_sys_code: 5193,
        coord_ref_sys_name: "Incheon height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1049, 
        base_crs_code: -1,
        remarks: "",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-30",
        deprecated: 0, 
    },

    "EPSG:5601" => CRS {
        coord_ref_sys_code: 5601,
        coord_ref_sys_name: "IGN 1966 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5196, 
        base_crs_code: -1,
        remarks: "Part of NGPF (CRS code 5600).",
        information_source: "Gouvernement de la Polynésie Française, Service de l&#x27;Urbanisme, Section topographie.",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:5608" => CRS {
        coord_ref_sys_code: 5608,
        coord_ref_sys_name: "IGLD 1955 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5204, 
        base_crs_code: -1,
        remarks: "Replaces several earlier systems. Replaced by IGLD 1985 (CRS code 5609).",
        information_source: "Coordinating Committee on Great Lakes Basic Hydraulic and Hydrologic Data",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:5609" => CRS {
        coord_ref_sys_code: 5609,
        coord_ref_sys_name: "IGLD 1985 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5205, 
        base_crs_code: -1,
        remarks: "Replaces IGLD 1955 (CRS code 5608).",
        information_source: "Coordinating Committee on Great Lakes Basic Hydraulic and Hydrologic Data",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:5616" => CRS {
        coord_ref_sys_code: 5616,
        coord_ref_sys_name: "IGN 1988 LS height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5210, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IGN Paris.",
        data_source: "EPSG",
        revision_date: "2008-10-13",
        deprecated: 0, 
    },

    "EPSG:5617" => CRS {
        coord_ref_sys_code: 5617,
        coord_ref_sys_name: "IGN 1988 MG height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5211, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IGN Paris.",
        data_source: "EPSG",
        revision_date: "2008-10-13",
        deprecated: 0, 
    },

    "EPSG:5618" => CRS {
        coord_ref_sys_code: 5618,
        coord_ref_sys_name: "IGN 1992 LD height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5212, 
        base_crs_code: -1,
        remarks: "Replaced by IGN 2008 LD height (CRS code 9130).",
        information_source: "IGN Paris.",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:5619" => CRS {
        coord_ref_sys_code: 5619,
        coord_ref_sys_name: "IGN 1988 SB height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5213, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IGN Paris.",
        data_source: "EPSG",
        revision_date: "2008-10-13",
        deprecated: 0, 
    },

    "EPSG:5620" => CRS {
        coord_ref_sys_code: 5620,
        coord_ref_sys_name: "IGN 1988 SM height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5214, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IGN Paris.",
        data_source: "EPSG",
        revision_date: "2008-10-13",
        deprecated: 0, 
    },

    "EPSG:6934" => CRS {
        coord_ref_sys_code: 6934,
        coord_ref_sys_name: "IGS08",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1141, 
        base_crs_code: -1,
        remarks: "Used for products from International GNSS Service (IGS) analysis centres from 2011-04-17 through 2012-10-06. Replaces IGS05 (code 9010). Replaced by IGb08 (code 9015). For most practical purposes IGS08 is equivalent to ITRF2008.",
        information_source: "http://www.ngs.noaa.gov/CORS/",
        data_source: "EPSG",
        revision_date: "2020-04-30",
        deprecated: 0, 
    },

    "EPSG:6978" => CRS {
        coord_ref_sys_code: 6978,
        coord_ref_sys_name: "IGD05",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1143, 
        base_crs_code: -1,
        remarks: "Replaced by IGD05/12 (CRS code 6985).",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2014-11-07",
        deprecated: 1, 
    },

    "EPSG:6979" => CRS {
        coord_ref_sys_code: 6979,
        coord_ref_sys_name: "IGD05",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1143, 
        base_crs_code: 6978,
        remarks: "Replaced by IGD05/12 (CRS code 6986).",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2014-11-07",
        deprecated: 1, 
    },

    "EPSG:6980" => CRS {
        coord_ref_sys_code: 6980,
        coord_ref_sys_name: "IGD05",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1143, 
        base_crs_code: 6979,
        remarks: "Replaces Israel 1993 (CRS code 4141) from January 2005. Replaced by IGD05/12 (CRS code 6987) from March 2012.",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2014-11-08",
        deprecated: 1, 
    },

    "EPSG:6985" => CRS {
        coord_ref_sys_code: 6985,
        coord_ref_sys_name: "IGD05/12",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1145, 
        base_crs_code: -1,
        remarks: "Replaces IGD05 (CRS code 6978).",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2014-11-07",
        deprecated: 1, 
    },

    "EPSG:6986" => CRS {
        coord_ref_sys_code: 6986,
        coord_ref_sys_name: "IGD05/12",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1145, 
        base_crs_code: 6985,
        remarks: "Replaces IGD05 (CRS code 6979).",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2014-11-07",
        deprecated: 1, 
    },

    "EPSG:6987" => CRS {
        coord_ref_sys_code: 6987,
        coord_ref_sys_name: "IGD05/12",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1145, 
        base_crs_code: 6986,
        remarks: "Replaces IGD05 (CRS code 6980) from March 2012.",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2014-11-08",
        deprecated: 1, 
    },

    "EPSG:7134" => CRS {
        coord_ref_sys_code: 7134,
        coord_ref_sys_name: "IGD05",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1114, 
        base_crs_code: -1,
        remarks: "Replaced by IGD05/12 (CRS code 7137).",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2015-06-18",
        deprecated: 0, 
    },

    "EPSG:7135" => CRS {
        coord_ref_sys_code: 7135,
        coord_ref_sys_name: "IGD05",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1114, 
        base_crs_code: 7134,
        remarks: "Replaced by IGD05/12 (CRS code 7138).",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2015-06-18",
        deprecated: 0, 
    },

    "EPSG:7136" => CRS {
        coord_ref_sys_code: 7136,
        coord_ref_sys_name: "IGD05",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1114, 
        base_crs_code: 7135,
        remarks: "Replaces Israel 1993 (CRS code 4141) from January 2005. Replaced by IGD05/12 (CRS code 7139) from March 2012.",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2015-06-18",
        deprecated: 0, 
    },

    "EPSG:7137" => CRS {
        coord_ref_sys_code: 7137,
        coord_ref_sys_name: "IGD05/12",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1115, 
        base_crs_code: -1,
        remarks: "Replaces IGD05 (CRS code 7134).",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2015-06-18",
        deprecated: 0, 
    },

    "EPSG:7138" => CRS {
        coord_ref_sys_code: 7138,
        coord_ref_sys_name: "IGD05/12",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1115, 
        base_crs_code: 7137,
        remarks: "Replaces IGD05 (CRS code 7135).",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2015-06-18",
        deprecated: 0, 
    },

    "EPSG:7139" => CRS {
        coord_ref_sys_code: 7139,
        coord_ref_sys_name: "IGD05/12",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1115, 
        base_crs_code: 7138,
        remarks: "Replaces IGD05 (CRS code 7136) from March 2012.",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2015-06-18",
        deprecated: 0, 
    },

    "EPSG:8227" => CRS {
        coord_ref_sys_code: 8227,
        coord_ref_sys_name: "IGS14",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1191, 
        base_crs_code: -1,
        remarks: "Used for products from the International GNSS Service (IGS) from 2017-01-29 to 2020-05-16. Replaces IGb08 (code 9015), replaced by IGb14 (code 9378). For most practical purposes IGS14 is equivalent to ITRF2014.",
        information_source: "http://www.igs.org/article/igs14-reference-frame-transition",
        data_source: "EPSG",
        revision_date: "2020-06-17",
        deprecated: 0, 
    },

    "EPSG:9001" => CRS {
        coord_ref_sys_code: 9001,
        coord_ref_sys_name: "IGS97",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1244, 
        base_crs_code: -1,
        remarks: "Adopted by the International GNSS Service (IGS) from 2000-06-04 through 2001-12-01. Replaced by IGS00 (CRS code 9004). For all practical purposes IGS97 is equivalent to ITRF97.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9002" => CRS {
        coord_ref_sys_code: 9002,
        coord_ref_sys_name: "IGS97",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1244, 
        base_crs_code: 9001,
        remarks: "Adopted by the International GNSS Service (IGS) from 2000-06-04 through 2001-12-01. Replaced by IGS00 (CRS code 9005). For all practical purposes IGS97 is equivalent to ITRF97.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9003" => CRS {
        coord_ref_sys_code: 9003,
        coord_ref_sys_name: "IGS97",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1244, 
        base_crs_code: 9002,
        remarks: "Adopted by the International GNSS Service (IGS) from 2000-06-04 through 2001-12-01. Replaced by IGS00 (CRS code 9006). For all practical purposes IGS97 is equivalent to ITRF97.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9004" => CRS {
        coord_ref_sys_code: 9004,
        coord_ref_sys_name: "IGS00",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1245, 
        base_crs_code: -1,
        remarks: "Adopted by the International GNSS Service (IGS) from 2001-12-02 through 2004-01-10. Replaces IGS97, replaced by IGb00 (CRS codes 9001 and 9007). For all practical purposes IGS00 is equivalent to ITRF2000.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9005" => CRS {
        coord_ref_sys_code: 9005,
        coord_ref_sys_name: "IGS00",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1245, 
        base_crs_code: 9004,
        remarks: "Adopted by the International GNSS Service (IGS) from 2001-12-02 through 2004-01-10. Replaces IGS97, replaced by IGb00 (CRS codes 9002 and 9008). For all practical purposes IGS00 is equivalent to ITRF2000.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9006" => CRS {
        coord_ref_sys_code: 9006,
        coord_ref_sys_name: "IGS00",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1245, 
        base_crs_code: 9005,
        remarks: "Adopted by the International GNSS Service (IGS) from 2001-12-02 through 2004-01-10. Replaces IGS97, replaced by IGb00 (CRS codes 9003 and 9009). For all practical purposes IGS00 is equivalent to ITRF2000.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9010" => CRS {
        coord_ref_sys_code: 9010,
        coord_ref_sys_name: "IGS05",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1247, 
        base_crs_code: -1,
        remarks: "Adopted by the International GNSS Service (IGS) from 2006-11-05 through 2011-04-16. Replaces IGb00, replaced by IGS08 (CRS codes 9007 and 6834). For all practical purposes IGS05 is equivalent to ITRF2005.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9011" => CRS {
        coord_ref_sys_code: 9011,
        coord_ref_sys_name: "IGS05",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1247, 
        base_crs_code: 9010,
        remarks: "Adopted by the International GNSS Service (IGS) from 2006-11-05 through 2011-04-16. Replaces IGb00, replaced by IGS08 (CRS codes 9008 and 9013). For all practical purposes IGS05 is equivalent to ITRF2005.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9012" => CRS {
        coord_ref_sys_code: 9012,
        coord_ref_sys_name: "IGS05",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1247, 
        base_crs_code: 9011,
        remarks: "Adopted by the International GNSS Service (IGS) from 2006-11-05 through 2011-04-16. Replaces IGb00, replaced by IGS08 (CRS codes 9009 and 9014). For all practical purposes IGS05 is equivalent to ITRF2005.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:24547" => CRS {
        coord_ref_sys_code: 24547,
        coord_ref_sys_name: "Kertau 1968 / UTM zone 47N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4245,
        remarks: "",
        information_source: "Defence Geographic Centre.",
        data_source: "EPSG",
        revision_date: "2006-07-24",
        deprecated: 0, 
    },

    "EPSG:9013" => CRS {
        coord_ref_sys_code: 9013,
        coord_ref_sys_name: "IGS08",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1141, 
        base_crs_code: 6934,
        remarks: "Used for products from International GNSS Service (IGS) analysis centres from 2011-04-17 through 2012-10-06. Replaces IGS05 (code 9011). Replaced by IGb08 (code 9016). For most practical purposes IGS08 is equivalent to ITRF2008.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9014" => CRS {
        coord_ref_sys_code: 9014,
        coord_ref_sys_name: "IGS08",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1141, 
        base_crs_code: 9013,
        remarks: "Used for products from International GNSS Service (IGS) analysis centres from 2011-04-17 through 2012-10-06. Replaces IGS05 (code 9012). Replaced by IGb08 (code 9017). For most practical purposes IGS08 is equivalent to ITRF2008.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9018" => CRS {
        coord_ref_sys_code: 9018,
        coord_ref_sys_name: "IGS14",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1191, 
        base_crs_code: 8227,
        remarks: "Used for products from the International GNSS Service (IGS) from 2017-01-29 to 2020-05-16. Replaces IGb08 (code 9016), replaced by IGb14 (code 9379). For most practical purposes IGS14 is equivalent to ITRF2014.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2020-06-17",
        deprecated: 0, 
    },

    "EPSG:9019" => CRS {
        coord_ref_sys_code: 9019,
        coord_ref_sys_name: "IGS14",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1191, 
        base_crs_code: 9018,
        remarks: "Used for products from the International GNSS Service (IGS) from 2017-01-29 to 2020-05-16. Replaces IGb08 (code 9017), replaced by IGb14 (code 9380). For most practical purposes IGS14 is equivalent to ITRF2014.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2020-06-17",
        deprecated: 0, 
    },

    "EPSG:9130" => CRS {
        coord_ref_sys_code: 9130,
        coord_ref_sys_name: "IGN 2008 LD height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1250, 
        base_crs_code: -1,
        remarks: "Replaces IGN 1992 LD height (CRS code 5618).",
        information_source: "IGN Paris.",
        data_source: "EPSG",
        revision_date: "2019-07-15",
        deprecated: 0, 
    },

    "EPSG:9380" => CRS {
        coord_ref_sys_code: 9380,
        coord_ref_sys_name: "IGb14",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1272, 
        base_crs_code: 9379,
        remarks: "Used for products from the International GNSS Service (IGS) from 2020-05-17. Replaces IGS14 (code 9019). For most practical purposes IGb14 is equivalent to ITRF2014.",
        information_source: "International GNSS Service (IGS).",
        data_source: "EPSG",
        revision_date: "2020-10-21",
        deprecated: 0, 
    },

    "EPSG:9471" => CRS {
        coord_ref_sys_code: 9471,
        coord_ref_sys_name: "INAGeoid2020 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1294, 
        base_crs_code: -1,
        remarks: "Physical height component of national vertical control network (JKVN). Orthometric heights.",
        information_source: "Geospatial Information Agency (BIG), https://srgi.big.go.id",
        data_source: "EPSG",
        revision_date: "2020-08-27",
        deprecated: 0, 
    },

    "EPSG:9716" => CRS {
        coord_ref_sys_code: 9716,
        coord_ref_sys_name: "IGM95 / UTM zone 34N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4670,
        remarks: "Replaced by RDN2008 / UTM zone 34N (CRS codes 6709 and 7793) from 2011-11-10.",
        information_source: "ENI",
        data_source: "EPSG",
        revision_date: "2021-05-13",
        deprecated: 0, 
    },

    "EPSG:23946" => CRS {
        coord_ref_sys_code: 23946,
        coord_ref_sys_name: "Indian 1954 / UTM zone 46N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4239,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1998-06-13",
        deprecated: 0, 
    },

    "EPSG:23947" => CRS {
        coord_ref_sys_code: 23947,
        coord_ref_sys_name: "Indian 1954 / UTM zone 47N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4239,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:23948" => CRS {
        coord_ref_sys_code: 23948,
        coord_ref_sys_name: "Indian 1954 / UTM zone 48N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4239,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2011-01-25",
        deprecated: 0, 
    },

    "EPSG:61316405" => CRS {
        coord_ref_sys_code: 61316405,
        coord_ref_sys_name: "Indian 1960 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4131,
        remarks: "See CRS code 4131 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62396405" => CRS {
        coord_ref_sys_code: 62396405,
        coord_ref_sys_name: "Indian 1954 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4239,
        remarks: "See CRS code 4239 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:66306405" => CRS {
        coord_ref_sys_code: 66306405,
        coord_ref_sys_name: "IGN72 Nuku Hiva (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4630,
        remarks: "See CRS code 4630 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:66336405" => CRS {
        coord_ref_sys_code: 66336405,
        coord_ref_sys_name: "IGN56 Lifou (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4633,
        remarks: "See CRS code 4633 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:66346405" => CRS {
        coord_ref_sys_code: 66346405,
        coord_ref_sys_name: "IGN72 Grande Terre (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4662,
        remarks: "See CRS code 4634 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2003-09-30",
        deprecated: 1, 
    },

    "EPSG:66416405" => CRS {
        coord_ref_sys_code: 66416405,
        coord_ref_sys_name: "IGN53 Mare (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4641,
        remarks: "See CRS code 4650 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:66676405" => CRS {
        coord_ref_sys_code: 66676405,
        coord_ref_sys_name: "IKBD-92 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4667,
        remarks: "See CRS code 4667 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2003-09-01",
        deprecated: 1, 
    },

    "EPSG:2039" => CRS {
        coord_ref_sys_code: 2039,
        coord_ref_sys_name: "Israel 1993 / Israeli TM Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4141,
        remarks: "Replaces Israeli CS Grid (EPSG code 28193) from June 1998. Replaced by Israeli Grid 05 (IG05) (CRS code 6984) for precise applications.",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2014-11-21",
        deprecated: 0, 
    },

    "EPSG:2157" => CRS {
        coord_ref_sys_code: 2157,
        coord_ref_sys_name: "IRENET95 / Irish Transverse Mercator",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4173,
        remarks: "Replaces TM75 / Irish Grid (code 29903) from 1/1/2001.",
        information_source: "Ordnance Survey of Ireland",
        data_source: "EPSG",
        revision_date: "2004-04-07",
        deprecated: 0, 
    },

    "EPSG:2158" => CRS {
        coord_ref_sys_code: 2158,
        coord_ref_sys_name: "IRENET95 / UTM zone 29N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4173,
        remarks: "",
        information_source: "Ordnance Survey of Ireland",
        data_source: "EPSG",
        revision_date: "2011-01-25",
        deprecated: 0, 
    },

    "EPSG:3057" => CRS {
        coord_ref_sys_code: 3057,
        coord_ref_sys_name: "ISN93 / Lambert 1993",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4659,
        remarks: "Replaces Hjorsey 1955 / Lambert 1955 (CRS code 3053). Replaced by ISN2004 / Lambert 2004 (CRS code 5325).",
        information_source: "Landmaelingar Islands (National Survey of Iceland).",
        data_source: "EPSG",
        revision_date: "2010-11-12",
        deprecated: 0, 
    },

    "EPSG:3148" => CRS {
        coord_ref_sys_code: 3148,
        coord_ref_sys_name: "Indian 1960 / UTM zone 48N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4131,
        remarks: "In Vietnam replaced by Hanoi 72 / Gauss zone 18 from 1988 onwards.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1998-11-11",
        deprecated: 0, 
    },

    "EPSG:3149" => CRS {
        coord_ref_sys_code: 3149,
        coord_ref_sys_name: "Indian 1960 / UTM zone 49N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4131,
        remarks: "Replaced by Hanoi 72 / Gauss zone 18 from 1988 onwards.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1998-11-11",
        deprecated: 0, 
    },

    "EPSG:3448" => CRS {
        coord_ref_sys_code: 3448,
        coord_ref_sys_name: "JAD2001 / Jamaica Metric Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4758,
        remarks: "Replaces JAD69 / Jamaica National Grid (CRS code 24200).",
        information_source: "National Land Agency.",
        data_source: "EPSG",
        revision_date: "2007-01-19",
        deprecated: 0, 
    },

    "EPSG:3449" => CRS {
        coord_ref_sys_code: 3449,
        coord_ref_sys_name: "JAD2001 / UTM zone 17N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4758,
        remarks: "See JAD2001 / Jamaica Metric Grid (CRS code 3448) for land applications.",
        information_source: "National Land Agency.",
        data_source: "EPSG",
        revision_date: "2007-01-19",
        deprecated: 0, 
    },

    "EPSG:4141" => CRS {
        coord_ref_sys_code: 4141,
        coord_ref_sys_name: "Israel 1993",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6141, 
        base_crs_code: -1,
        remarks: "Replaces Palestine 1923 (CRS code 4281) from June 1998. Replaced by IGD05 (CRS code 6980) from January 2005.",
        information_source: "Survey of Israel",
        data_source: "EPSG",
        revision_date: "2014-11-21",
        deprecated: 0, 
    },

    "EPSG:4173" => CRS {
        coord_ref_sys_code: 4173,
        coord_ref_sys_name: "IRENET95",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6173, 
        base_crs_code: 4943,
        remarks: "",
        information_source: "EPSG. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2012-01-05",
        deprecated: 0, 
    },

    "EPSG:4240" => CRS {
        coord_ref_sys_code: 4240,
        coord_ref_sys_name: "Indian 1975",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6240, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2011-01-25",
        deprecated: 0, 
    },

    "EPSG:4330" => CRS {
        coord_ref_sys_code: 4330,
        coord_ref_sys_name: "ITRF88 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6647, 
        base_crs_code: -1,
        remarks: "Superseded by ITRF89 (code 4331).",
        information_source: "International Earth Rotation Service (IERS) Annual Report for 1988. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:4331" => CRS {
        coord_ref_sys_code: 4331,
        coord_ref_sys_name: "ITRF89 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6648, 
        base_crs_code: -1,
        remarks: "Supersedes ITRF88 (code 4330). Superseded by ITRF91 (code 4332).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #6. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:4332" => CRS {
        coord_ref_sys_code: 4332,
        coord_ref_sys_name: "ITRF90 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6649, 
        base_crs_code: -1,
        remarks: "Supersedes ITRF89 (code 4331). Superseded by ITRF91 (code 4333).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #9. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:4333" => CRS {
        coord_ref_sys_code: 4333,
        coord_ref_sys_name: "ITRF91 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6650, 
        base_crs_code: -1,
        remarks: "Supersedes ITRF90 (code 4332). Superseded by ITRF92 (code 4334).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #12. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:4334" => CRS {
        coord_ref_sys_code: 4334,
        coord_ref_sys_name: "ITRF92 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6651, 
        base_crs_code: -1,
        remarks: "Supersedes ITRF91 (code 4333). Superseded by ITRF93 (code 4335).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #15. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:4335" => CRS {
        coord_ref_sys_code: 4335,
        coord_ref_sys_name: "ITRF93 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6652, 
        base_crs_code: -1,
        remarks: "Supersedes ITRF92 (code 4334). Superseded by ITRF94 (code 4336).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #18. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:4405" => CRS {
        coord_ref_sys_code: 4405,
        coord_ref_sys_name: "NAD27 / BLM 5N (ftUS)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4497,
        datum_code: -1, 
        base_crs_code: 4267,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2011-06-30",
        deprecated: 0, 
    },

    "EPSG:4336" => CRS {
        coord_ref_sys_code: 4336,
        coord_ref_sys_name: "ITRF94 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6653, 
        base_crs_code: -1,
        remarks: "Supersedes ITRF93 (code 4335). Superseded by ITRF96 (code 4337).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #20. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:4337" => CRS {
        coord_ref_sys_code: 4337,
        coord_ref_sys_name: "ITRF96 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6654, 
        base_crs_code: -1,
        remarks: "Supersedes ITRF94 (code 4336). Superseded by ITRF97 (code 4338).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #24. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:4338" => CRS {
        coord_ref_sys_code: 4338,
        coord_ref_sys_name: "ITRF97 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6655, 
        base_crs_code: -1,
        remarks: "Supersedes ITRF96 (code 4337). Superseded by ITRF2000 (code 4385).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #27. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:4351" => CRS {
        coord_ref_sys_code: 4351,
        coord_ref_sys_name: "IRENET95 (3D)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6401,
        datum_code: 6173, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Ordnance Survey of Ireland",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:4352" => CRS {
        coord_ref_sys_code: 4352,
        coord_ref_sys_name: "IRENET95 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6173, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Ordnance Survey of Ireland",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:4385" => CRS {
        coord_ref_sys_code: 4385,
        coord_ref_sys_name: "ITRF2000 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6656, 
        base_crs_code: -1,
        remarks: "Supersedes ITRF97 (code 4336).",
        information_source: "IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:4386" => CRS {
        coord_ref_sys_code: 4386,
        coord_ref_sys_name: "ISN93 (3D)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6401,
        datum_code: 6659, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Landmaelingar Islands (National Survey of Iceland).",
        data_source: "EPSG",
        revision_date: "2003-06-27",
        deprecated: 1, 
    },

    "EPSG:4387" => CRS {
        coord_ref_sys_code: 4387,
        coord_ref_sys_name: "ISN93 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6659, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Landmaelingar Islands (National Survey of Iceland).",
        data_source: "EPSG",
        revision_date: "2003-06-27",
        deprecated: 1, 
    },

    "EPSG:4659" => CRS {
        coord_ref_sys_code: 4659,
        coord_ref_sys_name: "ISN93",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6659, 
        base_crs_code: 4945,
        remarks: "Replaced by ISN2004 (CRS code 5324).",
        information_source: "EPSG. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2010-11-12",
        deprecated: 0, 
    },

    "EPSG:4709" => CRS {
        coord_ref_sys_code: 4709,
        coord_ref_sys_name: "Iwo Jima 1945",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6709, 
        base_crs_code: -1,
        remarks: "",
        information_source: "DMA / NIMA / NGA TR8350.2 (3rd edition, January 2000).",
        data_source: "EPSG",
        revision_date: "2006-01-26",
        deprecated: 0, 
    },

    "EPSG:4758" => CRS {
        coord_ref_sys_code: 4758,
        coord_ref_sys_name: "JAD2001",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6758, 
        base_crs_code: 4895,
        remarks: "Replaces JAD69 (CRS code 4242).",
        information_source: "OGP. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2007-08-27",
        deprecated: 0, 
    },

    "EPSG:4894" => CRS {
        coord_ref_sys_code: 4894,
        coord_ref_sys_name: "JAD2001",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6758, 
        base_crs_code: -1,
        remarks: "",
        information_source: "National Land Agency.",
        data_source: "EPSG",
        revision_date: "2007-01-19",
        deprecated: 0, 
    },

    "EPSG:4895" => CRS {
        coord_ref_sys_code: 4895,
        coord_ref_sys_name: "JAD2001",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6758, 
        base_crs_code: 4894,
        remarks: "",
        information_source: "National Land Agency.",
        data_source: "EPSG",
        revision_date: "2013-05-30",
        deprecated: 0, 
    },

    "EPSG:4896" => CRS {
        coord_ref_sys_code: 4896,
        coord_ref_sys_name: "ITRF2005",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6896, 
        base_crs_code: -1,
        remarks: "Replaces ITRF2000 (CRS code 4919). Replaced by ITRF2008 (CRS code 5332).",
        information_source: "IGN Paris http://itrf.ensg.ign.fr/itrs_itrf.php",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:4910" => CRS {
        coord_ref_sys_code: 4910,
        coord_ref_sys_name: "ITRF88",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6647, 
        base_crs_code: -1,
        remarks: "Replaced by ITRF89 (code 4911).",
        information_source: "International Earth Rotation Service (IERS) Annual Report for 1988. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:4911" => CRS {
        coord_ref_sys_code: 4911,
        coord_ref_sys_name: "ITRF89",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6648, 
        base_crs_code: -1,
        remarks: "Replaces ITRF88 (code 4910). Replaced by ITRF90 (code 4912).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #6. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:4912" => CRS {
        coord_ref_sys_code: 4912,
        coord_ref_sys_name: "ITRF90",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6649, 
        base_crs_code: -1,
        remarks: "Replaces ITRF89 (code 4911). Replaced by ITRF91 (code 4913).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #9. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:4913" => CRS {
        coord_ref_sys_code: 4913,
        coord_ref_sys_name: "ITRF91",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6650, 
        base_crs_code: -1,
        remarks: "Replaces ITRF90 (code 4912). Replaced by ITRF92 (code 4914).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #12. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:4914" => CRS {
        coord_ref_sys_code: 4914,
        coord_ref_sys_name: "ITRF92",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6651, 
        base_crs_code: -1,
        remarks: "Replaces ITRF91 (code 4913). Replaced by ITRF93 (code 4915).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #15. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:4915" => CRS {
        coord_ref_sys_code: 4915,
        coord_ref_sys_name: "ITRF93",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6652, 
        base_crs_code: -1,
        remarks: "Replaces ITRF92 (code 4914). Replaced by ITRF94 (code 4916).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #18. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:4916" => CRS {
        coord_ref_sys_code: 4916,
        coord_ref_sys_name: "ITRF94",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6653, 
        base_crs_code: -1,
        remarks: "Replaces ITRF93 (code 4915). Replaced by ITRF96 (code 4917).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #20. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:4917" => CRS {
        coord_ref_sys_code: 4917,
        coord_ref_sys_name: "ITRF96",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6654, 
        base_crs_code: -1,
        remarks: "Replaces ITRF94 (code 4916). Replaced by ITRF97 (code 4918).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #24. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:4918" => CRS {
        coord_ref_sys_code: 4918,
        coord_ref_sys_name: "ITRF97",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6655, 
        base_crs_code: -1,
        remarks: "Replaces ITRF96 (code 4917). Replaced by ITRF2000 (code 4919).",
        information_source: "International Earth Rotation Service (IERS) Technical Note #27. Also IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:4919" => CRS {
        coord_ref_sys_code: 4919,
        coord_ref_sys_name: "ITRF2000",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6656, 
        base_crs_code: -1,
        remarks: "Replaces ITRF97 (code 4918). Replaced by ITRF2005 (code 4896).",
        information_source: "IGN Paris http://lareg.ensg.ign.fr/ITRF",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:4942" => CRS {
        coord_ref_sys_code: 4942,
        coord_ref_sys_name: "IRENET95",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6173, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Ordnance Survey of Ireland",
        data_source: "EPSG",
        revision_date: "2012-01-05",
        deprecated: 0, 
    },

    "EPSG:4943" => CRS {
        coord_ref_sys_code: 4943,
        coord_ref_sys_name: "IRENET95",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6173, 
        base_crs_code: 4942,
        remarks: "",
        information_source: "Ordnance Survey of Ireland",
        data_source: "EPSG",
        revision_date: "2012-01-05",
        deprecated: 0, 
    },

    "EPSG:4944" => CRS {
        coord_ref_sys_code: 4944,
        coord_ref_sys_name: "ISN93",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6659, 
        base_crs_code: -1,
        remarks: "Replaced by ISN2004 (CRS code 5322).",
        information_source: "Landmaelingar Islands (National Survey of Iceland).",
        data_source: "EPSG",
        revision_date: "2010-11-12",
        deprecated: 0, 
    },

    "EPSG:4945" => CRS {
        coord_ref_sys_code: 4945,
        coord_ref_sys_name: "ISN93",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6659, 
        base_crs_code: 4944,
        remarks: "Replaced by ISN2004 (CRS code 5323).",
        information_source: "Landmaelingar Islands (National Survey of Iceland).",
        data_source: "EPSG",
        revision_date: "2010-11-12",
        deprecated: 0, 
    },

    "EPSG:5322" => CRS {
        coord_ref_sys_code: 5322,
        coord_ref_sys_name: "ISN2004",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1060, 
        base_crs_code: -1,
        remarks: "Replaces ISN93 (CRS code 4944).",
        information_source: "Landmaelingar Islands (National Land Survey of Iceland). http://www.lmi.is",
        data_source: "EPSG",
        revision_date: "2010-11-10",
        deprecated: 0, 
    },

    "EPSG:5323" => CRS {
        coord_ref_sys_code: 5323,
        coord_ref_sys_name: "ISN2004",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1060, 
        base_crs_code: 5322,
        remarks: "Replaces ISN93 (CRS code 4945).",
        information_source: "Landmaelingar Islands (National Land Survey of Iceland). http://www.lmi.is",
        data_source: "EPSG",
        revision_date: "2010-11-10",
        deprecated: 0, 
    },

    "EPSG:5324" => CRS {
        coord_ref_sys_code: 5324,
        coord_ref_sys_name: "ISN2004",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1060, 
        base_crs_code: 5323,
        remarks: "Replaces ISN93 (CRS code 4659).",
        information_source: "OGP. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2010-11-10",
        deprecated: 0, 
    },

    "EPSG:5325" => CRS {
        coord_ref_sys_code: 5325,
        coord_ref_sys_name: "ISN2004 / Lambert 2004",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 5324,
        remarks: "Replaces ISN93 / Lambert 1993 (CRS code 3057).",
        information_source: "Landmaelingar Islands (National Land Survey of Iceland). http://www.lmi.is",
        data_source: "EPSG",
        revision_date: "2010-11-10",
        deprecated: 0, 
    },

    "EPSG:5332" => CRS {
        coord_ref_sys_code: 5332,
        coord_ref_sys_name: "ITRF2008",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1061, 
        base_crs_code: -1,
        remarks: "Replaces ITRF2005 (CRS code 4896). Replaced by ITRF2014 (CRS code 7789).",
        information_source: "International Earth Rotation Service (IERS). http://itrf.ensg.ign.fr",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:5638" => CRS {
        coord_ref_sys_code: 5638,
        coord_ref_sys_name: "ISN2004 / LAEA Europe",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4532,
        datum_code: -1, 
        base_crs_code: 5324,
        remarks: "At applicable scales and usages, may be considered consistent with ETRS89-extended / LAEA Europe (CRS code 3035): see CT 9041. Replaced by ISN2016 / LAEA Europe (CRS code 9039).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:5639" => CRS {
        coord_ref_sys_code: 5639,
        coord_ref_sys_name: "ISN2004 / LCC Europe",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4500,
        datum_code: -1, 
        base_crs_code: 5324,
        remarks: "At applicable scales (1:500,000 and smaller) may be considered consistent with ETRS89-extended / LCC Europe (CRS code 3034): see CT 9042. Replaced by ISN2016 / LCC Europe (CRS code 9040).",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:5829" => CRS {
        coord_ref_sys_code: 5829,
        coord_ref_sys_name: "Instantaneous Water Level height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5113, 
        base_crs_code: -1,
        remarks: "Height relative to instantaneous water level uncorrected for tide. Not specific to any location or epoch.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-08-10",
        deprecated: 0, 
    },

    "EPSG:61416405" => CRS {
        coord_ref_sys_code: 61416405,
        coord_ref_sys_name: "Israel (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4141,
        remarks: "See CRS code 4141 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:5831" => CRS {
        coord_ref_sys_code: 5831,
        coord_ref_sys_name: "Instantaneous Water Level depth",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6498,
        datum_code: -1, 
        base_crs_code: 5829,
        remarks: "Depth relative to instantaneous water level uncorrected for tide. Not specific to any location or epoch.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-04-20",
        deprecated: 0, 
    },

    "EPSG:5863" => CRS {
        coord_ref_sys_code: 5863,
        coord_ref_sys_name: "ISLW depth",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6498,
        datum_code: 1085, 
        base_crs_code: -1,
        remarks: "Not specific to any location or epoch.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2012-08-10",
        deprecated: 0, 
    },

    "EPSG:6984" => CRS {
        coord_ref_sys_code: 6984,
        coord_ref_sys_name: "Israeli Grid 05",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 6983,
        remarks: "Replaces Israel 1993 / Israeli TM Grid (CRS code 2039). As an approximation at the one metre level these grids can be considered coincident. Replaced by Israeli Grid 05/12 (code 6991). Derived from IGD05 through official transformation details unknown.",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2014-11-19",
        deprecated: 0, 
    },

    "EPSG:6991" => CRS {
        coord_ref_sys_code: 6991,
        coord_ref_sys_name: "Israeli Grid 05/12",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 6990,
        remarks: "Derived from IGD05/12 (CRS code 6987) through official transformation (code 6993) and Israeli TM projection. Replaces IG05 (CRS code 6984). This grid IG05/12 coincident with IG05 at the 1dm level and with Israeli New Grid (CRS code 2039) at the 1m level.",
        information_source: "Survey of Israel.",
        data_source: "EPSG",
        revision_date: "2014-11-19",
        deprecated: 0, 
    },

    "EPSG:7789" => CRS {
        coord_ref_sys_code: 7789,
        coord_ref_sys_name: "ITRF2014",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1165, 
        base_crs_code: -1,
        remarks: "Replaces ITRF2008 (CRS code 5332).",
        information_source: "International Earth Rotation Service (IERS). http://itrf.ensg.ign.fr",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:7900" => CRS {
        coord_ref_sys_code: 7900,
        coord_ref_sys_name: "ITRF88",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6647, 
        base_crs_code: 4910,
        remarks: "Replaced by ITRF89 (code 7901).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:7901" => CRS {
        coord_ref_sys_code: 7901,
        coord_ref_sys_name: "ITRF89",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6648, 
        base_crs_code: 4911,
        remarks: "Replaces ITRF88 (code 7900). Replaced by ITRF90 (code 7902).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:7902" => CRS {
        coord_ref_sys_code: 7902,
        coord_ref_sys_name: "ITRF90",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6649, 
        base_crs_code: 4912,
        remarks: "Replaces ITRF89 (code 7901). Replaced by ITRF91 (code 7903).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:7903" => CRS {
        coord_ref_sys_code: 7903,
        coord_ref_sys_name: "ITRF91",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6650, 
        base_crs_code: 4912,
        remarks: "Replaces ITRF90 (code 7902). Replaced by ITRF92 (code 7904).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:7904" => CRS {
        coord_ref_sys_code: 7904,
        coord_ref_sys_name: "ITRF92",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6651, 
        base_crs_code: 4914,
        remarks: "Replaces ITRF91 (code 7903). Replaced by ITRF93 (code 7905).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:7905" => CRS {
        coord_ref_sys_code: 7905,
        coord_ref_sys_name: "ITRF93",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6652, 
        base_crs_code: 4915,
        remarks: "Replaces ITRF92 (code 7904). Replaced by ITRF94 (code 7906).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:7906" => CRS {
        coord_ref_sys_code: 7906,
        coord_ref_sys_name: "ITRF94",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6653, 
        base_crs_code: 4916,
        remarks: "Replaces ITRF93 (code 7905). Replaced by ITRF96 (code 7907).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:7907" => CRS {
        coord_ref_sys_code: 7907,
        coord_ref_sys_name: "ITRF96",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6654, 
        base_crs_code: 4917,
        remarks: "Replaces ITRF94 (code 7906). Replaced by ITRF97 (code 7908).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:7908" => CRS {
        coord_ref_sys_code: 7908,
        coord_ref_sys_name: "ITRF97",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6655, 
        base_crs_code: 4918,
        remarks: "Replaces ITRF96 (code 7907). Replaced by ITRF2000 (code 7909).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:7909" => CRS {
        coord_ref_sys_code: 7909,
        coord_ref_sys_name: "ITRF2000",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6656, 
        base_crs_code: 4919,
        remarks: "Replaces ITRF97 (code 7908). Replaced by ITRF2005 (code 7910).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:7910" => CRS {
        coord_ref_sys_code: 7910,
        coord_ref_sys_name: "ITRF2005",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6896, 
        base_crs_code: 4896,
        remarks: "Replaces ITRF2000 (code 7909). Replaced by ITRF2008 (code 7911).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:7911" => CRS {
        coord_ref_sys_code: 7911,
        coord_ref_sys_name: "ITRF2008",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1061, 
        base_crs_code: 5332,
        remarks: "Replaces ITRF2005 (code 7910). Replaced by ITRF2014 (code 7912).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:7912" => CRS {
        coord_ref_sys_code: 7912,
        coord_ref_sys_name: "ITRF2014",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1165, 
        base_crs_code: 7789,
        remarks: "Replaces ITRF2008 (code 7911).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:8084" => CRS {
        coord_ref_sys_code: 8084,
        coord_ref_sys_name: "ISN2016",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1187, 
        base_crs_code: -1,
        remarks: "Replaces ISN2004 (CRS code 5322) from September 2017.",
        information_source: "Landmaelingar Islands (National Land Survey of Iceland). http://www.lmi.is",
        data_source: "EPSG",
        revision_date: "2017-05-25",
        deprecated: 0, 
    },

    "EPSG:8085" => CRS {
        coord_ref_sys_code: 8085,
        coord_ref_sys_name: "ISN2016",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1187, 
        base_crs_code: 8084,
        remarks: "Replaces ISN2004 (CRS code 5323) from September 2017.",
        information_source: "Landmaelingar Islands (National Land Survey of Iceland). http://www.lmi.is",
        data_source: "EPSG",
        revision_date: "2017-05-25",
        deprecated: 0, 
    },

    "EPSG:8086" => CRS {
        coord_ref_sys_code: 8086,
        coord_ref_sys_name: "ISN2016",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1187, 
        base_crs_code: 8085,
        remarks: "Replaces ISN2004 (CRS code 5324) from September 2017.",
        information_source: "IOGP. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2017-05-25",
        deprecated: 0, 
    },

    "EPSG:8088" => CRS {
        coord_ref_sys_code: 8088,
        coord_ref_sys_name: "ISN2016 / Lambert 2016",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 8086,
        remarks: "Replaces ISN2004 / Lambert 2004 (CRS code 5325) from September 2017.",
        information_source: "Landmaelingar Islands (National Land Survey of Iceland). http://www.lmi.is",
        data_source: "EPSG",
        revision_date: "2017-05-25",
        deprecated: 0, 
    },

    "EPSG:8089" => CRS {
        coord_ref_sys_code: 8089,
        coord_ref_sys_name: "ISH2004 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1190, 
        base_crs_code: -1,
        remarks: "National system replacing older local systems from March 2011.",
        information_source: "Landmaelingar Islands (National Land Survey of Iceland). http://www.lmi.is",
        data_source: "EPSG",
        revision_date: "2017-05-25",
        deprecated: 0, 
    },

    "EPSG:8988" => CRS {
        coord_ref_sys_code: 8988,
        coord_ref_sys_name: "ITRF88",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6647, 
        base_crs_code: 7900,
        remarks: "Replaced by ITRF89 (code 8989).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:8989" => CRS {
        coord_ref_sys_code: 8989,
        coord_ref_sys_name: "ITRF89",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6648, 
        base_crs_code: 7901,
        remarks: "Replaces ITRF88 (code 8988). Replaced by ITRF90 (code 8990).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:8990" => CRS {
        coord_ref_sys_code: 8990,
        coord_ref_sys_name: "ITRF90",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6649, 
        base_crs_code: 7902,
        remarks: "Replaces ITRF89 (code 8989). Replaced by ITRF91 (code 8991).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:8991" => CRS {
        coord_ref_sys_code: 8991,
        coord_ref_sys_name: "ITRF91",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6650, 
        base_crs_code: 7903,
        remarks: "Replaces ITRF90 (code 8990). Replaced by ITRF92 (code 8992).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:8992" => CRS {
        coord_ref_sys_code: 8992,
        coord_ref_sys_name: "ITRF92",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6651, 
        base_crs_code: 7904,
        remarks: "Replaces ITRF91 (code 8991). Replaced by ITRF93 (code 8993).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:8993" => CRS {
        coord_ref_sys_code: 8993,
        coord_ref_sys_name: "ITRF93",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6652, 
        base_crs_code: 7905,
        remarks: "Replaces ITRF92 (code 8992). Replaced by ITRF94 (code 8994).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:8994" => CRS {
        coord_ref_sys_code: 8994,
        coord_ref_sys_name: "ITRF94",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6653, 
        base_crs_code: 7906,
        remarks: "Replaces ITRF93 (code 8993). Replaced by ITRF96 (code 8995).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:8995" => CRS {
        coord_ref_sys_code: 8995,
        coord_ref_sys_name: "ITRF96",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6654, 
        base_crs_code: 7907,
        remarks: "Replaces ITRF94 (code 8994). Replaced by ITRF97 (code 8996).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:8996" => CRS {
        coord_ref_sys_code: 8996,
        coord_ref_sys_name: "ITRF97",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6655, 
        base_crs_code: 7908,
        remarks: "Replaces ITRF96 (code 8995). Replaced by ITRF2000 (code 8997).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:8997" => CRS {
        coord_ref_sys_code: 8997,
        coord_ref_sys_name: "ITRF2000",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6656, 
        base_crs_code: 7909,
        remarks: "Replaces ITRF97 (code 8996). Replaced by ITRF2005 (code 8998).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:8998" => CRS {
        coord_ref_sys_code: 8998,
        coord_ref_sys_name: "ITRF2005",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6896, 
        base_crs_code: 7910,
        remarks: "Replaces ITRF2000 (code 8997). Replaced by ITRF2008 (code 8999).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:8999" => CRS {
        coord_ref_sys_code: 8999,
        coord_ref_sys_name: "ITRF2008",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1061, 
        base_crs_code: 7911,
        remarks: "Replaces ITRF2005 (code 8998). Replaced by ITRF2014 (code 9000).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9000" => CRS {
        coord_ref_sys_code: 9000,
        coord_ref_sys_name: "ITRF2014",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1165, 
        base_crs_code: 7912,
        remarks: "Replaces ITRF2008 (code 8999).",
        information_source: "IOGP.",
        data_source: "EPSG",
        revision_date: "2019-05-17",
        deprecated: 0, 
    },

    "EPSG:9039" => CRS {
        coord_ref_sys_code: 9039,
        coord_ref_sys_name: "ISN2016 / LAEA Europe",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4532,
        datum_code: -1, 
        base_crs_code: 8086,
        remarks: "At applicable scales and usages, may be considered consistent with ETRS89-extended / LAEA Europe (CRS code 3035): see CT 9043. Replaces ISN2004 / LAEA Europe (CRS code 5638).",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2019-05-16",
        deprecated: 0, 
    },

    "EPSG:9040" => CRS {
        coord_ref_sys_code: 9040,
        coord_ref_sys_name: "ISN2016 / LCC Europe",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4532,
        datum_code: -1, 
        base_crs_code: 8086,
        remarks: "At applicable scales (1:500,000 and smaller) may be considered consistent with ETRS89-extended / LCC Europe (CRS code 3034): see CT 9044. Replaces ISN2004 / LCC Europe (CRS code 5639).",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2019-05-16",
        deprecated: 0, 
    },

    "EPSG:9543" => CRS {
        coord_ref_sys_code: 9543,
        coord_ref_sys_name: "ITRF2005 + SA LLD height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2021-02-05",
        deprecated: 0, 
    },

    "EPSG:24047" => CRS {
        coord_ref_sys_code: 24047,
        coord_ref_sys_name: "Indian 1975 / UTM zone 47N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4240,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:61736405" => CRS {
        coord_ref_sys_code: 61736405,
        coord_ref_sys_name: "IRENET95 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4173,
        remarks: "See CRS code 4173 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61736413" => CRS {
        coord_ref_sys_code: 61736413,
        coord_ref_sys_name: "IRENET95 (3D deg)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6413,
        datum_code: -1, 
        base_crs_code: 4351,
        remarks: "See CRS code 4351 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:62406405" => CRS {
        coord_ref_sys_code: 62406405,
        coord_ref_sys_name: "Indian 1975 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4240,
        remarks: "See CRS code 4240 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:66596405" => CRS {
        coord_ref_sys_code: 66596405,
        coord_ref_sys_name: "ISN93 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4659,
        remarks: "See CRS code 4659 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2003-06-27",
        deprecated: 1, 
    },

    "EPSG:66596413" => CRS {
        coord_ref_sys_code: 66596413,
        coord_ref_sys_name: "ISN93 (3D deg)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6401,
        datum_code: -1, 
        base_crs_code: 4386,
        remarks: "See CRS code 4386 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2003-06-27",
        deprecated: 1, 
    },

    "EPSG:2443" => CRS {
        coord_ref_sys_code: 2443,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS I",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS I (code 30161).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2444" => CRS {
        coord_ref_sys_code: 2444,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS II",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS II (code 30162).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2445" => CRS {
        coord_ref_sys_code: 2445,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS III",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS III (code 30163).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2446" => CRS {
        coord_ref_sys_code: 2446,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS IV",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS IV (code 30164).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2447" => CRS {
        coord_ref_sys_code: 2447,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS V",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS V (code 30165).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2448" => CRS {
        coord_ref_sys_code: 2448,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS VI",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS VI (code 30166).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2449" => CRS {
        coord_ref_sys_code: 2449,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS VII",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS VII (code 30167).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2450" => CRS {
        coord_ref_sys_code: 2450,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS VIII",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS VIII (code 30168).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2451" => CRS {
        coord_ref_sys_code: 2451,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS IX",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS IX (code 30169).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2452" => CRS {
        coord_ref_sys_code: 2452,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS X",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS X (code 30170).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2453" => CRS {
        coord_ref_sys_code: 2453,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS XI",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS XI (code 30171).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2454" => CRS {
        coord_ref_sys_code: 2454,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS XII",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS XII (code 30172).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2455" => CRS {
        coord_ref_sys_code: 2455,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS XIII",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS XIII (code 30173).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2456" => CRS {
        coord_ref_sys_code: 2456,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS XIV",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS XIV (code 30174).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2457" => CRS {
        coord_ref_sys_code: 2457,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS XV",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS XV (code 30175).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2458" => CRS {
        coord_ref_sys_code: 2458,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS XVI",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS XVI (code 30176).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2459" => CRS {
        coord_ref_sys_code: 2459,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS XVII",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS XVII (code 30177).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2460" => CRS {
        coord_ref_sys_code: 2460,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS XVIII",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS XVIII (code 30178).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2461" => CRS {
        coord_ref_sys_code: 2461,
        coord_ref_sys_name: "JGD2000 / Japan Plane Rectangular CS XIX",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / Japan Plane Rectangular CS XIX (code 30179).",
        information_source: "Geographic Survey Institute; Japan. http://vldb.gsi.go.jp/sokuchi/datum/tokyodatum.html",
        data_source: "EPSG",
        revision_date: "2002-06-22",
        deprecated: 0, 
    },

    "EPSG:2979" => CRS {
        coord_ref_sys_code: 2979,
        coord_ref_sys_name: "K0 1949 / UTM zone 42S",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4631,
        remarks: "",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:3097" => CRS {
        coord_ref_sys_code: 3097,
        coord_ref_sys_name: "JGD2000 / UTM zone 51N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / UTM zone 51N (code 3092). Replaced by JGD2011 / UTM zone 51N (code 6688).",
        information_source: "Geographic Survey Institute; Japan.",
        data_source: "EPSG",
        revision_date: "2017-09-16",
        deprecated: 0, 
    },

    "EPSG:3098" => CRS {
        coord_ref_sys_code: 3098,
        coord_ref_sys_name: "JGD2000 / UTM zone 52N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / UTM zone 52N (code 3093). Replaced by JGD2011 / UTM zone 52N (code 6689).",
        information_source: "Geographic Survey Institute; Japan.",
        data_source: "EPSG",
        revision_date: "2017-09-16",
        deprecated: 0, 
    },

    "EPSG:3099" => CRS {
        coord_ref_sys_code: 3099,
        coord_ref_sys_name: "JGD2000 / UTM zone 53N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / UTM zone 53N (code 3094). Replaced by JGD2011 / UTM zone 53N (code 6690).",
        information_source: "Geographic Survey Institute; Japan.",
        data_source: "EPSG",
        revision_date: "2017-09-16",
        deprecated: 0, 
    },

    "EPSG:3100" => CRS {
        coord_ref_sys_code: 3100,
        coord_ref_sys_name: "JGD2000 / UTM zone 54N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / UTM zone 54N (code 3095). Replaced by JGD2011 / UTM zone 54N (code 6691).",
        information_source: "Geographic Survey Institute; Japan.",
        data_source: "EPSG",
        revision_date: "2017-09-16",
        deprecated: 0, 
    },

    "EPSG:3101" => CRS {
        coord_ref_sys_code: 3101,
        coord_ref_sys_name: "JGD2000 / UTM zone 55N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "Replaces Tokyo / UTM zone 55N (code 3096). Replaced by JGD2011 / UTM zone 55N (code 6692).",
        information_source: "Geographic Survey Institute; Japan.",
        data_source: "EPSG",
        revision_date: "2017-09-16",
        deprecated: 0, 
    },

    "EPSG:3450" => CRS {
        coord_ref_sys_code: 3450,
        coord_ref_sys_name: "JAD2001 / UTM zone 18N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4758,
        remarks: "See JAD2001 / Jamaica Metric Grid (CRS code 3448) for land applications.",
        information_source: "National Land Agency.",
        data_source: "EPSG",
        revision_date: "2007-01-19",
        deprecated: 0, 
    },

    "EPSG:4144" => CRS {
        coord_ref_sys_code: 4144,
        coord_ref_sys_name: "Kalianpur 1937",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6144, 
        base_crs_code: -1,
        remarks: "Adopts 1937 metric conversion of 0.30479841 metres per Indian foot.",
        information_source: "G. Bomford; &quot;The Re-adjustment of the Indian Triangulation&quot;; Survey of India Professional Paper 28; 1939.",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4145" => CRS {
        coord_ref_sys_code: 4145,
        coord_ref_sys_name: "Kalianpur 1962",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6145, 
        base_crs_code: -1,
        remarks: "Adopts 1962 metric conversion of 0.3047996 metres per Indian foot.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2018-04-20",
        deprecated: 0, 
    },

    "EPSG:4146" => CRS {
        coord_ref_sys_code: 4146,
        coord_ref_sys_name: "Kalianpur 1975",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6146, 
        base_crs_code: -1,
        remarks: "Adopts 1975 metric conversion of 0.3047995 metres per Indian foot.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4241" => CRS {
        coord_ref_sys_code: 4241,
        coord_ref_sys_name: "Jamaica 1875",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6241, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Survey Department, Government of Jamaica, 1983.",
        data_source: "EPSG",
        revision_date: "2004-09-12",
        deprecated: 0, 
    },

    "EPSG:4242" => CRS {
        coord_ref_sys_code: 4242,
        coord_ref_sys_name: "JAD69",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6242, 
        base_crs_code: -1,
        remarks: "Replaced by JAD2001 (CRS code 4758).",
        information_source: "Survey Department, Government of Jamaica, 1983.",
        data_source: "EPSG",
        revision_date: "2007-01-19",
        deprecated: 0, 
    },

    "EPSG:4243" => CRS {
        coord_ref_sys_code: 4243,
        coord_ref_sys_name: "Kalianpur 1880",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6243, 
        base_crs_code: -1,
        remarks: "",
        information_source: "G. Bomford; &quot;The Re-adjustment of the Indian Triangulation&quot;; Survey of India Professional Paper 28; 1939.",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4353" => CRS {
        coord_ref_sys_code: 4353,
        coord_ref_sys_name: "JGD2000 (3D)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6401,
        datum_code: 6612, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Japanese Survey Federation.",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:4354" => CRS {
        coord_ref_sys_code: 4354,
        coord_ref_sys_name: "JGD2000 (geocentric)",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6612, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Japanese Survey Federation.",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:4612" => CRS {
        coord_ref_sys_code: 4612,
        coord_ref_sys_name: "JGD2000",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6612, 
        base_crs_code: 4947,
        remarks: "Replaces Tokyo (CRS code 4301) from April 2002. From 21st October 2011 replaced by JGD2011 (CRS code 6668).",
        information_source: "EPSG. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:4631" => CRS {
        coord_ref_sys_code: 4631,
        coord_ref_sys_name: "K0 1949",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6631, 
        base_crs_code: -1,
        remarks: "",
        information_source: "IGN Paris",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 1, 
    },

    "EPSG:4679" => CRS {
        coord_ref_sys_code: 4679,
        coord_ref_sys_name: "Jouik 1961",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6679, 
        base_crs_code: -1,
        remarks: "Replaced by Mauritania 1999 (CRS code 4702).",
        information_source: "Woodside",
        data_source: "EPSG",
        revision_date: "2006-06-12",
        deprecated: 0, 
    },

    "EPSG:4725" => CRS {
        coord_ref_sys_code: 4725,
        coord_ref_sys_name: "Johnston Island 1961",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6725, 
        base_crs_code: -1,
        remarks: "",
        information_source: "DMA / NIMA / NGA TR8350.2 (3rd edition, January 2000).",
        data_source: "EPSG",
        revision_date: "2006-01-26",
        deprecated: 0, 
    },

    "EPSG:4946" => CRS {
        coord_ref_sys_code: 4946,
        coord_ref_sys_name: "JGD2000",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6612, 
        base_crs_code: -1,
        remarks: "From 21st October 2011 replaced by JGD2011 (CRS code 6666).",
        information_source: "Japanese Survey Federation.",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:4947" => CRS {
        coord_ref_sys_code: 4947,
        coord_ref_sys_name: "JGD2000",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6612, 
        base_crs_code: 4946,
        remarks: "From 21st October 2011 replaced by JGD2011 (CRS code 6667).",
        information_source: "Japanese Survey Federation.",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:5723" => CRS {
        coord_ref_sys_code: 5723,
        coord_ref_sys_name: "JSLD69 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5122, 
        base_crs_code: -1,
        remarks: "Replaces JSLD49. Replaced by JGD2000 (vertical) (CRS code 6694) with effect from April 2002.",
        information_source: "Geospatial Information Authority of Japan (GSI) bulletin volume 51 of March 2004.",
        data_source: "EPSG",
        revision_date: "2014-05-01",
        deprecated: 0, 
    },

    "EPSG:5890" => CRS {
        coord_ref_sys_code: 5890,
        coord_ref_sys_name: "JAXA Snow Depth Polar Stereographic North",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 1035,
        datum_code: -1, 
        base_crs_code: 4054,
        remarks: "The datum is unspecified. Uncertainty in location of over 1 km may result; at the coarse resolution and very small scales for which this system should be used this uncertainty may be insignificant.",
        information_source: "Japan Aerospace Exploration Agency.",
        data_source: "EPSG",
        revision_date: "2012-11-10",
        deprecated: 1, 
    },

    "EPSG:6666" => CRS {
        coord_ref_sys_code: 6666,
        coord_ref_sys_name: "JGD2011",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1128, 
        base_crs_code: -1,
        remarks: "Replaces JGD2000 (CRS code 4946) with effect from 21st October 2011.",
        information_source: "Geodetic Department, Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:6667" => CRS {
        coord_ref_sys_code: 6667,
        coord_ref_sys_name: "JGD2011",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 1128, 
        base_crs_code: 6666,
        remarks: "Replaces JGD2000 (CRS code 4947) with effect from 21st October 2011.",
        information_source: "Geodetic Department, Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:6668" => CRS {
        coord_ref_sys_code: 6668,
        coord_ref_sys_name: "JGD2011",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 1128, 
        base_crs_code: 6667,
        remarks: "Replaces JGD2000 (CRS code 4612) with effect from 21st October 2011.",
        information_source: "EPSG. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:6669" => CRS {
        coord_ref_sys_code: 6669,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS I",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS I (CRS code 2443) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6670" => CRS {
        coord_ref_sys_code: 6670,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS II",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS II (CRS code 2444) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6671" => CRS {
        coord_ref_sys_code: 6671,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS III",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS III (CRS code 2445) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6672" => CRS {
        coord_ref_sys_code: 6672,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS IV",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS IV (CRS code 2446) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6673" => CRS {
        coord_ref_sys_code: 6673,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS V",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS V (CRS code 2447) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6674" => CRS {
        coord_ref_sys_code: 6674,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS VI",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS VI (CRS code 2448) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6675" => CRS {
        coord_ref_sys_code: 6675,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS VII",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS VII (CRS code 2449) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6676" => CRS {
        coord_ref_sys_code: 6676,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS VIII",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS VIII (CRS code 2450) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6677" => CRS {
        coord_ref_sys_code: 6677,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS IX",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS IX (CRS code 2451) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-16",
        deprecated: 0, 
    },

    "EPSG:6678" => CRS {
        coord_ref_sys_code: 6678,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS X",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS X (CRS code 2452) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6679" => CRS {
        coord_ref_sys_code: 6679,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS XI",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS XI (CRS code 2453) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6680" => CRS {
        coord_ref_sys_code: 6680,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS XII",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS XII (CRS code 2454) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6681" => CRS {
        coord_ref_sys_code: 6681,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS XIII",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS XIII (CRS code 2455) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6682" => CRS {
        coord_ref_sys_code: 6682,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS XIV",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS IV (CRS code 2456) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6683" => CRS {
        coord_ref_sys_code: 6683,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS XV",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS XV (CRS code 2457) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:4406" => CRS {
        coord_ref_sys_code: 4406,
        coord_ref_sys_name: "NAD27 / BLM 6N (ftUS)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4497,
        datum_code: -1, 
        base_crs_code: 4267,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2011-06-30",
        deprecated: 0, 
    },

    "EPSG:6684" => CRS {
        coord_ref_sys_code: 6684,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS XVI",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS XVI (CRS code 2458) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6685" => CRS {
        coord_ref_sys_code: 6685,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS XVII",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS XVII (CRS code 2459) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6686" => CRS {
        coord_ref_sys_code: 6686,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS XVIII",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS XVIII (CRS code 2460) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6687" => CRS {
        coord_ref_sys_code: 6687,
        coord_ref_sys_name: "JGD2011 / Japan Plane Rectangular CS XIX",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / Japan Plane Rectangular CS XIX (CRS code 2461) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6688" => CRS {
        coord_ref_sys_code: 6688,
        coord_ref_sys_name: "JGD2011 / UTM zone 51N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / UTM zone 51N (CRS code 3097) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6689" => CRS {
        coord_ref_sys_code: 6689,
        coord_ref_sys_name: "JGD2011 / UTM zone 52N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / UTM zone 52N (CRS code 3098) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6690" => CRS {
        coord_ref_sys_code: 6690,
        coord_ref_sys_name: "JGD2011 / UTM zone 53N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / UTM zone 53N (CRS code 3099) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6691" => CRS {
        coord_ref_sys_code: 6691,
        coord_ref_sys_name: "JGD2011 / UTM zone 54N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / UTM zone 54N (CRS code 3100) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6692" => CRS {
        coord_ref_sys_code: 6692,
        coord_ref_sys_name: "JGD2011 / UTM zone 55N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 6668,
        remarks: "Replaces JGD2000 / UTM zone 55N (CRS code 3101) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2013-12-15",
        deprecated: 0, 
    },

    "EPSG:6693" => CRS {
        coord_ref_sys_code: 6693,
        coord_ref_sys_name: "JSLD72 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1129, 
        base_crs_code: -1,
        remarks: "Replaced by JGD2000 (vertical) (CRS code 6694) with effect from April 2002.",
        information_source: "Geospatial Information Authority of Japan (GSI) bulletin volume 51 of March 2004.",
        data_source: "EPSG",
        revision_date: "2013-12-16",
        deprecated: 0, 
    },

    "EPSG:6694" => CRS {
        coord_ref_sys_code: 6694,
        coord_ref_sys_name: "JGD2000 (vertical) height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1130, 
        base_crs_code: -1,
        remarks: "Replaced JSLD69 and JSLD72 (CRS codes 5723 and 6693) from April 2002. Replaced by JGD2011 (vertical) (CRS code 6695) with effect from 21st October 2011.",
        information_source: "Geospatial Information Authority of Japan (GSI) bulletin volume 51 of March 2004.",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:6695" => CRS {
        coord_ref_sys_code: 6695,
        coord_ref_sys_name: "JGD2011 (vertical) height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1131, 
        base_crs_code: -1,
        remarks: "Replaces JGD2000 (vertical) (CRS code 6694) with effect from 21st October 2011.",
        information_source: "Geodetic Department, Geospatial Information Authority of Japan (GSI).",
        data_source: "EPSG",
        revision_date: "2019-07-17",
        deprecated: 0, 
    },

    "EPSG:6696" => CRS {
        coord_ref_sys_code: 6696,
        coord_ref_sys_name: "JGD2000 + JGD2000 (vertical) height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaces Tokyo + JSLD69 height and Tokyo + JSLD72 height (CRS codes 7414 and 6700) from April 2002. Replaced by JGD2011 + JGD2011 (vertical) height (CRS code 6697) with effect from 21st October 2011.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2014-09-18",
        deprecated: 0, 
    },

    "EPSG:6697" => CRS {
        coord_ref_sys_code: 6697,
        coord_ref_sys_name: "JGD2011 + JGD2011 (vertical) height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaces JGD2000 + JGD2000 (vertical) height (CRS code 6696) with effect from 21st October 2011.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2014-09-18",
        deprecated: 0, 
    },

    "EPSG:7888" => CRS {
        coord_ref_sys_code: 7888,
        coord_ref_sys_name: "Jamestown 1971 height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1175, 
        base_crs_code: -1,
        remarks: "Replaced by SHVD2016 (CRS code 7890) from 2016.",
        information_source: "St. Helena Government, Environment and Natural Resources Directorate (ENRD).",
        data_source: "EPSG",
        revision_date: "2016-11-24",
        deprecated: 0, 
    },

    "EPSG:24100" => CRS {
        coord_ref_sys_code: 24100,
        coord_ref_sys_name: "Jamaica 1875 / Jamaica (Old Grid)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4403,
        datum_code: -1, 
        base_crs_code: 4241,
        remarks: "Replaced by JAD69 / Jamaica National Grid (CRS code 24200).",
        information_source: "Survey Department, Government of Jamaica, 1983.",
        data_source: "EPSG",
        revision_date: "1995-06-02",
        deprecated: 0, 
    },

    "EPSG:24200" => CRS {
        coord_ref_sys_code: 24200,
        coord_ref_sys_name: "JAD69 / Jamaica National Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4242,
        remarks: "Replaces Jamaica 1875 / Jamaica (Old Grid) (CRS code 24100). Replaced by JAD2001 / Jamaica Metric Grid (CRS code 3448).",
        information_source: "Survey Department, Government of Jamaica, 1983.",
        data_source: "EPSG",
        revision_date: "2007-01-19",
        deprecated: 0, 
    },

    "EPSG:24305" => CRS {
        coord_ref_sys_code: 24305,
        coord_ref_sys_name: "Kalianpur 1937 / UTM zone 45N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4144,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1999-10-20",
        deprecated: 0, 
    },

    "EPSG:24306" => CRS {
        coord_ref_sys_code: 24306,
        coord_ref_sys_name: "Kalianpur 1937 / UTM zone 46N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4144,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1999-10-20",
        deprecated: 0, 
    },

    "EPSG:24311" => CRS {
        coord_ref_sys_code: 24311,
        coord_ref_sys_name: "Kalianpur 1962 / UTM zone 41N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4145,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1999-10-20",
        deprecated: 0, 
    },

    "EPSG:24312" => CRS {
        coord_ref_sys_code: 24312,
        coord_ref_sys_name: "Kalianpur 1962 / UTM zone 42N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4145,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1999-10-20",
        deprecated: 0, 
    },

    "EPSG:24313" => CRS {
        coord_ref_sys_code: 24313,
        coord_ref_sys_name: "Kalianpur 1962 / UTM zone 43N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4145,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1999-10-20",
        deprecated: 0, 
    },

    "EPSG:24370" => CRS {
        coord_ref_sys_code: 24370,
        coord_ref_sys_name: "Kalianpur 1880 / India zone 0",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4408,
        datum_code: -1, 
        base_crs_code: 4243,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1999-10-20",
        deprecated: 0, 
    },

    "EPSG:24371" => CRS {
        coord_ref_sys_code: 24371,
        coord_ref_sys_name: "Kalianpur 1880 / India zone I",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4408,
        datum_code: -1, 
        base_crs_code: 4243,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1999-10-20",
        deprecated: 0, 
    },

    "EPSG:24372" => CRS {
        coord_ref_sys_code: 24372,
        coord_ref_sys_name: "Kalianpur 1880 / India zone IIa",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4408,
        datum_code: -1, 
        base_crs_code: 4243,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1999-10-20",
        deprecated: 0, 
    },

    "EPSG:24373" => CRS {
        coord_ref_sys_code: 24373,
        coord_ref_sys_name: "Kalianpur 1880 / India zone IIIa",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4408,
        datum_code: -1, 
        base_crs_code: 4243,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-05-08",
        deprecated: 0, 
    },

    "EPSG:24374" => CRS {
        coord_ref_sys_code: 24374,
        coord_ref_sys_name: "Kalianpur 1880 / India zone IVa",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4408,
        datum_code: -1, 
        base_crs_code: 4243,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2012-05-08",
        deprecated: 0, 
    },

    "EPSG:24375" => CRS {
        coord_ref_sys_code: 24375,
        coord_ref_sys_name: "Kalianpur 1937 / India zone IIb",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4144,
        remarks: "Used by Survey of Bangladesh since metrication to end of 1994. Metric conversion applies A.R.Clarke&#x27;s Indian foot-British foot ratio of 0.99999566 and Benoit&#x27;s 1895 British inch-metre ratio of 39.370115 rounded as Ind ft &#x3D; 0.30479841m exactly.",
        information_source: "Survey of Bangladesh.",
        data_source: "EPSG",
        revision_date: "2021-01-13",
        deprecated: 0, 
    },

    "EPSG:24376" => CRS {
        coord_ref_sys_code: 24376,
        coord_ref_sys_name: "Kalianpur 1962 / India zone I",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4145,
        remarks: "Used by Pakistan since metrication. Metric conversion applies A.R.Clarke&#x27;s Indian foot-British foot ratio of 0.99999566 and J.S.Clark&#x27;s 1865 British inch-metre ratio of 39.369971 rounded by G.Bomford with slight error as Ind ft &#x3D; 0.3047996m exactly.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2007-01-25",
        deprecated: 0, 
    },

    "EPSG:24377" => CRS {
        coord_ref_sys_code: 24377,
        coord_ref_sys_name: "Kalianpur 1962 / India zone IIa",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4145,
        remarks: "Used by Pakistan since metrication. Metric conversion applies A.R.Clarke&#x27;s Indian foot-British foot ratio of 0.99999566 and J.S.Clark&#x27;s 1865 British inch-metre ratio of 39.369971 rounded by G.Bomford with slight error as Ind ft &#x3D; 0.3047996m exactly.",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2007-01-25",
        deprecated: 0, 
    },

    "EPSG:24382" => CRS {
        coord_ref_sys_code: 24382,
        coord_ref_sys_name: "Kalianpur 1880 / India zone IIb",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4408,
        datum_code: -1, 
        base_crs_code: 4243,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "1999-10-20",
        deprecated: 0, 
    },

    "EPSG:61446405" => CRS {
        coord_ref_sys_code: 61446405,
        coord_ref_sys_name: "Kalianpur 1937 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4144,
        remarks: "See CRS code 4144 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61456405" => CRS {
        coord_ref_sys_code: 61456405,
        coord_ref_sys_name: "Kalianpur 1962 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4145,
        remarks: "See CRS code 4145 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:61466405" => CRS {
        coord_ref_sys_code: 61466405,
        coord_ref_sys_name: "Kalianpur 1975 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4146,
        remarks: "See CRS code 4146 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62416405" => CRS {
        coord_ref_sys_code: 62416405,
        coord_ref_sys_name: "Jamaica 1875 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4241,
        remarks: "See CRS code 4241 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:24548" => CRS {
        coord_ref_sys_code: 24548,
        coord_ref_sys_name: "Kertau 1968 / UTM zone 48N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4245,
        remarks: "",
        information_source: "Defence Geographic Centre.",
        data_source: "EPSG",
        revision_date: "2006-07-24",
        deprecated: 0, 
    },

    "EPSG:62426405" => CRS {
        coord_ref_sys_code: 62426405,
        coord_ref_sys_name: "JAD69 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4242,
        remarks: "See CRS code 4242 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:62436405" => CRS {
        coord_ref_sys_code: 62436405,
        coord_ref_sys_name: "Kalianpur 1880 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4243,
        remarks: "See CRS code 4243 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:66126405" => CRS {
        coord_ref_sys_code: 66126405,
        coord_ref_sys_name: "JGD2000 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4612,
        remarks: "See CRS code 4612 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-22",
        deprecated: 1, 
    },

    "EPSG:66126413" => CRS {
        coord_ref_sys_code: 66126413,
        coord_ref_sys_name: "JGD2000 (3D deg)",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6413,
        datum_code: -1, 
        base_crs_code: 4353,
        remarks: "See CRS code 4353 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-12-07",
        deprecated: 1, 
    },

    "EPSG:66316405" => CRS {
        coord_ref_sys_code: 66316405,
        coord_ref_sys_name: "K0 1949 (deg)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6405,
        datum_code: -1, 
        base_crs_code: 4631,
        remarks: "See CRS code 4631 for recommended coordinate axis representation for the human interface.",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2002-11-29",
        deprecated: 1, 
    },

    "EPSG:2096" => CRS {
        coord_ref_sys_code: 2096,
        coord_ref_sys_name: "Korean 1985 / East Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4162,
        remarks: "Replaces Tokyo 1892 / Korea East Belt (CRS code 5171). Although it was intended that the grids were for most practical purposes coincident, a 10.405&quot; (250m) longitude offset was introduced. Replaced by Korean 1985 / Modified East Belt (CRS code 5176).",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-30",
        deprecated: 0, 
    },

    "EPSG:2097" => CRS {
        coord_ref_sys_code: 2097,
        coord_ref_sys_name: "Korean 1985 / Central Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4162,
        remarks: "Replaces Tokyo 1892 / Korea Central Belt (CRS code 5170). Although it was intended that the grids were for most practical purposes coincident, a 10.405&quot; (250m) longitude offset was introduced. Replaced by Korean 1985 / Modified Central Belt (code 5174).",
        information_source: "National Geographic Information Centre (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-30",
        deprecated: 0, 
    },

    "EPSG:2098" => CRS {
        coord_ref_sys_code: 2098,
        coord_ref_sys_name: "Korean 1985 / West Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4162,
        remarks: "Replaces Tokyo 1892 / Korea West Belt (CRS code 5169). Although it was intended that the grids were for most practical purposes coincident, a 10.405&quot; (250m) longitude offset was introduced. Replaced by Korean 1985 / Modified West Belt (CRS code 5173).",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-30",
        deprecated: 0, 
    },

    "EPSG:2391" => CRS {
        coord_ref_sys_code: 2391,
        coord_ref_sys_name: "KKJ / Finland zone 1",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4123,
        remarks: "",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2006-03-31",
        deprecated: 0, 
    },

    "EPSG:2392" => CRS {
        coord_ref_sys_code: 2392,
        coord_ref_sys_name: "KKJ / Finland zone 2",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4123,
        remarks: "",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "1997-07-22",
        deprecated: 0, 
    },

    "EPSG:2393" => CRS {
        coord_ref_sys_code: 2393,
        coord_ref_sys_name: "KKJ / Finland Uniform Coordinate System",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4123,
        remarks: "Known as Uniform Coordinate System (YKJ) when used over all country and also as Basic Coordinate System zone 3 at larger scales.",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2010-05-02",
        deprecated: 0, 
    },

    "EPSG:2394" => CRS {
        coord_ref_sys_code: 2394,
        coord_ref_sys_name: "KKJ / Finland zone 4",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4123,
        remarks: "",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2006-03-31",
        deprecated: 0, 
    },

    "EPSG:3167" => CRS {
        coord_ref_sys_code: 3167,
        coord_ref_sys_name: "Kertau (RSO) / RSO Malaya (ch)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4410,
        datum_code: -1, 
        base_crs_code: 4751,
        remarks: "See Kertau (RSO) / RSO Malaya (m) (CRS code 3168) for metric equivalent. Replaced by GDM2000 / Peninsula RSO (CRS code 3375).",
        information_source: "Defence Geographic Centre.",
        data_source: "EPSG",
        revision_date: "2006-08-14",
        deprecated: 0, 
    },

    "EPSG:3168" => CRS {
        coord_ref_sys_code: 3168,
        coord_ref_sys_name: "Kertau (RSO) / RSO Malaya (m)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4751,
        remarks: "Adopts metric conversion of 0.914398 metres per yard. See Kertau (RSO) / RSO Malaya (ch) (CRS code 3168) for Imperial version. Replaced by GDM2000 / Peninsula RSO (CRS code 3375).",
        information_source: "Defence Geographic Centre.",
        data_source: "EPSG",
        revision_date: "2006-08-14",
        deprecated: 0, 
    },

    "EPSG:3314" => CRS {
        coord_ref_sys_code: 3314,
        coord_ref_sys_name: "Katanga 1955 / Katanga Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4695,
        remarks: "",
        information_source: "Clifford J. Mugnier in Photogrammetric Engineering and Remote Sensing, June 2005.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 1, 
    },

    "EPSG:3315" => CRS {
        coord_ref_sys_code: 3315,
        coord_ref_sys_name: "Katanga 1955 / Katanga TM",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4695,
        remarks: "",
        information_source: "Clifford J. Mugnier in Photogrammetric Engineering and Remote Sensing, June 2005.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 1, 
    },

    "EPSG:3316" => CRS {
        coord_ref_sys_code: 3316,
        coord_ref_sys_name: "Kasai 1953 / Congo TM zone 22",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4696,
        remarks: "",
        information_source: "Institut Geographique du Congo.",
        data_source: "EPSG",
        revision_date: "2020-03-30",
        deprecated: 0, 
    },

    "EPSG:3317" => CRS {
        coord_ref_sys_code: 3317,
        coord_ref_sys_name: "Kasai 1953 / Congo TM zone 24",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4696,
        remarks: "",
        information_source: "Institut Geographique du Congo.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 0, 
    },

    "EPSG:3386" => CRS {
        coord_ref_sys_code: 3386,
        coord_ref_sys_name: "KKJ / Finland zone 0",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4123,
        remarks: "",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2012-02-13",
        deprecated: 0, 
    },

    "EPSG:3387" => CRS {
        coord_ref_sys_code: 3387,
        coord_ref_sys_name: "KKJ / Finland zone 5",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4123,
        remarks: "",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2006-03-30",
        deprecated: 0, 
    },

    "EPSG:3391" => CRS {
        coord_ref_sys_code: 3391,
        coord_ref_sys_name: "Karbala 1979 / UTM zone 37N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4743,
        remarks: "Replaces Nahrwan 1934 / UTM zone 37N (CRS code 7005). Replaced by IGRS / UTM zone 37N (CRS code 3890). At time of record population, information regarding usage within oil sector is not available.",
        information_source: "Various industry sources.",
        data_source: "EPSG",
        revision_date: "2015-02-12",
        deprecated: 0, 
    },

    "EPSG:3392" => CRS {
        coord_ref_sys_code: 3392,
        coord_ref_sys_name: "Karbala 1979 / UTM zone 38N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4743,
        remarks: "Replaces Nahrwan 1934 / UTM zone 38N (projCRS code 7006). Replaced by IGRS / UTM zone 38N (CRS code 3891). At time of record population, information regarding usage within oil sector is not available.",
        information_source: "Various industry sources.",
        data_source: "EPSG",
        revision_date: "2015-02-12",
        deprecated: 0, 
    },

    "EPSG:3393" => CRS {
        coord_ref_sys_code: 3393,
        coord_ref_sys_name: "Karbala 1979 / UTM zone 39N",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4743,
        remarks: "Replaces Nahrwan 1934 / UTM zone 39N (projCRS code 7007). Replaced by IGRS / UTM zone 39N (CRS code 3892). At time of record population, information regarding usage within oil sector is not available.",
        information_source: "Various industry sources.",
        data_source: "EPSG",
        revision_date: "2015-02-12",
        deprecated: 0, 
    },

    "EPSG:3901" => CRS {
        coord_ref_sys_code: 3901,
        coord_ref_sys_name: "KKJ / Finland Uniform Coordinate System + N60 height",
        coord_ref_sys_kind: "compound",
        coord_sys_code: -1,
        datum_code: -1, 
        base_crs_code: -1,
        remarks: "Replaced by ETRS89 / TM35FIN(N,E) + N60 height (CRS code 3902).",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2010-04-23",
        deprecated: 0, 
    },

    "EPSG:3985" => CRS {
        coord_ref_sys_code: 3985,
        coord_ref_sys_name: "Katanga 1955 / Katanga Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4695,
        remarks: "",
        information_source: "J. van der Straeten, &quot;Triangulation du Katanga&quot;, Comite Special du Katanga (CSK), 1955.",
        data_source: "EPSG",
        revision_date: "2009-03-11",
        deprecated: 1, 
    },

    "EPSG:3986" => CRS {
        coord_ref_sys_code: 3986,
        coord_ref_sys_name: "Katanga 1955 / Katanga Gauss zone A",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4695,
        remarks: "",
        information_source: "J. van der Straeten, &quot;Triangulation du Katanga&quot;, Comite Special du Katanga (CSK), 1955.",
        data_source: "EPSG",
        revision_date: "2009-03-11",
        deprecated: 0, 
    },

    "EPSG:3987" => CRS {
        coord_ref_sys_code: 3987,
        coord_ref_sys_name: "Katanga 1955 / Katanga Gauss zone B",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4695,
        remarks: "",
        information_source: "J. van der Straeten, &quot;Triangulation du Katanga&quot;, Comite Special du Katanga (CSK), 1955.",
        data_source: "EPSG",
        revision_date: "2009-03-11",
        deprecated: 0, 
    },

    "EPSG:3988" => CRS {
        coord_ref_sys_code: 3988,
        coord_ref_sys_name: "Katanga 1955 / Katanga Gauss zone C",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4695,
        remarks: "",
        information_source: "J. van der Straeten, &quot;Triangulation du Katanga&quot;, Comite Special du Katanga (CSK), 1955.",
        data_source: "EPSG",
        revision_date: "2009-03-11",
        deprecated: 0, 
    },

    "EPSG:3989" => CRS {
        coord_ref_sys_code: 3989,
        coord_ref_sys_name: "Katanga 1955 / Katanga Gauss zone D",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4695,
        remarks: "",
        information_source: "J. van der Straeten, &quot;Triangulation du Katanga&quot;, Comite Special du Katanga (CSK), 1955.",
        data_source: "EPSG",
        revision_date: "2009-03-11",
        deprecated: 0, 
    },

    "EPSG:4123" => CRS {
        coord_ref_sys_code: 4123,
        coord_ref_sys_name: "KKJ",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6123, 
        base_crs_code: -1,
        remarks: "",
        information_source: "National Land Survey of Finland;
http://www.maanmittauslaitos.fi",
        data_source: "EPSG",
        revision_date: "2011-06-30",
        deprecated: 0, 
    },

    "EPSG:4162" => CRS {
        coord_ref_sys_code: 4162,
        coord_ref_sys_name: "Korean 1985",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6162, 
        base_crs_code: -1,
        remarks: "Replaces use of Tokyo datum.",
        information_source: "Clifford J. Mugnier; in Photogrammetric Engineering &amp; Remote Sensing November 1999. http://www.asprs.org",
        data_source: "EPSG",
        revision_date: "2005-09-29",
        deprecated: 0, 
    },

    "EPSG:4166" => CRS {
        coord_ref_sys_code: 4166,
        coord_ref_sys_name: "Korean 1995",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6166, 
        base_crs_code: -1,
        remarks: "",
        information_source: "NIMA TR8350.2 http://earth-info.nga.mil/GandG/",
        data_source: "EPSG",
        revision_date: "2005-09-29",
        deprecated: 0, 
    },

    "EPSG:4198" => CRS {
        coord_ref_sys_code: 4198,
        coord_ref_sys_name: "Kousseri",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6198, 
        base_crs_code: -1,
        remarks: "",
        information_source: "TotalFinaElf",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4407" => CRS {
        coord_ref_sys_code: 4407,
        coord_ref_sys_name: "NAD27 / BLM 7N (ftUS)",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4497,
        datum_code: -1, 
        base_crs_code: 4267,
        remarks: "",
        information_source: "OGP",
        data_source: "EPSG",
        revision_date: "2010-02-20",
        deprecated: 0, 
    },

    "EPSG:4244" => CRS {
        coord_ref_sys_code: 4244,
        coord_ref_sys_name: "Kandawala",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6244, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Abeyratne, Featherstone and Tantrigoda in Survey Review vol. 42 no. 317 (July 2010).",
        data_source: "EPSG",
        revision_date: "2010-09-24",
        deprecated: 0, 
    },

    "EPSG:4245" => CRS {
        coord_ref_sys_code: 4245,
        coord_ref_sys_name: "Kertau 1968",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6245, 
        base_crs_code: -1,
        remarks: "Not used for metrication of RSO grid - see Kertau (RSO) (CRS code 4751). Replaced by GDM2000 (CRS code 4742).",
        information_source: "Defence Geographic Centre.",
        data_source: "EPSG",
        revision_date: "2014-11-21",
        deprecated: 0, 
    },

    "EPSG:4246" => CRS {
        coord_ref_sys_code: 4246,
        coord_ref_sys_name: "KOC",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6246, 
        base_crs_code: -1,
        remarks: "",
        information_source: "",
        data_source: "EPSG",
        revision_date: "2004-01-06",
        deprecated: 0, 
    },

    "EPSG:4390" => CRS {
        coord_ref_sys_code: 4390,
        coord_ref_sys_name: "Kertau 1968 / Johor Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4245,
        remarks: "Replaced by GDM2000 / Johor Grid (CRS code 3377).",
        information_source: "Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-10-12",
        deprecated: 0, 
    },

    "EPSG:4391" => CRS {
        coord_ref_sys_code: 4391,
        coord_ref_sys_name: "Kertau 1968 / Sembilan and Melaka Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4245,
        remarks: "Replaced by GDM2000 / Sembilan and Melaka Grid (CRS code 3378).",
        information_source: "Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-10-12",
        deprecated: 0, 
    },

    "EPSG:4392" => CRS {
        coord_ref_sys_code: 4392,
        coord_ref_sys_name: "Kertau 1968 / Pahang Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4245,
        remarks: "Replaced by GDM2000 / Pahang Grid (CRS code 3379).",
        information_source: "Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-10-12",
        deprecated: 0, 
    },

    "EPSG:4393" => CRS {
        coord_ref_sys_code: 4393,
        coord_ref_sys_name: "Kertau 1968 / Selangor Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4245,
        remarks: "Replaced by GDM2000 / Selangor Grid (CRS code 3380).",
        information_source: "Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-10-12",
        deprecated: 0, 
    },

    "EPSG:4394" => CRS {
        coord_ref_sys_code: 4394,
        coord_ref_sys_name: "Kertau 1968 / Terengganu Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4245,
        remarks: "Replaced by GDM2000 / Terengganu Grid (CRS code 3381).",
        information_source: "Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-10-12",
        deprecated: 0, 
    },

    "EPSG:4395" => CRS {
        coord_ref_sys_code: 4395,
        coord_ref_sys_name: "Kertau 1968 / Pinang Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4245,
        remarks: "Replaced by GDM2000 / Pinang Grid (CRS code 3382).",
        information_source: "Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-10-12",
        deprecated: 0, 
    },

    "EPSG:4396" => CRS {
        coord_ref_sys_code: 4396,
        coord_ref_sys_name: "Kertau 1968 / Kedah and Perlis Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4245,
        remarks: "Replaced by GDM2000 / Kedah and Perlis Grid (CRS code 3383).",
        information_source: "Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-10-12",
        deprecated: 0, 
    },

    "EPSG:4397" => CRS {
        coord_ref_sys_code: 4397,
        coord_ref_sys_name: "Kertau 1968 / Perak Revised Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4245,
        remarks: "Replaces Kertau 1968 / Perak Grid. Replaced by GDM2000 / Perak Grid (CRS code 3384).",
        information_source: "Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-10-12",
        deprecated: 0, 
    },

    "EPSG:4398" => CRS {
        coord_ref_sys_code: 4398,
        coord_ref_sys_name: "Kertau 1968 / Kelantan Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4245,
        remarks: "Replaced by GDM2000 / Kelantan Grid (CRS code 3385).",
        information_source: "Department of Survey and Mapping Malaysia. www.jupem.gov.my",
        data_source: "EPSG",
        revision_date: "2014-10-12",
        deprecated: 0, 
    },

    "EPSG:4415" => CRS {
        coord_ref_sys_code: 4415,
        coord_ref_sys_name: "Katanga 1955 / Katanga Lambert",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4499,
        datum_code: -1, 
        base_crs_code: 4695,
        remarks: "",
        information_source: "J. van der Straeten, &quot;Triangulation du Katanga&quot;, Comite Special du Katanga (CSK), 1955.",
        data_source: "EPSG",
        revision_date: "2009-06-30",
        deprecated: 0, 
    },

    "EPSG:4695" => CRS {
        coord_ref_sys_code: 4695,
        coord_ref_sys_name: "Katanga 1955",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6695, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Clifford J. Mugnier, in Photogrammetric Engineering and Remote Sensing, June 2005.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 0, 
    },

    "EPSG:4696" => CRS {
        coord_ref_sys_code: 4696,
        coord_ref_sys_name: "Kasai 1953",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6696, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Institute Geographique du Congo.",
        data_source: "EPSG",
        revision_date: "2005-09-30",
        deprecated: 0, 
    },

    "EPSG:4737" => CRS {
        coord_ref_sys_code: 4737,
        coord_ref_sys_name: "Korea 2000",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6737, 
        base_crs_code: 4927,
        remarks: "",
        information_source: "OGP. See 3D CRS for original information source.",
        data_source: "EPSG",
        revision_date: "2007-08-27",
        deprecated: 0, 
    },

    "EPSG:4743" => CRS {
        coord_ref_sys_code: 4743,
        coord_ref_sys_name: "Karbala 1979",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6743, 
        base_crs_code: -1,
        remarks: "Geodetic network established by Polservice consortium. Replaces Nahrwan 1934 (CRS code 4744). Replaced by IGRS (CRS code 3889). At time of record population, information regarding usage within oil sector is not available.",
        information_source: "Various industry sources.",
        data_source: "EPSG",
        revision_date: "2015-02-12",
        deprecated: 0, 
    },

    "EPSG:4751" => CRS {
        coord_ref_sys_code: 4751,
        coord_ref_sys_name: "Kertau (RSO)",
        coord_ref_sys_kind: "geographic 2D",
        coord_sys_code: 6422,
        datum_code: 6751, 
        base_crs_code: -1,
        remarks: "Used only for metrication of RSO grid. See Kertau 1968 (CRS code 4245) for other purposes. Replaced by GDM2000 (CRS code 4742).",
        information_source: "Defence Geographic Centre.",
        data_source: "EPSG",
        revision_date: "2006-07-24",
        deprecated: 0, 
    },

    "EPSG:4926" => CRS {
        coord_ref_sys_code: 4926,
        coord_ref_sys_name: "Korea 2000",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 6737, 
        base_crs_code: -1,
        remarks: "",
        information_source: "Permanent Committee for GIS Infrastructure for Asia and the Pacific (PCGIAP), http://www.gsi.go.jp/PCGIAP",
        data_source: "EPSG",
        revision_date: "2006-01-30",
        deprecated: 0, 
    },

    "EPSG:4927" => CRS {
        coord_ref_sys_code: 4927,
        coord_ref_sys_name: "Korea 2000",
        coord_ref_sys_kind: "geographic 3D",
        coord_sys_code: 6423,
        datum_code: 6737, 
        base_crs_code: 4926,
        remarks: "",
        information_source: "Permanent Committee for GIS Infrastructure for Asia and the Pacific (PCGIAP), http://www.gsi.go.jp/PCGIAP",
        data_source: "EPSG",
        revision_date: "2007-08-27",
        deprecated: 0, 
    },

    "EPSG:5167" => CRS {
        coord_ref_sys_code: 5167,
        coord_ref_sys_name: "Korean 1985 / East Sea Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4162,
        remarks: "Replaces Tokyo 1892 / Korea East Sea Belt (CRS code 5172). Although it was intended that the grids were for most practical purposes coincident a 10.405&quot; (250m) longitude offset was introduced. Replaced by Korean 1985 / Modified East Sea Belt (code 5177).",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:5168" => CRS {
        coord_ref_sys_code: 5168,
        coord_ref_sys_name: "Korean 1985 / Central Belt Jeju",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4162,
        remarks: "Introduced to replace Korean 1985 / Central Belt (CRS code 2097) on Jeju to avoid negative northing coordinates. Replaced by Korean 1985 / Modified Central Belt Jeju (code 5175).",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:5173" => CRS {
        coord_ref_sys_code: 5173,
        coord_ref_sys_name: "Korean 1985 / Modified West Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4162,
        remarks: "Unofficial system replacing Korean 1985 / West Belt (code 2098) to correct 10.405&quot; longitude difference and make grid for most practical purposes coincident with Tokyo 1892 / West Belt (code 5169). Replaced by Korea 2000 / West Belt (code 5180).",
        information_source: "Korean Association of Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:5174" => CRS {
        coord_ref_sys_code: 5174,
        coord_ref_sys_name: "Korean 1985 / Modified Central Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4162,
        remarks: "Unofficial system replacing Korean 1985 / Central Belt (code 2097) to correct 10.405&quot; longitude difference and make grid for most practical purposes coincident with Tokyo 1892 / Central Belt (code 5170). Replaced by Korea 2000 / Central Belt (code 5181).",
        information_source: "Korean Association of Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:5175" => CRS {
        coord_ref_sys_code: 5175,
        coord_ref_sys_name: "Korean 1985 / Modified Central Belt Jeju",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4162,
        remarks: "Unofficial system replacing Korean 1985 / Central Belt Jeju (code 5168) to correct 10.405&quot; longitude difference. Replaced by Korea 2000 / Central Belt Jeju (code 5182).",
        information_source: "Korean Association of Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:5176" => CRS {
        coord_ref_sys_code: 5176,
        coord_ref_sys_name: "Korean 1985 / Modified East Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4162,
        remarks: "Unofficial system replacing Korean 1985 / East Belt (code 2096) to correct 10.405&quot; longitude difference and make grid for most practical purposes coincident with Tokyo 1892 / East Belt (code 5171). Replaced by Korea 2000 / East Belt (code 5183).",
        information_source: "Korean Association of Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:5177" => CRS {
        coord_ref_sys_code: 5177,
        coord_ref_sys_name: "Korean 1985 / Modified East Sea Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4162,
        remarks: "Unofficial system replacing Korean 1985 / E Sea Belt (code 5167) to correct 10.405&quot; longitude difference and make grid for most practical purposes coincident with Tokyo 1892 / E Sea Belt (code 5172). Replaced by Korea 2000 / East Sea Belt (code 5184).",
        information_source: "Korean Association of Surveying and Mapping.",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:5178" => CRS {
        coord_ref_sys_code: 5178,
        coord_ref_sys_name: "Korean 1985 / Unified CS",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4162,
        remarks: "Introduced in 2004. From 2010 replaced by Korea 2000 / Unified CS (CRS code 5179).",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2011-01-25",
        deprecated: 0, 
    },

    "EPSG:5179" => CRS {
        coord_ref_sys_code: 5179,
        coord_ref_sys_name: "Korea 2000 / Unified CS",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4737,
        remarks: "Replaces Korean 1985 / Unified CS (CRS code 5178) from 2010.",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:24500" => CRS {
        coord_ref_sys_code: 24500,
        coord_ref_sys_name: "Kertau 1968 / Singapore Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4245,
        remarks: "For cadastral purposes, replaced by SVY21 / Singapore TM (CRS code 3414) from August 2004.",
        information_source: "Defence Geographic Centre.",
        data_source: "EPSG",
        revision_date: "2006-10-13",
        deprecated: 0, 
    },

    "EPSG:5180" => CRS {
        coord_ref_sys_code: 5180,
        coord_ref_sys_name: "Korea 2000 / West Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4737,
        remarks: "Introduced 2003, withdrawn 2009-12-31. Replaces Korean 1985 / West Belt (CRS code 2098) and Korean 1985 / Modified West Belt (CRS code 5173).  Replaced by Korea 2000 / West Belt 2010 (CRS code 5185) .",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:5181" => CRS {
        coord_ref_sys_code: 5181,
        coord_ref_sys_name: "Korea 2000 / Central Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4737,
        remarks: "Introduced 2003, withdrawn 2009-12-31. Replaces Korean 1985 / Central Belt (CRS code 2097) and Korean 1985 / Modified Central Belt (CRS code 5174).  Replaced by Korea 2000 / Central Belt 2010 (CRS code 5186).",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:5182" => CRS {
        coord_ref_sys_code: 5182,
        coord_ref_sys_name: "Korea 2000 / Central Belt Jeju",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4737,
        remarks: "Introduced 2003, withdrawn 2009-12-31. Replaces Korean 1985 / Central Belt Jeju (CRS code 5168) and Korean 1985 / Modified Central Belt Jeju (CRS code 5175).  Replaced by Korea 2000 / Central Belt 2010 (CRS code 5186).",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:5183" => CRS {
        coord_ref_sys_code: 5183,
        coord_ref_sys_name: "Korea 2000 / East Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4737,
        remarks: "Introduced 2003, withdrawn 2009-12-31. Replaces Korean 1985 / East Belt (CRS code 2096) and Korean 1985 / Modified East Belt (CRS code 5176).  Replaced by Korea 2000 / East Belt 2010 (CRS code 5187).",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-30",
        deprecated: 0, 
    },

    "EPSG:5184" => CRS {
        coord_ref_sys_code: 5184,
        coord_ref_sys_name: "Korea 2000 / East Sea Belt",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4737,
        remarks: "Introduced 2003, withdrawn 2009-12-31. Replaces Korean 1985 / East Sea Belt (CRS code 5167) and Korean 1985 / Modified East Sea Belt (CRS code 5177).  Replaced by Korea 2000 / East Sea Belt 2010 (CRS code 5188).",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:5185" => CRS {
        coord_ref_sys_code: 5185,
        coord_ref_sys_name: "Korea 2000 / West Belt 2010",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4737,
        remarks: "Legally mandated CRS from 2010-01-01. Replaces Korean 1985 / West Belt (CRS code 2098), Korean 1985 / Modified West Belt (CRS code 5173) and Korea 2000 / West Belt (CRS code 5180).",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:5186" => CRS {
        coord_ref_sys_code: 5186,
        coord_ref_sys_name: "Korea 2000 / Central Belt 2010",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4737,
        remarks: "Legally mandated CRS from 2010-01-01. Replaces Korean 1985 / Central Belt, Central Belt Jeju, Modified Central Belt, Modified Central Belt Jeju (CRS codes 2097, 5168, 5174 and 5175) and Korea 2000 / Central Belt and Central Belt Jeju (CRS codes 5181-82).",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:5187" => CRS {
        coord_ref_sys_code: 5187,
        coord_ref_sys_name: "Korea 2000 / East Belt 2010",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4737,
        remarks: "Legally mandated CRS from 2010-01-01. Replaces Korean 1985 / East Belt (CRS code 2096), Korean 1985 / Modified East Belt (CRS code 5176) and Korea 2000 / East Belt (CRS code 5183).",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:5188" => CRS {
        coord_ref_sys_code: 5188,
        coord_ref_sys_name: "Korea 2000 / East Sea Belt 2010",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4530,
        datum_code: -1, 
        base_crs_code: 4737,
        remarks: "Legally mandated CRS from 2010-01-01. Replaces Korean 1985 / East Sea Belt (CRS code 5167), Korean 1985 / Modified East Sea Belt (CRS code 5177) and Korea 2000 / East Sea Belt (CRS code 5184).",
        information_source: "National Geographic Information Institute (NGII).",
        data_source: "EPSG",
        revision_date: "2010-06-28",
        deprecated: 0, 
    },

    "EPSG:5234" => CRS {
        coord_ref_sys_code: 5234,
        coord_ref_sys_name: "Kandawala / Sri Lanka Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4244,
        remarks: "",
        information_source: "Abeyratne, Featherstone and Tantrigoda in Survey Review vol. 42 no. 317 (July 2010).",
        data_source: "EPSG",
        revision_date: "2010-08-07",
        deprecated: 0, 
    },

    "EPSG:5614" => CRS {
        coord_ref_sys_code: 5614,
        coord_ref_sys_name: "KOC WD depth (ft)",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6495,
        datum_code: -1, 
        base_crs_code: 5789,
        remarks: "See CRS code 5789 for equivalent system in feet.",
        information_source: "KOC",
        data_source: "EPSG",
        revision_date: "2021-04-20",
        deprecated: 0, 
    },

    "EPSG:5789" => CRS {
        coord_ref_sys_code: 5789,
        coord_ref_sys_name: "KOC WD depth",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6498,
        datum_code: -1, 
        base_crs_code: 7979,
        remarks: "See CRS code 5614 for equivalent system in feet.",
        information_source: "KOC",
        data_source: "EPSG",
        revision_date: "2021-04-20",
        deprecated: 0, 
    },

    "EPSG:5790" => CRS {
        coord_ref_sys_code: 5790,
        coord_ref_sys_name: "KOC CD height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5188, 
        base_crs_code: -1,
        remarks: "",
        information_source: "KOC",
        data_source: "EPSG",
        revision_date: "2008-03-14",
        deprecated: 0, 
    },

    "EPSG:6646" => CRS {
        coord_ref_sys_code: 6646,
        coord_ref_sys_name: "Karbala 1979 / Iraq National Grid",
        coord_ref_sys_kind: "projected",
        coord_sys_code: 4400,
        datum_code: -1, 
        base_crs_code: 4743,
        remarks: "",
        information_source: "Survey Division, Ministry of Water Resources (MoWR).",
        data_source: "EPSG",
        revision_date: "2013-12-05",
        deprecated: 0, 
    },

    "EPSG:7652" => CRS {
        coord_ref_sys_code: 7652,
        coord_ref_sys_name: "Kiunga height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 1151, 
        base_crs_code: -1,
        remarks: "Kiunga height &#x3D; WGS 84 ellipsoid height - value of geoid undulation derived by bilinear interpolation of EGM2008 geoid model - 3.0m &#x3D; EGM2008 height - 3.0m.  See CRS code 3855 and transformation code 3858.",
        information_source: "Quickclose Geomatics.",
        data_source: "EPSG",
        revision_date: "2015-11-05",
        deprecated: 0, 
    },

    "EPSG:7979" => CRS {
        coord_ref_sys_code: 7979,
        coord_ref_sys_name: "KOC WD height",
        coord_ref_sys_kind: "vertical",
        coord_sys_code: 6499,
        datum_code: 5187, 
        base_crs_code: -1,
        remarks: "See CRS code 5789 for equivalent depth system.",
        information_source: "IOGP",
        data_source: "EPSG",
        revision_date: "2017-02-17",
        deprecated: 0, 
    },

    "EPSG:9138" => CRS {
        coord_ref_sys_code: 9138,
        coord_ref_sys_name: "KOSOVAREF01",
        coord_ref_sys_kind: "geocentric",
        coord_sys_code: 6500,
        datum_code: 1251, 
        base_crs_code: -1,
        remarks: "",
     