Epimetheus
==========

An easy-to-use prometheus-compatible metrics library.

Epimetheus is probably the easiest way to get your Rust application
serving metrics.  Just scatter your code with instrumentation and call
`spawn_http_server()` in `main()`.

Watch your hashmaps grow!

```rust
metric!(my_data_len).set(my_data.len());
metric!(my_data_cap).set(my_data.capacity());
```

Monitor the latency of your functions!

```rust
let start = Instant::now();
my_function();
metric!(my_function_duration_sum).add(start.elapsed().as_secs_f64());
metric!(my_function_duration_count).add(1.0);
```

Track the status codes of your responses!

```rust
let resp = compute_response();
metric!(responses{code=resp.status()}).add(1.0);
Ok(resp)
```

Then connect to port 9898 to see what's happening:

```
$ curl localhost:9898
my_data_cap 1024
my_data_len 764
my_function_duration_count 6032
my_function_duration_sum 8.32
responses{code="200 OK"} 5443
responses{code="404 Not Found"} 587
responses{code="500 Internal Server Error"} 2
```

Your program can also call `epimetheus::query()` to inspect its own metrics.
If you only plan to access the metrics this way, you can skip the call to
`spawn_http_server()`.

## Features

* Multi-threaded OK!  Everyone shares the same set of metrics.
* The format is prometheus-compatible.  Point prometheus' watchful eye at
  your program and get nice graphs.
* Updating metrics is fast (...mostly.  See below.)
* The port number can be customised via the `RUST_METRICS_PORT` environment
  variable.
* The code is very readable - under 200 sloc.  [Take a
  look!](https://docs.rs/epimetheus/latest/src/epimetheus/lib.rs.html)
* Public domain.

## Performance

* Updating an unlabelled metric (like `my_data_len` above) is fast (~500ns
  uncontended, ~5us contended).
* Updating a labelled metric (like `responses` above) is slower (~1us
  uncontended, ~10us contended).
* The HTTP server is extremely dumb and can only handle one client at a time;
  typically this doesn't matter.

## Other stuff

Epimetheus is the brother of Prometheus, hence the crate's name.

## Contributing

Please send bug reports to <~asayers/public-inbox@lists.sr.ht>.

Please send patches to <~asayers/public-inbox@lists.sr.ht>, and include
the following text:

> I dedicate any and all copyright interest in this software to the
> public domain. I make this dedication for the benefit of the public at
> large and to the detriment of my heirs and successors. I intend this
> dedication to be an overt act of relinquishment in perpetuity of all
> present and future rights to this software under copyright law.
