use super::common::*;
use super::multirasterdataset::*;
use super::singlerasterdataset::*;
#[allow(unused_imports)]
use super::*;
use env_logger::Env;
use gdal::Dataset;
use ndarray::{s, Array3, Array4};
use rayon::iter::IntoParallelIterator;
use rayon::iter::ParallelIterator;
use std::collections::BTreeMap;
use std::convert::TryInto;
use std::path::Path;

fn init_logger() {
    // Setup logger
    let env = Env::default().filter_or("LOG_LEVEL", "debug");
    env_logger::init_from_env(env);
}
// #[test]
// fn test_tmp() {
//     fn f(x: &Array3<i32>) -> Array3<i32> {
//         x / 2
//     }

//     let source = "data/LC08_L1TP_224078_20200518_20200518_01_RT.TIF";
//     let rds = SingleRasterDatasetBuilder::from_file(source)
//         .num_threads(4)
//         .build();
//     rds.oom_apply_i32(f);

//     //     // rds.iter().map(|b| println!("{}", b) ).collect();
//     // for b in &mut rds {
//     //     // let mut rds_block = rds.clone();
//     //     let b_data = rds.read::<f32>();
//     //     println!("{:?}", b_data);
//     // }
//     //     // let z = rds.next().unwrap();
//     //     // println!("Oh! {}", z);
//     //     // let z = rds.next().unwrap();
//     //     // println!("Oh! {}", z);
// }

// #[test]
// fn test_n_bands() {
//     let source = "data/LC08_L1TP_224078_20200518_20200518_01_RT.TIF";
//     assert_eq!(3, get_n_bands(source));
// }

#[test]
fn test_tmp() {
    let source = "data/cemsre_t55jfm_20200614_sub_abam5.tif";
    let rds_over_1: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
        // .epsg(3577)
        .overlap_size(1)
        .bands(vec![1])
        .build();
    println!("{}", rds_over_1);
}

#[test]
fn test_overlap() {
    let source = "data/LC08_L1TP_224078_20200518_20200518_01_RT.TIF";
    let rds_over_0: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
        .overlap_size(0)
        .build();

    let rds_over_1: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
        .overlap_size(1)
        .build();
    // The idea is that reading with no overlap should give the same result
    // as reading with overlap + trimming the excess
    let ids = vec![0, 1, 2, 3];
    for id in ids {
        println!("\n Testing id: -> {:?} \n", id);
        let block_attributes_overlap = rds_over_1.blocks_attributes[id];
        let array_with_overlap = rds_over_1.read_block::<i32>(block_attributes_overlap);
        let array_with_overlap_trimmed = trimm_array3(&array_with_overlap, 1);
        // // should be the same than with no overlaps!
        let block_attributes_no_overlap = rds_over_0.blocks_attributes[id];
        let array_with_no_overlap = rds_over_0.read_block::<i32>(block_attributes_no_overlap);
        assert_eq!(array_with_no_overlap, array_with_overlap_trimmed);
    }
}

#[test]
fn test_write() {
    let source = "data/LC08_L1TP_224078_20200518_20200518_01_RT.TIF";
    let rds: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
        .overlap_size(1)
        .build();

    let ids = vec![0, 1, 2, 3];
    for id in ids {
        let block_attributes = rds.blocks_attributes[id];
        let data = rds.read_block::<i32>(block_attributes);
        rds.write_block3(id, data, &format!("/tmp/test_{}.tif", id));
    }

    let source = "data/LC08_L1TP_224078_20200518_20200518_01_RT.TIF";
    let rds: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
        .overlap_size(1)
        .build();
}

#[test]
fn test_create_temp_file() {
    create_temp_file("vrt");
}

#[test]
fn test_write_warped() {
    let source = "data/LC08_L1TP_224078_20200518_20200518_01_RT.TIF";
    let rds: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
        .epsg(3577)
        .bands(vec![1, 2, 3])
        .overlap_size(0)
        .build();
    println!("{}", rds);
    println!("{:?}", rds);

    rayon::ThreadPoolBuilder::new()
        .num_threads(4)
        .build()
        .unwrap();

    (0..9).into_par_iter().for_each(|id| {
        let block_attributes = rds.blocks_attributes[id];
        let data = rds.read_block::<i32>(block_attributes);

        rds.write_block3(
            id,
            data,
            &format!("/export/home/hardtkel/tmp/test_{}.tif", id),
        );
    })
}

#[test]
fn test_add_col_left() {
    let original =
        Array3::from_shape_vec((3, 2, 2), vec![1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12]).unwrap();

    let added = add_cols_left(&original, &2);
    let expected = Array3::from_shape_vec(
        (3, 2, 4),
        vec![
            1, 2, 1, 2, 3, 4, 3, 4, 5, 6, 5, 6, 7, 8, 7, 8, 9, 10, 9, 10, 11, 12, 11, 12,
        ],
    )
    .unwrap();
    assert_eq!(added, expected);

    let added = add_cols_left(&original, &1);
    let expected = Array3::from_shape_vec(
        (3, 2, 3),
        vec![1, 1, 2, 3, 3, 4, 5, 5, 6, 7, 7, 8, 9, 9, 10, 11, 11, 12],
    )
    .unwrap();
    assert_eq!(added, expected);

    let added = add_cols_left(&original, &0);
    assert_eq!(added, original);

    let added = add_rows_top(&original, &1);
    println!("{:?}", added)
}

#[test]
fn test_single_ds_read_window() {
    let source = "data/LC08_L1TP_224078_20200518_20200518_01_RT.TIF";
    let rds_1: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source).build();
    let read_window = ReadWindow {
        offset: Offset { rows: 0, cols: 0 },
        size: Size { rows: 10, cols: 10 },
    };
    let data: Array3<f64> = rds_1.read_window(read_window);
    assert_eq!(data.shape(), vec![3, 10, 10]);

    let read_window = ReadWindow {
        offset: Offset { rows: 0, cols: 0 },
        size: Size {
            rows: 256,
            cols: 256,
        },
    };
    let data: Array3<f64> = rds_1.read_window(read_window);
    assert_eq!(data.shape(), vec![3, 256, 256]);
}

use std::time::{Duration, Instant};
#[test]
fn test_read_speed() {
    //matching
    let source = "/scratch/rsc8/hardtkel/biocondition_new/rasters/qld_aca_1518_greenstats_.tif";
    let rds: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
        .block_size(BlockSize {
            rows: 4096,
            cols: 4096,
        })
        .build();
    let read_window = rds.blocks_attributes[300].read_window;
    println!("{:?}", read_window);
    let start = Instant::now();
    let block_data = rds.read_window::<i16>(read_window);
    let duration = start.elapsed();
    println!("Reading matching block size:{:?}", duration);

    // not matching
    let source = "/scratch/rsc8/hardtkel/biocondition_new/rasters/qld_aca_1518_greenstats.tif";
    let rds: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
        .block_size(BlockSize {
            rows: 4096,
            cols: 4096,
        })
        .build();
    let read_window = rds.blocks_attributes[300].read_window;
    println!("{:?}", read_window);
    let start = Instant::now();
    let block_data = rds.read_window::<i16>(read_window);
    let duration = start.elapsed();
    println!("Reading not matching block size:{:?}", duration);
}

#[test]
fn test_extract_blowckwise() {
    init_logger();
    // define the expected values
    let mut expected: BTreeMap<i16, Vec<i16>> = BTreeMap::new();
    expected.insert(1, vec![7561]);
    expected.insert(2, vec![8030]);
    expected.insert(3, vec![7966]);
    expected.insert(4, vec![8277]);
    expected.insert(5, vec![8302]);
    expected.insert(6, vec![7398]);

    let source = "data/LC08_L1TP_224078_20200518_20200518_01_RT.TIF";
    let bands: &[isize] = &[1];
    let vector_path = "data/LC08_L1TP_224078_20200518_20200518_01_RT_points.gpkg";
    let id_col_name = "id";
    let method = Some("value");
    let srds = SingleRasterDatasetBuilder::from_file(source).build();
    let extracted = srds.extract_blockwise(bands, vector_path, id_col_name, method);
    assert_eq!(expected, extracted);
}
// #[test]
// fn test_single_ds_read() {
//     let source = "data/LC08_L1TP_224078_20200518_20200518_01_RT.TIF";
//     let rds_1: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source).build();
//     let data = rds_1.read::<f64>();
//     assert_eq!(data.shape(), vec![3, 2041, 1860]);
// }

#[test]
fn test_multirasterdataset() {
    let source = "data/LC08_L1TP_224078_20200518_20200518_01_RT.TIF";
    let sources = vec![source, source];
    let mut srds_list = Vec::new();
    for source in sources {
        srds_list.push(SingleRasterDatasetBuilder::from_file(source).build());
    }
    let mrds = MultiRasterDatasetBuilder::from_datasets(srds_list).build();
    println!("{:?}", mrds);
}
#[test]
fn test_singlerasterdataset_iter() {
    let source = "data/LC08_L1TP_224078_20200518_20200518_01_RT.TIF";
    let sources = vec![source, source];
    let srds = SingleRasterDatasetBuilder::from_file(source).build();
    println!("{:?}", srds);
    rayon::ThreadPoolBuilder::new()
        .num_threads(4)
        .build_global()
        .unwrap();

    for sub_srds in srds.into_iter() {
        println!("----------------------");
        println!("{:?}", sub_srds);
        println!("{:?}", sub_srds.read::<i16>());
        println!("----------------------");
    }
}

// #[test]
// fn test_singlerasterdataset_par_iter() {
//     use std::{thread, time};
//     use rayon::iter::ParallelBridge;

//     let source = "data/LC08_L1TP_224078_20200518_20200518_01_RT.TIF";
//     let sources = vec![source, source];
//     let srds = SingleRasterDatasetBuilder::from_file(source).build();
//     // println!("{:?}", srds);
//     let iter = srds.into_iter();
//     for sub_srds in srds.iter().par_bridge(){
//         println!("----------------------");
//         println!("{:?}", sub_srds);
//         thread::sleep(time::Duration::from_secs(5));
//         // println!("{:?}", sub_srds.read::<i16>());
//         println!("----------------------");
//     }
// }

#[test]
fn builder_test() {
    let source = "data/LC08_L1TP_224078_20200518_20200518_01_RT.TIF";
    // let block_size: BlockSize = BlockSize {
    //     rows: 1024,
    //     cols: 1024,
    // };
    // let image_size = get_image_size(source);
    // let n_blocks = n_blocks(image_size, block_size);
    // let mut blocks_attributes: Vec<BlockAttributes> = Vec::new();
    // // for each block define:
    // let block_index = 0;
    // let read_window = ReadWindow {
    //     offset: (0, 0),
    //     size: (1024, 1024),
    // };
    // let geo_transform = GeoTransform {
    //     x_ul: 717345.0,
    //     x_res: 30.0,
    //     x_rot: 0.,
    //     y_ul: -2776995.0,
    //     y_rot: 0.,
    //     y_res: -30.0,
    // };
    // blocks_attributes.push(BlockAttributes {
    //     block_index,
    //     read_window,
    //     geo_transform,
    // });

    // let block_index = 1;
    // let read_window = ReadWindow {
    //     offset: (1024, 0),
    //     size: (1017, 1024),
    // };
    // let geo_transform = GeoTransform {
    //     x_ul: 748065.0,
    //     x_res: 30.0,
    //     x_rot: 0.,
    //     y_ul: -2776995.0,
    //     y_rot: 0.,
    //     y_res: -30.0,
    // };
    // blocks_attributes.push(BlockAttributes {
    //     block_index,
    //     read_window,
    //     geo_transform,
    // });

    // let block_index = 2;
    // let read_window = ReadWindow {
    //     offset: (0, 1024),
    //     size: (1024, 836),
    // };
    // let geo_transform = GeoTransform {
    //     x_ul: 717345.0,
    //     x_res: 30.0,
    //     x_rot: 0.,
    //     y_ul: -2807715.0,
    //     y_rot: 0.,
    //     y_res: -30.0,
    // };
    // blocks_attributes.push(BlockAttributes {
    //     block_index,
    //     read_window,
    //     geo_transform,
    // });

    // let block_index = 3;
    // let read_window = ReadWindow {
    //     offset: (1024, 1024),
    //     size: (1017, 836),
    // };
    // let geo_transform = GeoTransform {
    //     x_ul: 748065.0,
    //     x_res: 30.0,
    //     x_rot: 0.,
    //     y_ul: -2807715.0,
    //     y_rot: 0.,
    //     y_res: -30.0,
    // };
    // blocks_attributes.push(BlockAttributes {
    //     block_index,
    //     read_window,
    //     geo_transform,
    // });
    // default
    // let geo_transform = get_geotransform(source);
    // let rds = SingleRasterDataset {
    //     source: String::from(source),
    //     bands: vec![1, 2, 3],
    //     num_threads: 4,
    //     block_size,
    //     n_blocks,
    //     geo_transform,
    //     blocks_attributes, // read_index: 0,
    // };
    let rds: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source).build();

    println!("From SingleRasterBuilder test: ");
    println!("{}", rds);
}
