#[cfg(test)]
mod test {
    use super::singlerasterdataset::*;
    use super::*;
    use env_logger::Env;
    use gdal::{raster::RasterBand, Dataset};
    use libc::{c_int, c_void};
    use ndarray::{s, Array};
    use std::collections::BTreeMap;
    use std::path::Path;
    use std::process::Command;

    #[cfg(feature = "use_opencv")]
    use opencv::{
        core::{Mat_AUTO_STEP, Scalar},
        imgproc,
        prelude::*,
    };

    fn test_ndarray_to_opencv() {
        let cv_type = i32::typ();
        let img_size = 6;
        let vals: Vec<i32> = (0..img_size * img_size).map(|x| x).collect();
        let vals_pts = vals.as_ptr() as *mut c_int as *mut c_void;
        let data = Array::from_shape_vec((img_size as usize, img_size as usize), vals).unwrap();
        let rows = data.shape()[0] as i32;
        let cols = data.shape()[1] as i32;

        let mat_from_array = common::arrayview2_to_mat(data.view());

        let step = Mat_AUTO_STEP;
        let mat_from_vec =
            unsafe { Mat::new_rows_cols_with_data(rows, cols, cv_type, vals_pts, step).unwrap() };
        println!("\n Back and forth conversion with functions: ");
        let vec_from_mat_from_array: Vec<Vec<i32>> = mat_from_array.to_vec_2d().unwrap();
        let vec_from_mat_from_vec: Vec<Vec<i32>> = mat_from_vec.to_vec_2d().unwrap();
        assert_eq!(vec_from_mat_from_array, vec_from_mat_from_vec);
    }

    #[test]
    fn test_default_single_raster_dataset() {
        use crate::common::GeoTransform;

        let source = "data/cemsre_t55jfm_20200614_sub_abam5.tif";
        let rds: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source).build();
        assert_eq!(vec![1, 2, 3, 4, 5, 6], rds.bands);
        assert_eq!(4, rds.n_blocks);
        assert_eq!(
            GeoTransform {
                x_res: 20.,
                y_res: -20.,
                x_ul: 622720.0,
                y_ul: 7191600.0,
                x_rot: 0.0,
                y_rot: 0.0
            },
            rds.geo_transform
        );
    }
    #[test]
    fn test_overlap() {
        let source = "data/cemsre_t55jfm_20200614_sub_abam5.tif";
        let rds_over_0: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
            .overlap_size(0)
            .build();

        let rds_over_1: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
            .overlap_size(1)
            .build();
        // The idea is that reading with no overlap should give the same result
        // as reading with overlap + trimming the excess
        for id in (0..rds_over_0.n_blocks).into_iter() {
            let block_attributes_overlap = rds_over_1.blocks_attributes[id];
            let array_with_overlap = rds_over_1.read_block::<i32>(block_attributes_overlap);
            // println!("{:?}",array_with_overlap);
            let array_with_overlap_trimmed = common::trimm_array3(&array_with_overlap, 1);
            // should be the same than with no overlaps!
            let block_attributes_no_overlap = rds_over_0.blocks_attributes[id];
            let array_with_no_overlap = rds_over_0.read_block::<i32>(block_attributes_no_overlap);
            assert_eq!(array_with_no_overlap, array_with_overlap_trimmed);
        }
    }

    #[test]
    fn test_extract_blockwise() {
        let source = "data/cemsre_t55jfm_20200614_sub_abam5.tif";
        let vector_path = "data/cemsre_t55jfm_20200614_sub_abam5.gpkg";
        let overlap_size = 0;
        // two band,
        let rds: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
            .overlap_size(overlap_size)
            .bands(vec![1, 3])
            .build();
        let extracted_2b = rds.extract_blockwise(vector_path, "id", Some("value"), None);
        println!("extracted bands 1 and 3{:?}", extracted_2b);

        // Using value
        let rds: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
            .overlap_size(overlap_size)
            .build();
        let extracted_1 = rds.extract_blockwise(vector_path, "id", Some("value"), None);
        println!("extracted all bands{:?}", extracted_1);

        let overlap_size = 1;
        let rds: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
            .overlap_size(overlap_size)
            .build();
        let extracted_2 = rds.extract_blockwise(vector_path, "id", Some("value"), None);
        assert_eq!(extracted_1, extracted_2);

        // Using mode
        let overlap_size = 0;
        let rds: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
            .overlap_size(overlap_size)
            .build();
        println!("{:?}", rds);
        let extracted_mode = rds.extract_blockwise(vector_path, "id", Some("mode"), None);
        assert_eq!(extracted_1, extracted_mode);
        println!("Extracted with mode {:?}", extracted_1);
    }

    #[test]
    fn test_write() {
        let source = "data/cemsre_t55jfm_20200614_sub_abam5.tif";
        let overlap_size = 1;
        let rds: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
            .overlap_size(overlap_size)
            .build();

        for id in (0..rds.n_blocks).into_iter() {
            let block_attributes = rds.blocks_attributes[id];
            let original_array = rds.read_block::<i32>(block_attributes);

            let data = rds.read_block::<i32>(block_attributes);
            rds.write_block3(id, data, &format!("/tmp/test_{}.tif", id));

            let srdsr_new = SingleRasterDatasetBuilder::from_file(&format!("/tmp/test_{}.tif", id))
                .overlap_size(1)
                .build();
            let new_array = srdsr_new.read::<i32>();
            assert_eq!(
                common::trimm_array3(&original_array, overlap_size),
                new_array
            )
        }
    }

    #[test]
    fn test_write_warped() {
        let source = "data/cemsre_t55jfm_20200614_sub_abam5.tif";
        let rds: SingleRasterDataset = SingleRasterDatasetBuilder::from_file(source)
            .epsg(3577)
            .bands(vec![1, 2, 3])
            .overlap_size(0)
            .build();

        (0..rds.n_blocks).into_iter().for_each(|id| {
            let block_attributes = rds.blocks_attributes[id];
            let data = rds.read_block::<i32>(block_attributes);
            rds.write_block3(id, data, &format!("/tmp/test_warped_{}.tif", id));
        });

        // create a vrt with the parts and compare to gdal_warp output
        let mut build_vrt = Command::new("gdalbuildvrt");
        build_vrt
            .arg("/tmp/test_warped.vrt")
            .arg("/tmp/test_warped_0.tif")
            .arg("/tmp/test_warped_1.tif")
            .arg("/tmp/test_warped_2.tif")
            .arg("/tmp/test_warped_3.tif");

        build_vrt.output().expect("failed to execute build_vrt");

        let vrt_dataset = Dataset::open(Path::new("/tmp/test_warped.vrt")).unwrap();
        let vrt_rasterband: RasterBand = vrt_dataset.rasterband(1).unwrap();
        let vrt_array = vrt_rasterband
            .read_as::<u8>((0, 0), (256, 256), (256, 256), None)
            .unwrap()
            .data;

        let mut warp = Command::new("gdalwarp");
        warp.arg("-t_srs")
            .arg("EPSG:3577")
            .arg(source)
            .arg("/tmp/source_3577.tif");
        warp.output().expect("failed to execute gdal_warp");
        let warp_dataset = Dataset::open(Path::new("/tmp/source_3577.tif")).unwrap();
        let warp_rasterband: RasterBand = warp_dataset.rasterband(1).unwrap();
        let warp_array = warp_rasterband
            .read_as::<u8>((0, 0), (256, 256), (256, 256), None)
            .unwrap()
            .data;
        assert_eq!(vrt_array, warp_array);
    }
}
