use crate::data::{EOByte, Serializeable, StreamReader};

#[derive(Debug, Default)]
pub struct InitFileMap {
    pub data: Vec<EOByte>,
}

impl InitFileMap {
    pub fn new() -> Self {
        Self::default()
    }
}

impl Serializeable for InitFileMap {
    fn deserialize(&mut self, reader: &StreamReader) {
        self.data = reader.get_vec(reader.remaining());
    }
    fn serialize(&self) -> Vec<EOByte> {
        self.data.clone()
    }
}

#[cfg(test)]
mod tests {
    use super::{EOByte, InitFileMap, Serializeable, StreamReader};

    #[test]
    fn deserialize() {
        let map_bytes: Vec<EOByte> = vec![
            0x45, 0x4D, 0x46, 0x81, 0x69, 0xDD, 0x1F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
            0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x59,
            0x2C, 0x68, 0x4B, 0x01, 0x01, 0x01, 0x01, 0x01, 0xFE, 0x03, 0x03, 0x01, 0xFE, 0x02,
            0x02, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x04, 0x02, 0x04, 0x02, 0x97,
            0xFE, 0x01, 0x96, 0xFE, 0x03, 0x96, 0xFE, 0x03, 0x04, 0x02, 0x96, 0xFE, 0x01, 0x96,
            0xFE, 0x03, 0x96, 0xFE, 0x01, 0x04, 0x01, 0x96, 0xFE, 0x02, 0x96, 0xFE, 0x03, 0x96,
            0xFE, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
        ];

        let mut init_file_map = InitFileMap::new();
        let reader = StreamReader::new(&map_bytes);
        init_file_map.deserialize(&reader);
        assert_eq!(init_file_map.data, map_bytes);
    }

    #[test]
    fn serialize() {
        let map_bytes = vec![
            0x45, 0x4D, 0x46, 0x81, 0x69, 0xDD, 0x1F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
            0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x59,
            0x2C, 0x68, 0x4B, 0x01, 0x01, 0x01, 0x01, 0x01, 0xFE, 0x03, 0x03, 0x01, 0xFE, 0x02,
            0x02, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x04, 0x02, 0x04, 0x02, 0x97,
            0xFE, 0x01, 0x96, 0xFE, 0x03, 0x96, 0xFE, 0x03, 0x04, 0x02, 0x96, 0xFE, 0x01, 0x96,
            0xFE, 0x03, 0x96, 0xFE, 0x01, 0x04, 0x01, 0x96, 0xFE, 0x02, 0x96, 0xFE, 0x03, 0x96,
            0xFE, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
        ];

        let mut init_file_map = InitFileMap::new();
        init_file_map.data = map_bytes.clone();
        assert_eq!(init_file_map.serialize(), map_bytes);
    }
}
