use crate::data::{EOByte, EOChar, Serializeable, StreamBuilder, StreamReader};

#[derive(Debug, Default)]
pub struct InitFileItem {
    pub id: EOChar,
    pub data: Vec<EOByte>,
}

impl InitFileItem {
    pub fn new() -> Self {
        Self::default()
    }
}

impl Serializeable for InitFileItem {
    fn deserialize(&mut self, reader: &StreamReader) {
        self.id = reader.get_char();
        self.data = reader.get_vec(reader.remaining());
    }
    fn serialize(&self) -> Vec<EOByte> {
        let mut builder = StreamBuilder::with_capacity(1 + self.data.len());
        builder.add_char(self.id);
        builder.append(&mut self.data.clone());
        builder.get()
    }
}

#[cfg(test)]
mod tests {
    use super::{EOByte, InitFileItem, Serializeable, StreamReader};

    #[test]
    fn deserialize() {
        let item_bytes: Vec<EOByte> = vec![
            0x02, 0x45, 0x49, 0x46, 0x02, 0xFE, 0xFE, 0xFE, 0x03, 0xFE, 0x01, 0x05, 0x47, 0x6F,
            0x6C, 0x64, 0x01, 0xFE, 0x03, 0x01, 0x01, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01,
            0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
            0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0xFE, 0xFE, 0x01, 0x01, 0x01, 0xFE, 0x01,
            0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01,
            0x01, 0x01, 0x01, 0x01, 0x04, 0x65, 0x6F, 0x66, 0x01, 0xFE, 0x01, 0x01, 0x01, 0x01,
            0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01,
            0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0xFE,
            0xFE, 0x01, 0x01, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01,
            0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0x01, 0x01, 0x01, 0x01,
        ];

        let mut init_file_item = InitFileItem::new();
        let reader = StreamReader::new(&item_bytes);
        init_file_item.deserialize(&reader);
        assert_eq!(init_file_item.id, 1);
        assert_eq!(init_file_item.data, item_bytes[1..]);
    }

    #[test]
    fn serialize() {
        let item_bytes: Vec<EOByte> = vec![
            0x02, 0x45, 0x49, 0x46, 0x02, 0xFE, 0xFE, 0xFE, 0x03, 0xFE, 0x01, 0x05, 0x47, 0x6F,
            0x6C, 0x64, 0x01, 0xFE, 0x03, 0x01, 0x01, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01,
            0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
            0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0xFE, 0xFE, 0x01, 0x01, 0x01, 0xFE, 0x01,
            0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01,
            0x01, 0x01, 0x01, 0x01, 0x04, 0x65, 0x6F, 0x66, 0x01, 0xFE, 0x01, 0x01, 0x01, 0x01,
            0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01,
            0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0xFE,
            0xFE, 0x01, 0x01, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01,
            0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0x01, 0x01, 0x01, 0x01,
        ];
        let mut init_file_item = InitFileItem::new();
        init_file_item.id = 1;
        init_file_item.data = item_bytes[1..].to_vec();

        assert_eq!(init_file_item.serialize(), item_bytes);
    }
}
