use alloc::boxed::Box;
use alloc::collections::BTreeMap;
use alloc::format;
use alloc::string::String;
use alloc::vec::Vec;
use core::convert::TryInto;

use crate::buffer::ReadBuffer;
use crate::error::EtError;
use crate::parsers::FromSlice;
use crate::record::Value;

/// Readers for formats generated by Agilent instruments
pub mod agilent;
/// Reader for FASTA bioinformatics format
pub mod fasta;
/// Reader for FASTQ bioinformatics format
pub mod fastq;
/// Reader for flow data
pub mod flow;
/// Reader for Inficon Hapsite MS formats
pub mod inficon;
/// Reader for FASTA/FASTQ formats that parse into "kmers"
pub mod kmers;
/// Reader for PNG image format
#[cfg(feature = "std")]
pub mod png;
/// Reader for BAM/SAM bioinformatics formats
pub mod sam;
/// Readers for Thermo mass spectral isotopic formats
pub mod thermo_iso;
/// Readers for tab-seperated text format
pub mod tsv;
/// Reader for generic XML
//pub mod xml;

/// Turn `rb` into a Reader of type `parser_type`
pub fn get_reader<'r, B>(parser_type: &str, data: B) -> Result<Box<dyn RecordReader + 'r>, EtError>
where
    B: TryInto<ReadBuffer<'r>>,
    EtError: From<<B as TryInto<ReadBuffer<'r>>>::Error>,
{
    Ok(match parser_type {
        "bam" => Box::new(sam::BamReader::new(data, ())?),
        "chemstation_fid" => Box::new(agilent::ChemstationFidReader::new(data, ())?),
        "chemstation_ms" => Box::new(agilent::ChemstationMsReader::new(data, ())?),
        "chemstation_mwd" => Box::new(agilent::ChemstationMwdReader::new(data, ())?),
        "chemstation_uv" => Box::new(agilent::ChemstationUvReader::new(data, ())?),
        "fasta" => Box::new(fasta::FastaReader::new(data, ())?),
        "fastq" => Box::new(fastq::FastqReader::new(data, ())?),
        "fcs" => Box::new(flow::FcsReader::new(data, ())?),
        "inficon" => Box::new(inficon::InficonReader::new(data, (Vec::new(), 0))?),
        #[cfg(feature = "std")]
        "png" => Box::new(png::PngReader::new(data, ())?),
        "sam" => Box::new(sam::SamReader::new(data, ())?),
        "thermo_cf" => Box::new(thermo_iso::ThermoCfReader::new(data, ())?),
        "thermo_dxf" => Box::new(thermo_iso::ThermoDxfReader::new(data, ())?),
        "tsv" => Box::new(tsv::TsvReader::new(data, (b'\t', b'"'))?),
        _ => return Err(format!("No parser available for the filetype {}", parser_type).into()),
    })
}

/// Set up a state and a `ReadBuffer` for parsing.
///
/// EXPERIMENTAL: To be used to support setup for multi-threading parsing.
#[doc(hidden)]
pub fn init_state<'r, S, B, P>(data: B, params: Option<P>) -> Result<(ReadBuffer<'r>, S), EtError>
where
    B: TryInto<ReadBuffer<'r>>,
    EtError: From<<B as TryInto<ReadBuffer<'r>>>::Error>,
    S: for<'a> FromSlice<'a, State = P>,
    P: Default,
{
    let mut buffer = data.try_into()?;
    if let Some(state) = buffer.next::<S>(params.unwrap_or_default())? {
        Ok((buffer, state))
    } else {
        Err(format!(
            "Could not initialize state {}",
            ::core::any::type_name::<S>()
        )
        .into())
    }
}

/// The trait that maps over "generic" `RecordReader`s
///
/// Structs that implement this trait should also implement a `new` method that
/// takes a `ReadBuffer` and a "state" for creation and a `next` method that
/// returns a "specialized" struct that can be turned into the "generic" struct
/// via the `next_record` method.
pub trait RecordReader: ::core::fmt::Debug {
    /// Returns the next record from the file.
    ///
    /// Roughly equivalent to Rust's `Iterator.next`, but obeys slightly
    /// looser lifetime requirements to allow zero-copy parsing.
    fn next_record(&mut self) -> Result<Option<Vec<Value>>, EtError>;

    /// The header titles that correspond to every item in the record
    fn headers(&self) -> Vec<String>;

    /// Extra metadata about the file or data in the file
    fn metadata(&self) -> BTreeMap<String, Value>;
}

/// Generates a `...Reader` struct for the associated state-based file parsers
/// along with the matching `RecordReader` for that struct.
#[macro_export]
macro_rules! impl_reader {
    ($(#[$attr:meta])* $reader: ident, $record:ty, $state:ty, $new_params:ty) => {
        $(#[$attr])*
        /// [this reader was autogenerated via macro]
        #[derive(Debug)]
        pub struct $reader<'r> {
            rb: $crate::buffer::ReadBuffer<'r>,
            state: $state,
            // record: $record,  // FIXME
        }

        impl<'r> $reader<'r> {
            /// Create a new instance of the reader
            ///
            /// # Errors
            /// If data could not be turned into a `ReadBuffer` successfully or if the initial state
            /// could not be extracted, returns an `EtError`.
            pub fn new<B>(data: B, params: $new_params) -> Result<Self, EtError> where
                B: ::core::convert::TryInto<$crate::buffer::ReadBuffer<'r>>,
                EtError: From<<B as ::core::convert::TryInto<$crate::buffer::ReadBuffer<'r>>>::Error>,
            {
                let mut rb = data.try_into()?;
                // let record = <$record>::default(); // FIXME
                match rb.next(params)? {
                    Some(state) => Ok($reader { rb, state }),
                    None => Err(::alloc::format!("Could not initialize state {}", ::core::any::type_name::<$state>()) .into())
                }
            }

            /// Return the specialized version of this record.
            ///
            /// To get the "generic" version, please use the `next_record`
            /// method from the `RecordReader` trait.
            ///
            /// # Errors
            /// If a value could not be extracted, return an `EtError`.
            #[allow(clippy::should_implement_trait)]
            pub fn next(&mut self) -> Result<Option<$record>, EtError> {
                self.rb.next::<$record>(&mut self.state)
            }

            // TODO: get the lifetimes working for this
            // /// Return the specialized version of this record.
            // ///
            // /// To get the "generic" version, please use the `next_record`
            // /// method from the `RecordReader` trait.
            // #[allow(clippy::should_implement_trait)]
            // pub fn next_into(&mut self, record: &'r mut $record) -> Result<bool, EtError> {
            //     self.rb.record_pos += 1;
            //     Ok(if record.from_buffer(&mut self.rb, &mut self.state)? {
            //         true
            //     } else {
            //         false
            //     })
            // }
        }

        impl<'r> $crate::readers::RecordReader for $reader<'r> {
            /// The next record, expressed as a `Vec` of `Value`s.
            fn next_record(
                &mut self,
            ) -> Result<Option<::alloc::vec::Vec<$crate::record::Value>>, EtError> {
                if let Some(record) = self.rb.next::<$record>(&mut self.state)? {
                    Ok(Some(record.into()))
                } else {
                    Ok(None)
                }
            }

            /// The headers for this Reader.
            fn headers(&self) -> ::alloc::vec::Vec<::alloc::string::String> {
                use $crate::record::RecordHeader;
                <$record>::header()
            }

            /// The metadata for this Reader.
            fn metadata(&self) -> ::alloc::collections::BTreeMap<::alloc::string::String, $crate::record::Value> {
                self.state.metadata()
            }
        }
    };
}
