use std::collections::HashMap;

use wgpu::{
    RenderPass
};

use winit::event::WindowEvent;

use crate::{
    Camera,
    Vec2,
    ComponentType,
    EnschinContext,
    Renderer,
    SceneController,
    SceneContext,
    Input,
    RenderData
};


pub struct Scene{
    controller: Box<dyn SceneController>,
    component_types: HashMap<u32, ComponentType>,
    camera: Camera,
    input: Input,
}

impl Scene {
    pub fn new(controller: Box<dyn SceneController>) -> Self {
        let scene = Self {
            component_types: Default::default(),
            camera: Camera::new(),
            input: Input::new(),
            controller
        };
        return scene;
    }

    pub fn start(&mut self, enschin: &mut EnschinContext) {
        let mut context = SceneContext::new(&mut self.camera, &mut self.input);
        self.controller.setup(&mut context, enschin);

        for ctype in context.new_components {
            if !self.component_types.contains_key(&ctype.id()) {
                self.component_types.insert(ctype.id(), ComponentType::new(enschin, ctype));
            } else {
                self.component_types.get_mut(&ctype.id()).unwrap().add_component(ctype);
            }
        }
        self.camera.start(enschin.window_ratio);
        self.camera.update(enschin.render_data);
    } 

    pub fn update(&mut self, enschin: &mut EnschinContext) {
        self.camera.update(enschin.render_data);

        let mut context = SceneContext::new(&mut self.camera, &mut self.input);
        self.controller.update(&mut context, enschin);

        for (_, component_type) in &mut self.component_types {
            for component in &mut component_type.components {
                component.update(&mut context, enschin);
            }
        }

        for ctype in context.new_components {
            if !self.component_types.contains_key(&ctype.id()) {
                self.component_types.insert(ctype.id(), ComponentType::new(enschin, ctype));
            } else {
                self.component_types.get_mut(&ctype.id()).unwrap().add_component(ctype);
            }
        }
    }

    pub fn input(&mut self, window_size: Vec2<u32>, event: &WindowEvent) {
        self.input.update(event, self.camera.pos, window_size, self.camera.get_fov());
    }

    pub fn render<'a>(&'a mut self, render_pass: &mut RenderPass<'a>, enschin: &'a EnschinContext)  {
        render_pass.bind_camera_buffer(&enschin.render_data);        
        for (_, component_type) in &mut self.component_types {
            component_type.render(self.camera.get_fov(), enschin, render_pass);
        }
    }

    pub fn resize(&mut self, ratio: f32, render_data: &RenderData) {
        self.camera.resize(ratio, render_data);
    }
}
