use std::ops::*;
use std::mem;
use crate::{
    Vec2,
    Vec3,
    Vec4
};


#[repr(C)]
#[derive(Copy, Clone)]
pub struct Matrix4x4{
    pub x: Vec4<f32>,
    pub y: Vec4<f32>,
    pub z: Vec4<f32>,
    pub w: Vec4<f32>,
}

impl Matrix4x4 {
    pub const fn new(x: Vec4<f32>, y: Vec4<f32>, z: Vec4<f32>, w: Vec4<f32>) -> Matrix4x4 {
        Self {
            x,
            y,
            z,
            w
        }
    }

    pub const fn empty() -> Matrix4x4 {
        Self {
            x: Vec4{x: 0.0, y: 0.0, z: 0.0, w: 0.0},
            y: Vec4{x: 0.0, y: 0.0, z: 0.0, w: 0.0},
            z: Vec4{x: 0.0, y: 0.0, z: 0.0, w: 0.0},
            w: Vec4{x: 0.0, y: 0.0, z: 0.0, w: 0.0},
        }
    }

    pub const fn from_translate(pos: Vec2<f32>) -> Matrix4x4 {
        Self {
            x: Vec4{x: 1.0, y: 0.0, z: 0.0, w: 0.0},
            y: Vec4{x: 0.0, y: 1.0, z: 0.0, w: 0.0},
            z: Vec4{x: 0.0, y: 0.0, z: 0.5, w: 0.0},
            w: Vec4{x: pos.x, y: pos.y, z: 0.5, w: 1.0},
        }
    }

    pub fn from_rotation(angle: f32) -> Matrix4x4 {
        let s = angle.sin();
        let c = angle.cos();

        Self::new(
            Vec4::new(c, s, 0.0, 0.0),
            Vec4::new(-s, c, 0.0, 0.0),
            Vec4::new(0.0, 0.0, 1.0, 0.0),
            Vec4::new(0.0, 0.0, 0.0, 1.0),
        )
    }
    
    pub fn from_frustum(left: f32, right: f32, bottom: f32, top: f32, near: f32, far: f32) -> Matrix4x4 {
        let r_width  = 1.0 / (right - left);
        let r_height = 1.0 / (top - bottom);
        let r_depth  = 1.0 / (near - far);
        let x = 2.0 * (near * r_width);
        let y = 2.0 * (near * r_height);
        let a = (right + left) * r_width;
        let b = (top + bottom) * r_height;
        let c = (far + near) * r_depth;
        let d = far * near * r_depth;

        Self::new(
            Vec4::new(x, 0.0, 0.0, 0.0),
            Vec4::new(0.0, y, 0.0, 0.0),
            Vec4::new(a, b, c, -1.0),
            Vec4::new(0.0, 0.0, d, 0.0),
        )
    }


    pub fn from_look(eye: Vec3<f32>, center: Vec3<f32>, up: Vec3<f32>) -> Matrix4x4 {
        let mut f = center - eye;
        let rlf: f32 = 1.0 / Self::length(f);
        f*=rlf;

        let mut s = Vec3 {
            x: f.y * up.z - f.z * up.y,
            y: f.z * up.x - f.x * up.z,
            z: f.x * up.y - f.y * up.x,
        };
        
        
        let rls: f32 = 1.0 / Self::length(s);
        s *= rls;
        let u = Vec3 {
            x: s.y * f.z - s.z * f.y,
            y: s.z * f.x - s.x * f.z,
            z: s.x * f.y - s.y * f.x,
        };

        let mut m = Self::new(
            Vec4::new(s.x, u.x, -f.x, 0.0),
            Vec4::new(s.y, u.y, -f.y, 0.0),
            Vec4::new(s.z, u.z, -f.z, 0.0),
            Vec4::new(0.0, 0.0, 0.0, 1.0),
        );
        
        for i in 0..4 {
            m[12 + i] += m[i] * -eye.x + m[4 + i] * -eye.y + m[8 + i] * -eye.z;
        }

        return m;
    }

    pub fn rotate(&mut self, angle: f32) {
        let rotation = Matrix4x4::from_rotation(angle);
        *self = rotation * *self;
    }

    pub fn translate(&mut self, pos: Vec2<f32>) {
        for i in 0..4 {
            self[12 + i] += self[i] * pos.x + self[4 + i] * pos.y + self[8 + i] * 0.0;
        }
    }

    pub fn scale(&mut self, scaling: Vec2<f32>) {
        for i in 0..4 {
            self[     i] *= scaling.x;
            self[ 4 + i] *= scaling.y;
        }
    }

    pub fn length(v: Vec3<f32>) -> f32 {
        let l: f32 = v.x * v.x + v.y * v.y + v.z * v.z;
        return l.sqrt();
    }

    pub fn print(&self) {
        println!("Matrix: ");
        for i in 0..4 {
            println!("{}\t{}\t{}\t{}", self[i*4], self[i*4+1], self[i*4+2], self[i*4+3]);
        }
    }
}


impl AsRef <[f32; 16]> for Matrix4x4 {
    #[inline]
    fn as_ref(&self) -> &[f32; 16] {
        unsafe {
            mem::transmute(self)
        }
    }
}

impl AsMut <[f32; 16]> for Matrix4x4 {
    #[inline]
    fn as_mut(&mut self) -> &mut [f32; 16] {
        unsafe { 
            mem::transmute(self)
        }
    }
}

impl Index <usize> for Matrix4x4 {
    type Output = f32;

    #[inline]
    fn index<'a>(&'a self, i: usize) -> &'a f32 {
        let v: &[f32; 16] = self.as_ref();
        &v[i]
    }
}

impl IndexMut<usize> for Matrix4x4 {
    #[inline]
    fn index_mut<'a>(&'a mut self, i: usize) -> &'a mut f32{
        let v: &mut [f32; 16] = self.as_mut();
        &mut v[i]
    }
}


impl Mul for Matrix4x4 {
    type Output = Matrix4x4;
    fn mul(self, m: Matrix4x4) -> Matrix4x4 {
        {
            let a = m.x;
            let b = m.y;
            let c = m.z;
            let d = m.w;

            #[cfg_attr(rustfmt, rustfmt_skip)]
            Matrix4x4::new(
                a*self[0] + b*self[1] + c*self[2] + d*self[3],
                a*self[4] + b*self[5] + c*self[6] + d*self[7],
                a*self[8] + b*self[9] + c*self[10] + d*self[11],
                a*self[12] + b*self[13] + c*self[14] + d*self[15],
            )
        }
    }
}
