use crate::{
    Vec3,
    Vec2,
    Matrix4x4,
    RenderData
};


pub struct Camera {
    view: Matrix4x4,
    proj: Matrix4x4,
    pub pos: Vec2<f32>,
    fov: f32,
    ratio: f32,
    fov_min: f32,
    fov_max: f32
}


impl Camera {
    pub fn new() -> Self {
        Camera {
            view: Matrix4x4::empty(),
            proj: Matrix4x4::empty(),
            pos: Vec2::new(0.0, 0.0),
            fov: 5.0,
            ratio: 1.0,
            fov_min: -100000.0,
            fov_max: 100000.0
        }
    }

    pub fn start(&mut self, ratio: f32) {
        self.ratio = ratio;
        self.reset_camera_projection();
        self.reset_camera_view();
    }

    pub fn update(&mut self, render_data: &RenderData) {
        self.pos.x *= -1.0;
        self.view.translate(self.pos);
        render_data.update_camera_buffer((self.view * self.proj).as_ref());
        self.view.translate(-self.pos);
        self.pos.x *= -1.0;
    }

    pub fn resize(&mut self, ratio: f32, render_data: &RenderData) {
        self.ratio = ratio;
        self.reset_camera_projection();
        render_data.update_camera_buffer((self.view * self.proj).as_ref());
    }


    // TODO faid functions

    // TODO Camera bounds

    pub fn set_max_fov(&mut self, new_max_fov: f32) {
        self.fov_max = new_max_fov;
    }

    pub fn set_min_fov(&mut self, new_min_fov: f32) {
        self.fov_min = new_min_fov;
    }

    pub fn get_fov(&self) -> Vec2<f32> {
        Vec2::new(self.fov * self.ratio, self.fov)
    }

    pub fn set_fov(&mut self, new_fov: f32) {
        if new_fov < self.fov_min {
            self.fov = self.fov_min;
        } else if new_fov > self.fov_max {
            self.fov = self.fov_max;
        } else {
            self.fov = new_fov;
        }
        self.reset_camera_projection();
    }

    pub fn increase_fov(&mut self, increment_fov: f32) {
        if self.fov + increment_fov < self.fov_min {
            self.fov = self.fov_min;
        } else if self.fov + increment_fov > self.fov_max {
            self.fov = self.fov_max;
        } else {
            self.fov += increment_fov;
        }
        self.reset_camera_projection();
    }

    fn reset_camera_projection(&mut self) {
        self.proj = Matrix4x4::from_frustum(self.ratio * self.fov, -self.ratio * self.fov, -1.0 * self.fov, 1.0 * self.fov, 3.0, 7.0);
    }

    fn reset_camera_view(&mut self) {
        self.view = Matrix4x4::from_look(Vec3::new(0.0, 0.0,-3.0), Vec3::new(0.0, 0.0, 0.0), Vec3::new(0.0, 1.0, 0.0));
    }
}
